//    Copyright 2022 IDA Markup, ida-markup.com
// 
//    Licensed under the Apache License, Version 2.0 (the "License");
//    you may not use this file except in compliance with the License.
//    You may obtain a copy of the License at
// 
//        http://www.apache.org/licenses/LICENSE-2.0
// 
//    Unless required by applicable law or agreed to in writing, software
//    distributed under the License is distributed on an "AS IS" BASIS,
//    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//    See the License for the specific language governing permissions and
//    limitations under the License.

// Namespace

#include <cstddef>
#include <string>
#include <optional>
#include <any>
#include <vector>
#include <iostream>
#include <fstream>
#include <functional>
#include <stdexcept>
#include <sstream>
#include <regex>
#include <filesystem>
#include <algorithm>

#include "IDA.h"

// Options
IDAOptions::IDAOptions()
{
	#ifdef IDA_DEBUG_MEMORY
	dmem->c(this);
	#endif
}
IDAOptions::IDAOptions(IDAOptions* options)
{
	if (options != nullptr) copyFrom(options);
	
	#ifdef IDA_DEBUG_MEMORY
	dmem->c(this);
	#endif
}

void IDAOptions::copyFrom(IDAOptions* options)
{
	// Copy RX
	this->rxXmlReparseStrings = options->rxXmlReparseStrings;
	
	// Copy TX
	this->txUnicode = options->txUnicode;
}

#ifdef IDA_DEBUG_MEMORY
IDA_Debug_Memory<IDAOptions>* IDAOptions::dmem = new IDA_Debug_Memory<IDAOptions>("Opts");
IDAOptions::~IDAOptions() {dmem->d(this);}
#endif

IDAOptions* IDAOptions::defaultOptions = new IDAOptions();

std::optional<std::string> IDAObj::idaName() {return std::nullopt;}
IDA* IDAObj::idaEncode(IDA* el)
{
	throw IDAOpException("IDA encode not available");
}
IDAObj* IDAObj::idaDecode(IDA* el)
{
	throw IDAOpException("IDA decode not available");
}

// Port begin
IDA_Iterator::IDA_Iterator()
{
	this->funcHas = []{return false;};
	#ifdef IDA_DEBUG_MEMORY
	dmem->c(this);
	#endif
}
#ifdef IDA_DEBUG_MEMORY
IDA_Debug_Memory<IDA_Iterator>* IDA_Iterator::dmem = new IDA_Debug_Memory<IDA_Iterator>("i-IDA");
IDA_Iterator::~IDA_Iterator() {dmem->d(this);}
#endif
bool IDA_Iterator::hasNext() {return (this->funcHas)();}
IDA* IDA_Iterator::getNext() {return (this->funcGet)();}
IDA* IDA_Iterator::next() {return this->getNext();}

// Port iterators
IDA_Iterator* IDA_Port::empty_i()
{
	return new IDA_Iterator();
}
IDA_Iterator* IDA_Port::list_i(std::vector<IDA*>* l)
{
	//if (l == nullptr) throw IDAOpException("Internal list iteration error");
	IDA_Iterator* i = new IDA_Iterator();
	i->n = 0;
	long n_max = l->size();
	i->funcHas = [i, n_max] {return i->n < n_max;};
	i->funcGet = [i, l] {return (*l)[i->n++];};
	return i;
}
IDA_Iterator* IDA_Port::repeat_i(IDA** obj)
{
	IDA_Iterator* seq = new IDA_Iterator();
	seq->funcHas = [] {return true;};
	seq->funcGet = [obj] {return *obj;};
	return seq;
}

// Port arrays
bool IDA_Port::list_contains(std::vector<IDA*>* l, IDA* item)
{
	return std::find(l->begin(), l->end(), item) != l->end();
}
bool IDA_Port::list_rem(std::vector<IDA*>* l, IDA* item)
{
	auto x = std::find(l->begin(), l->end(), item);
	if (x != l->end()) {l->erase(x); return true;} else return false;
}

// Port streams
bool IDA_Port::si_allMatch(IDA_Iterator* i, std::function<bool(IDA*)> predicate)
{
	bool b = true;
	while (i->hasNext()) if (!predicate(i->next())) {b = false; break;}
	delete i;
	return b;
}
bool IDA_Port::si_anyMatch(IDA_Iterator* i, std::function<bool(IDA*)> predicate)
{
	bool b = false;
	while (i->hasNext()) if (predicate(i->next())) {b = true; break;}
	delete i;
	return b;
}
bool IDA_Port::si_noneMatch(IDA_Iterator* i, std::function<bool(IDA*)> predicate)
{
	return !si_anyMatch(i, predicate);
}

// Port string operation
std::string IDA_Port::replace_all(std::string s, std::string act, std::string dmd)
{
	if (act.length() < 1) return s;
	
	std::string::size_type pos = s.find(act), ptr = 0;
	if (pos == std::string::npos) return s;
	
	std::string::size_type actLen = act.length();
	std::ostringstream buf;
	
	while (pos != std::string::npos)
	{
		buf << s.substr(ptr, pos);
		buf << dmd;
		pos = s.find(act, ptr = pos+actLen);
	}
	
	buf << s.substr(ptr);
	return buf.str();
}

std::string IDA_Port::substring(std::string s, long fromChar, long toChar)
{
	s = s.substr(str_units(s, 0, fromChar));
	return s.substr(0, str_units(s, 0, toChar-fromChar));
}
std::string IDA_Port::substring(std::string s, long fromChar)
{
	return s.substr(str_units(s, 0, fromChar));
}

std::string IDA_Port::trim(std::string s)
{
	std::string::size_type i1 = 0;
	std::string::size_type i2 = s.length();
	while (i1 < i2 && ((unsigned char)s[i1]) <= ' ') i1++;
	while (i1 < i2 && ((unsigned char)s[i2-1]) <= ' ') i2--;
	s.erase(i2);
	s.erase(0, i1);
	return s;
}
std::string IDA_Port::rtrim(std::string s)
{
	std::string::size_type i2 = s.length();
	while (0 < i2 && ((unsigned char)s[i2-1]) <= ' ') i2--;
	s.erase(i2);
	return s;
}
std::string IDA_Port::ltrim(std::string s)
{
	std::string::size_type i1 = 0;
	std::string::size_type i2 = s.length();
	while (i1 < i2 && ((unsigned char)s[i1]) <= ' ') i1++;
	s.erase(0, i1);
	return s;
}

// Port string tests
bool IDA_Port::ends_with(std::string s, std::string suffix)
{
	return s.rfind(suffix) == (s.length() - suffix.length());
}
bool IDA_Port::starts_with(std::string s, std::string prefix)
{
	return s.find(prefix) == 0;
}
bool IDA_Port::str_contains(std::string s, std::string mid)
{
	return s.find(mid) != std::string::npos;
}

// Port string info
long IDA_Port::str_index(std::optional<std::string> s, std::optional<std::string> mid)
{
	long i = s->find(*mid);
	return (i == std::string::npos) ? -1 : str_chars(s, 0, i);
}
long IDA_Port::str_index_from(std::optional<std::string> s, std::optional<std::string> mid, long fromChar)
{
	std::string::size_type fromUnit = str_units(s, 0, fromChar);
	std::string::size_type i = s->find(*mid, fromUnit);
	return (i == std::string::npos) ? -1 : (fromChar + str_chars(s, fromUnit, i));
}
bool IDA_Port::is_continuation(unsigned char c)
{
	return (c & 0xc0) == 0x80;
}
std::string::size_type IDA_Port::str_len(std::optional<std::string> s)
{
	return str_chars(s, 0, s->length());
}
std::string::size_type IDA_Port::str_chars(std::optional<std::string> s, long fromUnit, long toUnit)
{
	std::string::size_type n = toUnit - fromUnit;
	for (long i=toUnit-1; i>=fromUnit; i-=1) if (is_continuation((*s)[i])) n-=1;
	return n;
}
std::string::size_type IDA_Port::str_units(std::optional<std::string> s, long fromChar, long toChar)
{
	toChar -= fromChar;
	std::string::size_type top = s->length();
	
	// Skip
	std::string::size_type mark = 0;
	while (fromChar-- > 0) while ((++mark < top) && is_continuation((*s)[mark])) ;
	
	// Count
	std::string::size_type ptr = mark;
	while (toChar-- > 0) while ((++ptr < top) && is_continuation((*s)[ptr])) ;
	
	return ptr - mark;
}

// Port hex
std::string IDA_Port::char_2(unsigned char c)
{
	std::ostringstream txt;
	txt << std::hex << (int)c;
	std::string c_hex = txt.str();
	if (c_hex.length() < 2) {c_hex = "0"+c_hex;}
	return c_hex;
}
std::string IDA_Port::char_4(unsigned long c)
{
	std::ostringstream txt;
	txt << std::hex << c;
	std::string c_hex = txt.str();
	while (c_hex.length() < 4) {c_hex = "0"+c_hex;}
	return c_hex;
}
std::string IDA_Port::char_8(unsigned long c)
{
	std::ostringstream txt;
	txt << std::hex << c;
	std::string c_hex = txt.str();
	while (c_hex.length() < 8) {c_hex = "0"+c_hex;}
	return c_hex;
}

// Port end

// Format exception
IDAFormatException::IDAFormatException(IDA_RX* rx, std::string message)
{
	this->msg = "(line "+std::to_string(rx->ln)+") "+message;
	
	node = nullptr;
	
	// Extract problem index first
	this->pos = rx->lnBuf.length();
	
	// Read until end of line for context
	std::string ln = rx->lnBuf;
	try {ln += rx->get_to('\n');}
	catch (...) {}
	this->ln = ln;
	
}
IDAFormatException::IDAFormatException(IDA* nodeCause, std::string message)
{
	this->msg = (nodeCause->isAnon() ? "(anonymous element) " : "(node '"+nodeCause->getNameOr("null")+"') ") + message;
	
	this->ln = "";
	this->pos = -1;
	this->node = nodeCause;
}
const char* IDAFormatException::what()
{
	return this->msg.c_str();
}


// Manip begin
long IDA_Manip::rev(long n) {return -n;}

std::optional<std::string> IDA_Manip::m_str(IDA* el)
{
	if (el->isNull()) return std::nullopt;
	try {return el->asStr();}
	catch (...) {throw IDAFormatException(el, "String required");}
}

// Manip/iterate
IDA* IDA_Manip::manip_iterate(IDA* manip, IDA* node, IDA_Iterator* itemTemplate)
{
	IDA* last = nullptr;
	try { while (itemTemplate->hasNext())
	{
		IDA* templateNode = itemTemplate->next();
		
		if (last != nullptr) {node = new IDA(); last->linkNext(node);}
		
		node->copyName(templateNode);
		node->copyParams(templateNode);
		if (templateNode->isList())
		{
			node->putList();
			
			if (templateNode->numItems() > 0)
			{
				IDA* item = new IDA();
				IDA* item_last = nullptr;
				
				try {item_last = manip_iterate(manip, item, IDA::port->list_i(templateNode->asListOr()));}
				catch (...) {delete item; throw;}
				
				if (item_last != nullptr)
				{
					for (IDA* nextEl; item != nullptr; item = nextEl)
					{
						nextEl = item->getNextOr();
						node->addItem(item);
					}
				}
			}
		}
		else {node->copyContent(templateNode);}
		last = node;
		
		// Manip/iterate/main
		if (manip->isList())
		for (IDA* m : *manip->asList())
		{
			if (m->isAnon() || m->isNull()) {continue;} // Operate on nothing, or no operations
			if (!m->isList()) {throw IDAFormatException(m, "Manipulation operations must be a list");}
			
			std::string operateOn = m->getNameOr("");
			bool n = IDA::port->str_contains(operateOn, "n"); // n: name
			bool k = IDA::port->str_contains(operateOn, "k"); // k: param key
			bool v = IDA::port->str_contains(operateOn, "v"); // v: param value
			bool p = IDA::port->str_contains(operateOn, "p"); // p: parameter(s)
			bool c = IDA::port->str_contains(operateOn, "c"); // c: content
			
			// Manip/iterate/match
			// if-match: any of (n node-name, k param-key, v param-val, p(param-key param-val), c node-content)
			std::function<bool(IDA*)> paramMatch = [] (IDA* x) {return true;};
			if (m->hasParams()) for (IDA* mp : *m->getParamsOr())
			{
				std::string sel = mp->getNameOr("");
				
				if (sel.find_first_not_of("nkvc!") != std::string::npos) {throw IDAFormatException(m, "Invalid manipulation match (valid characters are [nkvc!])");}
				
				bool flagNot = IDA::port->str_contains(sel, "!");
				
				if (IDA::port->str_contains(sel, "n"))
				{
					if (((node->isAnon() && mp->isNull()) || node->isName(m_str(mp))) == flagNot) {goto manip;}
				}
				if (IDA::port->str_contains(sel, "c"))
				{
					if (mp->eqContent(node) == flagNot) {goto manip;}
				}
				
				if (IDA::port->str_contains(sel, "k"))
				{
					std::optional<std::string> tgt = m_str(mp);
					paramMatch = [paramMatch,mp,flagNot,tgt] (IDA* x) {return paramMatch(x) && (flagNot ^ ((x->isAnon() && mp->isNull()) || x->isName(tgt)));};
					break;
				}
				if (IDA::port->str_contains(sel, "v"))
				{
					paramMatch = [paramMatch,mp,flagNot] (IDA* x) {return paramMatch(x) && (flagNot ^ mp->eqContent(x));};
					break;
				}
				
			}
			
			// Manip/iterate/op
			for (IDA* op : *m->asList())
			{
				if (op->isAnon()) {throw IDAFormatException(m, "Missing operation code");}
				std::string opName = op->getNameOr("");
				
				if (false) {}
				
				// Manip/iterate/op/insert
				else if (opName == "i") // i: insert (nkvc): i (index) text-string
				{
					if (op->isNull()) {throw IDAFormatException(m, "Missing insert string");}
					if (op->isList()) {throw IDAFormatException(m, "Insert string cannot be a list");}
					
					IDA* param = op->getParamFirstOr();
					if (param != nullptr && !param->isNum()) {throw IDAFormatException(m, "Insert position must be a number");}
					long pos = (param != nullptr) ? param->asIntOr(0) : REVERSE;
					
					try
					{
						if (n) {node->setName(manip_ins(node->getNameOr(), op->asStr(), pos));}
						if (c) {node->putStr(manip_ins(node->asStr(), op->asStr(), pos));}
						if (k || v)
						{
							if (node->hasParams()) for (IDA* np : *node->getParams())
							{
								if (!paramMatch(np)) {continue;}
								if (k) {np->setName(manip_ins(np->getNameOr(), op->asStr(), pos));}
								if (v) {np->putStr(manip_ins(np->asStr(), op->asStr(), pos));}
							}
						}
					}
					catch (IDAOpException opex) {throw IDAFormatException(m, opex.getMessage());}
				}
				
				// Manip/iterate/op/delete
				else if (opName == "d") // d: delete (nkvc): d (index) length or d (index) bool
				{
					if (!(op->isNum() || op->isBool() || op->isNull())) {throw IDAFormatException(m, "Delete amount must be boolean or a number");}
					
					IDA* param = op->getParamFirstOr();
					if (param != nullptr && !param->isNum()) {throw IDAFormatException(m, "Delete position must be a number");}
					int pos = (param != nullptr) ? param->asIntOr(0) : REVERSE;
					int amt = op->isBool() ? (op->isTrue() ? FORWARD : REVERSE) : op->isNull() ? 1 : op->asIntOr(0);
					if (pos == REVERSE && amt > 0) {amt = rev(amt);}
					
					try
					{
						if (n) {node->setName(manip_del(node->getNameOr(), pos, amt));}
						if (c) {node->putStr(manip_del(node->asStr(), pos, amt));}
						if (k || v)
						{
							if (node->hasParams()) for (IDA* np : *node->getParams())
							{
								if (!paramMatch(np)) {continue;}
								if (k) {np->setName(manip_del(np->getNameOr(), pos, amt));}
								if (v) {np->putStr(manip_del(np->asStr(), pos, amt));}
							}
						}
					}
					catch (IDAOpException opex) {throw IDAFormatException(m, opex.getMessage());}
				}
				
				// Manip/iterate/op/add
				else if (opName == "a") // a: add (vpc): a (index) {nodes...}
				{
					// No need to check isList; addItem and insertItem already have exceptions for this
					IDA* param = op->getParamFirstOr();
					if (param != nullptr && !param->isNum()) {throw IDAFormatException(m, "Add position must be a number");}
					long pos = (param != nullptr) ? param->asIntOr(0) : REVERSE;
					
					try
					{
						if (p)
						{
							if (!node->hasParams()) {node->setParams();}
							manip_add([node](IDA* x, long y) {node->addParamAt(x, y);}, pos == REVERSE ? node->numParams() : pos, op);
						}
						if (c)
						{
							manip_add([node](IDA* x, long y) {node->addItemAt(x, y);}, pos == REVERSE ? node->numItems() : pos, op);
						}
						if (v)
						{
							if (node->hasParams()) for (IDA* np : *node->getParams())
							{
								if (!paramMatch(np)) {continue;}
								manip_add([np](IDA* x, long y) {np->addItemAt(x, y);}, pos == REVERSE ? np->numItems() : pos, op);
							}
						}
					}
					catch (IDAOpException opex) {throw IDAFormatException(m, opex.getMessage());}
				}
				
				// Manip/iterate/op/remove
				else if (opName == "r") // r: remove (vpc): r (index) length or r(index) bool or r {nodes...}
				{
					if (op->isNum() || op->isBool() || op->isNull()) // r (index) length or r(index) bool
					{
						IDA* param = op->getParamFirstOr();
						if (param != nullptr && !param->isNum()) {throw IDAFormatException(m, "Remove position must be a number");}
						long pos = (param != nullptr) ? param->asIntOr(0) : REVERSE;
						long amt = op->isBool() ? (op->isTrue() ? FORWARD : REVERSE) : (op->isNull() ? 1 : op->asIntOr(0));
						if (pos == REVERSE && amt > 0) {amt = rev(amt);}
						
						if (p)
						{
							manip_rem_pos(m, [node](long x){node->remParamAt(x);}, [node](){return node->numParams();}, pos, amt);
						}
						if (c)
						{
							manip_rem_pos(m, [node](long x){node->remItemAt(x);}, [node](){return node->numItems();}, pos, amt);
						}
						if (v)
						{
							if (node->hasParams()) for (IDA* np : *node->getParamsOr())
							{
								if (!paramMatch(np)) {continue;}
								manip_rem_pos(m, [np](long x){np->remItemAt(x);}, [np](){return np->numItems();}, pos, amt);
							}
						}
					}
					else if (op->isList()) // r {nodes...}
					{
						if (p) {manip_rem_item([node](IDA* x){node->remParam(x);}, op);}
						if (c) {manip_rem_item([node](IDA* x){node->remItem(x);}, op);}
						if (v)
						{
							if (node->hasParams()) for (IDA* np : *node->getParamsOr())
							{
								if (paramMatch(np)) manip_rem_item([np](IDA* x){np->remItem(x);}, op);
							}
						}
					}
					else {throw IDAFormatException(m, "Remove subject must be boolean, a number or a list");}
				}
				
				// Manip/iterate/op/undo
				else if (opName == "u") // u: unset/unlist (vpc): u text-key or u {text-key...}
				{
					if (op->isList()) for (IDA* un : *op->asList()) // u {text-key...}
					{
						std::optional<std::string> name = m_str(un);
						
						if (c) {node->unItem(name);}
						if (p) {node->unParam(name);}
						if (v && node->hasParams()) for (IDA* np : *node->getParamsOr())
						{
							if (paramMatch(np)) np->unItem(name);
						}
					}
					else // u text-key
					{
						std::optional<std::string> name = m_str(op);
						
						if (c) {node->unItem(name);}
						if (p) {node->unParam(name);}
						if (v && node->hasParams()) for (IDA* np : *node->getParamsOr())
						{
							if (paramMatch(np)) np->unItem(name);
						}
					}
				}
				
				// Manip/iterate/op/substitute
				else if (opName == "s") // s: substitute (nkvc): s (search-token replacement-string ...);
				{
					if (op->hasParams()) for (IDA* repl : *op->getParamsOr())
					{
						if (repl->isAnon() || repl->isNull()) {continue;}
						std::string act = repl->getNameOr(""), dmd = m_str(repl).value_or("");
						
						// nkvc
						if (n && node->hasName()) {node->setName(IDA::port->replace_all(node->getNameOr(""), act, dmd));}
						if (c && node->hasContent()) {node->putStr(IDA::port->replace_all(m_str(node).value_or(""), act, dmd));}
						if (k || v)
						{
							if (node->hasParams()) for (IDA* np : *node->getParamsOr())
							{
								if (!paramMatch(np)) {continue;}
								if (k && np->hasName()) {np->setName(IDA::port->replace_all(np->getNameOr(""), act, dmd));}
								if (v && np->hasContent()) {np->putStr(IDA::port->replace_all(m_str(np).value_or(""), act, dmd));}
							}
						}
					}
				}
				
				// Manip/iterate/op/flip
				else if (opName == "f") // f (vc): flip: f;
				{
					try
					{
						if (c) {manip_negate(node);}
						if (v && node->hasParams()) for (IDA* np : *node->getParams())
						{
							if (!paramMatch(np)) continue;
							manip_negate(np);
						}
					}
					catch (IDAOpException opex) {throw IDAFormatException(m, opex.getMessage());}
				}
				
				// Manip/iterate/op/write
				else if (opName == "w") // w (nkvpc): write: w replacement
				{
					// TODO: replace null-checks with asString returning nullopt?
					if (n) {if (op->isNull()) {node->setName(std::nullopt);} else {node->setName(m_str(op));}}
					if (c) {node->copyContent(op);}
					if (p)
					{
						if (op->isNull()) {node->clearParams();}
						else if (op->isList())
						{
							node->setParams();
							for (IDA* param : *op->asListOr()) {node->addParam(param->copy());}
						}
						else {throw IDAFormatException(m, "Parameter write must be list or null");}
					}
					if (k || v)
					{
						if (node->hasParams()) for (IDA* np : *node->getParamsOr())
						{
							if (!paramMatch(np)) {continue;}
							if (k) {if (op->isNull()) {np->setName(std::nullopt);} else {np->setName(m_str(op));}}
							if (v) {np->copyContent(op);}
						}
					} 
				}
				
				// Manip/iterate/op end
				else throw IDAFormatException(m, "Invalid operation code \""+opName+"\"");
			}
			
			manip: ;
		}
		
	// Manip/iterate end
	} }
	catch (...)
	{
		delete itemTemplate;
		throw;
	}
	
	delete itemTemplate;
	return last;
	
}

// Manip/negate
void IDA_Manip::manip_negate(IDA* node)
{
	if (node->isBool())
	{
		node->putBool(!node->asBool());
	}
	else if (node->isNum())
	{
		std::string num = node->getNumN();
		node->putNum(IDA::port->starts_with(num, "-") ? num.substr(1) : "-"+num, node->getNumD());
	}
	else if (node->hasContent()) {throw IDAOpException("Flip subject must be boolean or a number");}
}

// Manip/insert
std::string IDA_Manip::manip_ins(std::optional<std::string> orig_opt, std::string ins, long pos)
{
	if (!orig_opt.has_value()) {throw IDAOpException("Insert subject must not be null");}
	std::string orig = orig_opt.value();
	if (pos == REVERSE) {return orig + ins;}
	long l = IDA::port->str_len(orig);
	if (pos > 0)
	{
		if (pos > l) {throw IDAOpException("Cannot insert string beyond position "+std::to_string(l)+" (request was "+std::to_string(pos)+")");}
		return IDA::port->substring(orig, 0, pos) + ins + IDA::port->substring(orig, pos);
	}
	else if (pos < 0)
	{
		if (pos < -l) {throw IDAOpException("Cannot insert string before position -"+std::to_string(l)+" (request was "+std::to_string(pos)+")");}
		return IDA::port->substring(orig, 0, l+pos) + ins + IDA::port->substring(orig, l+pos);
	}
	else {return ins + orig;}
}

// Manip/delete
std::string IDA_Manip::manip_del(std::optional<std::string> orig_opt, long pos, long amt)
{
	if (!orig_opt.has_value()) {throw IDAOpException("Delete subject must not be null");}
	std::string orig = orig_opt.value();
	long l = IDA::port->str_len(orig);
	if (pos == REVERSE) {pos = l;}
	if (pos >= 0)
	{
		if (pos > l) {throw IDAOpException("Cannot delete from string beyond position "+std::to_string(l)+" (request was "+std::to_string(pos)+")");}
		if (amt == REVERSE) {return IDA::port->substring(orig, pos);}
		if (amt == FORWARD) {return IDA::port->substring(orig, 0, pos);}
		if (amt > 0)
		{
			if (pos + amt > l) {throw IDAOpException("Cannot delete more than "+std::to_string(l-pos)+" beyond position "+std::to_string(pos)+" (request was "+std::to_string(amt)+")");}
			return IDA::port->substring(orig, 0, pos) + IDA::port->substring(orig, pos+amt);
		}
		if (amt < 0)
		{
			if (pos + amt < 0) {throw IDAOpException("Cannot delete more than "+std::to_string(pos)+" before position "+std::to_string(pos)+" (request was "+std::to_string(-amt)+")");}
			return IDA::port->substring(orig, 0, pos+amt) + IDA::port->substring(orig, pos);
		}
		return orig;
	}
	else //if (pos < 0)
	{
		if (pos < -l) {throw IDAOpException("Cannot delete from string before position -"+std::to_string(l)+" (request was "+std::to_string(pos)+")");}
		if (amt == REVERSE) {return IDA::port->substring(orig, pos+l);}
		if (amt == FORWARD) {return IDA::port->substring(orig, 0, pos+l);}
		if (amt > 0)
		{
			if (pos+l + amt > l) {throw IDAOpException("Cannot delete more than "+std::to_string(-pos)+" beyond position "+std::to_string(pos)+" (request was "+std::to_string(amt)+")");}
			return IDA::port->substring(orig, 0, pos+l) + IDA::port->substring(orig, pos+l+amt);
		}
		if (amt < 0)
		{
			if (pos+l + amt < 0) {throw IDAOpException("Cannot delete more than "+std::to_string(l+pos)+" before position "+std::to_string(pos)+" (request was "+std::to_string(-amt)+")");}
			return IDA::port->substring(orig, 0, pos+l+amt) + IDA::port->substring(orig, pos+l);
		}
		return orig;
	}
	
}

// Manip/add
void IDA_Manip::manip_add(std::function<void(IDA*,long)> func, long pos, IDA* nodeFrom)
{
	if (nodeFrom->isList()) for (IDA* ins : *nodeFrom->asList())
	{
		IDA* cp = ins->copy();
		try {func(cp, pos);} catch (...) {delete cp; throw;}
		if (pos >= 0) {pos += 1;}
	}
}

// Manip/rem
void IDA_Manip::manip_rem_pos(IDA* opNode, std::function<void(long)> func, std::function<long()> size, long pos, long amt)
{
	long n = size();
	if (pos == REVERSE) {pos = n;}
	
	if (amt == FORWARD) {amt = n - pos;}
	else if (amt == REVERSE) {amt = -pos;}
	
	try
	{
		if (amt > 0)
		{
			for (long i=pos+amt-1; i>=pos; i-=1)
			{
				func(i);
			}
		}
		else if (amt < 0)
		{
			for (long i=pos-1; i>=pos+amt; i-=1)
			{
				func(i);
			}
		}
		
	}
	catch (IDAOpException opex) {throw IDAFormatException(opNode, opex.getMessage());}
}
void IDA_Manip::manip_rem_item(std::function<void(IDA*)> func, IDA* nodeFrom)
{
	if (nodeFrom->isList()) for (IDA* rem : *nodeFrom->asList())
	{
		func(rem);
	}
}

// Manip end

// RX begin
IDA_RX::IDA_RX(std::istream* inStr)
{
	this->inStr = inStr;
}

// RX basic input
long IDA_RX::read()
{
	if (!has_c1) more();
	return has_c1 ? (long) read1() : -1;
}

void IDA_RX::buf(char c)
{
	if (c == '\n') {ln+=1; lnBuf = "";} else {lnBuf += c;}
}
char IDA_RX::read1()
{
	char output = c1; c1 = c2; has_c1 = has_c2; has_c2 = false;
	more2();
	buf(output);
	return output;
}
void IDA_RX::skip2()
{
	if (has_c1) buf(c1);
	if (has_c2) buf(c2);
	has_c1 = has_c2 = false;
	if (more1()) return;
	more2();
}

char IDA_RX::shift()
{
	char output = c1;
	read();
	return output;
}
void IDA_RX::more()
{
	if (has_c2 || eof) return;
	if (!has_c1 && more1()) return;
	if (!has_c2 && more2()) return;
}
bool IDA_RX::more1()
{
	if (eof) return true;
	int b = inStr->get();
	if (b < 0 || b == 4) return eof = true;
	c1 = (char) b;
	has_c1 = true;
	return false;
}
bool IDA_RX::more2()
{
	if (eof) return true;
	int b = inStr->get();
	if (b < 0 || b == 4) return eof = true;
	c2 = (char) b;
	has_c2 = true;
	return false;
}

// RX stream advancement
bool IDA_RX::comment()
{
	if (!has_c1) return false;
	if (IDA::comment1(c1)) return true;
	return has_c2 && IDA::comment2(c1, c2);
}
void IDA_RX::adv()
{
	adv(false);
}
void IDA_RX::adv(bool checkLn)
{
	more();
	while (has_c1)
	{
		if (checkLn ? whiteTxt(c1) : white(c1)) {read1();}
		else if (c1 == '#') {read1(); advPast('\n');}
		else if (has_c2)
		{
			if (c1 == '/' && c2 == '/')
			{
				skip2(); advPast('\n');
			}
			else if (c1 == '/' && c2 == '*')
			{
				skip2(); advPast('*', '/');
			}
			else if (c1 == '<' && c2 == '?') // XML header compatibility
			{
				skip2(); advPast('?', '>');
			}
			else if (c1 == '<' && c2 == '!') // XML comment compatibility
			{
				skip2();
				do {advPast('-', '-');} while (has_c1 && !c1e('>')); read1();
			}
			else {return;}
		}
		else {return;}
	}
	
}
void IDA_RX::advPast(char c)
{
	more();
	while (has_c1 && c1 != c) {read1();}
	read1();
}
void IDA_RX::advPast(char cx, char cy)
{
	more();
	while (has_c2 && (c1 != cx || c2 != cy)) {read1();}
	skip2();
}

bool IDA_RX::abrupt(bool checkLn)
{
	adv(checkLn);
	return !has_c1;
}

// RX scan bytes
bool IDA_RX::c1e(unsigned char c) {return has_c1 && c1 == c;}
bool IDA_RX::c1n(unsigned char c) {return has_c1 && c1 != c;}
bool IDA_RX::c2e(unsigned char c) {return has_c2 && c2 == c;}
bool IDA_RX::c2n(unsigned char c) {return has_c2 && c2 != c;}
bool IDA_RX::whiteLn(unsigned char c) {return c == '\n';}
bool IDA_RX::whiteTxt(unsigned char c) {return c <= ' ' && c != '\n';}
bool IDA_RX::white(unsigned char c) {return c <= ' ';}

// RX scan numbers
bool IDA_RX::num0(char c) {return (c>='0' && c<='9') || c == '-' || c == '+' || c == '.';}
bool IDA_RX::num1(char c) {return num0(c) || c == 'e' || c == 'E';}
std::string IDA_RX::get_num()
{
	std::string output = "";
	
	if (has_c2 && c1=='0' && c2=='x')
	{
		output = "0x"; skip2();
		while (has_c1 && hex(c1)) {output += shift();}
	}
	else if (has_c2 && c1=='0' && c2=='b')
	{
		output = "0b"; skip2();
		while (has_c1 && bin(c1)) {output += shift();}
	}
	else
	{
		output += shift();
		while (has_c1 && num1(c1)) {output += shift();}
	}
	
	return output;
}
bool IDA_RX::hex(char c) {return (c>='0' && c<='9') || (c>='a' && c<='f') || (c>='A' && c<='F');}
bool IDA_RX::bin(char c) {return c=='0' || c=='1';}

// RX scan opts
bool IDA_RX::opt() {return c1e('~');}
std::string IDA_RX::get_opt()
{
	read1();
	std::string output = get_id();
	adv();
	return output;
}

// RX scan unicode
bool IDA_RX::is_u8_marker(unsigned char c) {return (c & 0xc0) == 0xc0;}
bool IDA_RX::is_u8_data(unsigned char c) {return (c & 0xc0) == 0x80;}

unsigned char IDA_RX::shift_u8_data()
{
	unsigned char data = shift();
	if (!is_u8_data(data)) throw IDAFormatException(this, "Invalid unicode data");
	return data & 0x3f;
}
std::string IDA_RX::get_u8()
{
	unsigned long codePoint = 0;
	unsigned char marker = shift();
	
	if ((marker & 0xe0) == 0xc0) // 2-byte
	{
		codePoint = (marker & 0x1f) << 6;
		codePoint |= shift_u8_data();
		if (codePoint < 0x0080) throw IDAFormatException(this, "Overlong unicode character");
	}
	else if ((marker & 0xf0) == 0xe0) // 3-byte
	{
		codePoint = (marker & 0x0f) << 12;
		codePoint |= shift_u8_data() << 6;
		codePoint |= shift_u8_data();
		if (codePoint < 0x0800) throw IDAFormatException(this, "Overlong unicode character");
	}
	else if ((marker & 0xf8) == 0xf0) // 4-byte
	{
		codePoint = (marker & 0x07) << 18;
		codePoint |= shift_u8_data() << 12;
		codePoint |= shift_u8_data() << 6;
		codePoint |= shift_u8_data();
		if (codePoint < 0x10000) throw IDAFormatException(this, "Overlong unicode character");
	}
	else throw IDAFormatException(this, "Invalid unicode marker");
	
	return to_u8(codePoint);
}
std::string IDA_RX::to_u8(unsigned long codePoint)
{
	if (codePoint < 0x80)
	{
		std::string output = "_";
		output[0] = (char)codePoint;
		return output;
	}
	else if (codePoint < 0x800)
	{
		std::string output = "__";
		output[0] = (char)(0xc0 | ((codePoint >> 6) & 0x1f));
		output[1] = (char)(0x80 | (codePoint & 0x3f));
		return output;
	}
	else if (codePoint < 0x10000)
	{
		if (codePoint >= 0xd800 && codePoint < 0xe000) throw IDAFormatException(this, "Invalid unicode surrogate");
		std::string output = "___";
		output[0] = (char)(0xe0 | ((codePoint >> 12) & 0x0f));
		output[1] = (char)(0x80 | ((codePoint >> 6) & 0x3f));
		output[2] = (char)(0x80 | (codePoint & 0x3f));
		return output;
	}
	else if (codePoint < 0x110000)
	{
		std::string output = "____";
		output[0] = (char)(0xf0 | ((codePoint >> 18) & 0x07));
		output[1] = (char)(0x80 | ((codePoint >> 12) & 0x3f));
		output[2] = (char)(0x80 | ((codePoint >> 6) & 0x3f));
		output[3] = (char)(0x80 | (codePoint & 0x3f));
		return output;
	}
	else throw IDAFormatException(this, "Invalid unicode range");
}

// RX scan escapes
bool IDA_RX::is_esc(char c) {return c == '\\';}
bool IDA_RX::esc() {return has_c1 && is_esc(c1);}

int IDA_RX::shift_hex(std::string ex_txt)
{
	char h = shift();
	if ('0' <= h && h <= '9') return h - '0';
	if ('A' <= h && h <= 'F') return 10 + h - 'A';
	if ('a' <= h && h <= 'f') return 10 + h - 'a';
	throw IDAFormatException(this, "Invalid "+ex_txt+" escape sequence");
}
int IDA_RX::get_esc_byte(std::string ex_txt)
{
	return (shift_hex(ex_txt) << 4) | shift_hex(ex_txt);
}
std::string IDA_RX::get_esc()
{
	read1();
	if (!has_c1) {throw IDAFormatException(this, "Incomplete escape sequence");}
	char esc = shift();
	switch (esc)
	{
		case '0': return std::string("\0",1);
		case 'n': return "\n";
		case 'r': return "\r";
		case 't': return "\t";
		case 'x':
		{
			if (!has_c2) {throw IDAFormatException(this, "Incomplete byte escape sequence");}
			return to_u8(get_esc_byte("byte"));
		}
		case 'u':
		{
			if (!has_c2) {throw IDAFormatException(this, "Incomplete unicode escape sequence");}
			unsigned long code = get_esc_byte("unicode") << 8;
			if (!has_c2) {throw IDAFormatException(this, "Incomplete unicode escape sequence");}
			return to_u8(code | get_esc_byte("unicode"));
		}
		case 'U':
		{
			if (!has_c2) {throw IDAFormatException(this, "Incomplete unicode extended escape sequence");}
			unsigned long code = get_esc_byte("unicode extended") << 24;
			if (!has_c2) {throw IDAFormatException(this, "Incomplete unicode extended escape sequence");}
			code |= get_esc_byte("unicode extended") << 16;
			if (!has_c2) {throw IDAFormatException(this, "Incomplete unicode extended escape sequence");}
			code |= get_esc_byte("unicode extended") << 8;
			if (!has_c2) {throw IDAFormatException(this, "Incomplete unicode extended escape sequence");}
			return to_u8(code | get_esc_byte("unicode extended"));
		}
		
		default: return std::string(&esc,1);
	}
}

// RX scan IDs
bool IDA_RX::id0() {return IDA::id0(c1);}
bool IDA_RX::id1() {return IDA::id1(c1);}
std::string IDA_RX::get_tok()
{
	if (is_str_d(c1)) {return get_str_d(c1);}
	else
	{
		std::string output = "";
		while (!comment())
		{
			if (esc()) {output += get_esc();}
			else if (has_c1 && !white(c1)) {output += shift();}
			else break;
		}
		return output;
	}
}

std::string IDA_RX::get_id(bool untilLn)
{
	std::string output = "";
	this->id_symbolic = true;
	std::string::size_type holdMin = -1;
	std::string::size_type holdMax = -1;
	
	// Must not end with - or + , that is reserved for boolean false or true
	while (this->has_c1 && !comment())
	{
		std::string::size_type ptr = this->id_symbolic ? output.length() : holdMin;
		bool hold = false;
		
		if (hold = is_esc(c1)) {output += get_esc();}
		else if (hold = is_str_d(c1)) {output += get_str_d(c1);}
		else if (hold = is_var(c1)) {output += get_var();}
		else if (hold = is_u8_marker(c1)) {output += get_u8();}
		else if ((IDA::id1(c1) || (untilLn && (c1 != ';') && !whiteLn(c1))) && !(c1 == '-' && (!has_c2 || !IDA::id1(c2)))) {output += shift();}
		else break;
		
		if (hold)
		{
			holdMin = ptr;
			holdMax = output.length();
			this->id_symbolic = false;
		}
		
	}
	
	if (untilLn)
	{
		if (this->id_symbolic) {return IDA::port->trim(output);}
		else return IDA::port->ltrim(output.substr(0,holdMin)) + output.substr(holdMin, holdMax-holdMin) + IDA::port->rtrim(output.substr(holdMax));
	}
	else return output;
}
bool IDA_RX::is_id_start(char c) {return is_str_d(c) || is_var(c) || is_esc(c) || is_u8_marker(c) || IDA::id0(c);}
bool IDA_RX::c1_id_start() {return has_c1 && is_id_start(c1);}

bool IDA_RX::is_var(char c) {return c == '$';}
std::string IDA_RX::get_var()
{
	read();
	std::string output = "$";
	
	char endChar = 0;
	if (this->c1e('{')) {endChar = '}';}
	// TODO: Other common uses of $(...)
	
	if (endChar != 0) {output += read1(); output += get_to(endChar) + endChar;}
	return output;
}

std::string IDA_RX::xml_att_key()
{
	return is_id_start(this->c1) ? this->get_id() : this->get_pr([this] (char c) {return c != '=' && c != '/' && c != '>' && !this->white(c);});
}

// RX scan strings
bool IDA_RX::c1_str() {return has_c1 && is_str(c1);}
bool IDA_RX::c1_tok() {return has_c1 && !white(c1);}
bool IDA_RX::is_str_d(char c) {return c == '"' || c == '`' || c == '\'';}
bool IDA_RX::is_str(char c) {return is_str_d(c) || IDA::id1(c);}

std::string IDA_RX::get_str()
{
	return is_str_d(c1) ? get_str_d(c1) : get_id();
}
std::string IDA_RX::get_str_d(char delim)
{
	std::string output = "";
	if (c1e(delim))
	{
		read1();
		while (c1n(delim))
		{
			if (esc()) {output += get_esc();}
			else if (is_u8_marker(c1)) {output += get_u8();}
			else {output += read1();}
		}
		read();
	}
	return output;
}

// RX scan logic
std::string IDA_RX::get_pr(std::function<bool(char)> p)
{
	std::ostringstream output;
	while (has_c1 && p(c1)) output << shift();
	return output.str();
}
std::string IDA_RX::get_prior_to(char delim)
{
	std::ostringstream output;
	while (has_c1 && c1 != delim) output << shift();
	return output.str();
	
	//std::string output = get_pr([delim](char c){return c != delim;});
	//return output;
}
std::string IDA_RX::get_to(char delim)
{
	std::string output = get_prior_to(delim);
	read();
	return output;
}

// RX class iteration
// See END for static CLASS_* init
IDA_Iterator* IDA_RX::classOf(std::vector<IDA*>* itemClassList)
{
	return itemClassList != nullptr ? IDA::port->list_i(itemClassList) : CLASS_NONE;
}

// RX entry
IDA* IDA_RX::inObj(IDA* output, std::istream* inStr)
{
	CLASS_NONE->singleton = true;
	CLASS_ARRAY->singleton = true;
	
	IDA_RX rx(inStr);
	IDA globalScope;
	bool entry = false;
	IDA* entryNode = (new IDA())->rxScope(&globalScope)->setIncludes(output->rxIncludePwd());
	entryNode->rxInputList(output->inputListDepth, output->inputListLength);
	entryNode->copyOptions(output);
	
	rx.adv();
	while (rx.opt())
	{
		std::string opt = rx.get_opt();
		if (opt == "") {entry = true;}
	}
	
	IDA* node;
	try {node = fill(entryNode, &rx, '\0', '\0', nullptr, CLASS_NONE, 0);}
	catch (...) {delete entryNode; throw;}
	
	while (rx.has_c1 && node != nullptr)
	{
		IDA* nextNode = (new IDA())->rxScope(&globalScope)->setIncludes(output->rxIncludePwd());
		nextNode->rxInputList(output->inputListDepth, output->inputListLength);
		nextNode->copyOptions(output);
		while (rx.opt())
		{
			std::string opt = rx.get_opt();
			if (opt == "")
			{
				if (!entry) {entry = true; entryNode = nextNode;}
			}
		}
		
		IDA* last;
		try {last = fill(nextNode, &rx, '\0', '\0', node, CLASS_NONE, 0);}
		catch (...) {delete entryNode; throw;}
		
		node->linkNext(nextNode);
		node = last;
	}
	
	output->reset();
	rx.transfer_nodes(entryNode, output);
	output->walk([](IDA* x){x->rxScope(nullptr);}, true, true);
	
	if (rx.has_refTarget) try {output->walk([&rx](IDA* x){rx.put_ref(x);}, true, true);} catch (...) {delete entryNode; throw;}
	
	entryNode->link_break();
	delete entryNode;
	
	return output;
}
void IDA_RX::transfer_nodes(IDA* src, IDA* dst)
{
	// Name
	dst->copyName(src);
	
	// Parameters
	if (src->hasParams())
	{
		dst->setParams();
		for (IDA* p : *src->getParamsOr()) dst->addParam(p);
		// Empty the original list first so that no objects get deleted by clearParams
		src->getParams()->clear();
		src->clearParams();
	}
	else dst->clearParams();
	
	// Content
	if (src->isList())
	{
		dst->putList();
		for (IDA* i : *src->asListOr()) dst->addItem(i);
		src->asListOr()->clear();
		src->putNull();
	}
	else dst->copyContent(src);
	
	// Other
	dst->copyOptions(src);
	if (src->hasRefName()) {refs[*src->getRefName()] = dst; dst->setRefName(*src->getRefName());}
	
	dst->link(src->getPrevOr(), src->getNextOr());
}
void IDA_RX::put_ref(IDA* el)
{
	std::optional<std::string> t = el->getRefTarget();
	if (t.has_value())
	{
		if (refs.count(t.value()) < 1) throw IDAFormatException(el, "Identifier not defined");
		el->putRef(refs[t.value()]);
	}
}

// RX fill
IDA* IDA_RX::fill(IDA* node, IDA_RX* rx, char paren_l, char paren_r, IDA* prev, IDA_Iterator* listClass, long depth)
{
	
	// RX fill/state
	bool jsonVal = false;
	bool forceContent = false; // Used for : and = in an array context
	bool needName = true;
	std::optional<std::string> prelimName = std::nullopt;
	bool nameSet = false;
	bool needContent = true;
	bool checkDelim = true; // For ; or , between nodes
	IDA* itemClass = nullptr;
	IDA* output = nullptr;
	bool lcn = listClass->hasNext();
	
	// RX read
	try { while (rx->has_c1 && needContent)
	{
		
		// RX read/string implicit
		// This should be first
		// NOTE: By testing this prior to num(), names/strings beginning "e" will not be a number
		if (rx->c1_id_start())
		{
			output = node;
			
			if (!needName)
			{
				if (listClass == CLASS_ARRAY && !forceContent)
				{
					nameSet = true;
					needName = false;
					node->putStr(prelimName);
				}
				else
				{
					// String content
					// Potentially also true/false/null if JSON-style following a :
					std::string s = rx->get_id(depth == 0);
					
					if (jsonVal && rx->id_symbolic)
					{
						if ("true" == s) {node->putTrue();}
						else if ("false" == s) {node->putFalse();}
						else if ("null" == s) {node->putNull();}
						else {node->putStr(s);}
					}
					else {node->putStr(s);}
				}
				
				needContent = false;
			}
			else
			{
				prelimName = rx->get_id();
				if (rx->abrupt(paren_r == '\n')) {node->putStr(prelimName); prelimName = std::nullopt; needContent = false;}
				else {needName = false;}
			}
		}
		
		// RX read/delim explicit
		else if (rx->c1e(';') || rx->c1e(','))
		{
			if (!((rx->c1 == ',') && (depth == 0)))
			{
				// Consume delimiter unless using CSV compatibility
				rx->read1();
				checkDelim = false;
			}
			
			node->putNull();
			needContent = false;
			output = node;
		}
		
		// RX read/delim implicit
		else if (rx->c1e('}') || rx->c1e(']') || rx->c1e(')'))
		{
			if (paren_l != ':')
			{
				if (rx->c1e('}')) {if (paren_l != '{') throw IDAFormatException(rx, "List mismatch");}
				else if (rx->c1e(']')) {if (paren_l != '[') throw IDAFormatException(rx, "Array mismatch");}
				else /*if (rx->c1e(')'))*/ {if (paren_l != '(') throw IDAFormatException(rx, "Parameters mismatch");}
			}
			
			if (!nameSet)
			{
				if (!needName && lcn)
				{
					lcn = false;
					IDA* lcnode = listClass->next();
					if (lcnode == nullptr) std::cerr << "Failure: Class provided a null node (RX read/delim implicit, !needName)\n";
					if (lcnode != nullptr) node->setName(lcnode->getNameOr());
					node->putStr(prelimName);
				}
				else {node->setName(prelimName);}
				nameSet = true;
			}
			else
			{
				node->putNull();
			}
			
			needContent = false;
			checkDelim = false;
		}
		
		// RX read/as-above
		else if (rx->c1e('^'))
		{
			if (prev == nullptr) {throw IDAFormatException(rx, "No previous element");}
			
			if (needName)
			{
				prelimName = prev->getNameOr();
				needName = false;
			}
			else
			{
				node->copyContent(prev);
				needContent = false;
			}
			
			if (rx->c2e('^')) {node->copyParams(prev); rx->skip2();} else {rx->read1();}
			output = node;
		}
		
		// RX read/identifier definition
		else if (rx->c1e('*'))
		{
			rx->read1();
			std::string key = (rx->c1_str()) ? rx->get_str() : "";
			rx->adv();
			
			if (rx->refs.count(key) > 0) throw IDAFormatException(rx, "Identifier already defined");
			rx->refs[key] = node;
			node->setRefName(key);
			
			output = node;
		}
		
		// RX read/identifier reference
		else if (rx->c1e('&'))
		{
			rx->read1();
			std::string key = (rx->c1_str()) ? rx->get_str() : "";
			rx->adv();
			
			node->setRefTarget(key);
			rx->has_refTarget = true;
			
			output = node;
			needContent = false;
		}
		
		// RX read/XML
		else if (rx->c1e('<')) // XML compatibility
		{
			output = node;
			
			if (!needName)
			{
				if (listClass == CLASS_ARRAY && !forceContent)
				{
					nameSet = true;
					needName = false;
					node->putStr(prelimName);
					needContent = false;
				}
				else if (paren_r == '<') {needContent = false;}
				else throw IDAFormatException(rx, "Cannot nest XML element header");
			}
			else
			{
				rx->read1();
				rx->adv();
				if (rx->has_c1 && rx->id1())
				{
					node->setName(rx->get_id());
					needName = false;
					nameSet = true;
				}
				else {throw IDAFormatException(rx, "Invalid XML element name");}
				rx->adv();
				
				while (rx->has_c1 && needContent) // XML inners
				{
					if (rx->c1e('>')) // XML child elements
					{
						rx->read1();
						std::vector<IDA*> l;
						IDA* last = nullptr;
						while (rx->has_c1 && needContent)
						{
							if (rx->c1e('<'))
							{
								if (rx->c2e('/')) // End node
								{
									rx->skip2(); rx->adv();
									std::string end = IDA::port->trim(rx->get_to('>'));
									if (!node->isName(end)) {throw IDAFormatException(rx, "Incorrect XML element footer");}
									needContent = false;
								}
								else if (rx->c2e('!')) // XML comment
								{
									rx->skip2();
									do {rx->advPast('-', '-');} while (rx->c1n('>')); rx->read1();
								}
								else // Child element
								{
									IDA* el = node->rxNew();
									fill(el, rx, '<', '>', last, CLASS_NONE, depth+1);
									last = el;
									l.push_back(el);
								}
							}
							else
							{
								if (node->opts->rxXmlReparseStrings)
								{
									rx->adv();
									
									IDA* el = node->rxNew();
									
									last = fill(el, rx, '>', '<', last, CLASS_NONE, 0);
									while (el != nullptr && (node->inputListLength < 0 || node->inputListLength > l.size()))
									{
										l.push_back(el);
										el = el->getNextOr();
									}
								}
								else
								{
									std::string content = rx->get_prior_to('<');
									
									if (IDA::port->trim(content).length() > 0)
									{
										IDA* el = (new IDA())->putStr(content);
										l.push_back(el);
									}
									
								}
							}
						}
						
						node->putList(l);
					}
					else if (rx->c1e('/') && rx->c2e('>')) // Empty XML node
					{
						rx->skip2();
						rx->adv();
						node->putNull();
						needContent = false;
					}
					else // XML attrib
					{
						std::string attKey = rx->xml_att_key();
						while (is_id_start(rx->c1) && (rx->c1n('/') || rx->c2n('>'))) attKey += rx->xml_att_key();
						rx->adv();
						
						if (rx->c1e('='))
						{
							rx->read1();
							std::string attValue = rx->get_str();
							rx->adv();
							node->addParam((new IDA(attKey))->putStr(attValue));
						}
						else
						{
							node->addParam((new IDA(attKey))->putNull());
						}
						
					}
				}
			}
		}
		
		// RX read/params
		else if (rx->c1e('('))
		{
			if (node->hasParams())
			{
				needContent = false;
			}
			else
			{
				output = node;
				
				// Parameters also cause content delim
				forceContent = true;
				
				rx->read1();
				rx->adv();
				if (rx->c1e('^'))
				{
					if (prev == nullptr) {throw IDAFormatException(rx, "No previous element");}
					node->copyParams(prev);
					rx->read1();
					rx->adv();
				}
				else
				{
					node->setParams();
				}
				
				IDA* last = nullptr;
				IDA_Iterator* cItems = classOf(itemClass != nullptr && itemClass->hasParams() ? itemClass->getParamsOr() : nullptr);
				while (rx->c1n(')'))
				{
					IDA* param = node->rxNew();
					try {last = fill(param, rx, '(', ')', last, cItems, depth+1);}
					catch (...) {delete param; throw;}
					
					if (last == nullptr) {delete param;}
					else while (param != nullptr)
					{
						// Snapshot the next before addParam breaks it
						IDA* param_next = param->getNextOr();
						node->addParam(param);
						param = param_next;
					}
				}
				while (cItems->hasNext())
				{
					IDA* param = cItems->next();
					if (param != nullptr) {param = param->copy()->rxScope(node); node->addParam(param);}
					else {std::cerr << "Fault: Class provided a null node (RX read/params, cItems)\n";}
				}
				rx->read();
				if (!cItems->singleton) delete cItems;
			}
		}
		
		// RX read/content-delim
		else if ((rx->c1e(':') && !needName) || rx->c1e('='))
		{
			if (rx->read1() == ':') {jsonVal = true;}
			needName = false;
			nameSet = true;
			node->setName(prelimName);
			forceContent = true;
			output = node;
		}
		
		// RX read/null
		else if (rx->c1e('?'))
		{
			output = node;
			
			if (needName)
			{
				rx->read1();
				prelimName = std::nullopt;
				needName = false;
			}
			else
			{
				if (listClass == CLASS_ARRAY && !forceContent)
				{
					nameSet = true;
					needName = false;
					node->putStr(prelimName);
				}
				else
				{
					rx->read1();
					node->putNull();
				}
				needContent = false;
			}
		}
		
		// RX read/number
		else if (num0(rx->c1))
		{
			if (!needName && listClass == CLASS_ARRAY && !forceContent)
			{
				nameSet = true;
				needName = false;
				node->putStr(prelimName);
			}
			else
			{
				std::string s = "";
				std::string num = "";
				std::string unit = "";
				std::string den = "1";
				
				s = num = rx->get_num();
				
				// Check for denominator
				if (rx->c1e('/'))
				{
					rx->read1();
					
					if (!rx->has_c1 || !num0(rx->c1)) {throw IDAFormatException(rx, "Missing denominator");}
					den = rx->get_num();
					s += "/"+den;
				}
				
				// Check for unit
				if (rx->has_c1 && rx->id0())
				{
					unit = rx->get_id();
					s += unit;
				}
				
				// Bools with +, -
				if (s == "+") {node->putTrue();}
				else if (s == "-") {node->putFalse();}
				
				// Actual number
				else
				{
					rx->testNumber(num, "numerator");
					if (den != "1") {rx->testNumber(den, "denominator");}
					node->putNum(num, den, unit);
				}
			}
			needContent = false;
			output = node;
		}
		
		// RX read/list array
		else if (rx->c1e('['))
		{
			output = node;
			
			if (!needName && listClass == CLASS_ARRAY && !forceContent)
			{
				nameSet = true;
				needName = false;
				node->putStr(prelimName);
			}
			else
			{
				rx->read1();
				rx->adv();
				std::vector<IDA*> listObj = {};
				
				// TODO: implement discarding
				//if (node->inputListDepth != 0)
				//{
					IDA* last = nullptr;
					while (rx->c1n(']'))
					{
						IDA* el = node->rxNew();
						try {last = fill(el, rx, '[', ']', last, CLASS_ARRAY, depth+1);}
						catch (...) {delete el; throw;}
						
						if (last == nullptr) {delete el;}
						else while (el != nullptr && (node->inputListLength < 0 || node->inputListLength > listObj.size()))
						{
							listObj.push_back(el);
							el = el->getNextOr();
						}
					}
				//}
				//else discard(rx, '{', '}');
				
				rx->read();
				node->putList(listObj);
			}
			needContent = false;
		}
		
		// RX read/list object
		else if (rx->c1e('{'))
		{
			output = node;
			
			if (!needName && listClass == CLASS_ARRAY && !forceContent)
			{
				nameSet = true;
				needName = false;
				node->putStr(prelimName);
			}
			else
			{
				rx->read1();
				rx->adv();
				std::vector<IDA*> listObj = {};
				
				// TODO: implement discarding
				//if (node->inputListDepth != 0)
				//{
					IDA* last = nullptr;
					IDA_Iterator* cItems = classOf(itemClass != nullptr && itemClass->isList() ? itemClass->asListOr() : nullptr);
					//IDA_Iterator* cItems = itemClass != nullptr && itemClass->isList() ? classOf(itemClass->asListOr()) : classOf(nullptr);
					while (rx->c1n('}'))
					{
						IDA* el = node->rxNew();
						try {last = fill(el, rx, '{', '}', last, cItems, depth+1);}
						catch (...) {delete el; throw;}
						
						if (last == nullptr) {delete el;}
						else while (el != nullptr && (node->inputListLength < 0 || node->inputListLength > listObj.size()))
						{
							listObj.push_back(el);
							el = el->getNextOr();
						}
					}
					while (cItems->hasNext() && (node->inputListLength < 0 || node->inputListLength > listObj.size()))
					{
						IDA* el = cItems->next();
						if (el != nullptr) {el = el->copy()->rxScope(node);} else {std::cerr << "Failure: Class provided a null node (RX read/list object, cItems)\n";}
						listObj.push_back(el);
					}
					if (!cItems->singleton) delete cItems;
				//}
				//else discard(rx, '{', '}');
				
				rx->read();
				node->putList(listObj);
			}
			needContent = false;
		}
		
			// RX read/class definition
			else if (rx->c1e(':') && rx->c2e(':'))
			{
				rx->skip2();
				std::string key = (rx->c1_str()) ? rx->get_str() : "";
				rx->adv();
				
				IDA* el = node->rxNew();
				try
				{
					fill(el, rx, ':', '\0', nullptr, CLASS_NONE, depth+1);
				}
				catch (...) {delete el; throw;}
				
				if (!node->rxAddItemClass(key, el)) delete el;
			}
			
			// RX read/class instance
			else if (rx->c1e(':'))
			{
				if (paren_l == ':')
				{
					needContent = false;
				}
				else
				{
					rx->read1();
					std::string key = (rx->c1_str()) ? rx->get_str() : "";
					rx->adv();
					
					itemClass = node->rxGetItemClass(key);
					if (itemClass == nullptr) {throw IDAFormatException(rx, "Item class \""+key+"\" not found");}
					prelimName = itemClass->getNameOr();
					output = node;
				}
			}
			
			// RX read/template definition
			else if (rx->c1e('!'))
			{
				if (!needName) {throw IDAFormatException(rx, "Incomplete element prior to template definition");}
				rx->read1();
				std::string key = (rx->c1_str()) ? rx->get_str() : "";
				rx->adv();
				
				IDA* itemTemplate = node->rxNew();
				try
				{
					fill(itemTemplate, rx, '\0', '\0', nullptr, CLASS_NONE, depth+1);
					if (!itemTemplate->isList()) {throw IDAFormatException(rx, "Template definition must be a list");}
				}
				catch (...) {delete itemTemplate; throw;}
				
				if (!node->rxAddTemplate(key, itemTemplate)) delete itemTemplate;
			}
			
			// RX read/template instance
			else if (rx->c1e('@'))
			{
				// Hand-over to file include for @? and @@ 
				bool inc_opt = rx->c2e('?');
				if (inc_opt || rx->c2e('@'))
				{
					if (!needName) {throw IDAFormatException(rx, "Incomplete element prior to include");}
					rx->skip2();
					if (!rx->c1_tok()) {throw IDAFormatException(rx, "Missing include path");}
					
					std::string path = rx->get_tok();
					rx->adv();
					
					bool abort = false;
					try
					{
						// Platform-specific check (C++)
						std::optional<std::string> pwd_opt = node->rxIncludePwd();
						if (abort = (!pwd_opt.has_value())) {throw IDAOpException("Includes are disabled");}
						std::string pwd = pwd_opt.value();
						
						bool chdir = false;
						if (IDA::port->starts_with(path, "+")) {chdir = true; path = path.substr(1);}
						
						std::filesystem::path inc(path);
						if (!inc.is_absolute()) {inc = std::filesystem::path(pwd)/inc;}
						
						std::filesystem::file_status inc_st = std::filesystem::status(inc);
						if (abort = !std::filesystem::exists(inc_st)) throw IDAOpException("Include path does not exist");
						if (abort = !std::filesystem::is_regular_file(inc_st)) throw IDAOpException("Include is not a file");
						
						IDA template_adj;
						template_adj.rxScope(node);
						template_adj.setIncludes(chdir ? inc.parent_path().string() : pwd);
						
						std::ifstream input;
						input.open(inc, std::ios_base::in);
						if (!input.is_open()) throw IDAOpException("Include file has no read access");
						template_adj.inStream(&input);
						input.close();
						
						IDA* last = manip(rx, node, template_adj.iterateNext());
						
						nameSet = true;
						checkDelim = false;
						needContent = false;
						output = last;
						
					}
					catch (IDAOpException opex)
					{
						// Hold off expending the RX line on an exception when the include was optional
						if (!inc_opt) throw IDAFormatException(rx, opex.getMessage());
						
						// Discard the manipulation node
						if (abort)
						{
							IDA discardNode;
							fill(&discardNode, rx, '\0', '\0', nullptr, CLASS_NONE, depth+1);
						}
						
					}
					
				}
				else
				{
					if (!needName) {throw IDAFormatException(rx, "Incomplete element prior to template instance");}
					rx->read1();
					std::string key = (rx->c1_str()) ? rx->get_str() : "";
					rx->adv();
					
					IDA* itemTemplate = node->rxGetTemplate(key);
					if (itemTemplate == nullptr) {throw IDAFormatException(rx, "Template \""+key+"\" not found");}
					
					IDA* last = manip(rx, node, IDA::port->list_i(itemTemplate->asListOr()));
					
					if (last != nullptr)
					{
						nameSet = true;
						checkDelim = false;
						needContent = false;
						output = last;
					}
					
					
				}
				
			}
		// RX read/invalid
		// This should be last
		else
		{
			std::ostringstream txt;
			txt << "Invalid character 0x" << ((int)rx->c1 < 16 ? "0" : "") << std::hex << (int)rx->c1 << " '" << rx->c1 << "'";
			throw IDAFormatException(rx, txt.str());
		}
		
		// RX read delimiter
		if (paren_r != '>')
		{
			bool csvLn = paren_r == '\n';
			rx->adv(csvLn);
			csvLn &= rx->c1e('\n');
			if (checkDelim && (rx->c1e(';') || rx->c1e(',') || csvLn) )
			{
				if (needContent && lcn)
				{
					lcn = false;
					IDA* lcnode = listClass->next();
					if (lcnode == nullptr) std::cerr << "Failure: Class provided a null node (RX read delimiter, needContent)\n";
					if (lcnode != nullptr) node->setName(lcnode->getNameOr());
					node->putStr(prelimName);
				}
				else if (needName && lcn)
				{
					lcn = false;
					IDA* lcnode = listClass->next();
					if (lcnode == nullptr) std::cerr << "Failure: Class provided a null node (RX read delimiter, needName)\n";
					if (lcnode != nullptr) node->setName(lcnode->getNameOr());
				}
				else if ((rx->c1 == ',') && (depth == 0))
				{
					// CSV compatibility
					rx->read1();
					rx->adv(true);
					
					// Convert to list with new first element
					IDA* first = new IDA();
					first->rxConfig(node);
					if (needContent) {first->putStr(prelimName);} else {node->copyTo(first);}
					
					// Complete the row
					std::vector<IDA*> listObj;
					listObj.push_back(first);
					IDA* last = nullptr;
					while (rx->c1n('\n'))
					{
						IDA* el = node->rxNew();
						last = fill(el, rx, '\0', '\n', last, CLASS_ARRAY, depth+1);
						if (last == nullptr) {delete el;}
						else while (el != nullptr && (node->inputListLength < 0 || node->inputListLength > listObj.size()))
						{
							listObj.push_back(el);
							el = el->getNextOr();
						}
					}
					
					node->putList(listObj);
					node->setName(std::nullopt);
					
				}
				else if (csvLn)
				{
					// Last CSV item
					if (needContent) {node->putStr(prelimName);}
				}
				else if (!nameSet)
				{
					node->setName(prelimName);
				}
				
				nameSet = true;
				needContent = false;
				
				if (!csvLn) {rx->read1(); rx->adv();}
				
			}
		}
		
		// RX read END
	} }
	catch (...)
	{
		node->rxClearTemplates();
		node->rxClearItemClasses();
		throw;
	}
	
	// RX fill/name
	if (!nameSet)
	{
		if (needName && lcn)
		{
			lcn = false;
			IDA* lcnode = listClass->next();
			if (lcnode != nullptr) {node->setName(lcnode->getNameOr());} else {std::cerr << "Failure: Class provided a null node (RX fill/name, needName)\n";}
		}
		else {node->setName(prelimName);}
	}
	
	// RX fill END
	if (lcn) listClass->next();
	
	node->rxClearTemplates();
	node->rxClearItemClasses();
	return output;
}

// TODO: RX discard
// RX number test
std::regex IDA_RX::p_num = std::regex("^[\\+\\-]?[0-9]*(?:\\.[0-9]+)?([eE][\\+\\-]?[0-9]+)?$");

void IDA_RX::testNumber(std::string num, std::string desc)
{
	// No need to test hex/bin here; invalid characters will have delimited the number, potentially becoming the unit
	if (IDA::port->starts_with(num, "0x") || IDA::port->starts_with(num, "0b")) {return;}
	
	if (!std::regex_search(num, p_num)) {throw IDAFormatException(this, "Invalid "+desc+" number");}
}

// RX manip
IDA* IDA_RX::manip(IDA_RX* rx, IDA* node, IDA_Iterator* itemTemplate)
{
	// Manipulations
	IDA manip;
	
	try
	{
		fill(&manip, rx, '\0', '\0', nullptr, CLASS_NONE, 1);
	}
	catch (...)
	{
		delete itemTemplate;
		throw;
	}
	
	if (manip.isList())
	{
		// TODO: any pre-process or validation of manipulations, or simplify logic if none needed
	}
	else if (manip.hasContent())
	{
		delete itemTemplate;
		throw IDAFormatException(rx, "Template manipulation must be a list");
	}
	
	IDA* last = IDA_Manip::manip_iterate(&manip, node, itemTemplate);
	return last;
}

// RX end

// Output interface
IDA_Output::IDA_Output(std::ostream* output)
{
	this->w = output;
	if (w->bad()) throw std::runtime_error("Stream closed");
}

void IDA_Output::send(std::string s)
{
	*w << s;
	if (w->bad()) throw std::runtime_error("Stream closed");
}

void IDA_Output::queue(std::string s) {queueBuf += s;}
long IDA_Output::length() {return queueBuf.length();}
void IDA_Output::reject() {queueBuf.clear();}
void IDA_Output::accept() {send(queueBuf); queueBuf.clear();}

void IDA_Output::idaWrite(IDA* el, bool adj)
{
	if (el == nullptr) {return;}
	
	el->txRefLabel(adj);
	
	ida_header(el, adj);
	
	if (adj)
	{
		// Rewind first
		while (el->hasPrev()) {el = el->getPrevOr();}
		
		// Output forward
		while (el != nullptr)
		{
			ida_element(el, true);
			el = el->getNextOr();
		}
	}
	else {ida_element(el, false);}
	ida_footer();
	w->flush();
	if (w->bad()) throw std::runtime_error("Stream closed");
}

bool IDA_Output::idaPrint(IDA* el, bool adj)
{
	try {idaWrite(el, adj); return !w->bad();}
	catch (...) {return false;}
}

// TX begin

// TX defaults

// TX instance
IDA_TX* IDA_TX::setOptsOverride(bool ov) {optsOverride = ov; return this;}
IDAOptions* IDA_TX::opts(IDA* el) {return optsOverride ? IDAOptions::defaultOptions : el->opts;}

IDA_TX::IDA_TX(std::ostream* output) : IDA_Output(output)
{
}

// TX interface
void IDA_TX::ida_header(IDA* el, bool adj)
{
	entry = el;
}
void IDA_TX::ida_element(IDA* el, bool adj)
{
	// Explicit entry point?
	if (adj && entry == el && el->hasPrev()) {send("~ ");}
	
	// As-above all?
	if (rootAsAboveFirst == nullptr)
	{
		rootAsAboveFirst = el;
		rootAsAboveAll = as_above_all_test(el);
	}
	
	// Main output wrapped with name as-above part
	if (rootAsAbovePart && !el->isName(rootAsAboveName)) {rootAsAbovePart = false;}
	ida(el, "", false, rootAsAbovePart || (adj && rootAsAboveAll && el->hasPrev()), false, nullptr);
	rootAsAboveName = el->getNameOr();
	if (!rootAsAbovePart) rootAsAbovePart = as_above_part_test(el);
}

// TX structure filter
bool IDA_TX::has_inner_structure(IDA* first)
{
	for (IDA* outer = first; outer != nullptr; outer = outer->getNextOr())
	{
		if (outer->isList()) for (IDA* inner : *outer->asListOr())
		{
			if (inner->hasParams() || inner->isList()) return true;
		}
	}
	return false;
}

// TX main begin
void IDA_TX::ida(IDA* node, std::string indent, bool parameters, bool nameAsAbove, bool nameOmit, IDA* itemClass)
{
	bool hasOutput = false;
	
	// TX main/indent
	send(indent);
	if (node->hasRefName()) send("*"+*node->getRefName()+" ");
	if (itemClass != nullptr) send(": ");
	
	// TX main/name
	if (!nameOmit)
	{
		std::string name = "";
		if (node->hasName()) {name = nameAsAbove ? "^" : id_str(node, node->getNameOr());}
		else if (!node->hasParams() && !node->isList()) {name = "?";}
		
		send(name);
		hasOutput = name.length() > 0;
	}
	
	// TX main/parameters
	bool chain = false;
	if (node->hasParams())
	{
		if (hasOutput) {send(" ");}
		hasOutput = true;
		send("(");
		IDA_Iterator* pNames = itemClass != nullptr && itemClass->hasParams() ? IDA::port->list_i(itemClass->getParams()) : IDA::port->empty_i();
		for (IDA* param = node->getParamFirstOr(); param != nullptr; param = param->getNextOr())
		{
			bool pNameOmit = pNames->hasNext() && pNames->next()->eqName(param);
			if (chain && !pNameOmit) {send(" ");}
			chain = true;
			ida(param, "", true, false, pNameOmit, nullptr);
		}
		send(")");
		delete pNames;
	}
	
	// TX main/content
	if (node->isNull() || broken_ref(node))
	{
		send(";");
		if (!parameters) {send("\n");}
	}
	
	// TX main/ref
	else if (node->isRef())
	{
		send(" &");
		send(*(node->asRefOr()->getRefName()));
		send(parameters ? (node->hasNext() ? "," : "") : "\n");
	}
	
	// TX main/content/bool
	else if (node->isBool())
	{
		if (nameOmit) send(" ");
		send(node->isTrue() ? "+" : "-");
		if (!parameters) {send("\n");}
	}
	
	// TX main/content/number
	else if (node->isNum())
	{
		if (hasOutput || node->hasPrev()) {send(" ");}
		send(node->toString());
		send(parameters ? (node->hasNext() ? "," : "") : "\n");
	}
	
	// TX main/content/list
	else if (node->isList())
	{
		if (node->numItems() == 0)
		{
			if (hasOutput) {send(" ");}
			send("{}\n");
		}
		else if (node->numItems() == 1 && !node->getItemFirstOr()->isList())
		{
			if (hasOutput) {send(" ");}
			IDA* item = node->getItemFirstOr();
			if (item->hasName() || item->hasParams())
			{
				send("{ ");
				ida(item, "", true, false, false, nullptr);
				send(" }\n");
			}
			else
			{
				send("[ ");
				send(value(item, true));
				send(" ]\n");
			}
		}
		else
		{
			bool oneLine = IDA::port->si_allMatch(IDA::port->list_i(node->asListOr()), [](IDA* n) {return !n->isList() && !n->hasParams();});
			
			if (oneLine)
			{
				if (hasOutput) {queue(" ");}
				if (IDA::port->si_anyMatch(IDA::port->list_i(node->asListOr()), [](IDA* n) {return n->hasName();}))
				{
					IDA_Iterator* pNames = itemClass != nullptr && itemClass->isList() ? IDA::port->list_i(itemClass->asListOr()) : IDA::port->empty_i();
					
					bool asAbove = as_above_all_test(node->getItemFirstOr());
					
					IDA_Iterator* i = IDA::port->list_i(node->asListOr());
					queue("{");
					{
						IDA* n = i->next();
						bool pNameOmit = pNames->hasNext() && pNames->next()->isName(n->getNameOr());
						if (n->hasRefName()) queue(" *"+*n->getRefName());
						if (!pNameOmit) queue(" "+id_str(n, n->getNameOr()));
						queue(value(n, false));
					}
					while (i->hasNext())
					{
						IDA* n = i->next();
						bool pNameOmit = pNames->hasNext() && pNames->next()->isName(n->getNameOr());
						if (n->hasPrev() && !n->getPrevOr()->isBool() && !n->getPrevOr()->isNull()) queue(",");
						if (n->hasRefName()) queue(" *"+*n->getRefName());
						if (!pNameOmit) queue(" "+(asAbove ? "^" : id_str(n, n->getNameOr())));
						queue(value(n, false));
					}
					queue(" }\n");
					delete i;
					delete pNames;
				}
				else
				{
					IDA_Iterator* i = IDA::port->list_i(node->asListOr());
					queue("[");
					while (i->hasNext()) {queue(" " + IDA::port->trim(value(i->next(), true)));}
					queue(" ]\n");
					delete i;
				}
				
				if (length() < 2*ONE_LINE_MAX) {accept();}
				else {reject(); oneLine = false;}
				
			}
			
			if (!oneLine)
			{
				// For now, only use name-as-above when all nodes have the same name,
				// and are not lists above size 1
				
				if (hasOutput) {send("\n"); send(indent);}
				send("{\n");
				std::string indent2 = indent + "\t";
				
				IDA* first = node->getItemFirstOr();
				
				// Check to see if a class could be used for parameters
				IDA* pClass = nullptr;
				std::optional<std::string> firstName = first->getNameOr();
				bool cName = false;
				if (!parameters && node->numItems() > 2)
				{
					cName = true;
					
					// Parameters
					std::vector<IDA*>* firstParams = first->getParamsOr();
					int ptrParams = firstParams != nullptr && first->hasParams() ? firstParams->size() : 0;
					
					// Check names separately
					for (IDA* n = first->getNextOr(); n != nullptr; n = n->getNextOr())
					{
						if (!n->isName(firstName)) {cName = false; break;}
					}
					
					// Check parameters
					for (IDA* n = first->getNextOr(); n != nullptr && ptrParams > 0; n = n->getNextOr())
					{
						std::vector<IDA*>* nps = n->getParamsOr();
						if (!n->hasParams()) {ptrParams = 0; break;}
						
						int i = 0;
						for (IDA* np : *nps)
						{
							if (i >= ptrParams) {break;}
							if (!np->isName((*firstParams)[i]->getNameOr())) {break;}
							i+=1;
						}
						
						ptrParams = std::min(ptrParams, i);
					}
					
					
					// Items (all nodes must be lists)
					std::vector<IDA*>* firstItems = first->asListOr();
					int ptrItems = 0;
					bool namedItem = false;
					
					// Emergency measure - avoid classing the items if there are ANY parameters or lists inside ANY nodes
					if (firstItems != nullptr && first->isList() && !has_inner_structure(first))
					{
						ptrItems = firstItems->size();
						for (IDA* n = first->getNextOr(); n != nullptr && ptrItems > 0; n = n->getNextOr())
						{
							std::vector<IDA*>* nis = n->asListOr();
							if (!n->isList()) {ptrItems = 0; break;}
							
							int i = 0;
							for (IDA* ni : *nis)
							{
								if (i >= ptrItems) {break;}
								std::optional<std::string> name = (*firstItems)[i]->getNameOr();
								if (!ni->isName(name)) {break;}
								if (name.has_value()) namedItem = true;
								i+=1;
							}
							
							ptrItems = std::min(ptrItems, i);
						}
					}
					
					// Combine
					if (ptrParams > 0 || (namedItem && ptrItems > 0))
					{
						// Class!
						send(indent2);
						send(":: ");
						if (cName) {send(id_str(first, firstName));}
						
						pClass = new IDA(firstName);
						
						if (ptrParams > 0)
						{
							pClass->setParams();
							
							send("(");
							IDA* np = (*firstParams)[0];
							send(id_str(np, np->getNameOr()));
							pClass->addParam(new IDA(np->getNameOr()));
							while (np->hasNext() && (--ptrParams > 0))
							{
								np = np->getNextOr();
								send(", " + id_str(np, np->getNameOr()));
								pClass->addParam(new IDA(np->getNameOr()));
							}
							send(")");
						}
						
						if (ptrItems > 0)
						{
							pClass->putList();
							
							send("{");
							IDA* ni = (*firstItems)[0];
							send(id_str(ni, ni->getNameOr()));
							pClass->addItem(new IDA(ni->getNameOr()));
							while (ni->hasNext() && (--ptrItems > 0))
							{
								ni = ni->getNextOr();
								send(", " + id_str(ni, ni->getNameOr()));
								pClass->addItem(new IDA(ni->getNameOr()));
							}
							send("}\n");
						}
						else {send(";\n");}
						
					}
					else {cName = false;}
				}
				
				IDA* item = first;
				bool asAbove = as_above_all_test(first);
				
				if (asAbove || cName)
				{
					ida(item, indent2, false, false, cName, pClass);
					while ((item = item->getNextOr()) != nullptr)
					{
						ida(item, indent2, false, true, cName, pClass);
					}
				}
				else
				{
					bool asAbovePart = false;
					std::optional<std::string> asAboveName = std::nullopt;
					
					for (; item != nullptr; item = item->getNextOr())
					{
						if (asAbovePart && !item->isName(asAboveName)) {asAbovePart = false;}
						ida(item, indent2, false, asAbovePart, false, pClass);
						
						asAboveName = item->getNameOr();
						if (!asAbovePart) asAbovePart = as_above_part_test(item);
						
					}
				}
				
				if (pClass != nullptr) delete pClass;
				
				send(indent);
				send("}\n");
				
			}
			
		}
		
	}
	
	// TX main/content/string
	else
	{
		if (hasOutput || node->hasPrev()) {send(" ");}
		send(id_str(node, node->toString()));
		send(parameters ? (node->hasNext() ? "," : "") : "\n");
		
	}
	
// TX main end
}

// TX utility
bool IDA_TX::broken_ref(IDA* el)
{
	return el->isRef() && !el->asRefOr()->hasRefName();
}

bool IDA_TX::as_above_all_test(IDA* firstItem)
{
	std::optional<std::string> itemName = firstItem->getNameOr();
	
	if (!itemName.has_value()) return false;
	if (itemName.value().length() < BREVITY_ALL) return false;
	
	for (IDA* c = firstItem; c != nullptr; c = c->getNextOr())
	{
		if (c->numItems() >= 2) return false;
		if (!c->isName(itemName.value())) return false;
	}
	return true;
}
bool IDA_TX::as_above_part_test(IDA* item)
{
	std::optional<std::string> asAboveName = item->getNameOr();
	if (!asAboveName.has_value() || asAboveName.value().length() < BREVITY_ALL) {return false;}
	long req = BREVITY_PART;
	for (IDA* n = item->getNextOr(); (--req > 0) && n != nullptr; n = n->getNextOr())
	{
		if (!n->isName(asAboveName.value())) {return false;}
	}
	return req <= 0;
}


std::string IDA_TX::value(IDA* node, bool inArray)
{
	if (node->isNull() || broken_ref(node)) {return inArray ? "?" : ";";}
	if (node->isBool()) {return node->isTrue() ? "+" : "-";}
	if (node->isRef()) {return (inArray ? "&" : " &") + *node->asRefOr()->getRefName();}
	return (inArray ? "" : " ") + (node->isNum() ? node->toString() : id_str(node, node->toString()));
}
std::string IDA_TX::id_str(IDA* refNode, std::optional<std::string> s_opt)
{
	if (!s_opt.has_value()) {return "?";}
	std::string s = s_opt.value();
	if (s.length() < 1) {return "\"\"";}
	
	if (id(s)) {return s;}
	else
	{
		bool quoted = IDA::port->ends_with(s, "-");
		std::ostringstream buf;
		
		std::string::size_type max = s.length();
		for (std::string::size_type i=0; i<max; i+=1)
		{
			unsigned char c = s[i];
			if (c < 0x20)
			{
				if (c == 0) {buf << "\\0";}
				else if (c == '\n') {buf << "\\n";}
				else if (c == '\r') {buf << "\\r";}
				else if (c == '\t') {buf << "\\t";}
				else {buf << "\\x" << IDA::port->char_2(c);}
			}
			else if (c < 0x7f)
			{
				if (c == '\\') {buf << "\\\\";}
				else if (c == '"') {buf << "\\\""; quoted = true;}
				else
				{
					if (!quoted)
					{
						if (i<1 ? !IDA::id0(c) : !IDA::id1(c)) {quoted = true;}
						else if (i>0 && IDA::comment2(s[i-1], c)) {quoted = true;}
					}
					buf << c;
				}
			}
			else
			{
				if (c == 0xc2 && s[i+1] < 0xa0)
				{
					buf << "\\x" << IDA::port->char_2((unsigned char) s[++i]);
				}
				else if (opts(refNode)->txUnicode)
				{
					buf << c << s[++i];
					if (c & 0x20) {buf << s[++i];}
					if (c & 0x10) {buf << s[++i];}
				}
				else
				{
					unsigned long u;
					if ((c & 0xe0) == 0xc0) {u = ((c & 0x1f) << 6) | (s[++i] & 0x3f);}
					else if ((c & 0xf0) == 0xe0) {u = ((c & 0x0f) << 12) | ((s[++i] & 0x3f) << 6) | (s[++i] & 0x3f);}
					else if ((c & 0xf8) == 0xf0) {u = ((c & 0x07) << 18) | ((s[++i] & 0x3f) << 12) | ((s[++i] & 0x3f) << 6) | (s[++i] & 0x3f);}
					if (u < 0x100) {buf << "\\x" << IDA::port->char_2((unsigned char)u);}
					else if (u < 0x10000) {buf << "\\u" << IDA::port->char_4(u);}
					else if (u < 0x110000) {buf << "\\U" << IDA::port->char_8(u);}
				}
			}
			
		}
		
		if (quoted)
		{
			buf << "\"";
			return "\"" + buf.str();
		}
		else return buf.str();
	}
	
}
bool IDA_TX::id(std::string s)
{
	if (s.length() < 1 || !IDA::id0(s[0]) || s[s.length()-1] == '-') {return false;}
	char c1 = s[0];
	for (int i=1; i<s.length(); i+=1)
	{
		char c2 = s[i];
		if (!IDA::id1(c2)) return false;
		if (IDA::comment2(c1, c2)) return false;
		c1 = c2;
	}
	return true;
}

// TX end

// CLI begin

// CLI
// g++ -std=c++17 -D IDA_MAIN -o IDA IDA.cpp
int IDA_CLI::cli_err(std::string msg, int exitCode)
{
	std::cerr << "Error: " << msg << '\n';
	return exitCode;
}
void IDA_CLI::cli_out(std::string ln) {std::cout << ln << '\n';}
int IDA_CLI::cli_help()
{
	//      ".........1.........2.........3.........4.........5.........6.........7.........8"
	cli_out("IDA markup parser");
	cli_out("Read markup from standard input and write IDA to standard output.");
	cli_out("");
	cli_out("Parser options:");
	cli_out(" -h, --help         Show usage");
	cli_out(" -i in-option       Set IDA input option");
	cli_out(" -o out-option      Set IDA output option");
	cli_out("");
	cli_out("IDA input options:");
	cli_out(" xml-reparse        Parse XML string content as additional inline markup");
	cli_out("");
	cli_out("IDA output options:");
	cli_out(" unicode            Use UTF-8 for string characters beyond U+0080");
	cli_out("");
	return 0;
}
int IDA_CLI::cli(int argc, char** argv)
{
	IDA rootEl;
	IDAOptions* opts = rootEl.opts;
	
	int i=0;
	while (i<argc)
	{
		std::string arg = argv[i++];
		
		if (arg == "-h" || arg == "--help")
		{
			return cli_help();
		}
		else if (arg == "-i")
		{
			if (i >= argc) return cli_err("No input option specified");
			arg = argv[i++];
			if (arg == "xml-reparse") {opts->rxXmlReparseStrings = true;}
			else return cli_err("Unknown input option: " + arg);
		}
		else if (arg == "-o")
		{
			if (i >= argc) return cli_err("No output option specified");
			arg = argv[i++];
			if (arg == "unicode") {opts->txUnicode = true;}
			else return cli_err("Unknown output option: " + arg);
		}
	}
	
	try {rootEl.inStream(&std::cin);}
	catch (IDAOpException ex) {return cli_err("Parse failed: " + ex.getMessage());}
	catch (IDAFormatException ex) {return cli_err("Parse failed: " + ex.getMessage());}
	catch (std::string ex) {return cli_err("Parse failed: " + ex);}
	catch (std::bad_optional_access ex) {return cli_err("Parse failed: " + std::string(ex.what()));}
	catch (std::exception ex) {return cli_err("Parse failed: " + std::string(ex.what()));}
	
	rootEl.outPrint(&std::cout, true);
	return 0;
}

// CLI end


// File includes
// File includes - Global PWD
void IDA::setGlobalIncludes(std::optional<std::string> pwd) {IDA::gIncludePwd = pwd;}
void IDA::setGlobalIncludes() {setGlobalIncludes("./");}

// File includes - this element
IDA* IDA::setIncludes(std::optional<std::string> pwd) {includePwd = pwd; return this;}
IDA* IDA::setIncludes() {return setIncludes("./");}
std::optional<std::string> IDA::rxIncludePwd() {return includePwd;}

// Input control
IDA* IDA::rxInputList(int depth, int length)
{
	this->inputListDepth = depth;
	this->inputListLength = length;
	return this;
}

// Options storage
IDAOptions* IDA::getOpts() {return opts;}
IDA* IDA::copyOptions(IDA* src)
{
	this->opts->copyFrom(src->opts);
	return this;
}

IDA* IDA::rxConfig(IDA* parent)
{
	rxInputList(parent->inputListDepth > 0 ? parent->inputListDepth - 1 : parent->inputListDepth, parent->inputListLength);
	copyOptions(parent);
	return this;
}
IDA* IDA::rxNew()
{
	IDA* el = (new IDA())->rxScope(this)->setIncludes(this->rxIncludePwd());
	el->rxConfig(this);
	return el;
}

// Scope storage
IDA* IDA::rxScope(IDA* scope) {this->scope = scope; return this;}

// Item classes
void IDA::rxClearItemClasses()
{
	for (auto i = itemClasses.begin(); i != itemClasses.end(); i++) delete i->second;
	itemClasses.clear();
}
IDA* IDA::rxGetItemClass(std::string key)
{
	auto i = itemClasses.find(key);
	if (i != itemClasses.end()) return i->second;
	return scope != nullptr ? scope->rxGetItemClass(key) : nullptr;
}
bool IDA::rxAddItemClass(std::string key, IDA* itemClass)
{
	if (scope == nullptr) {return false;}
	auto i = scope->itemClasses.find(key);
	if (i != scope->itemClasses.end()) {delete i->second; scope->itemClasses.erase(i);}
	scope->itemClasses.insert({key, itemClass});
	return true;
}

// Templates
void IDA::rxClearTemplates()
{
	for (auto i = itemTemplates.begin(); i != itemTemplates.end(); i++) delete i->second;
	itemTemplates.clear();
}
IDA* IDA::rxGetTemplate(std::string key)
{
	auto i = itemTemplates.find(key);
	if (i != itemTemplates.end()) return i->second;
	return scope != nullptr ? scope->rxGetTemplate(key) : nullptr;
}
bool IDA::rxAddTemplate(std::string key, IDA* itemTemplate)
{
	if (scope == nullptr) {return false;}
	auto i = scope->itemTemplates.find(key);
	if (i != scope->itemTemplates.end()) {delete i->second; scope->itemTemplates.erase(i);}
	scope->itemTemplates.insert({key, itemTemplate});
	return true;
}

// Adjacent nodes
bool IDA::hasNext() {return this->next != nullptr;}
bool IDA::hasPrev() {return this->prev != nullptr;}
bool IDA::isFirst() {return !this->hasPrev();}
bool IDA::isLast() {return !this->hasNext();}

IDA* IDA::getNext()
{
	if (isLast()) throw IDAOpException("Element is last");
	return next;
}
IDA* IDA::getNextOr()
{
	return next;
}
IDA* IDA::getNextOr(IDA* other)
{
	return hasNext() ? next : other;
}
IDA* IDA::getPrev()
{
	if (isFirst()) throw IDAOpException("Element is first");
	return prev;
}
IDA* IDA::getPrevOr()
{
	return prev;
}
IDA* IDA::getPrevOr(IDA* other)
{
	return hasPrev() ? prev : other;
}

IDA_Iterator* IDA::iterateNext()
{
	IDA_Iterator* i = new IDA_Iterator();
	i->node = this;
	i->funcHas = [i] {return i->node != nullptr;};
	i->funcGet = [i] {IDA* x = i->node; i->node = x->getNextOr(); return x;};
	return i;
}
IDA* IDA::linkNext(IDA* next)
{
	if (this->next != nullptr) {this->next->prev = nullptr;}
	this->next = next; if (next != nullptr) next->prev = this;
	return this;
}
IDA* IDA::linkPrev(IDA* prev)
{
	if (this->prev != nullptr) {this->prev->next = nullptr;}
	this->prev = prev; if (prev != nullptr) prev->next = this;
	return this;
}
IDA* IDA::link(IDA* prev, IDA* next)
{
	if (this->prev != nullptr) {this->prev->next = this->next;}
	if (this->next != nullptr) {this->next->prev = this->prev;}
	this->prev = prev; if (prev != nullptr) prev->next = this;
	this->next = next; if (next != nullptr) next->prev = this;
	return this;
}
void IDA::link_break()
{
	this->prev = this->next = nullptr;
}

// Node name
bool IDA::isAnon() {return !this->name.has_value();}
bool IDA::hasName() {return this->name.has_value();}
bool IDA::isName(std::optional<std::string> name)
{
	return IDA::eq_str(name, this->getNameOr());
}

std::string IDA::getName()
{
	if (isAnon()) throw IDAOpException("Element has no name");
	return name.value();
}
std::optional<std::string> IDA::getNameOr()
{
	return name;
}
std::string IDA::getNameOr(std::string other)
{
	return name.value_or(other);
}

IDA* IDA::clearName() {return setName(nullptr);}
IDA* IDA::setName(std::optional<std::string> name) {if (name.has_value()) {this->name = name.value();} else {this->name.reset();} return this;}
void IDA::copyName(IDA* src) {setName(src->name);}

// Parameters
IDA* IDA::clearParams()
{
	for (IDA* x : parameters) {x->link_break(); delete x;}
	parameters.clear(); parameters.shrink_to_fit(); parameters_ok = false; return this;
}
IDA* IDA::setParams() {clearParams(); parameters_ok = true; return this;}

std::vector<IDA*>* IDA::getParams()
{
	if (!hasParams()) throw IDAOpException("No parameters list");
	return &parameters;
}
std::vector<IDA*>* IDA::getParamsOr()
{
	return hasParams() ? &parameters : nullptr;
}
std::vector<IDA*>* IDA::getParamsOr(std::vector<IDA*>* other)
{
	return hasParams() ? &parameters : other;
}

bool IDA::hasParams()
{
	return parameters_ok;
}
long IDA::numParams()
{
	return parameters_ok ? parameters.size() : -1;
}
bool IDA::hasParam(std::optional<std::string> key)
{
	return getParamOr(key) != nullptr;
}

// Parameters/get
IDA* IDA::getParamAt(long index)
{
	if (numParams() <= index) throw IDAOpException("Invalid parameter index");
	return parameters[index];
}
IDA* IDA::getParamAtOr(long index)
{
	return getParamAtOr(index, nullptr);
}
IDA* IDA::getParamAtOr(long index, IDA* other)
{
	return numParams() > index ? parameters[index] : other;
}

IDA* IDA::getParam(std::optional<std::string> key)
{
	IDA* p = getParamOr(key);
	if (p == nullptr) throw IDAOpException("Parameter not found");
	return p;
}
IDA* IDA::getParamOr(std::optional<std::string> key)
{
	return getParamOr(key, nullptr);
}
IDA* IDA::getParamOr(std::optional<std::string> key, IDA* other)
{
	if (hasParams()) for (IDA* p : parameters) if (p->isName(key)) return p;
	return other;
}

std::string IDA::getParamStr(std::optional<std::string> key)
{
	return getParam(key)->toStr();
}
std::optional<std::string> IDA::getParamStrOr(std::optional<std::string> key)
{
	IDA* param = getParamOr(key);
	return param != nullptr ? param->toStrOr() : std::nullopt;
}
std::string IDA::getParamStrOr(std::optional<std::string> key, std::string other)
{
	IDA* param = getParamOr(key);
	return param != nullptr ? param->toStrOr(other) : other;
}

IDA* IDA::getParamFirst()
{
	if (numParams() < 1) throw IDAOpException(hasParams() ? "Parameters list is empty" : "Element has no parameters list");
	return parameters.front();
}
IDA* IDA::getParamFirstOr()
{
	return getParamFirstOr(nullptr);
}
IDA* IDA::getParamFirstOr(IDA* other)
{
	return numParams() > 0 ? parameters.front() : other;
}
IDA* IDA::getParamLast()
{
	if (numParams() < 1) throw IDAOpException(hasParams() ? "Parameters list is empty" : "Element has no parameters list");
	return parameters.back();
}
IDA* IDA::getParamLastOr()
{
	return getParamLastOr(nullptr);
}
IDA* IDA::getParamLastOr(IDA* other)
{
	return numParams() > 0 ? parameters.back() : other;
}

// Parameters/add
IDA* IDA::newParam(std::optional<std::string> name)
{
	IDA* el = new IDA(name);
	try {addParam(el);}
	catch (...) {delete el; throw;}
	return el;
}

IDA* IDA::addParam(IDA* param)
{
	if (!parameters_ok) {setParams();}
	param->link(getParamLastOr(), nullptr);
	parameters.push_back(param);
	return this;
}
IDA* IDA::addParamAt(IDA* param, long pos)
{
	if (!parameters_ok) {setParams();}
	IDA::list_add(getParams(), param, pos, "param");
	return this;
}
IDA* IDA::setParam(IDA* param)
{
	long n = numParams();
	
	try
	{
		for (long i=0; i<n; i+=1)
		{
			IDA* p = getParamAt(i);
			if (p->eqName(param))
			{
				remParamAt(i);
				return addParamAt(param, i);
			}
		}
	}
	catch (IDAOpException ex) {throw std::runtime_error("Unexpected parameters state");}
	
	return addParam(param);
}

IDA* IDA::clearParam(std::optional<std::string> key)
{
	return setParamNull(key);
}
IDA* IDA::setParamNull(std::optional<std::string> key)
{
	IDA* el = (new IDA(key))->putNull();
	try {return setParam(el);} catch (...) {delete el; throw;}
}
IDA* IDA::setParamStr(std::optional<std::string> key, std::optional<std::string> content)
{
	IDA* el = (new IDA(key))->putStr(content);
	try {return setParam(el);} catch (...) {delete el; throw;}
}
IDA* IDA::setParamBool(std::optional<std::string> key, bool content)
{
	IDA* el = (new IDA(key))->putBool(content);
	try {return setParam(el);} catch (...) {delete el; throw;}
}
IDA* IDA::setParamDouble(std::optional<std::string> key, double content)
{
	IDA* el = (new IDA(key))->putDouble(content);
	try {return setParam(el);} catch (...) {delete el; throw;}
}
IDA* IDA::setParamFloat(std::optional<std::string> key, float content)
{
	IDA* el = (new IDA(key))->putFloat(content);
	try {return setParam(el);} catch (...) {delete el; throw;}
}
IDA* IDA::setParamLong(std::optional<std::string> key, long long content)
{
	IDA* el = (new IDA(key))->putLong(content);
	try {return setParam(el);} catch (...) {delete el; throw;}
}
IDA* IDA::setParamInt(std::optional<std::string> key, long content)
{
	IDA* el = (new IDA(key))->putInt(content);
	try {return setParam(el);} catch (...) {delete el; throw;}
}
IDA* IDA::setParamRef(std::optional<std::string> key, IDA* content)
{
	IDA* el = (new IDA(key))->putRef(content);
	try {return setParam(el);} catch (...) {delete el; throw;}
}

// Parameters/remove
void IDA::remParamAt(int pos)
{
	if (!hasParams()) {throw IDAOpException("No parameters from which to remove");}
	delete IDA::list_del(&parameters, pos, "param");
}
bool IDA::remParam(IDA* param)
{
	if (hasParams() && IDA::list_rem(&parameters, param)) {param->link(nullptr, nullptr); return true;}
	return false;
}
bool IDA::unParam(std::optional<std::string> key)
{
	IDA* p = getParamOr(key);
	if (p != nullptr) {remParam(p); delete p; return true;}
	return false;
}

// Parameters/other
void IDA::copyParams(IDA* src)
{
	clearParams();
	if (src->hasParams())
	{
		setParams();
		for (IDA* p : src->parameters) this->addParam(p->copy_inner());
	}
}

// Content
void IDA::copyContent(IDA* src)
{
	if (src->cRaw.has_value()) {this->cRaw = src->cRaw.value();} else {this->cRaw.reset();}
	if (src->cStr.has_value()) {this->cStr = src->cStr.value();} else {this->cStr.reset();}
	this->cNum = src->cNum;
	this->cDen = src->cDen;
	this->cUnit = src->cUnit;
	if (src->cBool.has_value()) {this->cBool = src->cBool.value();} else {this->cBool.reset();}
	this->cRef = nullptr;
	this->cRefObj = nullptr;
	this->cRefTarget = src->cRefTarget;
	this->cRefName = src->cRefName;
	this->clearList();
	if (src->isList())
	{
		this->putList();
		for (IDA* x : *src->asList())
		{
			IDA* cp = x->copy_inner();
			try {this->addItem(cp);} catch (...) {delete cp; throw;}
		}
	}
}

// Content/ref
IDA* IDA::clearRef() {cRef = nullptr; cRefObj = nullptr; cRefTarget = std::nullopt; return this;}
bool IDA::isRef() {return cRef != nullptr;}
IDA* IDA::asRef()
{
	return cRef;
}
IDA* IDA::asRefOr()
{
	return asRefOr(nullptr);
}
IDA* IDA::asRefOr(IDA* other)
{
	return isRef() ? cRef : other;
}
IDA* IDA::putRef(IDA* ref) {putNull(); cRaw = "Ref"; cRef = ref; cRefObj = ref->repObj; cRefTarget = ref->cRefName; return this;}
IDA* IDA::setRefObj(IDAObj* refObj) {putNull(); cRaw = "Ref"; cRefObj = refObj; return this;}
void IDA::withRefObj(std::function<void(IDAObj*)> callback)
{
	if (cRefObj != nullptr) {callback(cRefObj);}
	else if (cRef == nullptr) throw IDAOpException("Element is not a reference");
	else {cRef->on_rep_obj(callback);}
}
IDAObj* IDA::getRefObj()
{
	if (cRefObj != nullptr) return cRefObj;
	if (cRef == nullptr) throw IDAOpException("Element is not a reference");
	if (cRef->repObj != nullptr) return cRef->repObj;
	throw IDAOpException("Referenced object not available");
}
void IDA::setRefTarget(std::string refTarget) {putNull(); cRaw = "Ref"; cRefTarget = refTarget;}
std::optional<std::string> IDA::getRefTarget() {return cRefTarget;}
bool IDA::hasRefTarget() {return cRefTarget.has_value();}
void IDA::setRefName(std::optional<std::string> refName) {cRefName = refName;}
std::optional<std::string> IDA::getRefName() {return cRefName;}
bool IDA::hasRefName() {return cRefName.has_value();}

long IDA::txRefLabel(bool adj)
{
	// Ref walk
	long rc = 0;
	std::vector<IDA*> rn;
	walk([&rn,&rc](IDA* x){ref_label_clear(x, rn);}, adj, adj);
	walk([&rn,&rc](IDA* x){ref_label_count(x, rn, rc);}, adj, adj);
	
	// Null any foreign references at this stage (see IDA.verify)
	for (IDA* r : rn) r->setRefName(std::nullopt);
	
	return rc;
}
void IDA::ref_label_clear(IDA* el, std::vector<IDA*> &rn)
{
	el->setRefName(std::nullopt);
	if (el->isRef())
	{
		IDA* r = el->asRefOr();
		if (!IDA::port->list_contains(&rn, r)) rn.push_back(r);
	}
}
void IDA::ref_label_count(IDA* el, std::vector<IDA*> &rn, long &rc)
{
	if (IDA::port->list_contains(&rn, el)) IDA::port->list_rem(&rn, el);
	
	if (el->isRef())
	{
		IDA* r = el->asRefOr();
		if (!r->hasRefName())
		{
			long n = rc++;
			unsigned char n0 = n % 52;
			std::ostringstream id;
			id << (unsigned char)(n0 < 26 ? 0x61+n0 : 0x41+n0) << (n/52);
			r->cRefName = id.str();
			el->cRefTarget = id.str();
		}
	}
}

void IDA::rxRefResolve(bool adj)
{
	std::map<std::string,IDA*> rs;
	walk([&rs](IDA* x) {ref_resolve_scan(x, rs);}, adj, adj);
	walk([&rs](IDA* x) {ref_resolve_put(x, rs);}, adj, adj);
}
void IDA::ref_resolve_scan(IDA* el, std::map<std::string,IDA*> &rs)
{
	if (el->hasRefName()) rs[*el->getRefName()] = el;
}
void IDA::ref_resolve_put(IDA* el, std::map<std::string,IDA*> &rs)
{
	if (el->hasRefTarget()) el->putRef(rs[*el->getRefTarget()]);
}

// Content/bool
bool IDA::isBool() {return cBool.has_value();}
bool IDA::asBool()
{
	if (!cBool.has_value()) throw IDAOpException("Content is not a boolean value");
	return cBool.value();
}
bool IDA::asBoolOr(bool other) {return cBool.has_value() ? cBool.value() : other;}
/** Bool type and value is true, intended as a convenience for function references. */
bool IDA::isTrue() {return cBool.has_value() && cBool.value();}
/** Bool type and value is false, intended as a convenience for function references. */
bool IDA::isFalse() {return cBool.has_value() && !cBool.value();}
IDA* IDA::putTrue() {putNull(); cRaw = "+"; cBool = true; return this;}
IDA* IDA::putFalse() {putNull(); cRaw = "-"; cBool = false; return this;}
IDA* IDA::putBool(bool b) {return b ? putTrue() : putFalse();}
bool IDA::canBool()
{
	if (isBool() || isNum()) return true;
	std::string s = this->toStrOr("");
	return s=="true" || s=="false";
}
bool IDA::toBool()
{
	if (isBool()) return isTrue();
	if (isNum()) return asDoubleOr(0.0) != 0.0;
	std::string s = this->toStrOr("");
	if (s=="true") return true;
	if (s=="false") return false;
	throw IDAOpException("Content cannot cast to a boolean value");
}
bool IDA::toBoolOr(bool other)
{
	try {return toBool();}
	catch (...) {return other;}
}

// Content/null
bool IDA::isNull() {return !this->cRaw.has_value();}
bool IDA::hasContent() {return this->cRaw.has_value();}
IDA* IDA::putNull() {return this->clearContent();}
IDA* IDA::clearContent()
{
	this->cRaw = std::nullopt;
	this->cStr = std::nullopt;
	this->cBool = std::nullopt;
	this->clearRef();
	this->clearList();
	this->clearNum();
	return this;
}

// Content/string
IDA* IDA::putStr(std::optional<std::string> s) {putNull(); this->cRaw = s; this->cStr = s; return this;}
bool IDA::isStr() {return cStr.has_value();}
std::string IDA::toString() {return cRaw.value_or("?");}
std::string IDA::asStr()
{
	if (!cStr.has_value()) throw IDAOpException("Content is not a string value");
	return cStr.value();
}
std::optional<std::string> IDA::asStrOr()
{
	return cStr;
}
std::string IDA::asStrOr(std::string other)
{
	return this->cStr.has_value() ? this->cStr.value() : other;
}
bool IDA::canStr()
{
	return hasContent() && !isList() && !isRef();
}
std::string IDA::toStr()
{
	if (!canStr()) throw IDAOpException("Content cannot cast to a string value");
	return cRaw.value();
}
std::optional<std::string> IDA::toStrOr()
{
	return canStr() ? cRaw : std::nullopt;
}
std::string IDA::toStrOr(std::string other)
{
	return canStr() ? cRaw.value() : other;
}

// Content/number/other
void IDA::clearNum() {this->cNum = ""; this->cDen = "1"; this->cUnit = "";}

// Content/number/set
void IDA::refresh_num() {cRaw = cNum + (IDA::eq_str(cDen, "1") ? "" : "/"+cDen) + cUnit; cStr = std::nullopt;}
IDA* IDA::setNumUnit(std::optional<std::string> unit)
{
	if (unit.has_value())
	{
		if (!this->isNum()) throw IDAOpException("Content is not numeric");
		cUnit = unit.value(); refresh_num();
	}
	return this;
}

IDA* IDA::putNum(std::string num, std::string den, std::string unit) {putNull(); cNum = num; cDen = den; cUnit = unit; refresh_num(); return this;}
IDA* IDA::putNum(std::string num, std::string den) {return putNum(num, den, cUnit);}
IDA* IDA::putNum(std::string num) {return putNum(num, "1");}

IDA* IDA::putDouble(double num, std::optional<std::string> unit)
{
	std::ostringstream buf;
	buf << num;
	return putNum(buf.str())->setNumUnit(unit);
}
IDA* IDA::putFloat(float num, std::optional<std::string> unit)
{
	std::ostringstream buf;
	buf << num;
	return putNum(buf.str())->setNumUnit(unit);
}
IDA* IDA::putLong(long long num, std::optional<std::string> unit) {return putNum(std::to_string(num))->setNumUnit(unit);}
IDA* IDA::putInt(long num, std::optional<std::string> unit) {return putNum(std::to_string(num))->setNumUnit(unit);}

// Content/number/get
bool IDA::isNum()
{
	return !cNum.empty();
}
std::string IDA::getNumUnit()
{
	if (!isNum()) throw IDAOpException("Content is not numeric");
	return cUnit;
}
std::string IDA::getNumN()
{
	if (!isNum()) throw IDAOpException("Content is not numeric");
	return cNum;
}
std::string IDA::getNumD()
{
	if (!isNum()) throw IDAOpException("Content is not numeric");
	return cDen;
}
double IDA::asDoubleOr(double other)
{
	return isNum() ? c_double() : other;
}
float IDA::asFloatOr(float other)
{
	return isNum() ? c_float() : other;
}
long long IDA::asLongOr(long long other)
{
	return isNum() ? c_long() : other;
}
long IDA::asIntOr(long other)
{
	return isNum() ? c_int() : other;
}
double IDA::asDouble()
{
	if (!isNum()) throw IDAOpException("Content is not numeric");
	return c_double();
}
float IDA::asFloat()
{
	if (!isNum()) throw IDAOpException("Content is not numeric");
	return c_float();
}
long long IDA::asLong()
{
	if (!isNum()) throw IDAOpException("Content is not numeric");
	return c_long();
}
long IDA::asInt()
{
	if (!isNum()) throw IDAOpException("Content is not numeric");
	return c_int();
}

double IDA::c_double()
{
	return IDA::eq_str(cDen, "1") ? n_double(cNum) : n_double(cNum)/n_double(cDen);
}
float IDA::c_float()
{
	return (float) c_double();
}
long long IDA::c_long()
{
	return IDA::eq_str(cDen, "1") ? n_long(cNum) : (long long)(std::stod(cNum)/std::stod(cDen));
}
long IDA::c_int()
{
	return IDA::eq_str(cDen, "1") ? (long)n_long(cNum) : (long)(std::stod(cNum)/std::stod(cDen));
}
long long IDA::n_long(std::string str)
{
	if (IDA::port->starts_with(str, "0x")) {return n_hex(str);}
	if (IDA::port->starts_with(str, "0b")) {return n_bin(str);}
	try
	{
		std::size_t n = 0;
		long long ll = std::stoll(str, &n);
		return n < str.length() ? (long long) std::stod(str) : ll;
	}
	catch (...) {return (long long) std::stod(str);}
}
double IDA::n_double(std::string str)
{
	if (IDA::port->starts_with(str, "0x")) {return (double) n_hex(str);}
	if (IDA::port->starts_with(str, "0b")) {return (double) n_bin(str);}
	return std::stod(str);
}
long long IDA::n_hex(std::string str) {return std::stoll(str.substr(2), nullptr, 16);}
long long IDA::n_bin(std::string str) {return std::stoll(str.substr(2), nullptr, 2);}

// Content/list
void IDA::clearList()
{
	for (IDA* x : cList) {x->link_break(); delete x;}
	cList.clear(); cList.shrink_to_fit(); cList_ok = false;
}
void IDA::refresh_list() {cRaw = "List("+std::to_string(this->numItems())+")"; cStr = std::nullopt;}

std::vector<IDA*>::iterator IDA::iterator() {return cList.begin();}

// Content/list/set
IDA* IDA::putList()
{
	putNull();
	cList_ok = true;
	refresh_list();
	return this;
}
IDA* IDA::putList(std::optional<std::vector<IDA*>> l)
{
	putNull();
	if (l.has_value())
	{
		putList();
		//cList = l.value(); refresh_list();
		IDA* last = nullptr;
		for (IDA* item : l.value())
		{
			cList.push_back(item);
			item->link(last, nullptr);
			last = item;
		}
		refresh_list();
	}
	return this;
}

// Content/list/get
bool IDA::isList()
{
	return cList_ok;
}
std::vector<IDA*>* IDA::asList()
{
	if (!cList_ok) throw IDAOpException("Content is not a list");
	return &cList;
}
std::vector<IDA*>* IDA::asListOr()
{
	return &cList;
}
std::vector<IDA*>* IDA::asListOr(std::vector<IDA*>* other)
{
	return isList() ? &cList : other;
}
long IDA::numItems()
{
	return isList() ? cList.size() : -1;
}

bool IDA::hasItem(std::optional<std::string> key) {return getItemOr(key) != nullptr;}

IDA* IDA::getItemAt(long index)
{
	if (numItems() <= index) throw IDAOpException("Invalid list item index");
	return cList[index];
}
IDA* IDA::getItemAtOr(long index)
{
	return getItemAtOr(index, nullptr);
}
IDA* IDA::getItemAtOr(long index, IDA* other)
{
	return numItems() > index ? cList[index] : other;
}

IDA* IDA::getItem(std::optional<std::string> key)
{
	IDA* item = getItemOr(key);
	if (item == nullptr) throw IDAOpException("Item not found");
	return item;
}
IDA* IDA::getItemOr(std::optional<std::string> key)
{
	return getItemOr(key, nullptr);
}
IDA* IDA::getItemOr(std::optional<std::string> key, IDA* other)
{
	if (isList()) for (IDA* x : cList) if (x->isName(key)) return x;
	return other;
}

std::string IDA::getItemStr(std::optional<std::string> key)
{
	return getItem(key)->toStr();
}
std::optional<std::string> IDA::getItemStrOr(std::optional<std::string> key)
{
	IDA* item = getItemOr(key);
	return item != nullptr ? item->toStrOr() : std::nullopt;
}
std::string IDA::getItemStrOr(std::optional<std::string> key, std::string other)
{
	IDA* item = getItemOr(key);
	return item != nullptr ? item->toStrOr(other) : other;
}

IDA* IDA::getItemFirst()
{
	if (numItems() < 1) throw IDAOpException(isList() ? "Content list is empty" : "Element content is not a list");
	return cList.front();
}
IDA* IDA::getItemFirstOr()
{
	return getItemFirstOr(nullptr);
}
IDA* IDA::getItemFirstOr(IDA* other)
{
	return numItems() > 0 ? cList.front() : other;
}
IDA* IDA::getItemLast()
{
	if (numItems() < 1) throw IDAOpException(isList() ? "Content list is empty" : "Element content is not a list");
	return cList.back();
}
IDA* IDA::getItemLastOr()
{
	return getItemLastOr(nullptr);
}
IDA* IDA::getItemLastOr(IDA* other)
{
	return numItems() > 0 ? cList.back() : other;
}

// Content/list/add
IDA* IDA::addItem(IDA* el, bool makeList)
{
	if (!isList() && !makeList) {throw IDAOpException("Cannot append items onto a non-list");}
	if (!isList()) putList();
	el->link(getItemLastOr(), nullptr);
	cList.push_back(el);
	refresh_list();
	return this;
}

IDA* IDA::addItemAt(IDA* el, int pos)
{
	if (!isList()) {throw IDAOpException("Cannot merge items into a non-list");}
	IDA::list_add(&cList, el, pos, "item");
	refresh_list();
	return this;
}

IDA* IDA::newItem(std::optional<std::string> name, bool makeList)
{
	IDA* el = new IDA(name);
	try {addItem(el, makeList);}
	catch (IDAOpException ex) {delete el; throw ex;}
	return el;
}

// Content/list/remove
void IDA::remItemAt(int pos)
{
	if (!isList()) {throw IDAOpException("Cannot remove items from a non-list");}
	delete IDA::list_del(&cList, pos, "item");
	refresh_list();
}
bool IDA::remItem(IDA* el)
{
	if (isList() && IDA::list_rem(&cList, el)) {refresh_list(); return true;}
	return false;
}
bool IDA::unItem(std::optional<std::string> name)
{
	IDA* item = getItemOr(name);
	if (item != nullptr && isList() && IDA::list_rem(&cList, item)) {delete item; refresh_list(); return true;}
	return false;
}

// Object/equals
bool IDA::eq_inner(IDA* obj)
{
	if (obj == nullptr) {return false;}
	if (obj == this) {return true;}
	if (!this->eqName(obj)) {return false;}
	if (!this->eqParams(obj)) {return false;}
	if (!this->eqContent(obj)) {return false;}
	return true;
}
bool IDA::eq(IDA* obj, bool adjacent)
{
	if (obj == this) return true;
	if (obj == nullptr) return false;
	if (this->txRefLabel(adjacent) != obj->txRefLabel(adjacent)) return false;
	if (!eq_inner(obj)) return false;
	
	if (adjacent)
	{
		IDA* a = this;
		IDA* b = obj;
		IDA* el = b;
		while (a->hasPrev() && b->hasPrev())
		{
			a = a->getPrevOr(); b = b->getPrevOr();
			if (!a->eq_inner(b)) {return false;}
		}
		if (a->hasPrev() || b->hasPrev()) {return false;}
		
		a = this;
		b = el;
		while (a->hasNext() && b->hasNext())
		{
			a = a->getNextOr(); b = b->getNextOr();
			if (!a->eq_inner(b)) {return false;}
		}
		if (a->hasNext() || b->hasNext()) {return false;}
	}
	
	return true;
}

bool IDA::eqName(IDA* obj) {return isName(obj->getNameOr());}
bool IDA::eqContent(IDA* obj)
{
	if (!IDA::eq_str(this->cRaw, obj->cRaw)) {return false;}
	if (!IDA::eq_str(this->cStr, obj->cStr)) {return false;}
	if (!IDA::eq_str(this->cNum, obj->cNum)) {return false;}
	//if (!IDA::eq_str(this->cDen, obj->cDen)) {return false;} // Assured by testing cRaw
	//if (!IDA::eq_str(this->cUnit, obj->cUnit)) {return false;} // Assured by testing cRaw
	if (!IDA::list_equals(this->asListOr(), obj->asListOr())) {return false;}
	if (!IDA::eq_bool(this->cBool, obj->cBool)) {return false;}
	if (!eqContent_ref(cRef, obj->cRef)) return false;
	
	return true;
}
bool IDA::eqContent_ref(IDA* ref_a, IDA* ref_b)
{
	if ((ref_a != nullptr) ^ (ref_b != nullptr)) return false;
	if (ref_a == ref_b) return true;
	if (ref_a->hasRefName() ^ ref_b->hasRefName()) return false;
	if (!ref_a->hasRefName()) return true;
	return *ref_a->getRefName() == *ref_b->getRefName();
}
bool IDA::eqParams(IDA* obj)
{
	return IDA::list_equals(&this->parameters, &obj->parameters);
}


// Object/capture
IDA* IDA::objCapture(IDAObj* obj)
{
	this->objEncode(obj);
	
	std::map<IDAObj*, IDA*> m;
	capture_objs(&m);
	capture_refs(&m);
	
	return this;
}
IDA* IDA::objEncode(IDAObj* obj)
{
	this->setName(obj->idaName());
	repObj = obj;
	obj->idaEncode(this);
	
	return this;
}
void IDA::capture_objs(std::map<IDAObj*, IDA*>* objs)
{
	if (repObj != nullptr) {(*objs)[repObj] = this;}
	if (hasParams()) for (IDA* p : parameters) p->capture_objs(objs);
	if (isList()) for (IDA* i : cList) i->capture_objs(objs);
}
void IDA::capture_refs(std::map<IDAObj*, IDA*>* refs)
{
	if (cRefObj != nullptr) {putRef((*refs)[cRefObj]);}
	if (hasParams()) for (IDA* p : parameters) p->capture_refs(refs);
	if (isList()) for (IDA* i : cList) i->capture_refs(refs);
}

// Object/restore
void IDA::on_rep_obj(std::function<void(IDAObj*)> callback)
{
	if (repObj != nullptr) {callback(repObj);}
	else
	{
		repObjActions.push_back(callback);
	}
}
void IDA::restore_check()
{
	if (!repObjActions.empty()) throw IDAOpException("Decode process failed to resolve an object reference");
}
template <typename T> T* IDA::objRestore(T* obj)
{
	objDecode(obj);
	this->walk([](IDA* x){x->restore_check();}, true, true);
	return obj;
}
template <typename T> T* IDA::objDecode(T* obj)
{
	repObj = obj;
	if (!repObjActions.empty())
	{
		for (std::function<void(IDAObj*)> c : repObjActions) c(obj);
		repObjActions.clear();
	}
	obj->idaDecode(this);
	return obj;
}

// Deserialisation
IDA* IDA::inStream(std::istream* input)
{
	return IDA_RX::inObj(this, input);
}
IDA* IDA::inBytes(std::string input)
{
	return inString(input);
}
IDA* IDA::inString(std::string input)
{
	std::istringstream s(input);
	return IDA_RX::inObj(this, &s);
}
IDA* IDA::inFile(std::string path)
{
	std::ifstream input;
	input.open(path, std::ios_base::in);
	// TODO: test input.is_open() ?
	inStream(&input);
	input.close();
	return this;
}

// Serialisation
void IDA::outStream(std::ostream* output, bool adjacents)
{
	IDA_TX tx(output);
	tx.idaWrite(this, adjacents);
}
bool IDA::outPrint(std::ostream* output, bool adjacents)
{
	IDA_TX tx(output);
	return tx.idaPrint(this, adjacents);
}

std::string IDA::outBytes(bool adjacents)
{
	return outString(adjacents);
}
std::string IDA::outString(bool adjacents)
{
	std::ostringstream buf = std::ostringstream("");
	outStream(&buf, adjacents);
	return buf.str();
}

void IDA::outFile(std::string path, bool adjacents)
{
	std::ofstream res(path, std::ios::binary);
	outStream(&res, adjacents);
}
void IDA::outFile(std::filesystem::path path, bool adjacents)
{
	std::ofstream res(path, std::ios::binary);
	outStream(&res, adjacents);
}

// Construction
#ifdef IDA_DEBUG_MEMORY
IDA_Debug_Memory<IDA>* IDA::dmem = new IDA_Debug_Memory<IDA>("IDA");
#endif
IDA::IDA(std::optional<std::string> name)
{
	this->includePwd = gIncludePwd;
	this->reset();
	this->setName(name);
	
	#ifdef IDA_DEBUG_MEMORY
	this->dmem->c(this);
	#endif
}
IDA::~IDA()
{
	rxClearTemplates();
	rxClearItemClasses();
	
	// Global scope is freed during inObj, and child scopes are the elements being freed now
	//if (scope != nullptr) delete scope;
	
	if (next != nullptr)
	{
		next->prev = nullptr;
		delete next;
	}
	
	if (prev != nullptr)
	{
		prev->next = nullptr;
		delete prev;
	}
	
	clearParams();
	clearList();
	
	#ifdef IDA_DEBUG_MEMORY
	this->dmem->d(this);
	#endif
}

// Initialisation
IDA* IDA::reset()
{
	putNull();
	cRefName.reset();
	repObj = nullptr;
	repObjActions.clear();
	
	rxClearItemClasses();
	name = std::nullopt;
	next = NULL;
	prev = NULL;
	clearParams();
	
	opts->copyFrom(IDAOptions::defaultOptions);
	scope = nullptr;
	rxClearTemplates();
	
	inputListDepth = -1;
	inputListLength = -1;
	
	return this;
}

// Copies
IDA* IDA::copy(bool adjacent) {return copyTo(new IDA(), adjacent);}
IDA* IDA::copyTo(IDA* copy, bool adjacent)
{
	bool has_ref = txRefLabel(adjacent) > 0;
	copy_impl(this, copy, adjacent, adjacent);
	if (has_ref) copy->rxRefResolve(adjacent);
	return copy;
}

IDA* IDA::copy_inner()
{
	return copy_impl(this, new IDA(), false, false);
}
IDA* IDA::copy_impl(IDA* orig, IDA* copy, bool scanPrev, bool scanNext)
{
	copy->copyName(orig);
	copy->copyParams(orig);
	copy->copyContent(orig);
	copy->copyOptions(orig);
	
	IDA* a;
	if (scanPrev && (a = orig->getPrevOr()) != nullptr) {copy->linkPrev(copy_impl(a, new IDA(), true, false));}
	if (scanNext && (a = orig->getNextOr()) != nullptr) {copy->linkNext(copy_impl(a, new IDA(), false, true));}
	
	return copy;
}

// Utility
void IDA::walk(std::function<void(IDA*)> func, bool rev, bool fwd)
{
	func(this);
	if (hasParams()) for (IDA* p : parameters) p->walk(func, false, false);
	if (isList()) for (IDA* i : cList) i->walk(func, false, false);
	if (fwd && hasNext()) getNextOr()->walk(func, false, true);
	if (rev && hasPrev()) getPrevOr()->walk(func, true, false);
}

void IDA::validate(bool adjacents)
{
	validate_loop(adjacents);
	validate_foreign(adjacents);
}
void IDA::validate_loop(bool adjacents)
{
	if (!adjacents) return;
	std::function<IDA*(IDA*)> fwd = [](IDA* x){return x->getNextOr();};
	std::function<IDA*(IDA*)> rev = [](IDA* x){return x->getPrevOr();};
	validate_loop_f(fwd, rev);
	validate_loop_f(rev, fwd);
}
void IDA::validate_loop_f(std::function<IDA*(IDA*)> step, std::function<IDA*(IDA*)> back)
{
	IDA* a = this;
	IDA* b = this;
	
	while (true)
	{
		IDA* last = b;
		b = step(b);
		if (b == nullptr) return;
		if (back(b) != last) throw IDAOpException("Article contains a broken link");
		
		last = b;
		b = step(b);
		if (b == nullptr) return;
		if (back(b) != last) throw IDAOpException("Article contains a broken link");
		
		a = step(a);
		if (a == b) throw IDAOpException("Article contains an adjacency loop");
	}
}
void IDA::validate_foreign(bool adjacents)
{
	std::vector<IDA*> foreign;
	walk([&foreign](IDA* x){if (x->isRef()) if (!IDA::port->list_contains(&foreign, x->asRefOr())) foreign.push_back(x->asRefOr());}, adjacents, adjacents);
	walk([&foreign](IDA* x){if (IDA::port->list_contains(&foreign, x)) IDA::port->list_rem(&foreign, x);}, adjacents, adjacents);
	if (!foreign.empty()) throw IDAOpException("Article contains a foreign reference");
}


void IDA::list_add(std::vector<IDA*>* list, IDA* el, long pos, std::string noun)
{
	const unsigned long size = list->size();
	
	if (pos > 0)
	{
		if (pos > size) {throw IDAOpException("Cannot add "+noun+" beyond position "+std::to_string(size)+" (request was "+std::to_string(pos)+")");}
	}
	else if (pos < 0)
	{
		if (pos < -size) {throw IDAOpException("Cannot add "+noun+" before position -"+std::to_string(size)+" (request was "+std::to_string(pos)+")");}
		pos += size;
	}
	
	if (pos == 0)
	{
		auto i = list->begin();
		if (!list->empty()) {el->link(nullptr, *i);}
		list->insert(i, el);
	}
	else if (pos == size)
	{
		el->link((*list)[size-1], nullptr);
		list->push_back(el);
	}
	else // size must be at least 2, pos cannot be at either end
	{
		auto i = list->begin() + pos;
		el->link(*(i-1), *i);
		list->insert(i, el);
	}
}
IDA* IDA::list_del(std::vector<IDA*>* list, long pos, std::string noun)
{
	const unsigned long size = list->size();
	
	if (pos > 0)
	{
		if (pos > size-1) {throw IDAOpException("Cannot remove item beyond position "+std::to_string(size-1)+" (request was "+std::to_string(pos)+")");}
	}
	else if (pos < 0)
	{
		if (pos < -size) {throw IDAOpException("Cannot remove item before position -"+std::to_string(size)+" (request was "+std::to_string(pos)+")");}
		pos += size;
	}
	
	auto i = list->begin() + pos;
	IDA* el = *i;
	list->erase(i);
	return el->link(nullptr, nullptr);
}

// Syntax
bool IDA::id0(char c) {return (c>='a' && c<='z') || (c>='A' && c<='Z') || c == '_' || c == '/';}
bool IDA::id1(char c)
{
	return IDA::id0(c) || (c>='0' && c<='9') || c == '-' || c == '.'
		|| c == '!' || c == '@' || c == '^' || c == '&' || c == '?';
}
bool IDA::comment1(char c) {return c == '#';}
bool IDA::comment2(char c1, char c2)
{
	return (c1 == '/' && (c2 == '/' || c2 == '*'))
		|| (c1 == '<' && (c2 == '?' || c2 == '!'));
}

// Compatibility/port through
IDA_Port* IDA::port = new IDA_Port();

// Compatibility
bool IDA::eq_obj(IDA* a, IDA* b)
{
	return a == nullptr ? b == nullptr : a->eq(b);
}
bool IDA::eq_obj(std::optional<std::string> a, std::optional<std::string> b)
{
	if (!a.has_value()) return !b.has_value();
	return b.has_value() && a.value() == b.value();
}

bool IDA::list_rem(std::vector<IDA*>* list, IDA* el)
{
	for (auto i = list->begin(); i != list->end(); i++)
	{
		IDA* x = *i;
		if (IDA::eq_obj(el, x))
		{
			x->link(nullptr, nullptr);
			list->erase(i);
			if (el != x) delete x;
			return true;
		}
	}
	return false;
}
bool IDA::list_equals(std::vector<IDA*>* la, std::vector<IDA*>* lb)
{
	if (la == nullptr) return lb == nullptr;
	if (lb == nullptr) return false;
	auto ia = la->begin();
	auto ib = lb->begin();
	auto ia_end = la->end();
	auto ib_end = lb->end();
	while ((ia != ia_end) && (ib != ib_end))
	{
		if (!(*ia++)->eq(*ib++)) {return false;}
	}
	return !((ia != ia_end) || (ib != ib_end));
}

IDA_Iterator* IDA_RX::CLASS_NONE = IDA::port->empty_i();
IDA* repeat_node = new IDA();
IDA_Iterator* IDA_RX::CLASS_ARRAY = IDA::port->repeat_i(&repeat_node);

#ifdef IDA_MAIN
int main(int argc, char** argv)
{
	return IDA_CLI::cli(argc-1, argv+1);
}
#endif

