//    Copyright 2022 IDA Markup, ida-markup.com
// 
//    Licensed under the Apache License, Version 2.0 (the "License");
//    you may not use this file except in compliance with the License.
//    You may obtain a copy of the License at
// 
//        http://www.apache.org/licenses/LICENSE-2.0
// 
//    Unless required by applicable law or agreed to in writing, software
//    distributed under the License is distributed on an "AS IS" BASIS,
//    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//    See the License for the specific language governing permissions and
//    limitations under the License.

// Namespace

class IDA_Manip;
class IDA_Output;
class IDA_Port;
class IDA_RX;
class IDA_TX;
class IDA;
class IDAFormatException;
class IDAOpException;
class IDAOptions;
class IDAObj;

#ifdef IDA_DEBUG_MEMORY
template <typename X>
class IDA_Debug_Memory
{
	public:
	std::string typeName;
	long creations = 0;
	long deletions = 0;
	std::vector<X*> objs;
	long total_unrecovered = 0;
	
	IDA_Debug_Memory(std::string typeName)
	{
		this->typeName = typeName;
	}
	void c(X* obj)
	{
		creations += 1;
		if (obj != nullptr) objs.push_back(obj);
	}
	void d(X* obj)
	{
		deletions += 1;
		
		for (auto i = objs.begin(); i != objs.end(); i++)
		{
			if (*i == obj) {objs.erase(i); break;}
		}
	}
	void reset(bool gc)
	{
		total_unrecovered += (creations - deletions);
		creations = deletions = 0;
		
		if (gc) while (!objs.empty())
		{
			auto i = objs.begin();
			objs.erase(i);
			delete *i;
		}
		else objs.clear();
	}
	void init() {total_unrecovered = creations = deletions = 0; objs.clear();}
	bool ok() {return creations <= deletions;}
	bool nok() {return creations > deletions;}
	void report(std::ostream& s)
	{
		s << " " << typeName << " " << deletions << "/" << creations << " ";
	}
};
#endif

// Options
class IDAOptions
{
	public:
	IDA* owner = nullptr;
	
	// RX
	public:
	bool rxXmlReparseStrings = false;
	
	// TX
	public:
	bool txUnicode = false;
	
	IDAOptions();
	IDAOptions(IDAOptions* options);
	
	void copyFrom(IDAOptions* options);
	
	static IDAOptions* defaultOptions;
	
	#ifdef IDA_DEBUG_MEMORY
	~IDAOptions();
	public:
	static IDA_Debug_Memory<IDAOptions>* dmem;
	#endif
};

// Object interface
class IDAObj
{
	public:
	/** Override to offer a default element name. */
	virtual std::optional<std::string> idaName();
	/** Override to enable capturing of this object into the provided storage element.<br>
	 * Usage note: If applicable, implementing classes can guarantee not to throw.<br>
	 * @return The input element. */
	virtual IDA* idaEncode(IDA* el);
	/** Override to enable restoration of this object from the provided storage element.<br>
	 * @return This object. */
	virtual IDAObj* idaDecode(IDA* el);
};

// Port begin

class IDA_Iterator
{
	public:
	IDA_Iterator();
	bool hasNext();
	IDA* getNext();
	IDA* next();
	
	std::function<bool()> funcHas;
	std::function<IDA*()> funcGet;
	
	bool singleton = false;
	long n = 0;
	IDA* buf = nullptr;
	IDA_Iterator* lbuf = nullptr;
	bool more = false;
	std::function<void()> preLoad = [](){};
	IDA* node = nullptr;
	
	#ifdef IDA_DEBUG_MEMORY
	~IDA_Iterator();
	static IDA_Debug_Memory<IDA_Iterator>* dmem;
	#endif
};
class IDA_Port
{
	
	// Port iterators
	// Iterator/MutableSeq in Port begin
	public:
	IDA_Iterator* empty_i();
	IDA_Iterator* list_i(std::vector<IDA*>* l);
	IDA_Iterator* repeat_i(IDA** obj);
	
	// Port arrays
	public:
	bool list_contains(std::vector<IDA*>* l, IDA* item);
	bool list_rem(std::vector<IDA*>* l, IDA* item);
	
	// Port streams
	public:
	bool si_allMatch(IDA_Iterator* i, std::function<bool(IDA*)> predicate);
	bool si_anyMatch(IDA_Iterator* i, std::function<bool(IDA*)> predicate);
	bool si_noneMatch(IDA_Iterator* i, std::function<bool(IDA*)> predicate);
	
	// Port string replace
	public:
	std::string replace_all(std::string s, std::string act, std::string dmd);
	
	std::string substring(std::string s, long fromChar, long toChar);
	std::string substring(std::string s, long fromChar);
	
	std::string trim(std::string s);
	std::string rtrim(std::string s);
	std::string ltrim(std::string s);
	
	// Port string tests
	public:
	bool ends_with(std::string s, std::string suffix);
	bool starts_with(std::string s, std::string prefix);
	bool str_contains(std::string s, std::string mid);
	
	// Port string info
	public:
	/** Character index */
	long str_index(std::optional<std::string> s, std::optional<std::string> mid);
	/** Character index */
	long str_index_from(std::optional<std::string> s, std::optional<std::string> mid, long fromChar);
	bool is_continuation(unsigned char c);
	/** The number of characters represented */
	std::string::size_type str_len(std::optional<std::string> s);
	/** The number of characters represented */
	std::string::size_type str_chars(std::optional<std::string> s, long fromUnit, long toUnit);
	/** The number of memory locations */
	std::string::size_type str_units(std::optional<std::string> s, long fromChar, long toChar);
	
	// Port hex
	public:
	std::string char_2(unsigned char c);
	std::string char_4(unsigned long c);
	std::string char_8(unsigned long c);

// Port end
};

// Format exception
class IDAFormatException : public std::exception
{
	private:
	std::string msg;
	public:
	std::string getMessage() {return msg;}
	
	// Syntax
	std::string ln;
	long pos;
	
	// Structure
	IDA* node;
	
	IDAFormatException(IDA_RX* rx, std::string message);
	IDAFormatException(IDA* nodeCause, std::string message);
	virtual const char* what();
	
};

// Op exception
class IDAOpException : public std::exception
{
	private:
	std::string msg;
	public:
	std::string getMessage() {return msg;}
	IDAOpException(std::string msg)
	{
		this->msg = msg;
	}
	virtual const char* what()
	{
		return this->msg.c_str();
	}
};

// Manip begin
class IDA_Manip
{
	static const long FORWARD = 2147483647;
	static const long REVERSE = -2147483647;
	static long rev(long n);
	
	static std::optional<std::string> m_str(IDA* el);
	
	// Manip/iterate
	/** Return the last node */
	public:
	static IDA* manip_iterate(IDA* manip, IDA* node, IDA_Iterator* itemTemplate);
	
	// Manip/negate
	static void manip_negate(IDA* node);
	
	// Manip/insert
	/** Usage note: the 'ins' string is never null, and the pos can be REVERSE in place of a null to mean the end (never FORWARD though) */
	static std::string manip_ins(std::optional<std::string> orig_opt, std::string ins, long pos);
	
	// Manip/delete
	/** Usage note: the pos can be REVERSE in place of a null to mean the end (never FORWARD though) */
	static std::string manip_del(std::optional<std::string> orig_opt, long pos, long amt);
	
	// Manip/add
	static void manip_add(std::function<void(IDA*,long)> func, long pos, IDA* nodeFrom);
	
	// Manip/rem
	static void manip_rem_pos(IDA* opNode, std::function<void(long)> func, std::function<long()> size, long pos, long amt);
	static void manip_rem_item(std::function<void(IDA*)> func, IDA* nodeFrom);
	
// Manip end
};

// RX begin
class IDA_RX
{
	protected:
	std::map<std::string,IDA*> refs;
	bool has_refTarget = false;
	
	// Data source
	private:
	std::istream* inStr;
	public:
	//IDA_RX(std::optional<std::istream> inStr = std::nullopt);
	IDA_RX(std::istream* inStr);
	//~IDA_RX();
	
	// RX basic input
	// InputStream
	private:
	char c1, c2;
	bool has_c1 = false, has_c2 = false, eof = false;
	
	public:
	int ln = 1;
	std::string lnBuf = "";
	long read();
	
	private:
	void buf(char c);
	/** Assumes has_c1 */
	char read1();
	/** Assumes has_c2 */
	void skip2();
	public:
	/** Check has_c1 first */
	char shift();
	private:
	void more();
	/** Returns eof */
	bool more1();
	/** Returns eof */
	bool more2();
	
	// RX stream advancement
	bool comment();
	void adv();
	void adv(bool checkLn);
	void advPast(char c);
	void advPast(char cx, char cy);
	
	bool abrupt(bool checkLn);
	
	// RX scan bytes
	public:
	bool c1e(unsigned char c);
	bool c1n(unsigned char c);
	bool c2e(unsigned char c);
	bool c2n(unsigned char c);
	bool whiteLn(unsigned char c);
	bool whiteTxt(unsigned char c);
	bool white(unsigned char c);
	
	// RX scan numbers
	static bool num0(char c);
	static bool num1(char c);
	std::string get_num();
	static bool hex(char c);
	static bool bin(char c);
	
	// RX scan opts
	bool opt();
	/** Consumes the /, reads the opt, then advances */
	std::string get_opt();
	
	// RX scan unicode
	public:
	static bool is_u8_marker(unsigned char c);
	static bool is_u8_data(unsigned char c);
	
	unsigned char shift_u8_data();
	std::string get_u8();
	std::string to_u8(unsigned long codePoint);
	
	// RX scan escapes
	static bool is_esc(char c);
	bool esc();
	
	int shift_hex(std::string txt);
	int get_esc_byte(std::string txt);
	std::string get_esc();
	
	// RX scan IDs
	bool id0();
	bool id1();
	std::string get_tok();
	
	bool id_symbolic = false;
	std::string get_id(bool untilLn = false);
	static bool is_id_start(char c);
	bool c1_id_start();
	
	static bool is_var(char c);
	std::string get_var();
	
	std::string xml_att_key();
	
	// RX scan strings
	bool c1_str();
	bool c1_tok();
	static bool is_str_d(char c);
	static bool is_str(char c);
	
	std::string get_str();
	std::string get_str_d(char delim);
	
	// RX scan logic
	public:
	std::string get_pr(std::function<bool(char)> p);
	std::string get_prior_to(char delim);
	std::string get_to(char delim);
	
	// RX class iteration
	public:
	// Empty iterator
	static IDA_Iterator* CLASS_NONE;
	// Infinite iterator of blank nodes
	static IDA_Iterator* CLASS_ARRAY;
	// Specific class node iterator
	static IDA_Iterator* classOf(std::vector<IDA*>* itemClassList);
	
	// RX entry
	/** Returns the output argument */
	static IDA* inObj(IDA* output, std::istream*inStr);
	void transfer_nodes(IDA* src, IDA* dst);
	void put_ref(IDA* el);
	
	// RX fill
	/** Decodes element nodes read from the RX, using the supplied object as a container for the first such element.
	 * Adjacent elements will be created as needed and linked to the first, accessible using its getNext method.<br>
	 * @return The last consecutive element decoded, or null if no elements were present.
	 * @param node A container into which the first decoded element should be written.
	 * @param rx The serial input to be read.
	 * @param paren_l The character which caused <i>fill</i> to be called recursively, or zero to admit no such structure.
	 * @param paren_r The character which should mark the end of a recursive, structured call to <i>fill</i>.
	 * @param prev An element to be considered prior to the decoded element for the purpose of interpreting <i>as-above</i>
	 * markup, or null if no such element exists.
	 * @param listClass A class element to use for copying implicit element names, or an empty iterator (such as CLASS_NONE).
	 * @param depth The effective recursion level.
	 * @throws IOException If a error occurs while reading input bytes.
	 * @throws IDAFormatException If the parser encounters a syntax error, or if the markup contained an invalid
	 * <i>manipulation</i> which would throw {@link IDAOpException} if called programmatically. */
	private:
	static IDA* fill(IDA* node, IDA_RX* rx, char paren_l, char paren_r, IDA* prev, IDA_Iterator* listClass, long depth);
	
// TODO: RX discard
	// RX number test
	public:
	static std::regex p_num;
	
	void testNumber(std::string num, std::string desc);
	
	// RX manip
	/** Return the last node */
	static IDA* manip(IDA_RX* rx, IDA* node, IDA_Iterator* itemTemplate);
	
// RX end
};

// Output interface
class IDA_Output
{
	private:
	std::ostream* w;
	
	public:
	IDA_Output(std::ostream* output);
	
	protected:
	void send(std::string s);
	
	private:
	std::string queueBuf = "";
	protected:
	void queue(std::string s);
	long length();
	void reject();
	void accept();
	
	
	/** Element will not be null<br>
	 * This will receive the entry element, which is not necessarily the first to be output<br>
	 * The adjacents-flag is provided if any special state is needed to handle multiple elements to follow */
	protected:
	virtual void ida_header(IDA* el, bool adj) {}
	/** Element will not be null<br>
	 * This is called potentially multiple times if adjacents are enabled */
	virtual void ida_element(IDA* el, bool adj) {}
	/** Called once after all elements are done */
	virtual void ida_footer() {}
	
	public:
	void idaWrite(IDA* el, bool adj);
	
	/** Same as idaWrite, but converts exceptions into a return value instead.<br>
	 * True if success, false if a IO problem occurred */
	bool idaPrint(IDA* el, bool adj);
	
};

// TX begin
class IDA_TX : public IDA_Output
{
	
	// TX defaults
	private:
	static const long BREVITY_ALL = 3;
	static const long BREVITY_PART = 6;
	static const long ONE_LINE_MAX = 132;
	
	private:
	IDA* entry = nullptr;
	bool rootAsAboveAll = false;
	IDA* rootAsAboveFirst = nullptr;
	bool rootAsAbovePart = false;
	std::optional<std::string> rootAsAboveName = std::nullopt;
	
	private: bool optsOverride = false;
	public: IDA_TX* setOptsOverride(bool ov);
	private: IDAOptions* opts(IDA* el);
	
	public:
	IDA_TX(std::ostream* output);
	
	// TX interface
	protected:
	void ida_header(IDA* el, bool adj);
	void ida_element(IDA* el, bool adj);
	
	// TX structure filter
	private:
	bool has_inner_structure(IDA* first);
	
	// TX main begin
	private:
	void ida(IDA* node, std::string indent, bool parameters, bool nameAsAbove, bool nameOmit, IDA* itemClass);
	
	// TX utility
	private:
	static bool broken_ref(IDA* el);
	static bool as_above_all_test(IDA* firstItem);
	static bool as_above_part_test(IDA* item);
	
	std::string value(IDA* node, bool inArray);
	
	std::string id_str(IDA* refNode, std::optional<std::string> s);
	static bool id(std::string s);
	
// TX end
};

// CLI begin
class IDA_CLI
{
	
	// CLI
	// g++ -std=c++17 -D IDA_MAIN -o IDA IDA.cpp
	private:
	static int cli_err(std::string msg, int exitCode = 1);
	static void cli_out(std::string ln);
	static int cli_help();
	public:
	static int cli(int argc, char** argv);
	
// CLI end
};


/** One data article element. */
class IDA
{
	
	// File includes
	// File includes - Global PWD
	private:
	inline static std::optional<std::string> gIncludePwd = std::nullopt;
	public:
	/** Enables file includes globally, as applied to articles created hereafter.<br>
	 * The provided working dir is used for resolving relative paths.<br>
	 * Set null to disable includes by default. */
	static void setGlobalIncludes(std::optional<std::string> pwd);
	/** Enables file includes globally, as applied to articles created hereafter.<br>
	 * The present runtime working dir is used for resolving relative paths.<br>*/
	static void setGlobalIncludes();
	
	// File includes - this element
	private:
	std::optional<std::string> includePwd = std::nullopt;
	public:
	/** Enables file includes for this article, as applied to input decoded hereafter.<br>
	 * The provided working dir is used for resolving relative paths.<br>
	 * Set null to disable includes. */
	IDA* setIncludes(std::optional<std::string> pwd);
	/** Enables file includes for this article, as applied to input decoded hereafter.<br>
	 * The present runtime working dir is used for resolving relative paths.<br> */
	IDA* setIncludes();
	/** May be null, if includes are disabled for this article. */
	std::optional<std::string> rxIncludePwd();
	
	// Input control
	public:
	/** Negative for unlimited */
	int inputListDepth = -1;
	/** Negative for unlimited */
	int inputListLength = -1;
	IDA* rxInputList(int depth, int length);
	
	// Options storage
	private:
	IDAOptions opts_deref = IDAOptions(IDAOptions::defaultOptions);
	public:
	IDAOptions* opts = &opts_deref;
	/** <p>Fetch the input/output options for this element.</p>
	 * @return Input/output options object. */
	IDAOptions* getOpts();
	/** <p>Set the input/output options of this element to match those of the provided element.</p>
	 * @param src An element from which to copy the options.
	 * @return This element. */
	IDA* copyOptions(IDA* src);
	
	IDA* rxConfig(IDA* parent);
	IDA* rxNew();
	
	// Scope storage
	private:
	IDA* scope = nullptr;
	public:
	IDA* rxScope(IDA* scope);
	
	// Item classes
	private:
	std::map<std::string, IDA*> itemClasses;
	public:
	void rxClearItemClasses();
	IDA* rxGetItemClass(std::string key);
	/** Actually adds to scope element for use by adjacent elements.<br>
	 * Returns false (and takes no action with the class element) if no scope existed */
	bool rxAddItemClass(std::string key, IDA* itemClass);
	
	// Templates
	private:
	std::map<std::string, IDA*> itemTemplates;
	public:
	void rxClearTemplates();
	IDA* rxGetTemplate(std::string key);
	/** Actually adds to scope element for use by adjacent elements.<br>
	 * Returns false (and takes no action with the template element) if no scope existed */
	bool rxAddTemplate(std::string key, IDA* itemTemplate);
	
	// Adjacent nodes
	private:
	IDA* next;
	IDA* prev;
	
	public:
	/** <p>Test whether an element is adjacent to this element in the forward direction.</p>
	 * @return True if an element exists after this element, otherwise false */
	bool hasNext();
	/** <p>Test whether an element is adjacent to this element in the reverse direction.</p>
	 * @return True if an element exists before this element, otherwise false */
	bool hasPrev();
	/** <p>Test whether no element is adjacent to this element in the reverse direction.</p>
	 * <p>This is the opposite of hasPrev, intended as a convenience for function references.</p>
	 * @return True if nothing exists this element, otherwise false */
	bool isFirst();
	/** <p>Test whether no element is adjacent to this element in the forward direction.</p>
	 * <p>This is the opposite of hasNext, intended as a convenience for function references.</p>
	 * @return True if nothing exists after this element, otherwise false */
	bool isLast();
	
	/** <p>Fetch the adjacent element after this one.</p>
	 * <p>If this element is last, an exception is thrown.
	 * To avoid that possibility, use getNextOr to provide a default value.</p>
	 * @exception IDAOpException if this element is last.
	 * @return The element after. */
	IDA* getNext();
	/** <p>Fetch the adjacent element after this one.</p>
	 * <p>If this element is last, null is returned.</p>
	 * @return The element after if present, otherwise null. */
	IDA* getNextOr();
	/** <p>Fetch the adjacent element after this one.</p>
	 * <p>If this element is last, the provided default value is returned.</p>
	 * @param other A default value to return if nothing follows this element.
	 * @return The element after if present, otherwise the default value. */
	IDA* getNextOr(IDA* other);
	/** <p>Fetch the adjacent element before this one.</p>
	 * <p>If this element is first, an exception is thrown.
	 * To avoid that possibility, use getPrevOr to provide a default value.</p>
	 * @exception IDAOpException if this element is first.
	 * @return The element before. */
	IDA* getPrev();
	/** <p>Fetch the adjacent element before this one.</p>
	 * <p>If this element is first, null is returned.</p>
	 * @return The element before if present, otherwise null. */
	IDA* getPrevOr();
	/** <p>Fetch the adjacent element before this one.</p>
	 * <p>If this element is first, the provided default value is returned.</p>
	 * @param other A default value to return if nothing precedes this element.
	 * @return The element before if present, otherwise the default value. */
	IDA* getPrevOr(IDA* other);
	
	/** This element, followed by everything adjacent after it */
	IDA_Iterator* iterateNext();
	/** <p>Places the provided element adjacent to this one, next in sequence.<br>
	 * If the element is null, this one will become last in sequence.</p>
	 * <p>Any existing forward link will be undone before this operation.</p>
	 * @param next An element to follow this one, or null
	 * @return This element */
	IDA* linkNext(IDA* next);
	/** <p>Places the provided element adjacent to this one, previous in sequence.<br>
	 * If the element is null, this one will become first in sequence.</p>
	 * <p>Any existing reverse link will be undone before this operation.</p>
	 * @param prev An element to precede this one, or null
	 * @return This element */
	IDA* linkPrev(IDA* prev);
	/** <p>Places the provided elements adjacent to this one, on either side.<br>
	 * If an element is null, this one will become the first/last in sequence.</p>
	 * <p>Before this operation occurs, any existing links will be cross-connected to remove this element from the chain.</p>
	 * @param prev An element to precede this one, or null
	 * @param next An element to follow this one, or null
	 * @return This element */
	IDA* link(IDA* prev, IDA* next);
	void link_break();
	
	// Node name
	protected:
	std::optional<std::string> name = std::nullopt;
	public:
	/** <p>Test whether this element has no name.
	 * The name must be null, not merely the empty string.</p>
	 * <p>This is the opposite of hasName, intended as a convenience for function references.</p>  
	 * @return True if this element has no name, otherwise false */
	bool isAnon();
	/** <p>Test whether this element has a name.
	 * The name can be any non-null string, including the empty string.</p>
	 * @return True if this element has a name, otherwise false */
	bool hasName();
	/** <p>Test whether this element has a specific name.
	 * Providing null will perform the same test as with isAnon.</p>
	 * @param name A name for comparison
	 * @return True if this element name matches the provided name, otherwise false */
	bool isName(std::optional<std::string> name);
	
	/** <p>Fetch this element name string.</p>
	 * <p>If this element content is anonymous, an exception is thrown.
	 * To avoid that possibility, use getNameOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no name.
	 * @return Element name. */
	std::string getName();
	/** <p>Fetch this element name string.</p>
	 * <p>If this element content is anonymous, null is returned.</p>
	 * <p>This method exists for platform compatibility where strings are not nullable/optional, such as with C++.
	 * The purpose is for this return type to be a nullable string container, while getNameOr(other) returns a string directly.
	 * On platforms where string values can be null, getNameOr() is equivalent to getNameOr(null), but only the former is fully portable.</p>
	 * @return Element name if present, otherwise null. */
	std::optional<std::string> getNameOr();
	/** <p>Fetch this element name string.</p>
	 * <p>If this element content is anonymous, the provided default value is returned.</p>
	 * <p>On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
	 * If possible, always call getNameOr() rather than getNameOr(null).</p>
	 * @param other A default value to return if this element has no name.
	 * @return Element name if present, otherwise the default value. */
	std::string getNameOr(std::string other);
	
	/** <p>Clear the name of this element, leaving it anonymous.</p>
	 * @return This element. */
	IDA* clearName();
	/** <p>Set the name of this element to be the provided string.</p>
	 * <p>If the provided string is null, this element will become anonymous as with calling clearName.</p>
	 * @param name Element name string.
	 * @return This element. */
	IDA* setName(std::optional<std::string> name);
	/** <p>Set the name of this element to match the name of the provided element.</p>
	 * @param src An element from which to copy the name */
	void copyName(IDA* src);
	
	// Parameters
	protected:
	std::vector<IDA*> parameters;
	bool parameters_ok;
	public:
	/** <p>Delete this element's parameters list.</p>
	 * <p>If this element has no parameters list, calling clearParams has no effect.</p>
	 * @return This element. */
	IDA* clearParams();
	/** <p>Set this element's parameters to be an empty list.</p>
	 * <p>Any existing parameter elements will be deleted.</p>
	 * @return This element. */
	IDA* setParams();
	
	/** <p>Fetch this element's parameters list.</p>
	 * <p>If this element has no parameters list, an exception is thrown.
	 * To avoid that possibility, use getParamsOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no parameters list.
	 * @return List of parameters. */
	std::vector<IDA*>* getParams();
	/** <p>Fetch this element's parameters list.</p>
	 * <p>If this element has no parameters list, null is returned.</p>
	 * <p>This method exists for platform compatibility where lists/vectors are not nullable/optional, such as with C++.
	 * The purpose is for this return type to be a nullable list container, while getParamsOr(other) returns a list directly.
	 * On platforms where lists can be null, getParamsOr() is equivalent to getParamsOr(null), but only the former is fully portable.</p>
	 * @return List of parameters if present, otherwise null. */
	std::vector<IDA*>* getParamsOr();
	/** <p>Fetch this element's parameters list.</p>
	 * <p>If this element has no parameters list, the provided default value is returned.</p>
	 * <p>On platforms where lists/vectors are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed list, such that code will not be fully portable if null is provided.
	 * If possible, always call getParamsOr() rather than getParamsOr(null).</p>
	 * @param other A default value to return if this element has no parameters list.
	 * @return List of parameters if present, otherwise the default value. */
	std::vector<IDA*>* getParamsOr(std::vector<IDA*>* other);
	
	/** <p>Test whether this element contains a parameters list.</p>
	 * <p>This reports whether a parameters list object exists at all, as opposed to whether at least one parameter is present.
	 * The primary purpose is to ensure the list exists before calling getParams and/or iterating over its elements.</p>
	 * <p>Use numParams to test for an empty list (0) or at least one parameter present (greater than 0).</p>
	 * @return True if a parameters list exists for this element, otherwise false */
	bool hasParams();
	/** <p>Fetch the number of parameters within this element.</p>
	 * <p>If no parameters list is present, the sentinel value -1 is returned.
	 * If 0 is returned, the list is present but empty.</p>
	 * @return The parameters list size if a list is present, otherwise -1 */
	long numParams();
	/** <p>Test whether this element contains a parameter with the provided name.
	 * If it does, an equivalent call to getParam will return an element.</p>
	 * <p>If false is returned, this can be either due to having no parameter of that name or due to having no parameters list.
	 * Use hasParams to separate those two cases.</p>
	 * @param key A parameter name to be checked
	 * @return True if a parameter exists with the provided name, otherwise false */
	bool hasParam(std::optional<std::string> key);
	
	// Parameters/get
	public:
	/** <p>Fetch the parameter element at the provided index in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has too few parameters in the list, an exception is thrown.
	 * To avoid that possibility, use getParamAtOr to provide a default value.</p>
	 * @param index A parameter index to be recalled.
	 * @exception IDAOpException if this element has no such parameter.
	 * @return The indexed parameter element. */
	IDA* getParamAt(long index);
	/** <p>Fetch the parameter element at the provided index in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has too few parameters in the list, null is returned.</p>
	 * @param index A parameter index to be recalled.
	 * @return The indexed parameter element if present, otherwise null. */
	IDA* getParamAtOr(long index);
	/** <p>Fetch the parameter element at the provided index in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has too few parameters in the list, the provided default value is returned.</p>
	 * @param index A parameter index to be recalled.
	 * @param other A default value to return if this element has no such parameter.
	 * @return The indexed parameter element if present, otherwise the default value. */
	IDA* getParamAtOr(long index, IDA* other);
	
	/** <p>Fetch the first parameter element with the provided name in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has no parameter of that name, an exception is thrown.
	 * To avoid that possibility, use getParamOr to provide a default value.</p>
	 * @param key A parameter name to be recalled.
	 * @exception IDAOpException if this element has no such parameter.
	 * @return The named parameter element. */
	IDA* getParam(std::optional<std::string> key);
	/** <p>Fetch the first parameter element with the provided name in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has no parameter of that name, null is returned.</p>
	 * @param key A parameter name to be recalled.
	 * @return The named parameter element if present, otherwise null. */
	IDA* getParamOr(std::optional<std::string> key);
	/** <p>Fetch the first parameter element with the provided name in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has no parameter of that name, the provided default value is returned.</p>
	 * @param key A parameter name to be recalled.
	 * @param other A default value to return if this element has no such parameter.
	 * @return The named parameter element if present, otherwise the default value. */
	IDA* getParamOr(std::optional<std::string> key, IDA* other);
	
	/** <p>Cast to a string value the first parameter element with the provided name in this element's parameters list.</p>
	 * <p>If this element has no parameters list, has no parameter of that name or can not be interpreted as a string value, an exception is thrown.
	 * To avoid that possibility, use getParamStrOr to provide a default value.</p>
	 * @param key A parameter name to be recalled.
	 * @exception IDAOpException if this element has no such parameter, or the parameter content cannot cast to a string value.
	 * @return String content interpretation of the named parameter element. */
	std::string getParamStr(std::optional<std::string> key);
	/** <p>Cast to a string value the first parameter element with the provided name in this element's parameters list.</p>
	 * <p>If this element has no parameters list, has no parameter of that name or can not be interpreted as a string value, null is returned.</p>
	 * <p>This method exists for platform compatibility where strings are not nullable/optional, such as with C++.
	 * The purpose is for this return type to be a nullable string container, while getParamStrOr(key, other) returns a string directly.
	 * On platforms where string values can be null, getParamStrOr(key) is equivalent to getParamStrOr(key, null), but only the former is fully portable.</p>
	 * @param key A parameter name to be recalled.
	 * @return String content interpretation of the named parameter element if possible, otherwise null. */
	std::optional<std::string> getParamStrOr(std::optional<std::string> key);
	/** <p>Cast to a string value the first parameter element with the provided name in this element's parameters list.</p>
	 * <p>If this element has no parameters list, has no parameter of that name or can not be interpreted as a string value, the provided default value is returned.</p>
	 * <p>On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
	 * If possible, always call getParamStrOr(key) rather than getParamStrOr(key, null).</p>
	 * @param key A parameter name to be recalled.
	 * @param other A default value to return if this element has no such parameter, or the parameter content cannot cast to a string value.
	 * @return String content interpretation of the named parameter element if possible, otherwise the default value. */
	std::string getParamStrOr(std::optional<std::string> key, std::string other);
	
	/** <p>Fetch the first parameter element in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has an empty parameters list, an exception is thrown.
	 * To avoid that possibility, use getParamFirstOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no such parameter.
	 * @return The first parameter element. */
	IDA* getParamFirst();
	/** <p>Fetch the first parameter element in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has an empty parameters list, null is returned.</p>
	 * @return The first parameter element if present, otherwise null. */
	IDA* getParamFirstOr();
	/** <p>Fetch the first parameter element in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has an empty parameters list, the provided default value is returned.</p>
	 * @param other A default value to return if this element has no such parameter.
	 * @return The first parameter element if present, otherwise the default value. */
	IDA* getParamFirstOr(IDA* other);
	/** <p>Fetch the last parameter element in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has an empty parameters list, an exception is thrown.
	 * To avoid that possibility, use getParamLastOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no such parameter.
	 * @return The last parameter element. */
	IDA* getParamLast();
	/** <p>Fetch the last parameter element in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has an empty parameters list, null is returned.</p>
	 * @return The last parameter element if present, otherwise null. */
	IDA* getParamLastOr();
	/** <p>Fetch the last parameter element in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has an empty parameters list, the provided default value is returned.</p>
	 * @param other A default value to return if this element has no such parameter.
	 * @return The last parameter element if present, otherwise the default value. */
	IDA* getParamLastOr(IDA* other);
	
	// Parameters/add
	public:
	/** Create and insert a new parameter at the end of the list, with the provided element name.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param name Name string for the created parameter element.
	 * @return The created parameter element. */
	IDA* newParam(std::optional<std::string> name);
	
	/** Insert a parameter at the end of the list.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param param The parameter element to be stored.
	 * @return This element. */
	IDA* addParam(IDA* param);
	/** Insert a parameter at the specified list index position.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param param The parameter element to be stored.
	 * @param pos The list index position at which the parameter should be inserted.
	 * @exception IDAOpException if the specified position is below zero or above the present list size.
	 * @return This element. */
	IDA* addParamAt(IDA* param, long pos);
	/** Replace the first matching parameter name, or insert a new parameter at the end of the list.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param param The parameter element to be stored.
	 * @return This element. */
	IDA* setParam(IDA* param);
	
	/** <p>Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.</p>
	 * <p>This is the same as calling setParamNull.</p>
	 * @param key The parameter element name.
	 * @return This element. */
	IDA* clearParam(std::optional<std::string> key);
	/** <p>Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.</p>
	 * <p>This is the same as calling clearParam.</p>
	 * @param key The parameter element name.
	 * @return This element. */
	IDA* setParamNull(std::optional<std::string> key);
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content String value to be stored.
	 * @return This element. */
	IDA* setParamStr(std::optional<std::string> key, std::optional<std::string> content);
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Boolean value to be stored.
	 * @return This element. */
	IDA* setParamBool(std::optional<std::string> key, bool content);
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Floating-point value to be stored.
	 * @return This element. */
	IDA* setParamDouble(std::optional<std::string> key, double content);
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Floating-point value to be stored.
	 * @return This element. */
	IDA* setParamFloat(std::optional<std::string> key, float content);
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Integer value to be stored.
	 * @return This element. */
	IDA* setParamLong(std::optional<std::string> key, long long content);
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Integer value to be stored.
	 * @return This element. */
	IDA* setParamInt(std::optional<std::string> key, long content);
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Referenced element to be stored.
	 * @return This element. */
	IDA* setParamRef(std::optional<std::string> key, IDA* content);
	
	// Parameters/remove
	public:
	/** Remove the parameter at the specified list index position.<br>
	 * If this element has no parameters list, an exception is thrown.<br>
	 * @param pos The list index position of a parameter which should be removed.
	 * @exception IDAOpException if the specified position is below zero or beyond the present list size, or this element has no parameters list. */
	void remParamAt(int pos);
	/** Remove the provided parameter.<br>
	 * If the parameter is absent or this element has no parameters list, the call has no effect.<br>
	 * @param param The parameter element to be removed.
	 * @return True if a parameter was removed, otherwise false. */
	bool remParam(IDA* param);
	/** Remove the first parameter with the provided element name.<br>
	 * If no such parameter exists or this element has no parameters list, the call has no effect.<br>
	 * @param key Name string of the parameter element to remove.
	 * @return True if a parameter was removed, otherwise false. */
	bool unParam(std::optional<std::string> key);
	
	// Parameters/other
	public:
	/** <p>Set the parameters of this element to be clones of the parameters of the provided element.</p>
	 * @param src An element from which to copy the parameters */
	void copyParams(IDA* src);
	
	// Content
	private:
	std::optional<std::string> cRaw;
	std::optional<std::string> cStr;
	std::string cNum, cDen, cUnit;
	std::vector<IDA*> cList;
	bool cList_ok;
	std::optional<bool> cBool;
	IDA* cRef;
	IDAObj* cRefObj;
	std::optional<std::string> cRefTarget;
	std::optional<std::string> cRefName;
	
	public:
	/** <p>Set the content of this element to match the content of the provided element.
	 * If the content in question is a list, its inner elements will be cloned.</p>
	 * @param src An element from which to copy the content */
	void copyContent(IDA* src);
	
	// Content/ref
	public:
	/** Returns this element */
	IDA* clearRef();
	/** <p>Test whether the content type of this element is a reference to another element.</p>
	 * @return True if this element content is a reference, otherwise false */
	bool isRef();
	/** <p>Fetch this element content as a reference (to another element).</p>
	 * <p>If this element content is not a reference, an exception is thrown.
	 * To avoid that possibility, use asRefOr to provide a default value.</p>
	 * @exception IDAOpException if this element does not contain reference content.
	 * @return Reference content. */
	IDA* asRef();
	/** <p>Fetch this element content as a reference (to another element).</p>
	 * <p>If this element content is not a reference, null is returned.</p>
	 * @return Reference content if present, otherwise null. */
	IDA* asRefOr();
	/** <p>Fetch this element content as a reference (to another element).</p>
	 * <p>If this element content is not a reference, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain reference content.
	 * @return Reference content if present, otherwise the default value. */
	IDA* asRefOr(IDA* other);
	/** Returns this element */
	IDA* putRef(IDA* ref);
	/** <p>Sets an object for which this element should reference the corresponding element as it is encoded.</p> 
	 * <p>During a restore/decode process, the rebuilt object can be retrieved using getRefObj or withRefObj.
	 * The IDAObj codec interface implementation should ensure that any required type information is captured.</p>
	 * @param refObj A target object which implements the IDAObj codec interface.
	 * @return This element. */
	IDA* setRefObj(IDAObj* refObj);
	/** <p>During a restore/decode process, the object corresponding with the referenced element will be sent to the callback when it becomes available.
	 * If that object is already available, the callback will be resolved before this call returns.</p>
	 * <p>If the object is needed immediately, the article structure must be designed such that the referenced object element is decoded in advance of this call.
	 * Use getRefObj instead if that design should be enforced.</p>
	 * <p>To ensure consistency, a top-level call to objRestore will throw an exception if any unresolved callbacks remain after decoding is complete.</p>
	 * <p>Calling code should use article information to establish the correct object type.</p>
	 * @param callback A function which receives the decoded object associated with the referenced element.
	 * @exception IDAOpException if this element content is not a reference. */
	void withRefObj(std::function<void(IDAObj*)> callback);
	/** <p>Fetches the object corresponding with the referenced element, after it has been encoded or decoded.
	 * During a restore/decode process, an exception is thrown if the object is not yet available.</p>
	 * <p>To retrieve the object this way during a restore/decode process, the article structure must be designed such that the referenced object element is decoded in advance of this call.
	 * Use withRefObj instead if that design is not the case.</p>
	 * <p>Calling code should use article information to establish the correct object type.</p>
	 * @exception IDAOpException if this element content is not a reference, or if no object is available immediately.
	 * @return The encoded or decoded object associated with the referenced element. */
	IDAObj* getRefObj();
	/** <p>Set a transient target for this element.</p>
	 * <p>The targets match the identifiers assigned via setRefName, and are used only for capturing and restoring element pointer information.
	 * The string content itself is not information, insofar as it need not survive a loop test.</p>
	 * @param refTarget A string to store as the transient target for this element. */
	void setRefTarget(std::string refTarget);
	/** <p>Fetch the transient target for this element.
	 * The returned value may be <i>stale</i>.</p>
	 * <p>The targets match the identifiers assigned via setRefName, and are used only for capturing and restoring element pointer information.
	 * The string content itself is not information, insofar as it need not survive a loop test.</p>
	 * @return The transient target string for this element if it has a reference target, otherwise null. */
	std::optional<std::string> getRefTarget();
	/** <p>Test whether this element has been assigned a transient target for encoding/decoding an outbound reference.
	 * The returned value may be <i>stale</i>.</p>
	 * <p>If an article has been decoded from markup, any reference target elements will have fresh identifiers from the markup document.
	 * Encoding, copying or testing equal a top-level article element will generate fresh identifiers.</p>
	 * <p>Call txRefLabel at top-level to generate fresh identifiers manually.</p>
	 * @return True if this element has a transient target, otherwise false. */
	bool hasRefTarget();
	/** <p>Set a transient identifier for this element.</p>
	 * <p>Identifiers are set automatically when encoding/decoding markup, testing equal, or calling txRefLabel directly.
	 * Encoding, copying or testing equal a top-level article element will generate fresh identifiers.</p>
	 * <p>These names are used only for capturing and restoring element pointer information.
	 * The string content itself is not information, insofar as it need not survive a loop test.</p>
	 * @param refName A string to store as the transient identifier for this element. */
	void setRefName(std::optional<std::string> refName);
	/** <p>Fetch the transient identifier for this element.
	 * The returned value may be <i>stale</i>.</p>
	 * <p>Identifiers are set automatically when encoding/decoding markup, testing equal, or calling txRefLabel directly.
	 * Encoding, copying or testing equal a top-level article element will generate fresh identifiers.</p>
	 * <p>These names are used only for capturing and restoring element pointer information.
	 * The string content itself is not information, insofar as it need not survive a loop test.</p>
	 * @return The transient identifier string for this element if it is a reference target, otherwise null. */
	std::optional<std::string> getRefName();
	/** <p>Test whether this element has been assigned a transient identifier for encoding/decoding inbound references.
	 * The returned value may be <i>stale</i>.</p>
	 * <p>If an article has been decoded from markup, any reference target elements will have fresh identifiers from the markup document.
	 * Encoding, copying or testing equal a top-level article element will generate fresh identifiers.</p>
	 * <p>Call txRefLabel at top-level to generate fresh identifiers manually.</p>
	 * @return True if this element has a transient identifier, otherwise false. */
	bool hasRefName();
	
	public:
	/** Returns the number of reference labels applied */
	long txRefLabel(bool adj);
	private:
	static void ref_label_clear(IDA* el, std::vector<IDA*> &rn);
	static void ref_label_count(IDA* el, std::vector<IDA*> &rn, long &rc);
	
	public:
	void rxRefResolve(bool adj);
	private:
	static void ref_resolve_scan(IDA* el, std::map<std::string,IDA*> &rs);
	static void ref_resolve_put(IDA* el, std::map<std::string,IDA*> &rs);
	
	// Content/bool
	public:
	/** <p>Test whether the content type of this element is a boolean value.</p>
	 * @return True if this element has boolean content, otherwise false */
	bool isBool();
	/** <p>Fetch this element content as a boolean value.</p>
	 * <p>If this element content is not a boolean value, an exception is thrown.
	 * To avoid that possibility, use asBoolOr to provide a default value.
	 * One may also use isTrue for false by default, or not-isFalse for true by default.</p>
	 * <p>Use toBool/toBoolOr for casting other content types to be a boolean value.</p>
	 * @exception IDAOpException if this element does not contain boolean content.
	 * @return Boolean content value. */
	bool asBool();
	/** <p>Fetch this element content as a boolean value.</p>
	 * <p>If this element content is not a boolean value, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain boolean content.
	 * @return Boolean content value if present, otherwise the default value. */
	bool asBoolOr(bool other);
	/** <p>Test whether the content of this element is boolean-true.</p>
	 * <p>If false is returned, this can be either due to being boolean-false or not being a boolean value.
	 * Use isBool and asBool for methods with those effects.</p>
	 * <p>This is intended as a convenience for function references.</p>
	 * @return True if this element has boolean content with value true, otherwise false */
	bool isTrue();
	/** <p>Test whether the content of this element is boolean-false.</p>
	 * <p>If false is returned, this can be either due to being boolean-true or not being a boolean value.
	 * Use isBool and (not-)asBool for methods with those effects.</p>
	 * <p>This is intended as a convenience for function references.</p>
	 * @return True if this element has boolean content with value false, otherwise false */
	bool isFalse();
	/** <p>Place boolean content into this element with value true, replacing any existing content.</p>
	 * @return This element */
	IDA* putTrue();
	/** <p>Place boolean content into this element with value false, replacing any existing content.</p>
	 * @return This element */
	IDA* putFalse();
	/** <p>Place boolean content into this element with the provided value, replacing any existing content.</p>
	 * @param b Boolean content value.
	 * @return This element */
	IDA* putBool(bool b);
	/** <p>Test whether this element content can be interpreted as a boolean value (see toBool).
	 * This will always be the case for natural boolean content (see isBool).</p>
	 * <p>All numeric content can be cast to boolean, which is true if non-zero.
	 * String content can be "true" or "false".</p>
	 * @return True if this element content can be cast to a boolean value, otherwise false */
	bool canBool();
	/** <p>Cast this element content to a boolean value.</p>
	 * <p>If this element content can not be interpreted as a boolean value, an exception is thrown.
	 * To avoid that possibility, use toBoolOr to provide a default value.</p>
	 * @exception IDAOpException if this element content cannot cast to a boolean value.
	 * @return Boolean content interpretation. */
	bool toBool();
	/** <p>Cast this element content to a boolean value.</p>
	 * <p>If this element content can not be interpreted as a boolean value, the provided default value is returned.</p>
	 * @param other A default value to return if this element content cannot cast to a boolean value.
	 * @return Boolean content interpretation if able, otherwise the default value. */
	bool toBoolOr(bool other);
	
	// Content/null
	public:
	/** <p>Test whether this element contains no content.</p>
	 * <p>This is the opposite of hasContent, intended as a convenience for function references.</p>
	 * @return True if this element has no content, otherwise false. */
	bool isNull();
	/** <p>Test whether this element contains some type of content.</p>
	 * <p>This is the opposite of isNull, intended as a convenience for function references.</p>
	 * @return True if this element has content present, otherwise false. */
	bool hasContent();
	/** <p>Erase any existing content from this element.</p>
	 * <p>This is the same as calling clearContent.</p>
	 * @return This element. */
	IDA* putNull();
	/** <p>Erase any existing content from this element.</p>
	 * <p>This is the same as calling putNull.</p>
	 * @return This element. */
	IDA* clearContent();
	
	// Content/string
	public:
	/** <p>Place string content into this element with the provided value, replacing any existing content.</p>
	 * @param s String content value. 
	 * @return This element */
	IDA* putStr(std::optional<std::string> s);
	/** <p>Test whether the content type of this element is a string value.</p>
	 * @return True if this element has string content, otherwise false */
	bool isStr();
	std::string toString();
	/** <p>Fetch this element content as a string value.</p>
	 * <p>If this element content is not a string value, an exception is thrown.
	 * To avoid that possibility, use asStrOr to provide a default value.</p>
	 * <p>Use toStr/toStrOr for casting other content types to be a string value.</p>
	 * @exception IDAOpException if this element does not contain string content.
	 * @return String content value. */
	std::string asStr();
	/** <p>Fetch this element content as a string value.</p>
	 * <p>If this element content is not a string value, null is returned.</p>
	 * <p>This method exists for platform compatibility where strings are not nullable/optional, such as with C++.
	 * The purpose is for this return type to be a nullable string container, while asStrOr(other) returns a string directly.
	 * On platforms where string values can be null, asStrOr() is equivalent to asStrOr(null), but only the former is fully portable.</p>
	 * @return String content value if present, otherwise null. */
	std::optional<std::string> asStrOr();
	/** <p>Fetch this element content as a string value.</p>
	 * <p>If this element content is not a string value, the provided default value is returned.</p>
	 * <p>On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
	 * If possible, always call asStrOr() rather than asStrOr(null).</p>
	 * @param other A default value to return if this element does not contain string content.
	 * @return String content value if present, otherwise the default value. */
	std::string asStrOr(std::string other);
	/** <p>Test whether this element content can be interpreted as a string value (see toStr).
	 * This will always be the case for natural string content (see isStr).</p>
	 * <p>Most content can be cast to string.
	 * Types which cannot be cast are lists, references and nulls.</p>
	 * @return True if this element content can be cast to a string value, otherwise false */
	bool canStr();
	/** <p>Cast this element content to a string value.</p>
	 * <p>If this element content can not be interpreted as a string value, an exception is thrown.
	 * To avoid that possibility, use toStrOr to provide a default value.</p>
	 * @exception IDAOpException if this element content cannot cast to a string value.
	 * @return String content interpretation. */
	std::string toStr();
	/** <p>Cast this element content to a string value.</p>
	 * <p>If this element content can not be interpreted as a string value, null is returned.</p>
	 * <p>This method exists for platform compatibility where strings are not nullable/optional, such as with C++.
	 * The purpose is for this return type to be a nullable string container, while toStrOr(other) returns a string directly.
	 * On platforms where string values can be null, toStrOr() is equivalent to toStrOr(null), but only the former is fully portable.</p>
	 * @return String content interpretation if able, otherwise null. */
	std::optional<std::string> toStrOr();
	/** <p>Cast this element content to a string value.</p>
	 * <p>If this element content can not be interpreted as a string value, the provided default value is returned.</p>
	 * <p>On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
	 * If possible, always call toStrOr() rather than toStrOr(null).</p>
	 * @param other A default value to return if this element content cannot cast to a string value.
	 * @return String content interpretation if able, otherwise the default value. */
	std::string toStrOr(std::string other);
	
	// Content/number/other
	private:
	void clearNum();
	
	// Content/number/set
	private:
	void refresh_num();
	public:
	/** <p>Set the unit string appended to the number represented by this element content.</p>
	 * <p>Passing null to setNumUnit has no effect.
	 * To clear the unit, call setNumUnit with an empty string.</p>
	 * <p>If this element content is not numeric, an exception is thrown.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return This element. */
	IDA* setNumUnit(std::optional<std::string> unit);
	
	IDA* putNum(std::string num, std::string den, std::string unit);
	IDA* putNum(std::string num, std::string den);
	IDA* putNum(std::string num);
	
	/** <p>Place 64-bit floating-point content into this element with the provided value, replacing any existing content.</p>
	 * <p>If the provided unit string is non-null, also updates the unit string for this number (see setNumUnit).</p>
	 * @param num 64-bit floating-point content value.
	 * @param unit Number unit string.
	 * @return This element */
	IDA* putDouble(double num, std::optional<std::string> unit = std::nullopt);
	/** <p>Place 32-bit floating-point content into this element with the provided value, replacing any existing content.</p>
	 * <p>If the provided unit string is non-null, also updates the unit string for this number (see setNumUnit).</p>
	 * @param num 32-bit floating-point content value.
	 * @param unit Number unit string.
	 * @return This element */
	IDA* putFloat(float num, std::optional<std::string> unit = std::nullopt);
	/** <p>Place 64-bit integer content into this element with the provided value, replacing any existing content.</p>
	 * <p>If the provided unit string is non-null, also updates the unit string for this number (see setNumUnit).</p>
	 * @param num 64-bit integer content value.
	 * @param unit Number unit string.
	 * @return This element */
	IDA* putLong(long long num, std::optional<std::string> unit = std::nullopt);
	/** <p>Place 32-bit integer content into this element with the provided value, replacing any existing content.</p>
	 * <p>If the provided unit string is non-null, also updates the unit string for this number (see setNumUnit).</p>
	 * @param num 32-bit integer content value.
	 * @param unit Number unit string.
	 * @return This element */
	IDA* putInt(long num, std::optional<std::string> unit = std::nullopt);
	
	// Content/number/get
	public:
	/** <p>Test whether the content type of this element is a numeric value.</p>
	 * @return True if this element has numeric content, otherwise false */
	bool isNum();
	/** <p>Fetch the unit string appended to the number represented by this element content.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * If numeric content had no unit specified, an empty string is returned.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return Numeric unit string. */
	std::string getNumUnit();
	/** <p>Fetch the numerator string for the number represented by this element content.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * If the number was not a quotient, calling getNumN will return the same result as calling toStr.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return Numerator as a string. */
	std::string getNumN();
	/** <p>Fetch the denominator string for the number represented by this element content.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * If the number was not a quotient, calling getNumD will return "1".</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return Denominator as a string. */
	std::string getNumD();
	/** <p>Fetch this element content as a 64-bit floating-point value.</p>
	 * <p>If this element content is not numeric, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain numeric content.
	 * @return 64-bit floating-point content value if present, otherwise the default value. */
	double asDoubleOr(double other);
	/** <p>Fetch this element content as a 32-bit floating-point value.</p>
	 * <p>If this element content is not numeric, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain numeric content.
	 * @return 32-bit floating-point content value if present, otherwise the default value. */
	float asFloatOr(float other);
	/** <p>Fetch this element content as a 64-bit integer value.</p>
	 * <p>If this element content is not numeric, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain numeric content.
	 * @return 64-bit integer content value if present, otherwise the default value. */
	long long asLongOr(long long other);
	/** <p>Fetch this element content as a 32-bit integer value.</p>
	 * <p>If this element content is not numeric, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain numeric content.
	 * @return 32-bit integer content value if present, otherwise the default value. */
	long asIntOr(long other);
	/** <p>Fetch this element content as a 64-bit floating-point value.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * To avoid that possibility, use asDoubleOr to provide a default value.</p>
	 * <p>Use toDouble/toDoubleOr for casting other content types to be a 64-bit floating-point value.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return 64-bit floating-point content value. */
	double asDouble();
	/** <p>Fetch this element content as a 32-bit floating-point value.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * To avoid that possibility, use asFloatOr to provide a default value.</p>
	 * <p>Use toFloat/toFloatOr for casting other content types to be a 32-bit floating-point value.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return 32-bit floating-point content value. */
	float asFloat();
	/** <p>Fetch this element content as a 64-bit integer value.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * To avoid that possibility, use asLongOr to provide a default value.</p>
	 * <p>Use toLong/toLongOr for casting other content types to be a 64-bit integer value.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return 64-bit integer content value. */
	long long asLong();
	/** <p>Fetch this element content as a 32-bit integer value.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * To avoid that possibility, use asIntOr to provide a default value.</p>
	 * <p>Use toInt/toIntOr for casting other content types to be a 32-bit integer value.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return 32-bit integer content value. */
	long asInt();
	
	private:
	double c_double();
	float c_float();
	long long c_long();
	long c_int();
	long long n_long(std::string str);
	double n_double(std::string str);
	long long n_hex(std::string str);
	long long n_bin(std::string str);
	
	// Content/list
	private:
	void clearList();
	void refresh_list();
	
	public:
	std::vector<IDA*>::iterator iterator();
	
	// Content/list/set
	public:
	/** Returns this element */
	IDA* putList();
	/** Returns this element */
	IDA* putList(std::optional<std::vector<IDA*>> l);
	
	// Content/list/get
	public:
	/** <p>Test whether the content type of this element is a list.</p>
	 * <p>This reports whether the content is a list or non-list, as opposed to whether at least one list item is present.
	 * The primary purpose is to ensure the list exists before calling asList and/or iterating over its elements.</p>
	 * <p>Use numItems to test for an empty list (0) or at least one item present (greater than 0).</p>
	 * @return True if this element has list content, otherwise false */
	bool isList();
	/** <p>Fetch this element content as a list (containing other elements).</p>
	 * <p>If this element content is not a list, an exception is thrown.
	 * To avoid that possibility, use asListOr to provide a default value.</p>
	 * @exception IDAOpException if this element does not contain list content.
	 * @return List content. */
	std::vector<IDA*>* asList();
	/** <p>Fetch this element content as a list (containing other elements).</p>
	 * <p>If this element content is not a list, null is returned.</p>
	 * <p>This method exists for platform compatibility where lists/vectors are not nullable/optional, such as with C++.
	 * The purpose is for this return type to be a nullable list container, while asListOr(other) returns a list directly.
	 * On platforms where lists can be null, asListOr() is equivalent to asListOr(null), but only the former is fully portable.</p>
	 * @return List content if present, otherwise null. */
	std::vector<IDA*>* asListOr();
	/** <p>Fetch this element content as a list (containing other elements).</p>
	 * <p>If this element content is not a list, the provided default value is returned.</p>
	 * <p>On platforms where lists/vectors are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed list, such that code will not be fully portable if null is provided.
	 * If possible, always call asListOr() rather than asListOr(null).</p>
	 * @param other A default value to return if this element does not contain list content.
	 * @return List content if present, otherwise the default value. */
	std::vector<IDA*>* asListOr(std::vector<IDA*>* other);
	/** <p>Fetch the number of content list items within this element.</p>
	 * <p>If no content list is present, the sentinel value -1 is returned.
	 * If 0 is returned, the list is present but empty.</p>
	 * @return The content list size if a list is present, otherwise -1 */
	long numItems();
	
	/** <p>Test whether this element contains a list item with the provided name.
	 * If it does, an equivalent call to getItem will return an element.</p>
	 * <p>If false is returned, this can be either due to having no list item of that name or due to being a non-list.
	 * Use isList to separate those two cases.</p>
	 * @param key A list item name to be checked
	 * @return True if a content list item exists with the provided name, otherwise false */
	bool hasItem(std::optional<std::string> key);
	
	/** <p>Fetch the item element at the provided index in this element's content list.</p>
	 * <p>If this element content is not a list or has too few items in the list, an exception is thrown.
	 * To avoid that possibility, use getItemAtOr to provide a default value.</p>
	 * @param index A list item index to be recalled.
	 * @exception IDAOpException if this element has no such list item.
	 * @return The indexed content list item element. */
	IDA* getItemAt(long index);
	/** <p>Fetch the item element at the provided index in this element's content list.</p>
	 * <p>If this element content is not a list or has too few items in the list, null is returned.</p>
	 * @param index A list item index to be recalled.
	 * @return The indexed content list item element if present, otherwise null. */
	IDA* getItemAtOr(long index);
	/** <p>Fetch the item element at the provided index in this element's content list.</p>
	 * <p>If this element content is not a list or has too few items in the list, the provided default value is returned.</p>
	 * @param index A list item index to be recalled.
	 * @param other A default value to return if this element has no such list item.
	 * @return The indexed content list item element if present, otherwise the default value. */
	IDA* getItemAtOr(long index, IDA* other);
	
	/** <p>Fetch the first item element with the provided name in this element's content list.</p>
	 * <p>If this element content is not a list or has no item of that name, an exception is thrown.
	 * To avoid that possibility, use getItemOr to provide a default value.</p>
	 * @param index A list item name to be recalled.
	 * @exception IDAOpException if this element has no such list item.
	 * @return The named content list item element. */
	IDA* getItem(std::optional<std::string> key);
	/** <p>Fetch the first item element with the provided name in this element's content list.</p>
	 * <p>If this element content is not a list or has no item of that name, null is returned.</p>
	 * @param key A list item name to be recalled.
	 * @return The named content list item element if present, otherwise null. */
	IDA* getItemOr(std::optional<std::string> key);
	/** <p>Fetch the first item element with the provided name in this element's content list.</p>
	 * <p>If this element content is not a list or has no item of that name, the provided default value is returned.</p>
	 * @param key A list item name to be recalled.
	 * @param other A default value to return if this element has no such list item.
	 * @return The named content list item element if present, otherwise the default value. */
	IDA* getItemOr(std::optional<std::string> key, IDA* other);
	
	/** <p>Cast to a string value the first item element with the provided name in this element's content list.</p>
	 * <p>If this element content is not a list, has no item of that name or can not be interpreted as a string value, an exception is thrown.
	 * To avoid that possibility, use getItemStrOr to provide a default value.</p>
	 * @param key A list item name to be recalled.
	 * @exception IDAOpException if this element has no such list item, or the item content cannot cast to a string value.
	 * @return String content interpretation of the named list item element. */
	std::string getItemStr(std::optional<std::string> key);
	/** <p>Cast to a string value the first item element with the provided name in this element's content list.</p>
	 * <p>If this element content is not a list, has no item of that name or can not be interpreted as a string value, null is returned.</p>
	 * <p>This method exists for platform compatibility where strings are not nullable/optional, such as with C++.
	 * The purpose is for this return type to be a nullable string container, while getItemStrOr(key, other) returns a string directly.
	 * On platforms where string values can be null, getItemStrOr(key) is equivalent to getItemStrOr(key, null), but only the former is fully portable.</p>
	 * @param key A list item name to be recalled.
	 * @return String content interpretation of the named list item element if possible, otherwise null. */
	std::optional<std::string> getItemStrOr(std::optional<std::string> key);
	/** <p>Cast to a string value the first item element with the provided name in this element's content list.</p>
	 * <p>If this element content is not a list, has no item of that name or can not be interpreted as a string value, the provided default value is returned.</p>
	 * <p>On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
	 * If possible, always call getItemStrOr(key) rather than getItemStrOr(key, null).</p>
	 * @param key A list item name to be recalled.
	 * @param other A default value to return if this element has no such list item, or the item content cannot cast to a string value.
	 * @return String content interpretation of the named list item element if possible, otherwise the default value. */
	std::string getItemStrOr(std::optional<std::string> key, std::string other);
	
	/** <p>Fetch the first item element in this element's content list.</p>
	 * <p>If this element content is not a list or is an empty list, an exception is thrown.
	 * To avoid that possibility, use getItemFirstOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no such list item.
	 * @return The first content list item element. */
	IDA* getItemFirst();
	/** <p>Fetch the first item element in this element's content list.</p>
	 * <p>If this element content is not a list or is an empty list, null is returned.</p>
	 * @return The first content list item element if present, otherwise null. */
	IDA* getItemFirstOr();
	/** <p>Fetch the first item element in this element's content list.</p>
	 * <p>If this element content is not a list or is an empty list, the provided default value is returned.</p>
	 * @param other A default value to return if this element has no such list item.
	 * @return The first content list item element if present, otherwise the default value. */
	IDA* getItemFirstOr(IDA* other);
	/** <p>Fetch the last item element in this element's content list.</p>
	 * <p>If this element content is not a list or is an empty list, an exception is thrown.
	 * To avoid that possibility, use getItemLastOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no such list item.
	 * @return The last content list item element. */
	IDA* getItemLast();
	/** <p>Fetch the last item element in this element's content list.</p>
	 * <p>If this element content is not a list or is an empty list, null is returned.</p>
	 * @return The last content list item element if present, otherwise null. */
	IDA* getItemLastOr();
	/** <p>Fetch the last item element in this element's content list.</p>
	 * <p>If this element content is not a list or is an empty list, the provided default value is returned.</p>
	 * @param other A default value to return if this element has no such list item.
	 * @return The last content list item element if present, otherwise the default value. */
	IDA* getItemLastOr(IDA* other);
	
	// Content/list/add
	public:
	/** Insert a item element at the end of the content list.<br>
	 * If this element does not have list content, a new list will be created if the makeList flag is set, otherwise an exception is thrown.<br>
	 * @param el The item element to be stored.
	 * @param makeList Whether to overwrite existing non-list content.
	 * @exception IDAOpException if this element content is not a list and the makeList flag is false.
	 * @return This element. */
	IDA* addItem(IDA* el, bool makeList = true);
	
	/** Insert a item element at the specified content list index position.<br>
	 * If this element does not have list content, an exception is thrown.<br>
	 * @param el The item element to be stored.
	 * @param pos The list index position at which the item should be inserted.
	 * @exception IDAOpException if the specified position is below zero or above the present list size, or if this element content is not a list.
	 * @return This element. */
	IDA* addItemAt(IDA* el, int pos);
	
	/** Create and insert a new item element at the end of the content list, with the provided element name.<br>
	 * If this element does not have list content, a new list will be created if the makeList flag is set, otherwise an exception is thrown.<br>
	 * @param name Name string for the created item element.
	 * @param makeList Whether to overwrite existing non-list content.
	 * @exception IDAOpException if this element content is not a list and the makeList flag is false.
	 * @return The created item element. */
	IDA* newItem(std::optional<std::string> name, bool makeList = true);
	
	// Content/list/remove
	public:
	/** Remove the item at the specified content list index position.<br>
	 * If this element does not have list content, an exception is thrown.<br>
	 * @param pos The list index position of an item which should be removed.
	 * @exception IDAOpException if the specified position is below zero or beyond the present list size, or this element does not have list content. */
	void remItemAt(int pos);
	/** Remove the provided content list item.<br>
	 * If the item is absent or this element does not have list content, the call has no effect.<br>
	 * @param el The item element to be removed.
	 * @return True if an item was removed, otherwise false. */
	bool remItem(IDA* el);
	/** Remove the first content list item with the provided element name.<br>
	 * If no such item exists or this element does not have list content, the call has no effect.<br>
	 * @param name Name string of the item element to remove.
	 * @return True if an item was removed, otherwise false. */
	bool unItem(std::optional<std::string> name);
	
	// Object/equals
	private:
	bool eq_inner(IDA* obj);
	public:
	/** <p>Perform a top-level comparison of this element and the provided element.</p>
	 * <p>To be equal, both elements must match when interpreted as full articles.
	 * The element names, parameters and content must all match recursively.</p>
	 * <p>Note that content matches consider whether the <i>information</i> is the same.
	 * See eqContent for details.</p>
	 * @param obj An article element for comparison
	 * @param adjacents Whether to include adjacent elements in the comparison
	 * @return True if the elements represent matching articles, otherwise false */
	bool eq(IDA* obj, bool adjacent=false);
	
	// C++ compatibility
	static bool eq_str(std::optional<std::string> a, std::optional<std::string> b)
	{
		return a.has_value() ? (b.has_value() && (*a == *b)) : !b.has_value();
	}
	static bool eq_bool(std::optional<bool> a, std::optional<bool> b)
	{
		return a.has_value() ? (b.has_value() && (*a == *b)) : !b.has_value();
	}
	
	/** Compare the names of this element and the provided element.
	 * @param obj An element for name comparison
	 * @return True if the names match, otherwise false */
	bool eqName(IDA* obj);
	/** <p>Compare the contents of this element and the provided element.</p>
	 * <p>The content type must match, and the contents must match in a type-dependent way
	 * such that each element contains the same <i>information</i>. Numbers must have the
	 * same precision, and scope references must target the same element relative to each
	 * scope. Foreign references are equal by default.</p>
	 * @param obj An element for content comparison
	 * @return True if the contents match, otherwise false */
	bool eqContent(IDA* obj);
	private:
	bool eqContent_ref(IDA* ref_a, IDA* ref_b);
	public:
	/** Compare recursively the parameters of this element and the provided element.
	 * @param obj An element for parameter comparison
	 * @return True if the parameters match, otherwise false */
	bool eqParams(IDA* obj);
	

	// Object/capture
	private:
	IDAObj* repObj = nullptr;
	public:
	/** <p>Top-level article encode operation for a full object tree.
	 * This method calls objEncode, so the top-level object must implement idaEncode.</p>
	 * 
	 * <p>After all encoding is complete, any object references declared via setRefObj will be established as element references in the article.</p>
	 * @param obj A top-level object, which must implement idaEncode, to encode and record as this article element.
	 * @return This article element. */
	IDA* objCapture(IDAObj* obj);
	/** <p>Encode one member object via idaEncode, with automatic element name via idaName.
	 * This element will become "representative" of the object, which is used to encode references via setRefObj.</p>
	 * 
	 * <p>This method should be called from within idaEncode itself, to record members which also implement the IDAObj codec interface.
	 * For the top-level operation, user code should call idaCapture instead to ensure that all object references are recorded.</p>
	 * @param obj An object, which must implement idaEncode, to encode and record as this element.
	 * @return This element. */
	IDA* objEncode(IDAObj* obj);
	private:
	void capture_objs(std::map<IDAObj*, IDA*>* objs);
	void capture_refs(std::map<IDAObj*, IDA*>* refs);
	
	// Object/restore
	private:
	std::vector<std::function<void(IDAObj*)>> repObjActions;
	void on_rep_obj(std::function<void(IDAObj*)> callback);
	void restore_check();
	public:
	/** <p>Top-level article decode operation for a full object tree.
	 * This method calls objDecode, so the top-level object must implement idaDecode.</p>
	 * 
	 * <p>After all decoding is complete, the integrity will be checked to ensure that all object reference requests were resolved.
	 * If any calls to withRefObj did not receive the deferred result, an exception is thrown.
	 * This occurs if the article contained an element reference for which no object was constructed via objDecode.</p>
	 * @param obj A top-level object, which must implement idaDecode, to become the result of decoding this article element.
	 * @return The input object. */
	template <typename T> T* objRestore(T* obj);
	/** <p>Decode one member object via idaDecode.
	 * This element will become "representative" of the object, which is used to decode references via getRefObj or withRefObj.</p>
	 * 
	 * <p>This method should be called from within idaDecode itself, to create members which also implement the IDAObj codec interface.
	 * For the top-level operation, user code should call idaRestore instead to verify that all object reference requests were resolved.</p>
	 * @param obj An object, which must implement idaDecode, to become the result of decoding this element.
	 * @return The input object. */
	template <typename T> T* objDecode(T* obj);
	
	// Deserialisation
	public:
	/** Read from the provided input handle, parse it as markup and use the result to build this element.<br>
	 * After the parser is done, this element will become the top-level entry element from the markup stream.
	 * @param input An open input handle for a stream containing markup data
	 * @return This element */
	IDA* inStream(std::istream* input);
	/** Read the provided string, parse it as markup and use the result to build this element.<br>
	 * After the parser is done, this element will become the top-level entry element from the markup stream.
	 * @param input A string containing markup data
	 * @return This element */
	IDA* inBytes(std::string input);
	/** Read the provided string, parse it as markup and use the result to build this element.<br>
	 * After the parser is done, this element will become the top-level entry element from the markup stream.
	 * @param input A string containing markup data
	 * @return This element */
	IDA* inString(std::string input);
	/** Read from the file at the provided path, parse it as markup and use the result to build this element.<br>
	 * After the parser is done, this element will become the top-level entry element from the markup stream.
	 * @param input A path to a file containing markup data
	 * @return This element */
	IDA* inFile(std::string path);
	
	// Serialisation
	public:
	/** Encode this element as top-level markup and write it to the provided output handle.<br>
	 * If <i>adjacents</i> is true, all preceding and following elements will also be encoded.<br>
	 * If a preceding element was encoded that way, the markup will contain an entry instruction.<br>
	 * Any write errors will be sent back to the caller.
	 * @param output An open output handle to receive a stream of markup data
	 * @param adjacents Whether to write markup for elements adjacent to this one */
	void outStream(std::ostream* output, bool adjacents = false);
	/** Encode this element as top-level markup and write it to the provided output handle.<br>
	 * If <i>adjacents</i> is true, all preceding and following elements will also be encoded.<br>
	 * If a preceding element was encoded that way, the markup will contain an entry instruction.
	 * @param output An open output handle to receive a stream of markup data
	 * @param adjacents Whether to write markup for elements adjacent to this one
	 * @return True on success, or false if a write error occurred */
	bool outPrint(std::ostream* output, bool adjacents = false);
	/** Encode this element as top-level markup and return it as a byte array.<br>
	 * If <i>adjacents</i> is true, all preceding and following elements will also be encoded.<br>
	 * If a preceding element was encoded that way, the markup will contain an entry instruction.
	 * @param adjacents Whether to write markup for elements adjacent to this one
	 * @return A byte array containing the markup data */
	std::string outBytes(bool adjacents = false);
	/** Encode this element as top-level markup and return it as a byte array.<br>
	 * If <i>adjacents</i> is true, all preceding and following elements will also be encoded.<br>
	 * If a preceding element was encoded that way, the markup will contain an entry instruction.
	 * @param adjacents Whether to write markup for elements adjacent to this one.
	 * @return A string containing the markup data. */
	std::string outString(bool adjacents = false);
	/** Encode this element as top-level markup and write it to the file at the provided path.<br>
	 * If <i>adjacents</i> is true, all preceding and following elements will also be encoded.<br>
	 * If a preceding element was encoded that way, the markup will contain an entry instruction.<br>
	 * Any write errors will be sent back to the caller.
	 * @param path A destination file path for the markup data.
	 * @param adjacents Whether to write markup for elements adjacent to this one. */
	void outFile(std::string path, bool adjacents = false);
	/** Encode this element as top-level markup and write it to the provided file.<br>
	 * If <i>adjacents</i> is true, all preceding and following elements will also be encoded.<br>
	 * If a preceding element was encoded that way, the markup will contain an entry instruction.<br>
	 * Any write errors will be sent back to the caller.
	 * @param path A destination file for the markup data.
	 * @param adjacents Whether to write markup for elements adjacent to this one. */
	void outFile(std::filesystem::path path, bool adjacents = false);
	
	// Construction
	public:
	IDA(std::optional<std::string> name = std::nullopt);
	~IDA();
	
	#ifdef IDA_DEBUG_MEMORY
	public:
	static IDA_Debug_Memory<IDA>* dmem;
	#endif
	
	// Initialisation
	public:
	/** <p>Initialisate this element anew.</p>
	 * <p>All existing information comprising this element will be discarded.
	 * The result will be an anonymous, null-content element with no parameters list, no adjacent elements and default input/output options.</p>
	 * @return This element. */
	IDA* reset();
	
	// Copies
	public:
	/** <p>Create a new element as a copy of all aspects of this element.
	 * Any parameters or list content will be cloned recursively.</p>
	 * @param adjacents Whether to include adjacent elements in the cloning process
	 * @return The copy element */
	IDA* copy(bool adjacent = false);
	/** <p>Modify the provided element to become a copy of all aspects of this element.
	 * Any parameters or list content will be cloned recursively.</p>
	 * @param copy An element to become a copy of this element
	 * @param adjacents Whether to include adjacent elements in the cloning process
	 * @return The copy element */
	IDA* copyTo(IDA* copy, bool adjacent = false);
	
	private:
	IDA* copy_inner();
	static IDA* copy_impl(IDA* orig, IDA* copy, bool scanPrev, bool scanNext);
	
	// Utility
	public:
	/** <p>Call the provided function once for each element within this article.</p>
	 * <p>This element is passed first, then its parameters (if any), then its list content items (if any).
	 * Parameters and list items are treated recursively until the full article has been traversed.</p>
	 * <p>Optionally, the walk may also traverse adjacent elements recursively, requested via the reverse and forward flags.</p>
	 * @param func A callback function which will be supplied with article elements.
	 * @param rev Whether to include elements adjacent to this one in the reverse direction.
	 * @param fwd Whether to include elements adjacent to this one in the forward direction. */
	void walk(std::function<void(IDA*)> func, bool rev, bool fwd);
	/** <p>Check the integrity of article element.
	 * If a problem is detected which would lead to errors or corrupted output, an exception is thrown with details in the exception text.</p>
	 * <p>The tests can be performed with adjacent elements enabled or disabled, matching the markup output modes.
	 * Note that disabling adjacents may avoid some problems (such as with linkage), but may also induce others (such as with references).</p>
	 * <p>The top-level element linkage is tested to ensure that there are no loops, and that adjacent elements are doubly-linked correctly.
	 * A bad double-link can disrupt the output flow, and a loop can cause infinite iteration.</p>
	 * <p>Any elements with reference content are tested for foreign references, which become lost/null during output.</p>
	 * @param adjacents Whether to extend the test to include elements adjacent to this one.
	 * @exception IDAOpException if bad integrity is detected. */
	void validate(bool adjacents);
	
	private:
	void validate_loop(bool adjacents);
	void validate_loop_f(std::function<IDA*(IDA*)> step, std::function<IDA*(IDA*)> back);
	void validate_foreign(bool adjacents);
	
	public:
	static void list_add(std::vector<IDA*>* list, IDA* el, long pos, std::string noun);
	static IDA* list_del(std::vector<IDA*>* list, long pos, std::string noun);
	
	// Syntax
	public:
	static bool id0(char c);
	static bool id1(char c);
	static bool comment1(char c);
	static bool comment2(char c1, char c2);
	
	// Compatibility/port through
	public:
	static IDA_Port* port;
	
	// Compatibility
	public:
	static bool eq_obj(IDA* a, IDA* b);
	static bool eq_obj(std::optional<std::string> a, std::optional<std::string> b);
	
	private:
	static bool list_rem(std::vector<IDA*>* list, IDA* el);
	static bool list_equals(std::vector<IDA*>* la, std::vector<IDA*>* lb);
	
};

