//    Copyright 2022 IDA Markup, ida-markup.com
// 
//    Licensed under the Apache License, Version 2.0 (the "License");
//    you may not use this file except in compliance with the License.
//    You may obtain a copy of the License at
// 
//        http://www.apache.org/licenses/LICENSE-2.0
// 
//    Unless required by applicable law or agreed to in writing, software
//    distributed under the License is distributed on an "AS IS" BASIS,
//    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//    See the License for the specific language governing permissions and
//    limitations under the License.

// Namespace

using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Text.RegularExpressions;
using System.Linq;

// Options
public class IDAOptions
{
	// RX
	public bool rxXmlReparseStrings = false;
	
	// TX
	public bool txUnicode = false;
	
	public IDAOptions() {}
	public IDAOptions(IDAOptions options)
	{
		copyFrom(options);
	}
	
	public void copyFrom(IDAOptions options)
	{
		// Copy RX
		this.rxXmlReparseStrings = options.rxXmlReparseStrings;
		
		// Copy TX
		this.txUnicode = options.txUnicode;
	}
	
	public static IDAOptions defaultOptions = new IDAOptions();
}

// Object interface
public interface IDAObj
{
	/** Override to offer a default element name. */
	string idaName();
	/** Override to enable capturing of this object into the provided storage element.<br>
	 * Usage note: If applicable, implementing classes can guarantee not to throw.<br>
	 * @return The input element. */
	IDA idaEncode(IDA el);
	/** Override to enable restoration of this object from the provided storage element.<br>
	 * @return This object. */
	IDAObj idaDecode(IDA el);
	
	// Default implementation
	/*
	public string idaName() {return null;}
	public IDA idaEncode(IDA el)
	{
		throw new IDAOpException("IDA encode not available");
	}
	public IDAObj idaDecode(IDA el)
	{
		throw new IDAOpException("IDA decode not available");
	}
	*/
	
}

// Port begin
public class IDA_Iterator<X>
{
	public virtual bool hasNext() {return false;}
	public virtual X getNext() {return default(X);}
	public X next() {return getNext();}
}
public class IDA_MutableSeq<X> : IDA_Iterator<X>
{
	public Func<bool> funcHas;
	public Func<X> funcGet;
	public IDA_MutableSeq() {this.funcHas = () => false; this.funcGet = () => default(X);}
	public override bool hasNext() {return (this.funcHas)();}
	public override X getNext() {return (this.funcGet)();}
}
public class IDA_Port
{

	// Port iterators
	public IDA_Iterator<X> empty_i<X>()
	{
		return new IDA_Iterator<X>();
	}
	public IDA_Iterator<X> list_i<X>(List<X> l)
	{
		if (l == null) throw new IDAOpException("Internal list iteration error");
		IDA_MutableSeq<X> i = new IDA_MutableSeq<X>();
		int n = 0;
		i.funcHas = () => n < l.Count;
		i.funcGet = () => l[n++];
		return i;
	}
	public IDA_Iterator<X> repeat_i<X>(X obj)
	{
		IDA_MutableSeq<X> seq = new IDA_MutableSeq<X>();
		seq.funcHas = () => true;
		seq.funcGet = () => obj;
		return seq;
	}
	
	// Port arrays
	public bool list_contains(List<IDA> l, IDA item)
	{
		return l.Contains(item);
	}
	public bool list_rem(List<IDA> l, IDA item)
	{
		for (int i=0; i<l.Count; i+=1)
		{
			if (IDA.eq_obj(item, l[i]))
			{
				l.RemoveAt(i);
				return true;
			}
		}
		return false;
	}
	
	// Port streams
	public bool si_allMatch<X>(IDA_Iterator<X> i, Func<X,bool> predicate)
	{
		while (i.hasNext()) if (!predicate(i.next())) {return false;}
		return true;
	}
	public bool si_anyMatch<X>(IDA_Iterator<X> i, Func<X,bool> predicate)
	{
		while (i.hasNext()) if (predicate(i.next())) {return true;}
		return false;
	}
	public bool si_noneMatch<X>(IDA_Iterator<X> i, Func<X,bool> predicate)
	{
		return !this.si_anyMatch<X>(i, predicate);
	}
	
	public IDA_Iterator<Y> i_map<X,Y>(IDA_Iterator<X> i, Func<X,Y> map)
	{
		IDA_MutableSeq<Y> j = new IDA_MutableSeq<Y>();
		j.funcHas = () => i.hasNext();
		j.funcGet = () => map(i.getNext());
		return j;
	}
	public IDA_Iterator<X> i_filter<X>(IDA_Iterator<X> i, Func<X,bool> filter)
	{
		IDA_MutableSeq<X> j = new IDA_MutableSeq<X>();
		X buf = default(X);
		bool more = false;
		Action preLoad = () =>
		{
			more = false;
			while (i.hasNext())
			{
				if (more = filter(buf = i.getNext())) return;
			}
		};
		j.funcHas = () => more;
		j.funcGet = () => {X x = buf; preLoad(); return x;};
		preLoad();
		return j;
	}
	public IDA_Iterator<Y> i_flatMap<X,Y>(IDA_Iterator<X> i, Func<X,IDA_Iterator<Y>> map)
	{
		IDA_MutableSeq<Y> j = new IDA_MutableSeq<Y>();
		IDA_Iterator<Y> lbuf = null;
		Y buf = default(Y);
		bool more = false;
		Action preLoad = () =>
		{
			more = false;
			while (lbuf == null || !lbuf.hasNext())
			{
				if (!i.hasNext()) {return;}
				lbuf = map(i.getNext());
			}
			
			more = true;
			buf = lbuf.getNext();
		};
		j.funcHas = () => more;
		j.funcGet = () => {Y y = buf; preLoad(); return y;};
		preLoad();
		return j;
	}
	
	// Port string operation
	public string replace_all(string s, string act, string dmd)
	{
		return s.Replace(act, dmd);
	}
	
	public string substring(string s, int fromChar, int toChar)
	{
		s = s.Substring(str_units(s, 0, fromChar));
		return s.Substring(0, str_units(s, 0, toChar-fromChar));
	}
	public string substring(string s, int fromChar)
	{
		return s.Substring(str_units(s, 0, fromChar));
	}
	
	public string trim(string s)
	{
		int i1 = 0;
		int i2 = s.Length;
		while (i1 < i2 && s[i1] <= ' ') i1++;
		while (i1 < i2 && s[i2-1] <= ' ') i2--;
		return s.Substring(i1, i2-i1);
	}
	public string rtrim(string s)
	{
		int i2 = s.Length;
		while (0 < i2 && s[i2-1] <= ' ') i2--;
		return s.Substring(0, i2);
	}
	public string ltrim(string s)
	{
		int i1 = 0;
		int i2 = s.Length;
		while (i1 < i2 && s[i1] <= ' ') i1++;
		return s.Substring(i1);
	}
	
	// Port string tests
	public bool ends_with(string s, string suffix)
	{
		return s.EndsWith(suffix);
	}
	public bool starts_with(string s, string prefix)
	{
		return s.StartsWith(prefix);
	}
	public bool str_contains(string s, string mid)
	{
		return s.Contains(mid);
	}
	
	// Port string info
	/** Character index */
	public int str_index(string s, string mid)
	{
		int i = s.IndexOf(mid);
		return i < 1 ? i : str_chars(s, 0, i);
	}
	/** Character index */
	public int str_index_from(string s, string mid, int fromChar)
	{
		int fromUnit = str_units(s, 0, fromChar);
		int i = s.IndexOf(mid, fromUnit);
		return i < 0 ? -1 : (fromChar + str_chars(s, fromUnit, i));
	}
	/** The number of characters represented */
	public int str_len(string s)
	{
		return str_chars(s, 0, s.Length);
	}
	/** The number of characters represented */
	public int str_chars(string s, int fromUnit, int toUnit)
	{
		int n = toUnit - fromUnit;
		for (int i=toUnit-1; i>=fromUnit; i-=1) if (Char.IsLowSurrogate(s[i])) n-=1;
		return n;
	}
	/** The number of memory locations */
	public int str_units(string s, int fromChar, int toChar)
	{
		toChar -= fromChar;
		
		// Skip
		int mark = 0;
		while (fromChar-- > 0) if (Char.IsLowSurrogate(s[++mark])) ++mark;
		
		// Count
		int ptr = mark;
		while (toChar > 0) if (!Char.IsHighSurrogate(s[ptr++])) toChar--;
		
		return ptr - mark;
	}
	
	// Port hex
	public string char_2(char c)
	{
		return ((int)c).ToString("x2");
	}
	public string char_4(char c)
	{
		return ((int)c).ToString("x4");
	}
	public string char_8(int c)
	{
		return c.ToString("x8");
	}
	
// Port end
}

// Format exception
public class IDAFormatException : Exception
{
	// Syntax
	public readonly string ln;
	public readonly int pos;
	
	// Structure
	public readonly IDA node;
	
	public IDAFormatException(IDA_RX rx, string message) : base("(line "+rx.ln+") "+message)
	{
		node = null;
		
		// Extract problem index first
		this.pos = rx.lnBuf.Length;
		
		// Read until end of line for context
		string ln = rx.lnBuf;
		try {ln += rx.get_to('\n');}
		catch (Exception) {}
		this.ln = ln;
		
	}
	
	public IDAFormatException(IDA nodeCause, string message) : base((nodeCause.isAnon() ? "(anonymous element) " : "(node "+nodeCause.getNameOr("null")+") ") + message)
	{
		ln = null;
		pos = -1;
		node = nodeCause;
		
	}
	
	public string getMessage() {return this.Message;}
}

// Op-exception
public class IDAOpException : Exception
{
	public IDAOpException(string message) : base(message) {}
	public string getMessage() {return this.Message;}
}

// Manip begin
public class IDA_Manip
{
	const int FORWARD = int.MaxValue;
	const int REVERSE = -int.MaxValue;
	static int rev(int n) {return -n;}
	
	static string m_str(IDA el)
	{
		if (el.isNull()) return null;
		try {return el.asStr();}
		catch (IDAOpException) // ex
		{throw new IDAFormatException(el, "String required");}
	}
	
	// Manip/iterate
	/** Return the last node */
	public static IDA manip_iterate(IDA manip, IDA node, IDA_Iterator<IDA> itemTemplate)
	{
		IDA last = null;
		while (itemTemplate.hasNext())
		{
			IDA templateNode = itemTemplate.next();
			
			if (last != null) {node = new IDA(); last.linkNext(node);}
			
			node.copyName(templateNode);
			node.copyParams(templateNode);
			if (templateNode.isList())
			{
				node.putList();
				
				if (templateNode.numItems() > 0)
				{
					IDA item = new IDA();
					if (manip_iterate(manip, item, IDA.port.list_i<IDA>(templateNode.asListOr())) != null)
					{
						for (IDA nextEl; item != null; item = nextEl)
						{
							nextEl = item.getNextOr();
							node.addItem(item);
						}
					}
					
				}
			}
			else {node.copyContent(templateNode);}
			last = node;
			
			// Manip/iterate/main
			if (manip.isList()) {
			foreach (IDA m in manip.asList())
			{
				if (m.isAnon() || m.isNull()) {continue;} // Operate on nothing, or no operations
				if (!m.isList()) {throw new IDAFormatException(m, "Manipulation operations must be a list");}
				
				string operateOn = m.getNameOr();
				bool n = operateOn.Contains("n"); // n: name
				bool k = operateOn.Contains("k"); // k: param key
				bool v = operateOn.Contains("v"); // v: param value
				bool p = operateOn.Contains("p"); // p: parameter(s)
				bool c = operateOn.Contains("c"); // c: content
				
				// Manip/iterate/match
				// if-match: any of (n node-name, k param-key, v param-val, p(param-key param-val), c node-content)
				Func<IDA,bool> tParamMatch = x => true;
				if (m.hasParams()) foreach (IDA mp in m.getParamsOr())
				{
					string sel = mp.getNameOr("");
					
					if (Regex.Replace(sel, "[nkvc!]+", "").Length > 0) {throw new IDAFormatException(m, "Invalid manipulation match (valid characters are [nkvc!])");}
					
					bool flagNot = sel.Contains("!");
					
					if (sel.Contains("n"))
					{
						if (node.isName(m_str(mp)) == flagNot) {goto manip_end;}
					}
					if (sel.Contains("c"))
					{
						if (mp.eqContent(node) == flagNot) {goto manip_end;}
					}
					
					if (sel.Contains("k"))
					{
						string tgt = m_str(mp);
						Func<IDA,bool> tParamMatchOld = tParamMatch;
						tParamMatch = x => tParamMatchOld(x) && (flagNot ^ x.isName(tgt));
						break;
					}
					if (sel.Contains("v"))
					{
						Func<IDA,bool> tParamMatchOld = tParamMatch;
						tParamMatch = x => tParamMatchOld(x) && (flagNot ^ mp.eqContent(x));
						break;
					}
					
				}
				Func<IDA,bool> paramMatch = tParamMatch;
				
				// Manip/iterate/op
				foreach (IDA op in m)
				{
					if (op.isAnon()) {throw new IDAFormatException(m, "Missing operation code");}
					string opName = op.getNameOr("");
					
					switch (opName)
					{
					
						// Manip/iterate/op/insert
						case "i": // i: insert (nkvc): i (index) text-string
						{
							if (op.isNull()) {throw new IDAFormatException(m, "Missing insert string");}
							if (op.isList()) {throw new IDAFormatException(m, "Insert string cannot be a list");}
							
							IDA param = op.getParamFirstOr();
							if (param != null && !param.isNum()) {throw new IDAFormatException(m, "Insert position must be a number");}
							int pos = (param != null) ? param.asIntOr(0) : REVERSE;
							
							try
							{
								if (n) {node.setName(manip_ins(node.getNameOr(), op.asStr(), pos));}
								if (c) {node.putStr(manip_ins(node.asStr(), op.asStr(), pos));}
								if (k || v)
								{
									if (node.hasParams()) foreach (IDA np in node.getParams())
									{
										if (!paramMatch(np)) {continue;}
										if (k) {np.setName(manip_ins(np.getNameOr(), op.asStr(), pos));}
										if (v) {np.putStr(manip_ins(np.asStr(), op.asStr(), pos));}
									}
								}
							}
							catch (IDAOpException opex) {throw new IDAFormatException(m, opex.Message);}
							
							break;
						}
						
						// Manip/iterate/op/delete
						case "d": // d: delete (nkvc): d (index) length or d (index) bool
						{
							if (!(op.isNum() || op.isBool() || op.isNull())) {throw new IDAFormatException(m, "Delete amount must be boolean or a number");}
							
							IDA param = op.getParamFirstOr();
							if (param != null && !param.isNum()) {throw new IDAFormatException(m, "Delete position must be a number");}
							int pos = (param != null) ? param.asIntOr(0) : REVERSE;
							int amt = op.isBool() ? (op.isTrue() ? FORWARD : REVERSE) : op.isNull() ? 1 : op.asIntOr(0);
							if (pos == REVERSE && amt > 0) {amt = rev(amt);}
							
							try
							{
								if (n) {node.setName(manip_del(node.getNameOr(), pos, amt));}
								if (c) {node.putStr(manip_del(node.asStr(), pos, amt));}
								if (k || v)
								{
									if (node.hasParams()) foreach (IDA np in node.getParams())
									{
										if (!paramMatch(np)) {continue;}
										if (k) {np.setName(manip_del(np.getNameOr(), pos, amt));}
										if (v) {np.putStr(manip_del(np.asStr(), pos, amt));}
									}
								}
							}
							catch (IDAOpException opex) {throw new IDAFormatException(m, opex.Message);}
							
							break;
						}
						
						// Manip/iterate/op/add
						case "a": // a: add (vpc): a (index) {nodes...}
						{
							// No need to check isList; addItem and insertItem already have exceptions for this
							IDA param = op.getParamFirstOr();
							if (param != null && !param.isNum()) {throw new IDAFormatException(m, "Add position must be a number");}
							int pos = (param != null) ? param.asIntOr(0) : REVERSE;
							
							try
							{
								if (p)
								{
									if (!node.hasParams()) {node.setParams();}
									manip_add(node.addParamAt, pos == REVERSE ? node.numParams() : pos, op);
								}
								if (c)
								{
									manip_add(node.addItemAt, pos == REVERSE ? node.numItems() : pos, op);
								}
								if (v)
								{
									if (node.hasParams()) foreach (IDA np in node.getParams())
									{
										if (!paramMatch(np)) {continue;}
										manip_add(np.addItemAt, pos == REVERSE ? np.numItems() : pos, op);
									}
								}
							}
							catch (IDAOpException opex) {throw new IDAFormatException(m, opex.Message);}
							
							break;
						}
						
						// Manip/iterate/op/remove
						case "r": // r: remove (vpc): r (index) length or r(index) bool or r {nodes...}
						{
							if (op.isNum() || op.isBool() || op.isNull()) // r (index) length or r(index) bool
							{
								IDA param = op.getParamFirstOr();
								if (param != null && !param.isNum()) {throw new IDAFormatException(m, "Remove position must be a number");}
								int pos = (param != null) ? param.asIntOr(0) : REVERSE;
								int amt = op.isBool() ? (op.isTrue() ? FORWARD : REVERSE) : op.isNull() ? 1 : op.asIntOr(0);
								if (pos == REVERSE && amt > 0) {amt = rev(amt);}
								
								if (p)
								{
									manip_rem_pos(m, node.remParamAt, node.numParams, pos, amt);
								}
								if (c)
								{
									manip_rem_pos(m, node.remItemAt, node.numItems, pos, amt);
								}
								if (v)
								{
									if (node.hasParams()) foreach (IDA np in node.getParamsOr())
									{
										if (!paramMatch(np)) {continue;}
										manip_rem_pos(m, np.remItemAt, np.numItems, pos, amt);
									}
								}
							}
							else if (op.isList()) // r {nodes...}
							{
								if (p) {manip_rem_item(node.remParam, op);}
								if (c) {manip_rem_item(node.remItem, op);}
								if (v)
								{
									if (node.hasParams()) foreach (IDA np in node.getParamsOr())
									{
										if (paramMatch(np)) manip_rem_item(np.remItem, op);
									}
								}
							}
							else {throw new IDAFormatException(m, "Remove subject must be boolean, a number or a list");}
							
							break;
						}
						
						// Manip/iterate/op/undo
						case "u": // u: unset/unlist (vpc): u text-key or u {text-key...}
						{
							if (op.isList()) foreach (IDA un in op) // u {text-key...}
							{
								string name = m_str(un);
								
								if (c) {node.unItem(name);}
								if (p) {node.unParam(name);}
								if (v && node.hasParams()) foreach (IDA np in node.getParamsOr())
								{
									if (paramMatch(np)) np.unItem(name);
								}
							}
							else // u text-key
							{
								string name = m_str(op);
								
								if (c) {node.unItem(name);}
								if (p) {node.unParam(name);}
								if (v && node.hasParams()) foreach (IDA np in node.getParamsOr())
								{
									if (paramMatch(np)) np.unItem(name);
								}
							}
							break;
						}
						
						// Manip/iterate/op/substitute
						case "s": // s: substitute (nkvc): s (search-token replacement-string ...);
						{
							if (op.hasParams()) foreach (IDA repl in op.getParamsOr())
							{
								if (repl.isAnon() || repl.isNull()) {continue;}
								string act = repl.getNameOr(), dmd = m_str(repl);
								
								// nkvc
								if (n && node.hasName()) {node.setName(node.getNameOr().Replace(act, dmd));}
								if (c && node.hasContent()) {node.putStr(m_str(node).Replace(act, dmd));}
								if (k || v)
								{
									if (node.hasParams()) foreach (IDA np in node.getParamsOr())
									{
										if (!paramMatch(np)) {continue;}
										if (k && np.hasName()) {np.setName(np.getNameOr().Replace(act, dmd));}
										if (v && np.hasContent()) {np.putStr(m_str(np).Replace(act, dmd));}
									}
								}
							}
							break;
						}
						
						// Manip/iterate/op/flip
						case "f": // f (vc): flip: f;
						{
							try
							{
								if (c) {manip_negate(node);}
								if (v && node.hasParams()) foreach (IDA np in node.getParams())
								{
									if (!paramMatch(np)) {continue;}
									manip_negate(np);
								}
							}
							catch (IDAOpException opex) {throw new IDAFormatException(m, opex.Message);}
							break;
						}
						
						// Manip/iterate/op/write
						case "w": // w (nkvpc): write: w replacement
						{
							if (n) {node.setName(m_str(op));}
							if (c) {node.copyContent(op);}
							if (p)
							{
								if (op.isNull()) {node.clearParams();}
								else if (op.isList())
								{
									node.setParams();
									foreach (IDA param in op.asListOr()) {node.addParam(param.copy());}
								}
								else {throw new IDAFormatException(m, "Parameter write must be list or null");}
							}
							if (k || v)
							{
								if (node.hasParams()) foreach (IDA np in node.getParamsOr())
								{
									if (!paramMatch(np)) {continue;}
									if (k) {np.setName(m_str(op));}
									if (v) {np.copyContent(op);}
								}
							} 
							break;
						}
						
						// Manip/iterate/op end
						default: throw new IDAFormatException(m, "Invalid operation code \""+opName+"\"");
					}
				}
				
			// Manip/iterate end
			manip_end: {} }}
		
		}
		
		return last;
		
	}
	
	// Manip/negate
	static void manip_negate(IDA node)
	{
		if (node.isBool())
		{
			node.putBool(!node.asBool());
		}
		else if (node.isNum())
		{
			string num = node.getNumN();
			node.putNum(num.StartsWith("-") ? num.Substring(1) : "-"+num, node.getNumD());
		}
		else if (node.hasContent()) {throw new IDAOpException("Flip subject must be boolean or a number");}
	}
	
	// Manip/insert
	/** Usage note: the 'ins' string is never null, and the pos can be REVERSE in place of a null to mean the end (never FORWARD though) */
	static string manip_ins(string orig, string ins, int pos)
	{
		if (orig == null) {throw new IDAOpException("Insert subject must not be null");}
		if (pos == REVERSE) {return orig + ins;}
		int l = IDA.port.str_len(orig);
		if (pos > 0)
		{
			if (pos > l) {throw new IDAOpException("Cannot insert string beyond position "+l+" (request was "+pos+")");}
			return IDA.port.substring(orig, 0, pos) + ins + IDA.port.substring(orig, pos);
		}
		else if (pos < 0)
		{
			if (pos < -l) {throw new IDAOpException("Cannot insert string before position -"+l+" (request was "+pos+")");}
			return IDA.port.substring(orig, 0, l+pos) + ins + IDA.port.substring(orig, l+pos);
		}
		else {return ins + orig;}
	}
	
	// Manip/delete
	/** Usage note: the pos can be REVERSE in place of a null to mean the end (never FORWARD though) */
	static String manip_del(string orig, int pos, int amt)
	{
		if (orig == null) {throw new IDAOpException("Delete subject must not be null");}
		int l = IDA.port.str_len(orig);
		if (pos == REVERSE) {pos = l;}
		if (pos >= 0)
		{
			if (pos > l) {throw new IDAOpException("Cannot delete from string beyond position "+l+" (request was "+pos+")");}
			if (amt == REVERSE) {return IDA.port.substring(orig, pos);}
			if (amt == FORWARD) {return IDA.port.substring(orig, 0, pos);}
			if (amt > 0)
			{
				if (pos + amt > l) {throw new IDAOpException("Cannot delete more than "+(l-pos)+" beyond position "+pos+" (request was "+amt+")");}
				return IDA.port.substring(orig, 0, pos) + IDA.port.substring(orig, pos+amt);
			}
			if (amt < 0)
			{
				if (pos + amt < 0) {throw new IDAOpException("Cannot delete more than "+pos+" before position "+pos+" (request was "+(-amt)+")");}
				return IDA.port.substring(orig, 0, pos+amt) + IDA.port.substring(orig, pos);
			}
			return orig;
		}
		else //if (pos < 0)
		{
			if (pos < -l) {throw new IDAOpException("Cannot delete from string before position -"+l+" (request was "+pos+")");}
			if (amt == REVERSE) {return IDA.port.substring(orig, pos+l);}
			if (amt == FORWARD) {return IDA.port.substring(orig, 0, pos+l);}
			if (amt > 0)
			{
				if (pos+l + amt > l) {throw new IDAOpException("Cannot delete more than "+(-pos)+" beyond position "+pos+" (request was "+amt+")");}
				return IDA.port.substring(orig, 0, pos+l) + IDA.port.substring(orig, pos+l+amt);
			}
			if (amt < 0)
			{
				if (pos+l + amt < 0) {throw new IDAOpException("Cannot delete more than "+(l+pos)+" before position "+pos+" (request was "+(-amt)+")");}
				return IDA.port.substring(orig, 0, pos+l+amt) + IDA.port.substring(orig, pos+l);
			}
			return orig;
		}
		
	}
	
	// Manip/add
	static void manip_add(Func<IDA,int,IDA> func_insert, int pos, IDA nodeFrom)
	{
		foreach (IDA ins in nodeFrom)
		{
			func_insert(ins.copy(), pos);
			if (pos >= 0) {pos += 1;}
		}
	}
	
	// Manip/rem
	static void manip_rem_pos(IDA opNode, Action<int> func_delete, Func<int> size, int pos, int amt)
	{
		int n = size();
		if (pos == REVERSE) {pos = n;}
		
		if (amt == FORWARD) {amt = n - pos;}
		else if (amt == REVERSE) {amt = -pos;}
		
		try
		{
			if (amt > 0)
			{
				for (int i=pos+amt-1; i>=pos; i-=1)
				{
					func_delete(i);
				}
			}
			else if (amt < 0)
			{
				for (int i=pos-1; i>=pos+amt; i-=1)
				{
					func_delete(i);
				}
			}
			
		}
		catch (IDAOpException opex) {throw new IDAFormatException(opNode, opex.Message);}
	}
	static void manip_rem_item(Func<IDA,bool> func_remove, IDA nodeFrom)
	{
		foreach (IDA rem in nodeFrom)
		{
			func_remove(rem);
		}
	}
	
// Manip end
}

// RX begin
public class IDA_RX
{
	protected Dictionary<string,IDA> refs = new Dictionary<string,IDA>();
	protected bool has_refTarget = false;
	
	// Data source
	private Stream inStr;
	public IDA_RX(Stream inStr)
	{
		if (inStr == null) {throw new NullReferenceException();}
		this.inStr = inStr;
		// TODO: Impose buffering if possible. Java case is -
		//this.inStr = inStr instanceof BufferedInputStream ? inStr : new BufferedInputStream(inStr);
	}
	
	// RX basic input
	private char c1, c2;
	private bool has_c1 = false, has_c2 = false, eof = false;
	public int ln = 1;
	public string lnBuf = "";
	public int read()
	{
		if (!has_c1) more();
		return has_c1 ? (int) read1() : -1;
	}
	
	private void buf(char c)
	{
		if (c == '\n') {ln+=1; lnBuf = "";} else {lnBuf += c;}
	}
	/** Assumes has_c1 */
	private char read1()
	{
		char output = c1; c1 = c2; has_c1 = has_c2; has_c2 = false;
		more2();
		buf(output);
		return output;
	}
	/** Assumes has_c2 */
	private void skip2()
	{
		if (has_c1) buf(c1);
		if (has_c2) buf(c2);
		has_c1 = has_c2 = false;
		if (more1()) return;
		more2();
	}
	/** Check has_c1 first */
	public char shift()
	{
		char output = c1;
		read();
		return output;
	}
	private void more()
	{
		if (has_c2 || eof) return;
		if (!has_c1 && more1()) return;
		if (!has_c2 && more2()) return;
	}
	/** Returns eof */
	private bool more1()
	{
		if (eof) return true;
		int b = inStr.ReadByte();
		if (b < 0 || b == 4) return eof = true;
		c1 = (char) b;
		has_c1 = true;
		return false;
	}
	/** Returns eof */
	private bool more2()
	{
		if (eof) return true;
		int b = inStr.ReadByte();
		if (b < 0 || b == 4) return eof = true;
		c2 = (char) b;
		has_c2 = true;
		return false;
	}
	
	// RX stream advancement
	bool comment()
	{
		if (!has_c1) return false;
		if (IDA.comment1(c1)) return true;
		return has_c2 && IDA.comment2(c1, c2);
	}
	void adv()
	{
		adv(false);
	}
	void adv(bool checkLn)
	{
		more();
		while (has_c1)
		{
			if (checkLn ? whiteTxt(c1) : white(c1)) {read1();}
			else if (c1 == '#') {read1(); advPast('\n');}
			else if (has_c2)
			{
				if (c1 == '/' && c2 == '/')
				{
					skip2(); advPast('\n');
				}
				else if (c1 == '/' && c2 == '*')
				{
					skip2(); advPast('*', '/');
				}
				else if (c1 == '<' && c2 == '?') // XML header compatibility
				{
					skip2(); advPast('?', '>');
				}
				else if (c1 == '<' && c2 == '!') // XML comment compatibility
				{
					skip2();
					do {advPast('-', '-');} while (has_c1 && !c1e('>')); read1();
				}
				else {return;}
			}
			else {return;}
		}
		
	}
	void advPast(char c)
	{
		more();
		while (has_c1 && c1 != c) {read1();}
		read1();
	}
	void advPast(char cx, char cy)
	{
		more();
		while (has_c2 && (c1 != cx || c2 != cy)) {read1();}
		skip2();
	}
	
	bool abrupt(bool checkLn)
	{
		adv(checkLn);
		return !has_c1;
	}
	
	// RX scan bytes
	bool c1e(char c) {return has_c1 && c1 == c;}
	bool c1n(char c) {return has_c1 && c1 != c;}
	bool c2e(char c) {return has_c2 && c2 == c;}
	bool c2n(char c) {return has_c2 && c2 != c;}
	bool whiteLn(char c) {return c == '\n';}
	bool whiteTxt(char c) {return c <= ' ' && c != '\n';}
	bool white(char c) {return c <= ' ';}
	
	// RX scan numbers
	static bool num0(char c) {return (c>='0' && c<='9') || c == '-' || c == '+' || c == '.';}
	static bool num1(char c) {return num0(c) || c == 'e' || c == 'E';}
	string get_num()
	{
		string output;
		
		if (has_c2 && c1=='0' && c2=='x')
		{
			output = "0x"; skip2();
			while (has_c1 && hex(c1)) {output += shift();}
		}
		else if (has_c2 && c1=='0' && c2=='b')
		{
			output = "0b"; skip2();
			while (has_c1 && bin(c1)) {output += shift();}
		}
		else
		{
			output = "" + shift();
			while (has_c1 && num1(c1)) {output += shift();}
		}
		
		return output;
	}
	static bool hex(char c) {return (c>='0' && c<='9') || (c>='a' && c<='f') || (c>='A' && c<='F');}
	static bool bin(char c) {return c=='0' || c=='1';}
	
	// RX scan opts
	bool opt() {return c1e('~');}
	/** Consumes the /, reads the opt, then advances */
	string get_opt()
	{
		read1();
		string output = get_id();
		adv();
		return output;
	}
	
	// RX scan unicode
	static bool is_u8_marker(char c) {return (c & 0xc0) == 0xc0;}
	static bool is_u8_data(char c) {return (c & 0xc0) == 0x80;}
	
	int shift_u8_data()
	{
		char data = shift();
		if (!is_u8_data(data)) throw new IDAFormatException(this, "Invalid unicode data");
		return data & 0x3f;
	}
	string get_u8()
	{
		int codePoint = 0;
		char marker = shift();
		
		if ((marker & 0xe0) == 0xc0) // 2-byte
		{
			codePoint = (marker & 0x1f) << 6;
			codePoint |= shift_u8_data();
			if (codePoint < 0x0080) throw new IDAFormatException(this, "Overlong unicode character");
		}
		else if ((marker & 0xf0) == 0xe0) // 3-byte
		{
			codePoint = (marker & 0x0f) << 12;
			codePoint |= shift_u8_data() << 6;
			codePoint |= shift_u8_data();
			if (codePoint < 0x0800) throw new IDAFormatException(this, "Overlong unicode character");
		}
		else if ((marker & 0xf8) == 0xf0) // 4-byte
		{
			codePoint = (marker & 0x07) << 18;
			codePoint |= shift_u8_data() << 12;
			codePoint |= shift_u8_data() << 6;
			codePoint |= shift_u8_data();
			if (codePoint < 0x10000) throw new IDAFormatException(this, "Overlong unicode character");
		}
		else throw new IDAFormatException(this, "Invalid unicode marker");
		
		return to_u8(codePoint);
	}
	string to_u8(int codePoint)
	{
		if (codePoint >= 0xd800 && codePoint < 0xe000)
		{
			throw new IDAFormatException(this, "Invalid unicode surrogate");
		}
		else if (codePoint < 0x10000)
		{
			return "" + (char)codePoint;
		}
		else if (codePoint < 0x110000)
		{
			codePoint -= 0x10000;
			return "" + (char)(0xd800 | (codePoint >> 10)) + (char)(0xdc00 | (codePoint & 0x03ff));
		}
		else throw new IDAFormatException(this, "Invalid unicode range");
	}
	
	// RX scan escapes
	static bool is_esc(char c) {return c == '\\';}
	bool esc() {return has_c1 && is_esc(c1);}
	
	int shift_hex(string ex_txt)
	{
		char h = shift();
		if ('0' <= h && h <= '9') return h - '0';
		if ('A' <= h && h <= 'F') return 10 + h - 'A';
		if ('a' <= h && h <= 'f') return 10 + h - 'a';
		throw new IDAFormatException(this, "Invalid "+ex_txt+" escape sequence");
	}
	int get_esc_byte(string ex_txt)
	{
		return (shift_hex(ex_txt) << 4) | shift_hex(ex_txt);
	}
	string get_esc()
	{
		read1();
		if (!has_c1) {throw new IDAFormatException(this, "Incomplete escape sequence");}
		char esc = shift();
		switch (esc)
		{
			case '0': return "\0";
			case 'n': return "\n";
			case 'r': return "\r";
			case 't': return "\t";
			case 'x':
			{
				if (!has_c2) {throw new IDAFormatException(this, "Incomplete byte escape sequence");}
				return to_u8(get_esc_byte("byte"));
			}
			case 'u':
			{
				if (!has_c2) {throw new IDAFormatException(this, "Incomplete unicode escape sequence");}
				int code = get_esc_byte("unicode") << 8;
				if (!has_c2) {throw new IDAFormatException(this, "Incomplete unicode escape sequence");}
				code |= get_esc_byte("unicode");
				return to_u8(code);
			}
			case 'U':
			{
				if (!has_c2) {throw new IDAFormatException(this, "Incomplete unicode extended escape sequence");}
				int code = get_esc_byte("unicode extended") << 24;
				if (!has_c2) {throw new IDAFormatException(this, "Incomplete unicode extended escape sequence");}
				code |= get_esc_byte("unicode extended") << 16;
				if (!has_c2) {throw new IDAFormatException(this, "Incomplete unicode extended escape sequence");}
				code |= get_esc_byte("unicode extended") << 8;
				if (!has_c2) {throw new IDAFormatException(this, "Incomplete unicode extended escape sequence");}
				code |= get_esc_byte("unicode extended");
				return to_u8(code);
			}
			
			default: return ""+esc;
		}
	}
	
	// RX scan IDs
	bool id0() {return IDA.id0(c1);}
	bool id1() {return IDA.id1(c1);}
	string get_tok()
	{
		if (is_str_d(c1)) {return get_str_d(c1);}
		else
		{
			string output = "";
			while (!comment())
			{
				if (esc()) {output += get_esc();}
				else if (has_c1 && !white(c1)) {output += shift();}
				else break;
			}
			return output;
		}
	}
	
	public bool id_symbolic = false;
	string get_id()
	{
		return get_id(false);
	}
	string get_id(bool untilLn)
	{
		string output = "";
		this.id_symbolic = true;
		int holdMin = -1;
		int holdMax = -1;
		
		// Must not end with - or + , that is reserved for boolean false or true
		while (this.has_c1 && !comment())
		{
			int ptr = this.id_symbolic ? output.Length : holdMin;
			bool hold = false;
			
			if (hold = is_esc(c1)) {output += get_esc();}
			else if (hold = is_str_d(c1)) {output += get_str_d(c1);}
			else if (hold = is_var(c1)) {output += get_var();}
			else if (hold = is_u8_marker(c1)) {output += get_u8();}
			else if ((IDA.id1(c1) || (untilLn && (c1 != ';') && !whiteLn(c1))) && !(c1 == '-' && (!has_c2 || !IDA.id1(c2)))) {output += shift();}
			else break;
			
			if (hold)
			{
				holdMin = ptr;
				holdMax = output.Length;
				this.id_symbolic = false;
			}
			
		}
		
		if (untilLn)
		{
			if (this.id_symbolic) {return IDA.port.trim(output);}
			else return IDA.port.ltrim(output.Substring(0,holdMin)) + output.Substring(holdMin, holdMax-holdMin) + IDA.port.rtrim(output.Substring(holdMax));
		}
		else return output;
	}
	static bool is_id_start(char c) {return is_str_d(c) || is_var(c) || is_esc(c) || is_u8_marker(c) || IDA.id0(c);}
	bool c1_id_start() {return has_c1 && is_id_start(c1);}
	
	static bool is_var(char c) {return c == '$';}
	string get_var()
	{
		read();
		string output = "$";
		
		char endChar = '\0';
		if (this.c1e('{')) {endChar = '}';}
		// TODO: Other common uses of $(...)
		
		if (endChar != '\0') output += read1() + get_to(endChar) + endChar;
		return output;
	}
	
	string xml_att_key()
	{
		return is_id_start(this.c1) ? this.get_id() : this.get_pr(c => c != '=' && c != '/' && c != '>' && !this.white(c));
	}
	
	// RX scan strings
	bool c1_str() {return has_c1 && is_str(c1);}
	bool c1_tok() {return has_c1 && !white(c1);}
	static bool is_str_d(char c) {return c == '"' || c == '`' || c == '\'';}
	static bool is_str(char c) {return is_str_d(c) || IDA.id1(c);}
	
	string get_str()
	{
		return is_str_d(c1) ? get_str_d(c1) : get_id();
	}
	string get_str_d(char delim)
	{
		string output = "";
		if (c1e(delim))
		{
			read1();
			while (c1n(delim))
			{
				if (esc()) {output += get_esc();}
				else if (is_u8_marker(c1)) {output += get_u8();}
				else {output += read1();}
			}
			read();
		}
		return output;
	}
	
	// RX scan logic
	public string get_pr(Func<char,bool> p)
	{
		string output = "";
		while (has_c1 && p(c1)) {output += shift();}
		return output;
	}
	public string get_prior_to(char delim)
	{
		string output = get_pr(c => c != delim);
		return output;
	}
	public string get_to(char delim)
	{
		string output = get_prior_to(delim);
		read();
		return output;
	}
	
	// RX class iteration
	// Empty iterator
	static IDA_Iterator<IDA> CLASS_NONE = IDA.port.empty_i<IDA>();
	// Infinite iterator of blank nodes
	private static IDA_Iterator<IDA> CLASS_ARRAY = IDA.port.repeat_i<IDA>(new IDA());
	// Specific class node iterator
	private static IDA_Iterator<IDA> classOf(List<IDA> itemClassList)
	{
		return itemClassList != null ? IDA.port.list_i(itemClassList) : CLASS_NONE;
	}
	
	// RX entry
	/** Returns the output argument */
	public static IDA inObj(IDA output, Stream inStr)
	{
		IDA_RX rx = new IDA_RX(inStr);
		IDA globalScope = new IDA();
		bool entry = false;
		IDA entryNode = new IDA().rxScope(globalScope).setIncludes(output.rxIncludePwd());
		entryNode.rxInputList(output.inputListDepth, output.inputListLength);
		entryNode.copyOptions(output);
		
		rx.adv();
		while (rx.opt()) switch (rx.get_opt())
		{
			case "": entry = true; break;
		}
		IDA node = fill(entryNode, rx, '\0', '\0', null, CLASS_NONE, 0);
		
		while (rx.has_c1 && node != null)
		{
			IDA nextNode = new IDA().rxScope(globalScope).setIncludes(output.rxIncludePwd());
			nextNode.rxInputList(output.inputListDepth, output.inputListLength);
			nextNode.copyOptions(output);
			while (rx.opt()) switch (rx.get_opt())
			{
				case "":
				{
					if (!entry) {entry = true; entryNode = nextNode;}
					break;
				}
			}
			
			IDA last = fill(nextNode, rx, '\0', '\0', node, CLASS_NONE, 0);
			node.linkNext(nextNode);
			node = last;
		}
		
		output.reset();
		rx.transfer_nodes(entryNode, output);
		output.walk(x => x.rxScope(null), true, true);
		
		if (rx.has_refTarget) output.walk(x => rx.put_ref(x), true, true);
		
		return output;
	}
	void transfer_nodes(IDA src, IDA dst)
	{
		// Name
		dst.copyName(src);
		
		// Parameters
		if (src.hasParams())
		{
			dst.setParams();
			foreach (IDA p in src.getParamsOr()) dst.addParam(p);
			src.clearParams();
		}
		else dst.clearParams();
		
		// Content
		if (src.isList())
		{
			dst.putList();
			foreach (IDA i in src.asListOr()) dst.addItem(i);
			src.putNull();
		}
		else dst.copyContent(src);
		
		// Other
		dst.copyOptions(src);
		if (src.hasRefName()) {refs[src.getRefName()] = dst; dst.setRefName(src.getRefName());}
		
		dst.link(src.getPrevOr(), src.getNextOr());
	}
	void put_ref(IDA el)
	{
		string t = el.getRefTarget();
		if (t != null)
		{
			if (!refs.ContainsKey(t)) throw new IDAFormatException(el, "Identifier not defined");
			el.putRef(refs[t]);
		}
	}
	
	// RX fill
	/** Decodes element nodes read from the RX, using the supplied object as a container for the first such element.
	 * Adjacent elements will be created as needed and linked to the first, accessible using its getNext method.<br>
	 * @return The last consecutive element decoded, or null if no elements were present.
	 * @param node A container into which the first decoded element should be written.
	 * @param rx The serial input to be read.
	 * @param paren_l The character which caused <i>fill</i> to be called recursively, or zero to admit no such structure.
	 * @param paren_r The character which should mark the end of a recursive, structured call to <i>fill</i>.
	 * @param prev An element to be considered prior to the decoded element for the purpose of interpreting <i>as-above</i>
	 * markup, or null if no such element exists.
	 * @param listClass A class element to use for copying implicit element names, or an empty iterator (such as CLASS_NONE).
	 * @param depth The effective recursion level.
	 * @throws IOException If a error occurs while reading input bytes.
	 * @throws IDAFormatException If the parser encounters a syntax error, or if the markup contained an invalid
	 * <i>manipulation</i> which would throw {@link IDAOpException} if called programmatically. */
	private static IDA fill(IDA node, IDA_RX rx, char paren_l, char paren_r, IDA prev, IDA_Iterator<IDA> listClass, int depth)
	{
		
		// RX fill/state
		bool jsonVal = false;
		bool forceContent = false; // Used for : and = in an array context
		bool needName = true;
		string prelimName = null;
		bool nameSet = false;
		bool needContent = true;
		bool checkDelim = true; // For ; or , between nodes
		IDA itemClass = null;
		IDA output = null;
		bool lcn = listClass.hasNext();
		
		// RX read
		while (rx.has_c1 && needContent)
		{
			
			// RX read/string implicit
			// This should be first
			// NOTE: By testing this prior to num(), names/strings beginning "e" will not be a number
			if (rx.c1_id_start())
			{
				output = node;
				
				if (!needName)
				{
					if (listClass == CLASS_ARRAY && !forceContent)
					{
						nameSet = true;
						needName = false;
						node.putStr(prelimName);
					}
					else
					{
						// String content
						// Potentially also true/false/null if JSON-style following a :
						string s = rx.get_id(depth == 0);
						
						if (jsonVal && rx.id_symbolic)
						{
							if ("true" == s) {node.putTrue();}
							else if ("false" == s) {node.putFalse();}
							else if ("null" == s) {node.putNull();}
							else {node.putStr(s);}
						}
						else {node.putStr(s);}
						
					}
					
					needContent = false;
				}
				else
				{
					prelimName = rx.get_id();
					if (rx.abrupt(paren_r == '\n')) {node.putStr(prelimName); prelimName = null; needContent = false;}
					else {needName = false;}
				}
			}
			
			// RX read/delim explicit
			else if (rx.c1e(';') || rx.c1e(','))
			{
				if (!((rx.c1 == ',') && (depth == 0)))
				{
					// Consume delimiter unless using CSV compatibility
					rx.read1();
					checkDelim = false;
				}
				
				node.putNull();
				needContent = false;
				output = node;
			}
			
			// RX read/delim implicit
			else if (rx.c1e('}') || rx.c1e(']') || rx.c1e(')'))// || rx.c('<'))
			{
				if (paren_l != ':')
				{
					if (rx.c1e('}')) {if (paren_l != '{') throw new IDAFormatException(rx, "List mismatch");}
					else if (rx.c1e(']')) {if (paren_l != '[') throw new IDAFormatException(rx, "Array mismatch");}
					else /*if (rx.c1e(')'))*/ {if (paren_l != '(') throw new IDAFormatException(rx, "Parameters mismatch");}
				}
				
				if (!nameSet)
				{
					if (!needName && lcn) {lcn = false; node.setName(listClass.next().getNameOr()); node.putStr(prelimName);}
					else {node.setName(prelimName);}
					nameSet = true;
				}
				else
				{
					node.putNull();
				}
				
				needContent = false;
				checkDelim = false;
			}
			
			// RX read/as-above
			else if (rx.c1e('^'))
			{
				if (prev == null) {throw new IDAFormatException(rx, "No previous element");}
				
				if (needName)
				{
					prelimName = prev.getNameOr();
					needName = false;
				}
				else
				{
					node.copyContent(prev);
					needContent = false;
				}
				
				if (rx.c2e('^')) {node.copyParams(prev); rx.skip2();} else {rx.read1();}
				output = node;
			}
			
			// RX read/identifier definition
			else if (rx.c1e('*'))
			{
				rx.read1();
				string key = (rx.c1_str()) ? rx.get_str() : "";
				rx.adv();
				
				if (rx.refs.ContainsKey(key)) throw new IDAFormatException(rx, "Identifier already defined");
				rx.refs[key] = node;
				node.setRefName(key);
				
				output = node;
			}
			
			// RX read/identifier reference
			else if (rx.c1e('&'))
			{
				rx.read1();
				string key = (rx.c1_str()) ? rx.get_str() : "";
				rx.adv();
				
				node.setRefTarget(key);
				rx.has_refTarget = true;
				
				output = node;
				needContent = false;
			}
			
			// RX read/XML
			else if (rx.c1e('<')) // XML compatibility
			{
				output = node;
				
				if (!needName)
				{
					if (listClass == CLASS_ARRAY && !forceContent)
					{
						nameSet = true;
						needName = false;
						node.putStr(prelimName);
						needContent = false;
					}
					else if (paren_r == '<') {needContent = false;}
					else throw new IDAFormatException(rx, "Cannot nest XML element header");
				}
				else
				{
					rx.read1();
					rx.adv();
					if (rx.has_c1 && rx.id1())
					{
						node.setName(rx.get_id());
						needName = false;
						nameSet = true;
					}
					else {throw new IDAFormatException(rx, "Invalid XML element name");}
					rx.adv();
					
					while (rx.has_c1 && needContent) // XML inners
					{
						if (rx.c1e('>')) // XML child elements
						{
							rx.read1();
							List<IDA> l = new List<IDA>();
							IDA last = null;
							while (rx.has_c1 && needContent)
							{
								if (rx.c1e('<'))
								{
									if (rx.c2e('/')) // End node
									{
										rx.skip2(); rx.adv();
										string end = rx.get_to('>').Trim();
										if (!node.isName(end)) {throw new IDAFormatException(rx, "Incorrect XML element footer");}
										needContent = false;
									}
									else if (rx.c2e('!')) // XML comment
									{
										rx.skip2();
										do {rx.advPast('-', '-');} while (rx.c1n('>')); rx.read1();
									}
									else // Child element
									{
										IDA el = node.rxNew();
										fill(el, rx, '<', '>', last, CLASS_NONE, depth+1);
										last = el;
										l.Add(el);
									}
								}
								else
								{
									if (node.opts.rxXmlReparseStrings)
									{
										rx.adv();
										
										IDA el = node.rxNew();
										
										last = fill(el, rx, '>', '<', last, CLASS_NONE, 0);
										while (el != null && (node.inputListLength < 0 || node.inputListLength > l.Count))
										{
											l.Add(el);
											el = el.getNextOr();
										}
									}
									else
									{
										string content = rx.get_prior_to('<');
										
										if (content.Trim().Length > 0)
										{
											IDA el = new IDA().putStr(content);
											l.Add(el);
										}
										
									}
								}
							}
							
							node.putList(l);
						}
						else if (rx.c1e('/') && rx.c2e('>')) // Empty XML node
						{
							rx.skip2();
							rx.adv();
							node.putNull();
							needContent = false;
						}
						else // XML attrib
						{
							string attKey = rx.xml_att_key();
							while (is_id_start(rx.c1) && (rx.c1n('/') || rx.c2n('>'))) attKey += rx.xml_att_key();
							rx.adv();
							string attValue;
							
							if (rx.c1e('='))
							{
								rx.read1();
								attValue = rx.get_str();
								rx.adv();
							}
							else
							{
								attValue = null;
							}
							
							IDA att = new IDA(attKey).putStr(attValue);
							node.addParam(att);
							
						}
					}
				}
			}
			
			// RX read/params
			else if (rx.c1e('('))
			{
				if (node.hasParams())
				{
					needContent = false;
				}
				else
				{
					output = node;
					
					// Parameters also cause content delim
					forceContent = true;
					
					rx.read1();
					rx.adv();
					if (rx.c1e('^'))
					{
						if (prev == null) {throw new IDAFormatException(rx, "No previous element");}
						node.copyParams(prev);
						rx.read1();
						rx.adv();
					}
					else
					{
						node.setParams();
					}
					
					IDA last = null;
					IDA_Iterator<IDA> cItems = classOf(itemClass != null ? itemClass.getParamsOr() : null);
					while (rx.c1n(')'))
					{
						IDA param = node.rxNew();
						last = fill(param, rx, '(', ')', last, cItems, depth+1);
						if (last != null) while (param != null)
						{
							// Snapshot the next before addParam breaks it
							IDA param_next = param.getNextOr();
							node.addParam(param);
							param = param_next;
						}
					}
					while (cItems.hasNext())
					{
						IDA param = cItems.next().copy().rxScope(node);
						node.addParam(param);
					}
					rx.read();
				}
			}
			
			// RX read/content-delim
			else if ((rx.c1e(':') && !needName) || rx.c1e('='))
			{
				if (rx.read1() == ':') {jsonVal = true;}
				needName = false;
				nameSet = true;
				node.setName(prelimName);
				forceContent = true;
				output = node;
			}
			
			// RX read/null
			else if (rx.c1e('?'))
			{
				output = node;
				
				if (needName)
				{
					rx.read1();
					prelimName = null;
					needName = false;
				}
				else
				{
					if (listClass == CLASS_ARRAY && !forceContent)
					{
						nameSet = true;
						needName = false;
						node.putStr(prelimName);
					}
					else
					{
						rx.read1();
						node.putNull();
					}
					needContent = false;
				}
			}
			
			// RX read/number
			else if (num0(rx.c1))
			{
				if (!needName && listClass == CLASS_ARRAY && !forceContent)
				{
					nameSet = true;
					needName = false;
					node.putStr(prelimName);
				}
				else
				{
					string s, num, unit = "", den = "1";
					
					s = num = rx.get_num();
					
					// Check for denominator
					if (rx.c1e('/'))
					{
						rx.read1();
						
						if (!rx.has_c1 || !num0(rx.c1)) {throw new IDAFormatException(rx, "Missing denominator");}
						den = rx.get_num();
						s += "/"+den;
					}
					
					// Check for unit
					if (rx.has_c1 && rx.id0())
					{
						unit = rx.get_id();
						s += unit;
					}
					
					switch (s)
					{
						// Bools with +, -
						case "+": node.putTrue(); break;
						case "-": node.putFalse(); break;
						
						// Actual number
						default:
						{
							rx.testNumber(num, "numerator");
							if (den != "1") {rx.testNumber(den, "denominator");}
							node.putNum(num, den, unit);
							break;
						}
					}
				}
				needContent = false;
				output = node;
			}
			
			// RX read/list array
			else if (rx.c1e('['))
			{
				output = node;
				
				if (!needName && listClass == CLASS_ARRAY && !forceContent)
				{
					nameSet = true;
					needName = false;
					node.putStr(prelimName);
				}
				else
				{
					rx.read1();
					rx.adv();
					List<IDA> listObj = new List<IDA>();
					
					if (node.inputListDepth != 0)
					{
						IDA last = null;
						while (rx.c1n(']'))
						{
							IDA el = node.rxNew();
							last = fill(el, rx, '[', ']', last, CLASS_ARRAY, depth+1);
							if (last != null) while (el != null && (node.inputListLength < 0 || node.inputListLength > listObj.Count))
							{
								listObj.Add(el);
								el = el.getNextOr();
							}
						}
					}
					else discard(rx, '[', ']');
					
					rx.read();
					node.putList(listObj);
				}
				needContent = false;
			}
			
			// RX read/list object
			else if (rx.c1e('{'))
			{
				output = node;
				
				if (!needName && listClass == CLASS_ARRAY && !forceContent)
				{
					nameSet = true;
					needName = false;
					node.putStr(prelimName);
				}
				else
				{
					rx.read1();
					rx.adv();
					List<IDA> listObj = new List<IDA>();
					
					if (node.inputListDepth != 0)
					{
						IDA last = null;
						IDA_Iterator<IDA> cItems = classOf(itemClass != null ? itemClass.asListOr() : null);
						while (rx.c1n('}'))
						{
							IDA el = node.rxNew();
							last = fill(el, rx, '{', '}', last, cItems, depth+1);
							if (last != null) while (el != null && (node.inputListLength < 0 || node.inputListLength > listObj.Count))
							{
								listObj.Add(el);
								el = el.getNextOr();
							}
						}
						while (cItems.hasNext() && (node.inputListLength < 0 || node.inputListLength > listObj.Count))
						{
							IDA el = cItems.next().copy().rxScope(node);
							listObj.Add(el);
						}
					}
					else discard(rx, '{', '}');
					
					rx.read();
					node.putList(listObj);
				}
				needContent = false;
			}
			
			// RX read/class definition
			else if (rx.c1e(':') && rx.c2e(':'))
			{
				rx.skip2();
				string key = (rx.c1_str()) ? rx.get_str() : "";
				rx.adv();
				
				IDA el = node.rxNew();
				fill(el, rx, ':', '\0', null, CLASS_NONE, depth+1);
				node.rxAddItemClass(key, el);
			}
			
			// RX read/class instance
			else if (rx.c1e(':'))
			{
				if (paren_l == ':')
				{
					needContent = false;
				}
				else
				{
					rx.read1();
					string key = (rx.c1_str()) ? rx.get_str() : "";
					rx.adv();
					
					itemClass = node.rxGetItemClass(key);
					if (itemClass == null) {throw new IDAFormatException(rx, "Item class \""+key+"\" not found");}
					prelimName = itemClass.getNameOr();
					output = node;
				}
			}
			
			// RX read/template definition
			else if (rx.c1e('!'))
			{
				if (!needName) {throw new IDAFormatException(rx, "Incomplete element prior to template definition");}
				rx.read1();
				string key = (rx.c1_str()) ? rx.get_str() : "";
				rx.adv();
				
				IDA itemTemplate = node.rxNew();
				fill(itemTemplate, rx, '\0', '\0', null, CLASS_NONE, depth+1);
				if (!itemTemplate.isList()) {throw new IDAFormatException(rx, "Template definition must be a list");}
				node.rxAddTemplate(key, itemTemplate);
			}
			
			// RX read/template instance
			else if (rx.c1e('@'))
			{
				// Hand-over to file include for @? and @@ 
				bool inc_opt = rx.c2e('?');
				if (inc_opt || rx.c2e('@'))
				{
					if (!needName) {throw new IDAFormatException(rx, "Incomplete element prior to include");}
					rx.skip2();
					if (!rx.c1_tok()) {throw new IDAFormatException(rx, "Missing include path");}
					
					string path = rx.get_tok();
					rx.adv();
					
					bool abort = false;
					try
					{
						// Platform-specific check (C#)
						string pwd = node.rxIncludePwd();
						if (abort = pwd == null) {throw new IDAOpException("Includes are disabled");}
						
						bool chdir = false;
						if (path.StartsWith("+")) {chdir = true; path = path.Substring(1);}
						
						// TODO: Missing aspects of file includes
						string inc = path;
						if (!Path.IsPathRooted(inc)) {inc = Path.Combine(pwd, path);}
						
						if (abort = !File.Exists(inc))
						{
							if (Directory.Exists(inc)) throw new IDAOpException("Include is not a file");
							throw new IDAOpException("Include path does not exist");
						}
						
						IDA template_adj = new IDA().rxScope(node);
						if (chdir)
						{
							DirectoryInfo info = Directory.GetParent(inc);
							template_adj.setIncludes(info != null ? info.FullName : null);
						}
						else {template_adj.setIncludes(node.rxIncludePwd());}
						using (FileStream fs = File.OpenRead(inc))
						{
							if (abort = !fs.CanRead) {throw new IDAOpException("Include file has no read access");}
							template_adj.inStream(fs);
						}
						
						IDA last = manip(rx, node, template_adj.iterateNext());
						
						nameSet = true;
						checkDelim = false;
						needContent = false;
						output = last;
						
					}
					catch (IDAOpException opex)
					{
						// Hold off expending the RX line on an exception when the include was optional
						if (!inc_opt) throw new IDAFormatException(rx, opex.Message);
						
						// Discard the manipulation node
						if (abort) {fill(new IDA(), rx, '\0', '\0', null, CLASS_NONE, depth+1);}
						
					}
					
				}
				else
				{
					if (!needName) {throw new IDAFormatException(rx, "Incomplete element prior to template instance");}
					rx.read1();
					string key = (rx.c1_str()) ? rx.get_str() : "";
					rx.adv();
					
					IDA itemTemplate = node.rxGetTemplate(key);
					if (itemTemplate == null) {throw new IDAFormatException(rx, "Template \""+key+"\" not found");}
					
					IDA last = manip(rx, node, IDA.port.list_i<IDA>(itemTemplate.asListOr()));
					
					if (last != null)
					{
						nameSet = true;
						checkDelim = false;
						needContent = false;
						output = last;
					}
					
					
				}
				
			}
			
			// RX read/invalid
			// This should be last
			else
			{
				throw new IDAFormatException(rx, String.Format("Invalid character 0x{0:x2} '{1}'", (int)rx.c1, rx.c1));
			}
			
			// RX read delimiter
			if (paren_r != '>')
			{
				bool csvLn = paren_r == '\n';
				rx.adv(csvLn);
				csvLn &= rx.c1e('\n');
				if (checkDelim && (rx.c1e(';') || rx.c1e(',') || csvLn) )
				{
					if (needContent && lcn)
					{
						lcn = false;
						node.setName(listClass.next().getNameOr());
						node.putStr(prelimName);
					}
					else if (needName && lcn)
					{
						lcn = false;
						node.setName(listClass.next().getNameOr());
					}
					else if ((rx.c1 == ',') && (depth == 0))
					{
						// CSV compatibility
						rx.read1();
						rx.adv(true);
						
						// Convert to list with new first element
						IDA first = new IDA();
						first.rxConfig(node);
						if (needContent) {first.putStr(prelimName);} else {node.copyTo(first);}
						
						// Complete the row
						List<IDA> listObj = new List<IDA>();
						listObj.Add(first);
						IDA last = null;
						IDA_Iterator<IDA> cItems = CLASS_ARRAY;
						while (rx.c1n('\n'))
						{
							IDA el = node.rxNew();
							last = fill(el, rx, '\0', '\n', last, cItems, depth+1);
							if (last != null) while (el != null && (node.inputListLength < 0 || node.inputListLength > listObj.Count))
							{
								listObj.Add(el);
								el = el.getNextOr();
							}
						}
						
						node.putList(listObj);
						node.setName(null);
						
					}
					else if (csvLn)
					{
						// Last CSV item
						if (needContent) {node.putStr(prelimName);}
					}
					else if (!nameSet)
					{
						node.setName(prelimName);
					}
					
					nameSet = true;
					needContent = false;
					
					if (!csvLn) {rx.read1(); rx.adv();}
					
				}
			}
			
			// RX read END
		}
		
		// RX fill name
		if (!nameSet)
		{
			if (needName && lcn) {lcn = false; node.setName(listClass.next().getNameOr());}
			else {node.setName(prelimName);}
		}
		
		// RX fill END
		if (lcn) listClass.next();
		
		node.rxClearTemplates();
		node.rxClearItemClasses();
		return output;
	}
	
	// RX discard
	private static void discard(IDA_RX rx, char paren_l, char paren_r)
	{
		while (rx.c1n(paren_r))
		{
			if (rx.c1e('}') || rx.c1e(']') || rx.c1e(')'))// || rx.c('<'))
			{
				if (paren_l != ':')
				{
					if (rx.c1e('}')) {if (paren_l != '{') throw new IDAFormatException(rx, "List mismatch");}
					else if (rx.c1e(']')) {if (paren_l != '[') throw new IDAFormatException(rx, "Array mismatch");}
					else /*if (rx.c1e(')'))*/ {if (paren_l != '(') throw new IDAFormatException(rx, "Parameters mismatch");}
				}
			}
			
			
			//else if (rx.c1e('<')) // XML compatibility
			//{
			//}
			else if (rx.c1e('(')) // Params
			{
				rx.read1();
				rx.adv();
				while (rx.c1n(')')) discard(rx, '(', ')');
				rx.read();
			}
			else if (rx.c1e('"') || rx.c1e('\'') || rx.c1e('`'))
			{
				rx.get_str();
				rx.adv();
			}
			else if (rx.c1e('['))
			{
				rx.read1();
				rx.adv();
				while (rx.c1n(']')) discard(rx, '[', ']');
				rx.read();
			}
			else if (rx.c1e('{'))
			{
				rx.read1();
				rx.adv();
				while (rx.c1n('}')) discard(rx, '{', '}');
				rx.read();
			}
			else if (rx.c1e(':'))
			{
				rx.read1();
				if (rx.c1_str()) rx.get_str();
				rx.adv();
				discard(rx, ':', '\0');
			}
			else if (rx.c1e('!'))
			{
				rx.read1();
				if (rx.c1_str()) rx.get_str();
				rx.adv();
				discard(rx, '\0', '\0');
			}
			else
			{
				rx.read1();
			}
		}
	}
	
	// RX number test
	private static Regex p_num = new Regex("^[\\+\\-]?[0-9]*(?:\\.[0-9]+)?([eE][\\+\\-]?[0-9]+)?$");
	void testNumber(string num, string desc)
	{
		// No need to test hex/bin here; invalid characters will have delimited the number, potentially becoming the unit
		if (IDA.port.starts_with(num, "0x") || IDA.port.starts_with(num, "0b")) {return;}
		
		if (!p_num.IsMatch(num)) {throw new IDAFormatException(this, "Invalid "+desc+" number");}
	}
	
	// RX manip
	/** Return the last node */
	static IDA manip(IDA_RX rx, IDA node, IDA_Iterator<IDA> itemTemplate)
	{
		// Manipulations
		IDA manip = new IDA();
		fill(manip, rx, '\0', '\0', null, CLASS_NONE, 1);
		if (manip.isList())
		{
			// TODO: any pre-process or validation of manipulations, or simplify logic if none needed
		}
		else if (manip.hasContent()) {throw new IDAFormatException(rx, "Template manipulation must be a list");}
		
		return IDA_Manip.manip_iterate(manip, node, itemTemplate);
	}
	
// RX end
}

// Output interface
public class IDA_Output
{
	private TextWriter w;
	
	public IDA_Output(TextWriter output)
	{
		this.w = output;
		// TODO: Impose buffering if possible. Java case is -
		//w = new BufferedWriter(underlying_stream);
	}
	
	protected void send(string s) {w.Write(s);}
	
	private string queueBuf = "";
	protected void queue(string s) {queueBuf += s;}
	protected int length() {return queueBuf.Length;}
	protected void reject() {queueBuf = "";}
	protected void accept() {send(queueBuf); queueBuf = "";}
	
	
	/** Element will not be null<br>
	 * This will receive the entry element, which is not necessarily the first to be output<br>
	 * The adjacents-flag is provided if any special state is needed to handle multiple elements to follow */
	protected virtual void ida_header(IDA el, bool adj) {}
	/** Element will not be null<br>
	 * This is called potentially multiple times if adjacents are enabled */
	protected virtual void ida_element(IDA el, bool adj) {}
	/** Called once after all elements are done */
	protected virtual void ida_footer() {}
	
	public void idaWrite(IDA el, bool adj)
	{
		if (el == null) {return;}
		
		el.txRefLabel(adj);
		
		ida_header(el, adj);
		
		if (adj)
		{
			// Rewind first
			while (el.hasPrev()) {el = el.getPrevOr();}
			
			// Output forward
			while (el != null)
			{
				ida_element(el, true);
				el = el.getNextOr();
			}
		}
		else {ida_element(el, false);}
		
		ida_footer();
		
		w.Flush();
	}
	
	/** Same as idaWrite, but converts exceptions into a return value instead.<br>
	 * True if success, false if a IO problem occurred */
	public bool idaPrint(IDA el, bool adj)
	{
		try {idaWrite(el, adj); return true;}
		catch (Exception) {return false;}
	}
	
}

// TX begin
public class IDA_TX : IDA_Output
{
	
	// TX defaults
	private static int BREVITY_ALL = 3;
	private static int BREVITY_PART = 6;
	private static int ONE_LINE_MAX = 132;
	
	// TX instance
	private IDA entry = null;
	private bool rootAsAboveAll = false;
	private IDA rootAsAboveFirst = null;
	private bool rootAsAbovePart = false;
	private string rootAsAboveName = null;
	
	private bool optsOverride = false;
	public IDA_TX setOptsOverride(bool ov) {optsOverride = ov; return this;}
	private IDAOptions opts(IDA el) {return optsOverride ? IDAOptions.defaultOptions : el.opts;}
	
	public IDA_TX(TextWriter output) : base(output)
	{
	}
	
	// TX interface
	protected override void ida_header(IDA el, bool adj)
	{
		entry = el;
	}
	protected override void ida_element(IDA el, bool adj)
	{
		// Explicit entry point?
		if (adj && entry == el && el.hasPrev()) {send("~ ");}
		
		// As-above all?
		if (rootAsAboveFirst == null)
		{
			rootAsAboveFirst = el;
			rootAsAboveAll = as_above_all_test(el);
		}
		
		// Main output wrapped with name as-above part
		if (rootAsAbovePart && !el.isName(rootAsAboveName)) {rootAsAbovePart = false;}
		ida(el, "", false, rootAsAbovePart || (adj && rootAsAboveAll && el.hasPrev()), false, null);
		rootAsAboveName = el.getNameOr();
		if (!rootAsAbovePart) rootAsAbovePart = as_above_part_test(el);
	}
	
	// TX structure filter
	private bool has_inner_structure(IDA first)
	{
		for (IDA outer = first; outer != null; outer = outer.getNextOr())
		{
			if (outer.isList()) foreach (IDA inner in outer.asListOr())
			{
				if (inner.hasParams() || inner.isList()) return true;
			}
		}
		return false;
	}
	
	// TX main begin
	private void ida(IDA node, string indent, bool parameters, bool nameAsAbove, bool nameOmit, IDA itemClass)
	{
		bool hasOutput = false;
		
		// TX main indent
		send(indent);
		if (node.hasRefName()) send("*"+node.getRefName()+" ");
		if (itemClass != null) send(": ");
		
		// TX main name
		if (!nameOmit)
		{
			string name = "";
			if (node.hasName()) {name = nameAsAbove ? "^" : id_str(node, node.getNameOr());}
			else if (!node.hasParams() && !node.isList()) {name = "?";}
			
			send(name);
			hasOutput = name.Length > 0;
		}
		
		// TX main parameters
		bool chain = false;
		if (node.hasParams())
		{
			if (hasOutput) {send(" ");}
			hasOutput = true;
			send("(");
			IDA_Iterator<IDA> pNames = itemClass != null && itemClass.hasParams() ? IDA.port.list_i<IDA>(itemClass.getParams()) : IDA.port.empty_i<IDA>();
			for (IDA param = node.getParamFirstOr(); param != null; param = param.getNextOr())
			{
				bool pNameOmit = pNames.hasNext() && pNames.next().eqName(param);
				if (chain && !pNameOmit) {send(" ");}
				chain = true;
				ida(param, "", true, false, pNameOmit, null);
			}
			send(")");
		}
		
		// TX main/content
		if (node.isNull() || broken_ref(node))
		{
			send(";");
			if (!parameters) {send("\n");}
		}
		
		// TX main/ref
		else if (node.isRef())
		{
			send(" &");
			send(node.asRefOr().getRefName());
			send(parameters ? (node.hasNext() ? "," : "") : "\n");
		}
		
		// TX main/content/bool
		else if (node.isBool())
		{
			if (nameOmit) send(" ");
			send(node.isTrue() ? "+" : "-");
			if (!parameters) {send("\n");}
		}
		
		// TX main/content/number
		else if (node.isNum())
		{
			if (hasOutput || node.hasPrev()) {send(" ");}
			send(node.toString());
			send(parameters ? (node.hasNext() ? "," : "") : "\n");
		}
		
		// TX main/content/list
		else if (node.isList())
		{
			if (node.numItems() == 0)
			{
				if (hasOutput) {send(" ");}
				send("{}\n");
			}
			else if (node.numItems() == 1 && !node.getItemFirstOr().isList())
			{
				if (hasOutput) {send(" ");}
				IDA item = node.getItemFirstOr();
				if (item.hasName() || item.hasParams())
				{
					send("{ ");
					ida(item, "", true, false, false, null);
					send(" }\n");
				}
				else
				{
					send("[ ");
					send(value(item, true));
					send(" ]\n");
				}
			}
			else
			{
				bool oneLine = IDA.port.si_allMatch(node.iterator(), n => !n.isList() && !n.hasParams());
				
				if (oneLine)
				{
					if (hasOutput) {queue(" ");}
					if (IDA.port.si_anyMatch(node.iterator(), n => n.hasName()))
					{
						IDA_Iterator<string> pNames = itemClass != null && itemClass.isList() ? IDA.port.i_map<IDA,string>(itemClass.iterator(), n => n.getNameOr()) : IDA.port.empty_i<string>();
						
						bool asAbove = as_above_all_test(node.getItemFirstOr());
						
						IDA_Iterator<IDA> i = node.iterator();
						queue("{");
						{
							IDA n = i.next();
							bool pNameOmit = pNames.hasNext() && (pNames.next() == n.getNameOr());
							if (n.hasRefName()) queue(" *"+n.getRefName());
							if (!pNameOmit) queue(" "+id_str(n, n.getNameOr()));
							queue(value(n, false));
						}
						while (i.hasNext())
						{
							IDA n = i.next();
							bool pNameOmit = pNames.hasNext() && (pNames.next() == n.getNameOr());
							if (n.hasPrev() && !n.getPrevOr().isBool() && !n.getPrevOr().isNull()) queue(",");
							if (n.hasRefName()) queue(" *"+n.getRefName());
							if (!pNameOmit) queue(" "+(asAbove ? "^" : id_str(n, n.getNameOr())));
							queue(value(n, false));
						}
						queue(" }\n");
					}
					else
					{
						IDA_Iterator<IDA> i = node.iterator();
						queue("[");
						while (i.hasNext()) {queue(" " + value(i.next(), true).Trim());}
						queue(" ]\n");
					}
					
					if (length() < 2*ONE_LINE_MAX) {accept();}
					else {reject(); oneLine = false;}
					
				}
				
				if (!oneLine)
				{
					// For now, only use name-as-above when all nodes have the same name,
					// and are not lists above size 1
					
					if (hasOutput) {send("\n"); send(indent);}
					send("{\n");
					string indent2 = indent + "\t";
					
					IDA first = node.getItemFirstOr();
					
					// Check to see if a class could be used for parameters
					IDA pClass = null;
					string firstName = first.getNameOr();
					bool cName = false;
					if (!parameters && node.numItems() > 2)
					{
						cName = true;
						
						// Parameters
						List<IDA> firstParams = first.getParamsOr();
						int ptrParams = firstParams != null ? firstParams.Count : 0;
						
						// Check names separately
						for (IDA n = first.getNextOr(); n != null; n = n.getNextOr())
						{
							if (!n.isName(firstName)) {cName = false; break;}
						}
						
						// Check parameters
						for (IDA n = first.getNextOr(); n != null && ptrParams > 0; n = n.getNextOr())
						{
							List<IDA> nps = n.getParamsOr();
							if (nps == null) {ptrParams = 0; break;}
							
							int i = 0;
							foreach (IDA np in nps)
							{
								if (i >= ptrParams) {break;}
								if (!np.isName(firstParams[i].getNameOr())) {break;}
								i+=1;
							}
							
							ptrParams = Math.Min(ptrParams, i);
						}
						
						
						// Items (all nodes must be lists)
						List<IDA> firstItems = first.asListOr();
						int ptrItems = 0;
						bool namedItem = false;
						
						// Emergency measure - avoid classing the items if there are ANY parameters or lists inside ANY nodes
						if (firstItems != null && !has_inner_structure(first))
						{
							ptrItems = firstItems.Count;
							for (IDA n = first.getNextOr(); n != null && ptrItems > 0; n = n.getNextOr())
							{
								List<IDA> nis = n.asListOr();
								if (nis == null) {firstItems = null; ptrItems = 0; break;}
								
								int i = 0;
								foreach (IDA ni in nis)
								{
									if (i >= ptrItems) {break;}
									string name = firstItems[i].getNameOr();
									if (!ni.isName(name)) {break;}
									if (name != null) namedItem = true;
									i+=1;
								}
								
								ptrItems = Math.Min(ptrItems, i);
							}
						}
						
						
						// Combine
						if (ptrParams > 0 || (namedItem && ptrItems > 0))
						{
							// Class!
							send(indent2);
							send(":: ");
							if (cName) {send(id_str(first, firstName));}
							
							pClass = new IDA(firstName);
							
							if (ptrParams > 0)
							{
								pClass.setParams();
								
								send("(");
								IDA np = firstParams[0];
								send(id_str(np, np.getNameOr()));
								pClass.addParam(new IDA(np.getNameOr()));
								while (np.hasNext() && (--ptrParams > 0))
								{
									np = np.getNextOr();
									send(", " + id_str(np, np.getNameOr()));
									pClass.addParam(new IDA(np.getNameOr()));
								}
								send(")");
							}
							
							if (ptrItems > 0)
							{
								pClass.putList();
								
								send("{");
								IDA ni = firstItems[0];
								send(id_str(ni, ni.getNameOr()));
								pClass.addItem(new IDA(ni.getNameOr()));
								while (ni.hasNext() && (--ptrItems > 0))
								{
									ni = ni.getNextOr();
									send(", " + id_str(ni, ni.getNameOr()));
									pClass.addItem(new IDA(ni.getNameOr()));
								}
								send("}\n");
							}
							else {send(";\n");}
							
						}
						else {cName = false;}
					}
					
					IDA item = first;
					bool asAbove = as_above_all_test(first);
					
					if (asAbove || cName)
					{
						ida(item, indent2, false, false, cName, pClass);
						while ((item = item.getNextOr()) != null)
						{
							ida(item, indent2, false, true, cName, pClass);
						}
					}
					else
					{
						bool asAbovePart = false;
						string asAboveName = null;
						
						for (; item != null; item = item.getNextOr())
						{
							if (asAbovePart && !item.isName(asAboveName)) {asAbovePart = false;}
							ida(item, indent2, false, asAbovePart, false, pClass);
							
							asAboveName = item.getNameOr();
							if (!asAbovePart) asAbovePart = as_above_part_test(item);
							
						}
					}
					
					
					
					send(indent);
					send("}\n");
					
				}
				
			}
			
		}
		
		// TX main/content/string
		else
		{
			if (hasOutput || node.hasPrev()) {send(" ");}
			send(id_str(node, node.toString()));
			send(parameters ? (node.hasNext() ? "," : "") : "\n");
			
		}
		
	// TX main end
	}
	
	// TX utility
	private static bool broken_ref(IDA el)
	{
		return el.isRef() && !el.asRefOr().hasRefName();
	}
	
	private static bool as_above_all_test(IDA firstItem)
	{
		string itemName = firstItem.getNameOr();
		
		if (itemName == null) return false;
		if (itemName.Length < BREVITY_ALL) return false;
		
		for (IDA c = firstItem; c != null; c = c.getNextOr())
		{
			if (c.numItems() >= 2) return false;
			if (!c.isName(itemName)) return false;
		}
		return true;
	}
	private static bool as_above_part_test(IDA item)
	{
		string asAboveName = item.getNameOr();
		if (asAboveName == null || asAboveName.Length < BREVITY_ALL) {return false;}
		int req = BREVITY_PART;
		for (IDA n = item.getNextOr(); (--req > 0) && n != null; n = n.getNextOr())
		{
			if (!n.isName(asAboveName)) {return false;}
		}
		return req <= 0;
	}
	
	private string value(IDA node, bool inArray)
	{
		if (node.isNull() || broken_ref(node)) {return inArray ? "?" : ";";}
		if (node.isBool()) {return node.isTrue() ? "+" : "-";}
		if (node.isRef()) {return (inArray ? "&" : " &") + node.asRefOr().getRefName();}
		return (inArray ? "" : " ") + (node.isNum() ? node.toString() : id_str(node, node.toString()));
	}
	
	private static string replace_ux(string s)
	{
		int l = s.Length;
		int ptr = 0;
		string output = "";
		for (int i=0; i<l; i+=1)
		{
			char c = s[i];
			if (c < 0x20 || (c >= 0x7f && c < 0x100))
			{
				output += s.Substring(ptr, i-ptr);
				output += "\\x";
				output += IDA.port.char_2(c);
				ptr = i+1;
			}
			else if (c >= 0x100 && c <= 0xffff)
			{
				output += s.Substring(ptr, i-ptr);
				output += "\\u";
				output += IDA.port.char_4(c);
				ptr = i+1;
			}
		}
		return output + s.Substring(ptr);
	}
	private string id_str(IDA refNode, string s)
	{
		if (s == null) {return "?";}
		else if (s.Length < 1) {return "\"\"";}
		else if (id(s)) {return s;}
		else
		{
			bool quoted = IDA.port.ends_with(s, "-");
			using (StringWriter buf = new StringWriter())
			{
				int max = s.Length;
				for (int i=0; i<max; i+=1)
				{
					char c = s[i];
					if (c < 0x20)
					{
						if (c == 0) {buf.Write("\\0");}
						else if (c == '\n') {buf.Write("\\n");}
						else if (c == '\r') {buf.Write("\\r");}
						else if (c == '\t') {buf.Write("\\t");}
						else {buf.Write("\\x"); buf.Write(IDA.port.char_2(c));}
					}
					else if (c < 0x7f)
					{
						if (c == '\\') {buf.Write("\\\\");}
						else if (c == '"') {buf.Write("\\\""); quoted = true;}
						else
						{
							if (!quoted)
							{
								if (i<1 ? !IDA.id0(c) : !IDA.id1(c)) {quoted = true;}
								else if (i>0 && IDA.comment2(s[i-1], c)) {quoted = true;}
							}
							buf.Write(c);
						}
					}
					else if (c < 0x100)
					{
						if (c < 0xa0 || !opts(refNode).txUnicode) {buf.Write("\\x"); buf.Write(IDA.port.char_2(c));}
						else {buf.Write(c);}
					}
					else if (Char.IsHighSurrogate(c))
					{
						int code = Char.ConvertToUtf32(c, s[i+=1]);
						if (opts(refNode).txUnicode)
						{
							buf.Write(Char.ConvertFromUtf32(code));
						}
						else
						{
							buf.Write("\\U");
							buf.Write(IDA.port.char_8(code));
						}
					}
					else if (c >= 0x100)
					{
						if (opts(refNode).txUnicode)
						{
							buf.Write(c);
						}
						else
						{
							buf.Write("\\u");
							buf.Write(IDA.port.char_4(c));
						}
					}
					
				}
				
				if (quoted)
				{
					buf.Write("\"");
					return "\"" + buf.ToString();
				}
				else return buf.ToString();
			}
		}
		
	}
	private static bool id(string s)
	{
		if ((s.Length < 1) || !IDA.id0(s[0]) || s[s.Length-1] == '-') {return false;}
		char c1 = s[0];
		for (int i=1; i<s.Length; i+=1)
		{
			char c2 = s[i];
			if (!IDA.id1(c2)) return false;
			if (IDA.comment2(c1, c2)) return false;
			c1 = c2;
		}
		return true;
	}
	
// TX end
}

// CLI begin
public class IDA_CLI
{
	
	// CLI
	static int cli_err(string msg) {return cli_err(msg, 1);}
	static int cli_err(string msg, int exitCode)
	{
		Console.Error.WriteLine("Error: " + msg);
		return exitCode;
	}
	static void cli_out(string ln) {Console.Out.WriteLine(ln);}
	static int cli_help()
	{
		//      ".........1.........2.........3.........4.........5.........6.........7.........8"
		cli_out("IDA markup parser");
		cli_out("Read markup from standard input and write IDA to standard output.");
		cli_out("");
		cli_out("Parser options:");
		cli_out(" -h, --help         Show usage");
		cli_out(" -i in-option       Set IDA input option");
		cli_out(" -o out-option      Set IDA output option");
		cli_out("");
		cli_out("IDA input options:");
		cli_out(" xml-reparse        Parse XML string content as additional inline markup");
		cli_out("");
		cli_out("IDA output options:");
		cli_out(" unicode            Use UTF-8 for string characters beyond U+0080");
		cli_out("");
		return 0;
	}
	public static int cli(int argc, string[] argv)
	{
		IDA rootEl = new IDA();
		IDAOptions opts = rootEl.opts;
		
		int i=0;
		while (i<argc)
		{
			string arg = argv[i++];
			
			if (arg == "-h" || arg == "--help")
			{
				return cli_help();
			}
			else if (arg == "-i")
			{
				if (i >= argc) return cli_err("No input option specified");
				arg = argv[i++];
				if (arg == "xml-reparse") {opts.rxXmlReparseStrings = true;}
				else return cli_err("Unknown input option: " + arg);
			}
			else if (arg == "-o")
			{
				if (i >= argc) return cli_err("No output option specified");
				arg = argv[i++];
				if (arg == "unicode") {opts.txUnicode = true;}
				else return cli_err("Unknown output option: " + arg);
			}
		}
		
		try {rootEl.inStream(Console.OpenStandardInput());}
		catch (Exception ex) {return cli_err("Parse failed: " + ex.Message);}
		
		rootEl.outPrint(Console.Out, true);
		return 0;
	}
	
// CLI end
}


/** One data article element. */
public class IDA : IEnumerable<IDA>
{
	
	// File includes
	// File includes - Global PWD
	private static string gIncludePwd = null;
	/** Enables file includes globally, as applied to articles created hereafter.<br>
	 * The provided working dir is used for resolving relative paths.<br>
	 * Set null to disable includes by default. */
	public static void setGlobalIncludes(string pwd) {gIncludePwd = pwd;}
	/** Enables file includes globally, as applied to articles created hereafter.<br>
	 * The present runtime working dir is used for resolving relative paths.<br>*/
	public static void setGlobalIncludes() {setGlobalIncludes("./");}
	
	// File includes - this element
	private string includePwd;
	/** Enables file includes for this article, as applied to input decoded hereafter.<br>
	 * The provided working dir is used for resolving relative paths.<br>
	 * Set null to disable includes. */
	public IDA setIncludes(string pwd) {includePwd = pwd; return this;}
	/** Enables file includes for this article, as applied to input decoded hereafter.<br>
	 * The present runtime working dir is used for resolving relative paths.<br> */
	public IDA setIncludes() {return setIncludes("./");}
	/** May be null, if includes are disabled for this article. */
	public string rxIncludePwd() {return includePwd;}
	
	// Input control
	/** Negative for unlimited */
	public int inputListDepth = -1;
	/** Negative for unlimited */
	public int inputListLength = -1;
	public IDA rxInputList(int depth, int length)
	{
		this.inputListDepth = depth;
		this.inputListLength = length;
		return this;
	}
	
	// Options storage
	public IDAOptions opts = new IDAOptions(IDAOptions.defaultOptions);
	/** <p>Fetch the input/output options for this element.</p>
	 * @return Input/output options object. */
	public IDAOptions getOpts() {return opts;}
	/** <p>Set the input/output options of this element to match those of the provided element.</p>
	 * @param src An element from which to copy the options.
	 * @return This element. */
	public IDA copyOptions(IDA src)
	{
		this.opts.copyFrom(src.opts);
		return this;
	}
	
	public IDA rxConfig(IDA parent)
	{
		rxInputList(parent.inputListDepth > 0 ? parent.inputListDepth - 1 : parent.inputListDepth, parent.inputListLength);
		copyOptions(parent);
		return this;
	}
	public IDA rxNew()
	{
		IDA el = new IDA().rxScope(this).setIncludes(this.rxIncludePwd());
		el.rxConfig(this);
		return el;
	}
	
	// Scope storage
	private IDA scope = null;
	public IDA rxScope(IDA scope) {this.scope = scope; return this;}
	
	// Item classes
	private Dictionary<string, IDA> itemClasses = null;
	public void rxClearItemClasses()
	{
		itemClasses = null;
	}
	public IDA rxGetItemClass(string key)
	{
		IDA c = itemClasses != null ? itemClasses[key] : null;
		return c != null ? c : scope != null ? scope.rxGetItemClass(key) : null;
	}
	/** Actually adds to scope element for use by adjacent elements.<br>
	 * Returns false (and takes no action with the class element) if no scope existed */
	public bool rxAddItemClass(string key, IDA itemClass)
	{
		if (scope == null) {return false;}
		if (scope.itemClasses == null) {scope.itemClasses = new Dictionary<string, IDA>();}
		scope.itemClasses[key] = itemClass;
		return true;
	}
	
	// Templates
	private Dictionary<string, IDA> itemTemplates = null;
	public void rxClearTemplates()
	{
		itemTemplates = null;
	}
	public IDA rxGetTemplate(string key)
	{
		IDA c = itemTemplates != null ? itemTemplates[key] : null;
		return c != null ? c : scope != null ? scope.rxGetTemplate(key) : null;
	}
	/** Actually adds to scope element for use by adjacent elements.<br>
	 * Returns false (and takes no action with the template element) if no scope existed */
	public bool rxAddTemplate(string key, IDA itemTemplate)
	{
		if (scope == null) {return false;}
		if (scope.itemTemplates == null) {scope.itemTemplates = new Dictionary<string, IDA>();}
		scope.itemTemplates[key] = itemTemplate;
		return true;
	}
	
	// Adjacent nodes
	private IDA next;
	private IDA prev;
	/** <p>Test whether an element is adjacent to this element in the forward direction.</p>
	 * @return True if an element exists after this element, otherwise false */
	public bool hasNext() {return next != null;}
	/** <p>Test whether an element is adjacent to this element in the reverse direction.</p>
	 * @return True if an element exists before this element, otherwise false */
	public bool hasPrev() {return prev != null;}
	/** <p>Test whether no element is adjacent to this element in the reverse direction.</p>
	 * <p>This is the opposite of hasPrev, intended as a convenience for function references.</p>
	 * @return True if nothing exists this element, otherwise false */
	public bool isFirst() {return !hasPrev();}
	/** <p>Test whether no element is adjacent to this element in the forward direction.</p>
	 * <p>This is the opposite of hasNext, intended as a convenience for function references.</p>
	 * @return True if nothing exists after this element, otherwise false */
	public bool isLast() {return !hasNext();}
	
	/** <p>Fetch the adjacent element after this one.</p>
	 * <p>If this element is last, an exception is thrown.
	 * To avoid that possibility, use getNextOr to provide a default value.</p>
	 * @exception IDAOpException if this element is last.
	 * @return The element after. */
	public IDA getNext()
	{
		if (isLast()) throw new IDAOpException("Element is last");
		return next;
	}
	/** <p>Fetch the adjacent element after this one.</p>
	 * <p>If this element is last, null is returned.</p>
	 * @return The element after if present, otherwise null. */
	public IDA getNextOr()
	{
		return next;
	}
	/** <p>Fetch the adjacent element after this one.</p>
	 * <p>If this element is last, the provided default value is returned.</p>
	 * @param other A default value to return if nothing follows this element.
	 * @return The element after if present, otherwise the default value. */
	public IDA getNextOr(IDA other)
	{
		return hasNext() ? next : other;
	}
	/** <p>Fetch the adjacent element before this one.</p>
	 * <p>If this element is first, an exception is thrown.
	 * To avoid that possibility, use getPrevOr to provide a default value.</p>
	 * @exception IDAOpException if this element is first.
	 * @return The element before. */
	public IDA getPrev()
	{
		if (isFirst()) throw new IDAOpException("Element is first");
		return prev;
	}
	/** <p>Fetch the adjacent element before this one.</p>
	 * <p>If this element is first, null is returned.</p>
	 * @return The element before if present, otherwise null. */
	public IDA getPrevOr()
	{
		return prev;
	}
	/** <p>Fetch the adjacent element before this one.</p>
	 * <p>If this element is first, the provided default value is returned.</p>
	 * @param other A default value to return if nothing precedes this element.
	 * @return The element before if present, otherwise the default value. */
	public IDA getPrevOr(IDA other)
	{
		return hasPrev() ? prev : other;
	}
	
	/** This element, followed by everything adjacent after it */
	public IDA_Iterator<IDA> iterateNext()
	{
		IDA_MutableSeq<IDA> i = new IDA_MutableSeq<IDA>();
		IDA n = this;
		i.funcHas = () => n != null;
		i.funcGet = () => {IDA x = n; n = x.getNextOr(); return x;};
		return i;
	}
	/** <p>Places the provided element adjacent to this one, next in sequence.<br>
	 * If the element is null, this one will become last in sequence.</p>
	 * <p>Any existing forward link will be undone before this operation.</p>
	 * @param next An element to follow this one, or null
	 * @return This element */
	public IDA linkNext(IDA next)
	{
		if (this.next != null) {this.next.prev = null;}
		this.next = next; if (next != null) next.prev = this;
		return this;
	}
	/** <p>Places the provided element adjacent to this one, previous in sequence.<br>
	 * If the element is null, this one will become first in sequence.</p>
	 * <p>Any existing reverse link will be undone before this operation.</p>
	 * @param prev An element to precede this one, or null
	 * @return This element */
	public IDA linkPrev(IDA prev)
	{
		if (this.prev != null) {this.prev.next = null;}
		this.prev = prev; if (prev != null) prev.next = this;
		return this;
	}
	/** <p>Places the provided elements adjacent to this one, on either side.<br>
	 * If an element is null, this one will become the first/last in sequence.</p>
	 * <p>Before this operation occurs, any existing links will be cross-connected to remove this element from the chain.</p>
	 * @param prev An element to precede this one, or null
	 * @param next An element to follow this one, or null
	 * @return This element */
	public IDA link(IDA prev, IDA next)
	{
		if (this.prev != null) {this.prev.next = this.next;}
		if (this.next != null) {this.next.prev = this.prev;}
		this.prev = prev; if (prev != null) prev.next = this;
		this.next = next; if (next != null) next.prev = this;
		return this;
	}
	
	// Node name
	string name = null;
	/** <p>Test whether this element has no name.
	 * The name must be null, not merely the empty string.</p>
	 * <p>This is the opposite of hasName, intended as a convenience for function references.</p>  
	 * @return True if this element has no name, otherwise false */
	public bool isAnon() {return name == null;}
	/** <p>Test whether this element has a name.
	 * The name can be any non-null string, including the empty string.</p>
	 * @return True if this element has a name, otherwise false */
	public bool hasName() {return name != null;}
	/** <p>Test whether this element has a specific name.
	 * Providing null will perform the same test as with isAnon.</p>
	 * @param name A name for comparison
	 * @return True if this element name matches the provided name, otherwise false */
	public bool isName(string name)
	{
		return getNameOr() == name;
	}
	
	/** <p>Fetch this element name string.</p>
	 * <p>If this element content is anonymous, an exception is thrown.
	 * To avoid that possibility, use getNameOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no name.
	 * @return Element name. */
	public string getName()
	{
		if (isAnon()) throw new IDAOpException("Element has no name");
		return name;
	}
	/** <p>Fetch this element name string.</p>
	 * <p>If this element content is anonymous, null is returned.</p>
	 * <p>This method exists for platform compatibility where strings are not nullable/optional, such as with C++.
	 * The purpose is for this return type to be a nullable string container, while getNameOr(other) returns a string directly.
	 * On platforms where string values can be null, getNameOr() is equivalent to getNameOr(null), but only the former is fully portable.</p>
	 * @return Element name if present, otherwise null. */
	public string getNameOr()
	{
		return getNameOr(null);
	}
	/** <p>Fetch this element name string.</p>
	 * <p>If this element content is anonymous, the provided default value is returned.</p>
	 * <p>On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
	 * If possible, always call getNameOr() rather than getNameOr(null).</p>
	 * @param other A default value to return if this element has no name.
	 * @return Element name if present, otherwise the default value. */
	public string getNameOr(string other)
	{
		return hasName() ? name : other;
	}
	
	/** <p>Clear the name of this element, leaving it anonymous.</p>
	 * @return This element. */
	public IDA clearName() {return setName(null);}
	/** <p>Set the name of this element to be the provided string.</p>
	 * <p>If the provided string is null, this element will become anonymous as with calling clearName.</p>
	 * @param name Element name string.
	 * @return This element. */
	public IDA setName(string name) {this.name = name; return this;}
	/** <p>Set the name of this element to match the name of the provided element.</p>
	 * @param src An element from which to copy the name */
	public void copyName(IDA src) {this.name = src.name;}
	
	// Parameters
	protected List<IDA> parameters;
	/** <p>Delete this element's parameters list.</p>
	 * <p>If this element has no parameters list, calling clearParams has no effect.</p>
	 * @return This element. */
	public IDA clearParams() {parameters = null; return this;}
	/** <p>Set this element's parameters to be an empty list.</p>
	 * <p>Any existing parameter elements will be deleted.</p>
	 * @return This element. */
	public IDA setParams() {parameters = new List<IDA>(); return this;}
	
	/** <p>Fetch this element's parameters list.</p>
	 * <p>If this element has no parameters list, an exception is thrown.
	 * To avoid that possibility, use getParamsOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no parameters list.
	 * @return List of parameters. */
	public List<IDA> getParams()
	{
		if (!hasParams()) throw new IDAOpException("No parameters list");
		return parameters;
	}
	/** <p>Fetch this element's parameters list.</p>
	 * <p>If this element has no parameters list, null is returned.</p>
	 * <p>This method exists for platform compatibility where lists/vectors are not nullable/optional, such as with C++.
	 * The purpose is for this return type to be a nullable list container, while getParamsOr(other) returns a list directly.
	 * On platforms where lists can be null, getParamsOr() is equivalent to getParamsOr(null), but only the former is fully portable.</p>
	 * @return List of parameters if present, otherwise null. */
	public List<IDA> getParamsOr()
	{
		return hasParams() ? parameters : null;
	}
	/** <p>Fetch this element's parameters list.</p>
	 * <p>If this element has no parameters list, the provided default value is returned.</p>
	 * <p>On platforms where lists/vectors are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed list, such that code will not be fully portable if null is provided.
	 * If possible, always call getParamsOr() rather than getParamsOr(null).</p>
	 * @param other A default value to return if this element has no parameters list.
	 * @return List of parameters if present, otherwise the default value. */
	public List<IDA> getParamsOr(List<IDA> other)
	{
		return hasParams() ? parameters : other;
	}
	
	/** <p>Test whether this element contains a parameters list.</p>
	 * <p>This reports whether a parameters list object exists at all, as opposed to whether at least one parameter is present.
	 * The primary purpose is to ensure the list exists before calling getParams and/or iterating over its elements.</p>
	 * <p>Use numParams to test for an empty list (0) or at least one parameter present (greater than 0).</p>
	 * @return True if a parameters list exists for this element, otherwise false */
	public bool hasParams()
	{
		return parameters != null;
	}
	/** <p>Fetch the number of parameters within this element.</p>
	 * <p>If no parameters list is present, the sentinel value -1 is returned.
	 * If 0 is returned, the list is present but empty.</p>
	 * @return The parameters list size if a list is present, otherwise -1 */
	public int numParams()
	{
		return parameters != null ? parameters.Count : -1;
	}
	/** <p>Test whether this element contains a parameter with the provided name.
	 * If it does, an equivalent call to getParam will return an element.</p>
	 * <p>If false is returned, this can be either due to having no parameter of that name or due to having no parameters list.
	 * Use hasParams to separate those two cases.</p>
	 * @param key A parameter name to be checked
	 * @return True if a parameter exists with the provided name, otherwise false */
	public bool hasParam(string key)
	{
		return getParamOr(key) != null;
	}
	
	// Parameters/get
	/** <p>Fetch the parameter element at the provided index in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has too few parameters in the list, an exception is thrown.
	 * To avoid that possibility, use getParamAtOr to provide a default value.</p>
	 * @param index A parameter index to be recalled.
	 * @exception IDAOpException if this element has no such parameter.
	 * @return The indexed parameter element. */
	public IDA getParamAt(int index)
	{
		if (numParams() <= index) throw new IDAOpException("Invalid parameter index");
		return parameters[index];
	}
	/** <p>Fetch the parameter element at the provided index in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has too few parameters in the list, null is returned.</p>
	 * @param index A parameter index to be recalled.
	 * @return The indexed parameter element if present, otherwise null. */
	public IDA getParamAtOr(int index)
	{
		return getParamAtOr(index, null);
	}
	/** <p>Fetch the parameter element at the provided index in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has too few parameters in the list, the provided default value is returned.</p>
	 * @param index A parameter index to be recalled.
	 * @param other A default value to return if this element has no such parameter.
	 * @return The indexed parameter element if present, otherwise the default value. */
	public IDA getParamAtOr(int index, IDA other)
	{
		return numParams() > index ? parameters[index] : other;
	}
	
	/** <p>Fetch the first parameter element with the provided name in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has no parameter of that name, an exception is thrown.
	 * To avoid that possibility, use getParamOr to provide a default value.</p>
	 * @param key A parameter name to be recalled.
	 * @exception IDAOpException if this element has no such parameter.
	 * @return The named parameter element. */
	public IDA getParam(string key)
	{
		IDA p = getParamOr(key);
		if (p == null) throw new IDAOpException("Parameter not found");
		return p;
	}
	/** <p>Fetch the first parameter element with the provided name in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has no parameter of that name, null is returned.</p>
	 * @param key A parameter name to be recalled.
	 * @return The named parameter element if present, otherwise null. */
	public IDA getParamOr(string key)
	{
		return getParamOr(key, null);
	}
	/** <p>Fetch the first parameter element with the provided name in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has no parameter of that name, the provided default value is returned.</p>
	 * @param key A parameter name to be recalled.
	 * @param other A default value to return if this element has no such parameter.
	 * @return The named parameter element if present, otherwise the default value. */
	public IDA getParamOr(string key, IDA other)
	{
		if (hasParams()) foreach (IDA p in parameters) if (p.isName(key)) return p;
		return other;
	}
	
	/** <p>Cast to a string value the first parameter element with the provided name in this element's parameters list.</p>
	 * <p>If this element has no parameters list, has no parameter of that name or can not be interpreted as a string value, an exception is thrown.
	 * To avoid that possibility, use getParamStrOr to provide a default value.</p>
	 * @param key A parameter name to be recalled.
	 * @exception IDAOpException if this element has no such parameter, or the parameter content cannot cast to a string value.
	 * @return String content interpretation of the named parameter element. */
	public string getParamStr(string key)
	{
		return getParam(key).toStr();
	}
	/** <p>Cast to a string value the first parameter element with the provided name in this element's parameters list.</p>
	 * <p>If this element has no parameters list, has no parameter of that name or can not be interpreted as a string value, null is returned.</p>
	 * <p>This method exists for platform compatibility where strings are not nullable/optional, such as with C++.
	 * The purpose is for this return type to be a nullable string container, while getParamStrOr(key, other) returns a string directly.
	 * On platforms where string values can be null, getParamStrOr(key) is equivalent to getParamStrOr(key, null), but only the former is fully portable.</p>
	 * @param key A parameter name to be recalled.
	 * @return String content interpretation of the named parameter element if possible, otherwise null. */
	public string getParamStrOr(string key)
	{
		IDA param = getParamOr(key);
		return param != null ? param.toStrOr() : null;
	}
	/** <p>Cast to a string value the first parameter element with the provided name in this element's parameters list.</p>
	 * <p>If this element has no parameters list, has no parameter of that name or can not be interpreted as a string value, the provided default value is returned.</p>
	 * <p>On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
	 * If possible, always call getParamStrOr(key) rather than getParamStrOr(key, null).</p>
	 * @param key A parameter name to be recalled.
	 * @param other A default value to return if this element has no such parameter, or the parameter content cannot cast to a string value.
	 * @return String content interpretation of the named parameter element if possible, otherwise the default value. */
	public string getParamStrOr(string key, string other)
	{
		IDA param = getParamOr(key);
		return param != null ? param.toStrOr(other) : other;
	}
	
	/** <p>Fetch the first parameter element in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has an empty parameters list, an exception is thrown.
	 * To avoid that possibility, use getParamFirstOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no such parameter.
	 * @return The first parameter element. */
	public IDA getParamFirst()
	{
		if (numParams() < 1) throw new IDAOpException(hasParams() ? "Parameters list is empty" : "Element has no parameters list");
		return parameters[0];
	}
	/** <p>Fetch the first parameter element in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has an empty parameters list, null is returned.</p>
	 * @return The first parameter element if present, otherwise null. */
	public IDA getParamFirstOr()
	{
		return getParamFirstOr(null);
	}
	/** <p>Fetch the first parameter element in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has an empty parameters list, the provided default value is returned.</p>
	 * @param other A default value to return if this element has no such parameter.
	 * @return The first parameter element if present, otherwise the default value. */
	public IDA getParamFirstOr(IDA other)
	{
		return numParams() > 0 ? parameters[0] : other;
	}
	/** <p>Fetch the last parameter element in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has an empty parameters list, an exception is thrown.
	 * To avoid that possibility, use getParamLastOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no such parameter.
	 * @return The last parameter element. */
	public IDA getParamLast()
	{
		int n = numParams();
		if (n < 1) throw new IDAOpException(hasParams() ? "Parameters list is empty" : "Element has no parameters list");
		return parameters[n-1];
	}
	/** <p>Fetch the last parameter element in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has an empty parameters list, null is returned.</p>
	 * @return The last parameter element if present, otherwise null. */
	public IDA getParamLastOr()
	{
		return getParamLastOr(null);
	}
	/** <p>Fetch the last parameter element in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has an empty parameters list, the provided default value is returned.</p>
	 * @param other A default value to return if this element has no such parameter.
	 * @return The last parameter element if present, otherwise the default value. */
	public IDA getParamLastOr(IDA other)
	{
		int n = numParams();
		return n > 0 ? parameters[n-1] : other;
	}
	
	// Parameters/add
	/** Create and insert a new parameter at the end of the list, with the provided element name.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param name Name string for the created parameter element.
	 * @return The created parameter element. */
	public IDA newParam(string name)
	{
		IDA el = new IDA(name);
		addParam(el);
		return el;
	}
	
	/** Insert a parameter at the end of the list.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param param The parameter element to be stored.
	 * @return This element. */
	public IDA addParam(IDA param)
	{
		if (parameters == null) {setParams();}
		param.link(getParamLastOr(), null);
		parameters.Add(param);
		return this;
	}
	/** Insert a parameter at the specified list index position.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param param The parameter element to be stored.
	 * @param pos The list index position at which the parameter should be inserted.
	 * @exception IDAOpException if the specified position is below zero or above the present list size.
	 * @return This element. */
	public IDA addParamAt(IDA param, int pos)
	{
		if (parameters == null) {setParams();}
		list_add(getParams(), param, pos, "param");
		return this;
	}
	/** Replace the first matching parameter name, or insert a new parameter at the end of the list.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param param The parameter element to be stored.
	 * @return This element. */
	public IDA setParam(IDA param)
	{
		int n = numParams();
		
		try
		{
			for (int i=0; i<n; i+=1)
			{
				IDA p = getParamAt(i);
				if (p.eqName(param))
				{
					remParamAt(i);
					return addParamAt(param, i);
				}
			}
		}
		catch (IDAOpException ex) {throw new Exception("Unexpected parameters state", ex);}
		
		return addParam(param);
	}
	
	/** <p>Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.</p>
	 * <p>This is the same as calling setParamNull.</p>
	 * @param key The parameter element name.
	 * @return This element. */
	public IDA clearParam(string key) {return setParamNull(key);}
	/** <p>Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.</p>
	 * <p>This is the same as calling clearParam.</p>
	 * @param key The parameter element name.
	 * @return This element. */
	public IDA setParamNull(string key) {return setParam(new IDA(key).putNull());}
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content String value to be stored.
	 * @return This element. */
	public IDA setParamStr(string key, string content) {return setParam(new IDA(key).putStr(content));}
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Boolean value to be stored.
	 * @return This element. */
	public IDA setParamBool(string key, bool content) {return setParam(new IDA(key).putBool(content));}
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Floating-point value to be stored.
	 * @return This element. */
	public IDA setParamDouble(string key, double content) {return setParam(new IDA(key).putDouble(content));}
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Floating-point value to be stored.
	 * @return This element. */
	public IDA setParamFloat(string key, float content) {return setParam(new IDA(key).putFloat(content));}
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Integer value to be stored.
	 * @return This element. */
	public IDA setParamLong(string key, long content) {return setParam(new IDA(key).putLong(content));}
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Integer value to be stored.
	 * @return This element. */
	public IDA setParamInt(string key, int content) {return setParam(new IDA(key).putInt(content));}
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Referenced element to be stored.
	 * @return This element. */
	public IDA setParamRef(string key, IDA content) {return setParam(new IDA(key).putRef(content));}

	// Parameters/remove
	/** Remove the parameter at the specified list index position.<br>
	 * If this element has no parameters list, an exception is thrown.<br>
	 * @param pos The list index position of a parameter which should be removed.
	 * @exception IDAOpException if the specified position is below zero or beyond the present list size, or this element has no parameters list. */
	public void remParamAt(int pos)
	{
		if (!hasParams()) {throw new IDAOpException("No parameters from which to remove");}
		list_del(parameters, pos, "param");
	}
	/** Remove the provided parameter.<br>
	 * If the parameter is absent or this element has no parameters list, the call has no effect.<br>
	 * @param param The parameter element to be removed.
	 * @return True if a parameter was removed, otherwise false. */
	public bool remParam(IDA param)
	{
		if (hasParams() && list_rem(parameters, param)) {param.link(null, null); return true;}
		return false;
	}
	/** Remove the first parameter with the provided element name.<br>
	 * If no such parameter exists or this element has no parameters list, the call has no effect.<br>
	 * @param key Name string of the parameter element to remove.
	 * @return True if a parameter was removed, otherwise false. */
	public bool unParam(string key)
	{
		IDA p = getParamOr(key);
		if (p != null) {remParam(p); return true;}
		return false;
	}
	
	// Parameters/other
	/** <p>Set the parameters of this element to be clones of the parameters of the provided element.</p>
	 * @param src An element from which to copy the parameters */
	public void copyParams(IDA src)
	{
		clearParams();
		if (src.hasParams())
		{
			setParams();
			foreach (IDA p in src.parameters) this.addParam(p.copy_inner());
		}
	}
	
	// Content
	private string cRaw;
	private string cStr;
	private string cNum, cDen, cUnit;
	private List<IDA> cList;
	private bool? cBool;
	private IDA cRef;
	private IDAObj cRefObj;
	private string cRefTarget;
	private string cRefName;
	
	/** <p>Set the content of this element to match the content of the provided element.
	 * If the content in question is a list, its inner elements will be cloned.</p>
	 * @param src An element from which to copy the content */
	public void copyContent(IDA src)
	{
		this.cRaw = src.cRaw;
		this.cStr = src.cStr;
		this.cNum = src.cNum;
		this.cDen = src.cDen;
		this.cUnit = src.cUnit;
		this.cBool = src.cBool;
		this.cRef = null;
		this.cRefObj = null;
		this.cRefTarget = src.cRefTarget;
		this.cRefName = src.cRefName;
		this.clearList();
		if (src.isList())
		{
			this.putList();
			foreach (IDA x in src.asList()) this.addItem(x.copy_inner());
		}
	}
	
	// Content/ref
	/** Returns this element */
	public IDA clearRef() {cRef = null; cRefObj = null; cRefTarget = null; return this;}
	/** <p>Test whether the content type of this element is a reference to another element.</p>
	 * @return True if this element content is a reference, otherwise false */
	public bool isRef() {return cRef != null;}
	/** <p>Fetch this element content as a reference (to another element).</p>
	 * <p>If this element content is not a reference, an exception is thrown.
	 * To avoid that possibility, use asRefOr to provide a default value.</p>
	 * @exception IDAOpException if this element does not contain reference content.
	 * @return Reference content. */
	public IDA asRef()
	{
		return cRef;
	}
	/** <p>Fetch this element content as a reference (to another element).</p>
	 * <p>If this element content is not a reference, null is returned.</p>
	 * @return Reference content if present, otherwise null. */
	public IDA asRefOr()
	{
		return asRefOr(null);
	}
	/** <p>Fetch this element content as a reference (to another element).</p>
	 * <p>If this element content is not a reference, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain reference content.
	 * @return Reference content if present, otherwise the default value. */
	public IDA asRefOr(IDA other)
	{
		return isRef() ? cRef : other;
	}
	/** Returns this element */
	public IDA putRef(IDA refEl) {putNull(); cRaw = "Ref"; cRef = refEl; cRefObj = refEl.repObj; cRefTarget = refEl.cRefName; return this;}
	/** <p>Sets an object for which this element should reference the corresponding element as it is encoded.</p> 
	 * <p>During a restore/decode process, the rebuilt object can be retrieved using getRefObj or withRefObj.
	 * The IDAObj codec interface implementation should ensure that any required type information is captured.</p>
	 * @param refObj A target object which implements the IDAObj codec interface.
	 * @return This element. */
	public IDA setRefObj(IDAObj refObj) {putNull(); cRaw = "Ref"; cRefObj = refObj; return this;}
	/** <p>During a restore/decode process, the object corresponding with the referenced element will be sent to the callback when it becomes available.
	 * If that object is already available, the callback will be resolved before this call returns.</p>
	 * <p>If the object is needed immediately, the article structure must be designed such that the referenced object element is decoded in advance of this call.
	 * Use getRefObj instead if that design should be enforced.</p>
	 * <p>To ensure consistency, a top-level call to objRestore will throw an exception if any unresolved callbacks remain after decoding is complete.</p>
	 * <p>Calling code should use article information to establish the correct object type.</p>
	 * @param callback A function which receives the decoded object associated with the referenced element.
	 * @exception IDAOpException if this element content is not a reference. */
	public void withRefObj(Action<IDAObj> callback)
	{
		if (cRefObj != null) {callback(cRefObj);}
		else if (cRef == null) throw new IDAOpException("Element is not a reference");
		else {cRef.on_rep_obj(callback);}
	}
	/** <p>Fetches the object corresponding with the referenced element, after it has been encoded or decoded.
	 * During a restore/decode process, an exception is thrown if the object is not yet available.</p>
	 * <p>To retrieve the object this way during a restore/decode process, the article structure must be designed such that the referenced object element is decoded in advance of this call.
	 * Use withRefObj instead if that design is not the case.</p>
	 * <p>Calling code should use article information to establish the correct object type.</p>
	 * @exception IDAOpException if this element content is not a reference, or if no object is available immediately.
	 * @return The encoded or decoded object associated with the referenced element. */
	public IDAObj getRefObj()
	{
		if (cRefObj != null) return cRefObj;
		if (cRef == null) throw new IDAOpException("Element is not a reference");
		if (cRef.repObj != null) return cRef.repObj;
		throw new IDAOpException("Referenced object not available");
	}
	/** <p>Set a transient target for this element.</p>
	 * <p>The targets match the identifiers assigned via setRefName, and are used only for capturing and restoring element pointer information.
	 * The string content itself is not information, insofar as it need not survive a loop test.</p>
	 * @param refTarget A string to store as the transient target for this element. */
	public void setRefTarget(string refTarget) {putNull(); cRaw = "Ref"; cRefTarget = refTarget;}
	/** <p>Fetch the transient target for this element.
	 * The returned value may be <i>stale</i>.</p>
	 * <p>The targets match the identifiers assigned via setRefName, and are used only for capturing and restoring element pointer information.
	 * The string content itself is not information, insofar as it need not survive a loop test.</p>
	 * @return The transient target string for this element if it has a reference target, otherwise null. */
	public string getRefTarget() {return cRefTarget;}
	/** <p>Test whether this element has been assigned a transient target for encoding/decoding an outbound reference.
	 * The returned value may be <i>stale</i>.</p>
	 * <p>If an article has been decoded from markup, any reference target elements will have fresh identifiers from the markup document.
	 * Encoding, copying or testing equal a top-level article element will generate fresh identifiers.</p>
	 * <p>Call txRefLabel at top-level to generate fresh identifiers manually.</p>
	 * @return True if this element has a transient target, otherwise false. */
	public bool hasRefTarget() {return cRefTarget != null;}
	/** <p>Set a transient identifier for this element.</p>
	 * <p>Identifiers are set automatically when encoding/decoding markup, testing equal, or calling txRefLabel directly.
	 * Encoding, copying or testing equal a top-level article element will generate fresh identifiers.</p>
	 * <p>These names are used only for capturing and restoring element pointer information.
	 * The string content itself is not information, insofar as it need not survive a loop test.</p>
	 * @param refName A string to store as the transient identifier for this element. */
	public void setRefName(string refName) {cRefName = refName;}
	/** <p>Fetch the transient identifier for this element.
	 * The returned value may be <i>stale</i>.</p>
	 * <p>Identifiers are set automatically when encoding/decoding markup, testing equal, or calling txRefLabel directly.
	 * Encoding, copying or testing equal a top-level article element will generate fresh identifiers.</p>
	 * <p>These names are used only for capturing and restoring element pointer information.
	 * The string content itself is not information, insofar as it need not survive a loop test.</p>
	 * @return The transient identifier string for this element if it is a reference target, otherwise null. */
	public string getRefName() {return cRefName;}
	/** <p>Test whether this element has been assigned a transient identifier for encoding/decoding inbound references.
	 * The returned value may be <i>stale</i>.</p>
	 * <p>If an article has been decoded from markup, any reference target elements will have fresh identifiers from the markup document.
	 * Encoding, copying or testing equal a top-level article element will generate fresh identifiers.</p>
	 * <p>Call txRefLabel at top-level to generate fresh identifiers manually.</p>
	 * @return True if this element has a transient identifier, otherwise false. */
	public bool hasRefName() {return cRefName != null;}
	
	/** Returns the number of reference labels applied */
	public int txRefLabel(bool adj)
	{
		// Ref walk
		int[] rc = {0};
		List<IDA> rn = new List<IDA>();
		walk(x => ref_label_clear(x, rn), adj, adj);
		walk(x => ref_label_count(x, rn, rc), adj, adj);
		
		// Null any foreign references at this stage (see IDA.verify)
		foreach (IDA r in rn) r.setRefName(null);
		
		return rc[0];
	}
	private static void ref_label_clear(IDA el, List<IDA> rn)
	{
		el.setRefName(null);
		if (el.isRef())
		{
			IDA r = el.asRefOr();
			if (!IDA.port.list_contains(rn, r)) rn.Add(r);
		}
	}
	private static void ref_label_count(IDA el, List<IDA> rn, int[] rc)
	{
		if (IDA.port.list_contains(rn, el)) IDA.port.list_rem(rn, el);
		
		if (el.isRef())
		{
			IDA r = el.asRefOr();
			if (!r.hasRefName())
			{
				int n = rc[0]++;
				int n0 = n % 52;
				string id = "" + (char) (n0 < 26 ? 0x61+n0 : 0x41+n0) + (n/52);
				r.cRefName = id;
				el.cRefTarget = id;
			}
		}
	}
	
	public void rxRefResolve(bool adj)
	{
		Dictionary<string,IDA> rs = new Dictionary<string,IDA>();
		walk(x => ref_resolve_scan(x, rs), adj, adj);
		walk(x => ref_resolve_put(x, rs), adj, adj);
	}
	private static void ref_resolve_scan(IDA el, Dictionary<string,IDA> rs)
	{
		if (el.hasRefName()) rs[el.getRefName()] = el;
	}
	private static void ref_resolve_put(IDA el, Dictionary<string,IDA> rs)
	{
		if (el.hasRefTarget()) el.putRef(rs[el.getRefTarget()]);
	}
	
	// Content/bool
	/** <p>Test whether the content type of this element is a boolean value.</p>
	 * @return True if this element has boolean content, otherwise false */
	public bool isBool() {return cBool.HasValue;}
	/** <p>Fetch this element content as a boolean value.</p>
	 * <p>If this element content is not a boolean value, an exception is thrown.
	 * To avoid that possibility, use asBoolOr to provide a default value.
	 * One may also use isTrue for false by default, or not-isFalse for true by default.</p>
	 * <p>Use toBool/toBoolOr for casting other content types to be a boolean value.</p>
	 * @exception IDAOpException if this element does not contain boolean content.
	 * @return Boolean content value. */
	public bool asBool()
	{
		if (!this.cBool.HasValue) throw new IDAOpException("Content is not a boolean value");
		return cBool.Value;
	}
	/** <p>Fetch this element content as a boolean value.</p>
	 * <p>If this element content is not a boolean value, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain boolean content.
	 * @return Boolean content value if present, otherwise the default value. */
	public bool asBoolOr(bool other) {return cBool.HasValue ? cBool.Value : other;}
	/** <p>Test whether the content of this element is boolean-true.</p>
	 * <p>If false is returned, this can be either due to being boolean-false or not being a boolean value.
	 * Use isBool and asBool for methods with those effects.</p>
	 * <p>This is intended as a convenience for function references.</p>
	 * @return True if this element has boolean content with value true, otherwise false */
	public bool isTrue() {return cBool.HasValue && cBool.Value;}
	/** <p>Test whether the content of this element is boolean-false.</p>
	 * <p>If false is returned, this can be either due to being boolean-true or not being a boolean value.
	 * Use isBool and (not-)asBool for methods with those effects.</p>
	 * <p>This is intended as a convenience for function references.</p>
	 * @return True if this element has boolean content with value false, otherwise false */
	public bool isFalse() {return cBool.HasValue && !cBool.Value;}
	/** <p>Place boolean content into this element with value true, replacing any existing content.</p>
	 * @return This element */
	public IDA putTrue() {putNull(); cRaw = "+"; cBool = true; return this;}
	/** <p>Place boolean content into this element with value false, replacing any existing content.</p>
	 * @return This element */
	public IDA putFalse() {putNull(); cRaw = "-"; cBool = false; return this;}
	/** <p>Place boolean content into this element with the provided value, replacing any existing content.</p>
	 * @param b Boolean content value.
	 * @return This element */
	public IDA putBool(bool b) {return b ? putTrue() : putFalse();}
	/** <p>Test whether this element content can be interpreted as a boolean value (see toBool).
	 * This will always be the case for natural boolean content (see isBool).</p>
	 * <p>All numeric content can be cast to boolean, which is true if non-zero.
	 * String content can be "true" or "false".</p>
	 * @return True if this element content can be cast to a boolean value, otherwise false */
	public bool canBool()
	{
		if (isBool() || isNum()) return true;
		string s = this.toStrOr("");
		return s=="true" || s=="false";
	}
	/** <p>Cast this element content to a boolean value.</p>
	 * <p>If this element content can not be interpreted as a boolean value, an exception is thrown.
	 * To avoid that possibility, use toBoolOr to provide a default value.</p>
	 * @exception IDAOpException if this element content cannot cast to a boolean value.
	 * @return Boolean content interpretation. */
	public bool toBool()
	{
		if (isBool()) return isTrue();
		if (isNum()) return asDoubleOr(0d) != 0d;
		string s = this.toStrOr("");
		if (s=="true") return true;
		if (s=="false") return false;
		throw new IDAOpException("Content cannot cast to a boolean value");
	}
	/** <p>Cast this element content to a boolean value.</p>
	 * <p>If this element content can not be interpreted as a boolean value, the provided default value is returned.</p>
	 * @param other A default value to return if this element content cannot cast to a boolean value.
	 * @return Boolean content interpretation if able, otherwise the default value. */
	public bool toBoolOr(bool other)
	{
		try {return toBool();}
		catch (IDAOpException) // opex
		{return other;}
	}
	
	// Content/null
	/** <p>Test whether this element contains no content.</p>
	 * <p>This is the opposite of hasContent, intended as a convenience for function references.</p>
	 * @return True if this element has no content, otherwise false. */
	public bool isNull() {return cRaw == null;}
	/** <p>Test whether this element contains some type of content.</p>
	 * <p>This is the opposite of isNull, intended as a convenience for function references.</p>
	 * @return True if this element has content present, otherwise false. */
	public bool hasContent() {return cRaw != null;}
	/** <p>Erase any existing content from this element.</p>
	 * <p>This is the same as calling clearContent.</p>
	 * @return This element. */
	public IDA putNull() {return clearContent();}
	/** <p>Erase any existing content from this element.</p>
	 * <p>This is the same as calling putNull.</p>
	 * @return This element. */
	public IDA clearContent()
	{
		cRaw = null;
		cStr = null;
		cBool = null;
		clearRef();
		clearList();
		clearNum();
		return this;
	}
	
	// Content/string
	/** <p>Place string content into this element with the provided value, replacing any existing content.</p>
	 * @param s String content value. 
	 * @return This element */
	public IDA putStr(string s) {putNull(); cRaw = s; cStr = s; return this;}
	/** <p>Test whether the content type of this element is a string value.</p>
	 * @return True if this element has string content, otherwise false */
	public bool isStr() {return cStr != null;}
	public string toString() {return cRaw != null ? cRaw : "?";}
	/** <p>Fetch this element content as a string value.</p>
	 * <p>If this element content is not a string value, an exception is thrown.
	 * To avoid that possibility, use asStrOr to provide a default value.</p>
	 * <p>Use toStr/toStrOr for casting other content types to be a string value.</p>
	 * @exception IDAOpException if this element does not contain string content.
	 * @return String content value. */
	public string asStr()
	{
		if (cStr == null) throw new IDAOpException("Content is not a string value");
		return cStr;
	}
	/** <p>Fetch this element content as a string value.</p>
	 * <p>If this element content is not a string value, null is returned.</p>
	 * <p>This method exists for platform compatibility where strings are not nullable/optional, such as with C++.
	 * The purpose is for this return type to be a nullable string container, while asStrOr(other) returns a string directly.
	 * On platforms where string values can be null, asStrOr() is equivalent to asStrOr(null), but only the former is fully portable.</p>
	 * @return String content value if present, otherwise null. */
	public string asStrOr()
	{
		return cStr;
	}
	/** <p>Fetch this element content as a string value.</p>
	 * <p>If this element content is not a string value, the provided default value is returned.</p>
	 * <p>On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
	 * If possible, always call asStrOr() rather than asStrOr(null).</p>
	 * @param other A default value to return if this element does not contain string content.
	 * @return String content value if present, otherwise the default value. */
	public string asStrOr(string other)
	{
		return cStr != null ? cStr : other;
	}
	/** <p>Test whether this element content can be interpreted as a string value (see toStr).
	 * This will always be the case for natural string content (see isStr).</p>
	 * <p>Most content can be cast to string.
	 * Types which cannot be cast are lists, references and nulls.</p>
	 * @return True if this element content can be cast to a string value, otherwise false */
	public bool canStr()
	{
		return hasContent() && !isList() && !isRef();
	}
	/** <p>Cast this element content to a string value.</p>
	 * <p>If this element content can not be interpreted as a string value, an exception is thrown.
	 * To avoid that possibility, use toStrOr to provide a default value.</p>
	 * @exception IDAOpException if this element content cannot cast to a string value.
	 * @return String content interpretation. */
	public string toStr()
	{
		if (!canStr()) throw new IDAOpException("Content cannot cast to a string value");
		return cRaw;
	}
	/** <p>Cast this element content to a string value.</p>
	 * <p>If this element content can not be interpreted as a string value, null is returned.</p>
	 * <p>This method exists for platform compatibility where strings are not nullable/optional, such as with C++.
	 * The purpose is for this return type to be a nullable string container, while toStrOr(other) returns a string directly.
	 * On platforms where string values can be null, toStrOr() is equivalent to toStrOr(null), but only the former is fully portable.</p>
	 * @return String content interpretation if able, otherwise null. */
	public string toStrOr()
	{
		return canStr() ? cRaw : null;
	}
	/** <p>Cast this element content to a string value.</p>
	 * <p>If this element content can not be interpreted as a string value, the provided default value is returned.</p>
	 * <p>On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
	 * If possible, always call toStrOr() rather than toStrOr(null).</p>
	 * @param other A default value to return if this element content cannot cast to a string value.
	 * @return String content interpretation if able, otherwise the default value. */
	public string toStrOr(string other)
	{
		return canStr() ? cRaw : other;
	}
	
	// Content/number/other
	private void clearNum() {cNum = ""; cDen = "1"; cUnit = "";}
	
	// Content/number/set
	private void refresh_num() {cRaw = cNum + ((cDen == "1") ? "" : "/"+cDen) + cUnit; cStr = null;}
	/** <p>Set the unit string appended to the number represented by this element content.</p>
	 * <p>Passing null to setNumUnit has no effect.
	 * To clear the unit, call setNumUnit with an empty string.</p>
	 * <p>If this element content is not numeric, an exception is thrown.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return This element. */
	public IDA setNumUnit(string unit)
	{
		if (unit != null)
		{
			if (!this.isNum()) throw new IDAOpException("Content is not numeric");
			cUnit = unit; refresh_num();
		}
		return this;
	}
	
	public IDA putNum(string num, string den, string unit) {putNull(); cNum = num; cDen = den; cUnit = unit; refresh_num(); return this;}
	public IDA putNum(string num, string den) {return putNum(num, den, cUnit);}
	public IDA putNum(string num) {return putNum(num, "1");}
	
	/** <p>Place 64-bit floating-point content into this element with the provided value, replacing any existing content.</p>
	 * <p>If the provided unit string is non-null, also updates the unit string for this number (see setNumUnit).</p>
	 * @param num 64-bit floating-point content value.
	 * @param unit Number unit string.
	 * @return This element */
	public IDA putDouble(double num, string unit) {return putDouble(num).setNumUnit(unit);}
	/** <p>Place 32-bit floating-point content into this element with the provided value, replacing any existing content.</p>
	 * <p>If the provided unit string is non-null, also updates the unit string for this number (see setNumUnit).</p>
	 * @param num 32-bit floating-point content value.
	 * @param unit Number unit string.
	 * @return This element */
	public IDA putFloat(float num, string unit) {return putFloat(num).setNumUnit(unit);}
	/** <p>Place 64-bit integer content into this element with the provided value, replacing any existing content.</p>
	 * <p>If the provided unit string is non-null, also updates the unit string for this number (see setNumUnit).</p>
	 * @param num 64-bit integer content value.
	 * @param unit Number unit string.
	 * @return This element */
	public IDA putLong(long num, string unit) {return putLong(num).setNumUnit(unit);}
	/** <p>Place 32-bit integer content into this element with the provided value, replacing any existing content.</p>
	 * <p>If the provided unit string is non-null, also updates the unit string for this number (see setNumUnit).</p>
	 * @param num 32-bit integer content value.
	 * @param unit Number unit string.
	 * @return This element */
	public IDA putInt(int num, string unit) {return putInt(num).setNumUnit(unit);}
	/** <p>Place 64-bit floating-point content into this element with the provided value, replacing any existing content.</p>
	 * @param num 64-bit floating-point content value.
	 * @return This element */
	public IDA putDouble(double num) {return putNum(""+num);}
	/** <p>Place 32-bit floating-point content into this element with the provided value, replacing any existing content.</p>
	 * @param num 32-bit floating-point content value.
	 * @return This element */
	public IDA putFloat(float num) {return putNum(""+num);}
	/** <p>Place 64-bit integer content into this element with the provided value, replacing any existing content.</p>
	 * @param num 64-bit integer content value.
	 * @return This element */
	public IDA putLong(long num) {return putNum(""+num);}
	/** <p>Place 32-bit integer content into this element with the provided value, replacing any existing content.</p>
	 * @param num 32-bit integer content value.
	 * @return This element */
	public IDA putInt(int num) {return putNum(""+num);}
	
	// Content/number/get
	/** <p>Test whether the content type of this element is a numeric value.</p>
	 * @return True if this element has numeric content, otherwise false */
	public bool isNum()
	{
		return cNum != "";
	}
	/** <p>Fetch the unit string appended to the number represented by this element content.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * If numeric content had no unit specified, an empty string is returned.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return Numeric unit string. */
	public string getNumUnit()
	{
		if (!isNum()) throw new IDAOpException("Content is not numeric");
		return cUnit;
	}
	/** <p>Fetch the numerator string for the number represented by this element content.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * If the number was not a quotient, calling getNumN will return the same result as calling toStr.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return Numerator as a string. */
	public string getNumN()
	{
		if (!isNum()) throw new IDAOpException("Content is not numeric");
		return cNum;
	}
	/** <p>Fetch the denominator string for the number represented by this element content.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * If the number was not a quotient, calling getNumD will return "1".</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return Denominator as a string. */
	public string getNumD()
	{
		if (!isNum()) throw new IDAOpException("Content is not numeric");
		return cDen;
	}
	/** <p>Fetch this element content as a 64-bit floating-point value.</p>
	 * <p>If this element content is not numeric, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain numeric content.
	 * @return 64-bit floating-point content value if present, otherwise the default value. */
	public double asDoubleOr(double other)
	{
		return isNum() ? c_double() : other;
	}
	/** <p>Fetch this element content as a 32-bit floating-point value.</p>
	 * <p>If this element content is not numeric, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain numeric content.
	 * @return 32-bit floating-point content value if present, otherwise the default value. */
	public float asFloatOr(float other)
	{
		return isNum() ? c_float() : other;
	}
	/** <p>Fetch this element content as a 64-bit integer value.</p>
	 * <p>If this element content is not numeric, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain numeric content.
	 * @return 64-bit integer content value if present, otherwise the default value. */
	public long asLongOr(long other)
	{
		return isNum() ? c_long() : other;
	}
	/** <p>Fetch this element content as a 32-bit integer value.</p>
	 * <p>If this element content is not numeric, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain numeric content.
	 * @return 32-bit integer content value if present, otherwise the default value. */
	public int asIntOr(int other)
	{
		return isNum() ? c_int() : other;
	}
	/** <p>Fetch this element content as a 64-bit floating-point value.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * To avoid that possibility, use asDoubleOr to provide a default value.</p>
	 * <p>Use toDouble/toDoubleOr for casting other content types to be a 64-bit floating-point value.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return 64-bit floating-point content value. */
	public double asDouble()
	{
		if (!isNum()) throw new IDAOpException("Content is not numeric");
		return c_double();
	}
	/** <p>Fetch this element content as a 32-bit floating-point value.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * To avoid that possibility, use asFloatOr to provide a default value.</p>
	 * <p>Use toFloat/toFloatOr for casting other content types to be a 32-bit floating-point value.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return 32-bit floating-point content value. */
	public float asFloat()
	{
		if (!isNum()) throw new IDAOpException("Content is not numeric");
		return c_float();
	}
	/** <p>Fetch this element content as a 64-bit integer value.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * To avoid that possibility, use asLongOr to provide a default value.</p>
	 * <p>Use toLong/toLongOr for casting other content types to be a 64-bit integer value.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return 64-bit integer content value. */
	public long asLong()
	{
		if (!isNum()) throw new IDAOpException("Content is not numeric");
		return c_long();
	}
	/** <p>Fetch this element content as a 32-bit integer value.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * To avoid that possibility, use asIntOr to provide a default value.</p>
	 * <p>Use toInt/toIntOr for casting other content types to be a 32-bit integer value.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return 32-bit integer content value. */
	public int asInt()
	{
		if (!isNum()) throw new IDAOpException("Content is not numeric");
		return c_int();
	}
	
	private double c_double()
	{
		return (cDen == "1") ? n_double(cNum) : n_double(cNum)/n_double(cDen);
	}
	private float c_float()
	{
		return (float) c_double();
	}
	private long c_long()
	{
		return (cDen == "1") ? n_long(cNum) : (long)(Convert.ToDouble(cNum)/Convert.ToDouble(cDen));
	}
	private int c_int()
	{
		return (cDen == "1") ? (int)n_long(cNum) : (int)(Convert.ToDouble(cNum)/Convert.ToDouble(cDen));
	}
	private long n_long(string str)
	{
		if (str.StartsWith("0x")) {return n_hex(str);}
		if (str.StartsWith("0b")) {return n_bin(str);}
		try {return Convert.ToInt64(str);}
		catch (Exception) {return (long) Convert.ToDouble(str);}
	}
	private double n_double(string str)
	{
		if (str.StartsWith("0x")) {return (double) n_hex(str);}
		if (str.StartsWith("0b")) {return (double) n_bin(str);}
		return Convert.ToDouble(str);
	}
	private long n_hex(string str) {return Convert.ToInt64(str.Substring(2), 16);}
	private long n_bin(string str) {return Convert.ToInt64(str.Substring(2), 2);}
	
	// Content/list
	private void clearList() {cList = null;}
	private void refresh_list() {cRaw = "List("+numItems()+")"; cStr = null;}
	
	public IDA_Iterator<IDA> iterator() {return IDA.port.list_i<IDA>(this.cList);}
	public IEnumerator<IDA> GetEnumerator() {return cList.GetEnumerator();}
	IEnumerator IEnumerable.GetEnumerator() {return this.GetEnumerator();}
	public IDA this[int i]
	{
		get {return cList[i];}
	}
	
	// Content/list/set
	/** Returns this element */
	public IDA putList()
	{
		putNull();
		cList = new List<IDA>();
		refresh_list();
		return this;
	}
	/** Returns this element */
	public IDA putList(List<IDA> l)
	{
		putNull();
		if (l != null)
		{
			cList = l; refresh_list();
			IDA last = null;
			foreach (IDA item in l)
			{
				item.link(last, null);
				last = item;
			}
		}
		return this;
	}
	
	// Content/list/get
	/** <p>Test whether the content type of this element is a list.</p>
	 * <p>This reports whether the content is a list or non-list, as opposed to whether at least one list item is present.
	 * The primary purpose is to ensure the list exists before calling asList and/or iterating over its elements.</p>
	 * <p>Use numItems to test for an empty list (0) or at least one item present (greater than 0).</p>
	 * @return True if this element has list content, otherwise false */
	public bool isList() {return cList != null;}
	/** <p>Fetch this element content as a list (containing other elements).</p>
	 * <p>If this element content is not a list, an exception is thrown.
	 * To avoid that possibility, use asListOr to provide a default value.</p>
	 * @exception IDAOpException if this element does not contain list content.
	 * @return List content. */
	public List<IDA> asList()
	{
		if (cList == null) throw new IDAOpException("Content is not a list");
		return cList;
	}
	/** <p>Fetch this element content as a list (containing other elements).</p>
	 * <p>If this element content is not a list, null is returned.</p>
	 * <p>This method exists for platform compatibility where lists/vectors are not nullable/optional, such as with C++.
	 * The purpose is for this return type to be a nullable list container, while asListOr(other) returns a list directly.
	 * On platforms where lists can be null, asListOr() is equivalent to asListOr(null), but only the former is fully portable.</p>
	 * @return List content if present, otherwise null. */
	public List<IDA> asListOr()
	{
		return cList;
	}
	/** <p>Fetch this element content as a list (containing other elements).</p>
	 * <p>If this element content is not a list, the provided default value is returned.</p>
	 * <p>On platforms where lists/vectors are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed list, such that code will not be fully portable if null is provided.
	 * If possible, always call asListOr() rather than asListOr(null).</p>
	 * @param other A default value to return if this element does not contain list content.
	 * @return List content if present, otherwise the default value. */
	public List<IDA> asListOr(List<IDA> other)
	{
		return isList() ? cList : other;
	}
	/** <p>Fetch the number of content list items within this element.</p>
	 * <p>If no content list is present, the sentinel value -1 is returned.
	 * If 0 is returned, the list is present but empty.</p>
	 * @return The content list size if a list is present, otherwise -1 */
	public int numItems()
	{
		return isList() ? cList.Count : -1;
	}
	
	/** <p>Test whether this element contains a list item with the provided name.
	 * If it does, an equivalent call to getItem will return an element.</p>
	 * <p>If false is returned, this can be either due to having no list item of that name or due to being a non-list.
	 * Use isList to separate those two cases.</p>
	 * @param key A list item name to be checked
	 * @return True if a content list item exists with the provided name, otherwise false */
	public bool hasItem(string key) {return getItemOr(key) != null;}
	
	/** <p>Fetch the item element at the provided index in this element's content list.</p>
	 * <p>If this element content is not a list or has too few items in the list, an exception is thrown.
	 * To avoid that possibility, use getItemAtOr to provide a default value.</p>
	 * @param index A list item index to be recalled.
	 * @exception IDAOpException if this element has no such list item.
	 * @return The indexed content list item element. */
	public IDA getItemAt(int index)
	{
		if (numItems() <= index) throw new IDAOpException("Invalid list item index");
		return cList[index];
	}
	/** <p>Fetch the item element at the provided index in this element's content list.</p>
	 * <p>If this element content is not a list or has too few items in the list, null is returned.</p>
	 * @param index A list item index to be recalled.
	 * @return The indexed content list item element if present, otherwise null. */
	public IDA getItemAtOr(int index)
	{
		return getItemAtOr(index, null);
	}
	/** <p>Fetch the item element at the provided index in this element's content list.</p>
	 * <p>If this element content is not a list or has too few items in the list, the provided default value is returned.</p>
	 * @param index A list item index to be recalled.
	 * @param other A default value to return if this element has no such list item.
	 * @return The indexed content list item element if present, otherwise the default value. */
	public IDA getItemAtOr(int index, IDA other)
	{
		return numItems() > index ? cList[index] : other;
	}
	
	/** <p>Fetch the first item element with the provided name in this element's content list.</p>
	 * <p>If this element content is not a list or has no item of that name, an exception is thrown.
	 * To avoid that possibility, use getItemOr to provide a default value.</p>
	 * @param index A list item name to be recalled.
	 * @exception IDAOpException if this element has no such list item.
	 * @return The named content list item element. */
	public IDA getItem(String key)
	{
		IDA item = getItemOr(key);
		if (item == null) throw new IDAOpException("Item not found");
		return item;
	}
	/** <p>Fetch the first item element with the provided name in this element's content list.</p>
	 * <p>If this element content is not a list or has no item of that name, null is returned.</p>
	 * @param key A list item name to be recalled.
	 * @return The named content list item element if present, otherwise null. */
	public IDA getItemOr(string key)
	{
		return getItemOr(key, null);
	}
	/** <p>Fetch the first item element with the provided name in this element's content list.</p>
	 * <p>If this element content is not a list or has no item of that name, the provided default value is returned.</p>
	 * @param key A list item name to be recalled.
	 * @param other A default value to return if this element has no such list item.
	 * @return The named content list item element if present, otherwise the default value. */
	public IDA getItemOr(string key, IDA other)
	{
		if (isList()) foreach (IDA x in asList()) if (x.isName(key)) return x;
		return other;
	}
	
	/** <p>Cast to a string value the first item element with the provided name in this element's content list.</p>
	 * <p>If this element content is not a list, has no item of that name or can not be interpreted as a string value, an exception is thrown.
	 * To avoid that possibility, use getItemStrOr to provide a default value.</p>
	 * @param key A list item name to be recalled.
	 * @exception IDAOpException if this element has no such list item, or the item content cannot cast to a string value.
	 * @return String content interpretation of the named list item element. */
	public string getItemStr(string key)
	{
		return getItem(key).toStr();
	}
	/** <p>Cast to a string value the first item element with the provided name in this element's content list.</p>
	 * <p>If this element content is not a list, has no item of that name or can not be interpreted as a string value, null is returned.</p>
	 * <p>This method exists for platform compatibility where strings are not nullable/optional, such as with C++.
	 * The purpose is for this return type to be a nullable string container, while getItemStrOr(key, other) returns a string directly.
	 * On platforms where string values can be null, getItemStrOr(key) is equivalent to getItemStrOr(key, null), but only the former is fully portable.</p>
	 * @param key A list item name to be recalled.
	 * @return String content interpretation of the named list item element if possible, otherwise null. */
	public string getItemStrOr(string key)
	{
		IDA item = getItemOr(key);
		return item != null ? item.toStrOr() : null;
	}
	/** <p>Cast to a string value the first item element with the provided name in this element's content list.</p>
	 * <p>If this element content is not a list, has no item of that name or can not be interpreted as a string value, the provided default value is returned.</p>
	 * <p>On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
	 * If possible, always call getItemStrOr(key) rather than getItemStrOr(key, null).</p>
	 * @param key A list item name to be recalled.
	 * @param other A default value to return if this element has no such list item, or the item content cannot cast to a string value.
	 * @return String content interpretation of the named list item element if possible, otherwise the default value. */
	public string getItemStrOr(string key, string other)
	{
		IDA item = getItemOr(key);
		return item != null ? item.toStrOr(other) : other;
	}
	
	/** <p>Fetch the first item element in this element's content list.</p>
	 * <p>If this element content is not a list or is an empty list, an exception is thrown.
	 * To avoid that possibility, use getItemFirstOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no such list item.
	 * @return The first content list item element. */
	public IDA getItemFirst()
	{
		if (numItems() < 1) throw new IDAOpException(isList() ? "Content list is empty" : "Element content is not a list");
		return cList[0];
	}
	/** <p>Fetch the first item element in this element's content list.</p>
	 * <p>If this element content is not a list or is an empty list, null is returned.</p>
	 * @return The first content list item element if present, otherwise null. */
	public IDA getItemFirstOr()
	{
		return getItemFirstOr(null);
	}
	/** <p>Fetch the first item element in this element's content list.</p>
	 * <p>If this element content is not a list or is an empty list, the provided default value is returned.</p>
	 * @param other A default value to return if this element has no such list item.
	 * @return The first content list item element if present, otherwise the default value. */
	public IDA getItemFirstOr(IDA other)
	{
		return numItems() > 0 ? cList[0] : other;
	}
	/** <p>Fetch the last item element in this element's content list.</p>
	 * <p>If this element content is not a list or is an empty list, an exception is thrown.
	 * To avoid that possibility, use getItemLastOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no such list item.
	 * @return The last content list item element. */
	public IDA getItemLast()
	{
		int n = numItems();
		if (n < 1) throw new IDAOpException(isList() ? "Content list is empty" : "Element content is not a list");
		return cList[n-1];
	}
	/** <p>Fetch the last item element in this element's content list.</p>
	 * <p>If this element content is not a list or is an empty list, null is returned.</p>
	 * @return The last content list item element if present, otherwise null. */
	public IDA getItemLastOr()
	{
		return getItemLastOr(null);
	}
	/** <p>Fetch the last item element in this element's content list.</p>
	 * <p>If this element content is not a list or is an empty list, the provided default value is returned.</p>
	 * @param other A default value to return if this element has no such list item.
	 * @return The last content list item element if present, otherwise the default value. */
	public IDA getItemLastOr(IDA other)
	{
		int n = numItems();
		return n > 0 ? cList[n-1] : other;
	}
	
	// Content/list/add
	/** Insert a item element at the end of the content list.<br>
	 * If this element does not have list content, a new list will be created.<br>
	 * @param el The item element to be stored.
	 * @return This element. */
	public IDA addItem(IDA el)
	{
		if (!isList()) putList();
		el.link(getItemLastOr(), null);
		cList.Add(el);
		refresh_list();
		return this;
	}
	/** Insert a item element at the end of the content list.<br>
	 * If this element does not have list content, a new list will be created if the makeList flag is set, otherwise an exception is thrown.<br>
	 * @param el The item element to be stored.
	 * @param makeList Whether to overwrite existing non-list content.
	 * @exception IDAOpException if this element content is not a list and the makeList flag is false.
	 * @return This element. */
	public IDA addItem(IDA el, bool makeList)
	{
		if (!isList() && !makeList) {throw new IDAOpException("Cannot append items onto a non-list");}
		return addItem(el);
	}
	
	/** Insert a item element at the specified content list index position.<br>
	 * If this element does not have list content, an exception is thrown.<br>
	 * @param el The item element to be stored.
	 * @param pos The list index position at which the item should be inserted.
	 * @exception IDAOpException if the specified position is below zero or above the present list size, or if this element content is not a list.
	 * @return This element. */
	public IDA addItemAt(IDA el, int pos)
	{
		if (!isList()) {throw new IDAOpException("Cannot merge items into a non-list");}
		list_add(cList, el, pos, "item");
		refresh_list();
		return this;
	}
	
	/** Create and insert a new item element at the end of the content list, with the provided element name.<br>
	 * If this element does not have list content, a new list will be created, overwriting existing content.<br>
	 * @param name Name string for the created item element.
	 * @return The created item element. */
	public IDA newItem(string name)
	{
		IDA el = new IDA(name);
		addItem(el);
		return el;
	}
	/** Create and insert a new item element at the end of the content list, with the provided element name.<br>
	 * If this element does not have list content, a new list will be created if the makeList flag is set, otherwise an exception is thrown.<br>
	 * @param name Name string for the created item element.
	 * @param makeList Whether to overwrite existing non-list content.
	 * @exception IDAOpException if this element content is not a list and the makeList flag is false.
	 * @return The created item element. */
	public IDA newItem(string name, bool makeList)
	{
		IDA el = new IDA(name);
		addItem(el, makeList);
		return el;
	}
	
	// Content/list/remove
	/** Remove the item at the specified content list index position.<br>
	 * If this element does not have list content, an exception is thrown.<br>
	 * @param pos The list index position of an item which should be removed.
	 * @exception IDAOpException if the specified position is below zero or beyond the present list size, or this element does not have list content. */
	public void remItemAt(int pos)
	{
		if (!isList()) {throw new IDAOpException("Cannot remove items from a non-list");}
		list_del(cList, pos, "item");
		refresh_list();
	}
	/** Remove the provided content list item.<br>
	 * If the item is absent or this element does not have list content, the call has no effect.<br>
	 * @param el The item element to be removed.
	 * @return True if an item was removed, otherwise false. */
	public bool remItem(IDA el)
	{
		if (isList() && list_rem(cList, el)) {refresh_list(); return true;}
		return false;
	}
	/** Remove the first content list item with the provided element name.<br>
	 * If no such item exists or this element does not have list content, the call has no effect.<br>
	 * @param name Name string of the item element to remove.
	 * @return True if an item was removed, otherwise false. */
	public bool unItem(string name)
	{
		IDA item = getItemOr(name);
		if (item != null) {remItem(item); return true;}
		return false;
	}
	
	// Object/equals
	public override bool Equals(object obj)
	{
		return (obj is IDA) && eq_inner((IDA) obj);
	}
	private bool eq_inner(IDA obj)
	{
		if (obj == null) {return false;}
		if (obj == this) {return true;}
		if (!eqName(obj)) {return false;}
		if (!eqParams(obj)) {return false;}
		if (!eqContent(obj)) {return false;}
		return true;
	}
	/** <p>Perform a top-level comparison of this element and the provided element, excluding adjacent elements.</p>
	 * <p>To be equal, both elements must match when interpreted as full articles.
	 * The element names, parameters and content must all match recursively.</p>
	 * <p>Note that content matches consider whether the <i>information</i> is the same.
	 * See eqContent for details.</p>
	 * @param obj An article element for comparison
	 * @return True if the elements represent matching articles, otherwise false */
	public bool eq(IDA obj)
	{
		return eq(obj, false);
	}
	/** <p>Perform a top-level comparison of this element and the provided element.</p>
	 * <p>To be equal, both elements must match when interpreted as full articles.
	 * The element names, parameters and content must all match recursively.</p>
	 * <p>Note that content matches consider whether the <i>information</i> is the same.
	 * See eqContent for details.</p>
	 * @param obj An article element for comparison
	 * @param adjacents Whether to include adjacent elements in the comparison
	 * @return True if the elements represent matching articles, otherwise false */
	public bool eq(IDA obj, bool adjacent)
	{
		if (obj == this) return true;
		if (obj == null) return false;
		if (this.txRefLabel(adjacent) != obj.txRefLabel(adjacent)) return false;
		if (!eq_inner(obj)) return false;
		
		if (adjacent)
		{
			IDA a = this;
			IDA b = (IDA)obj, el = b;
			while (a.hasPrev() && b.hasPrev())
			{
				a = a.getPrevOr(); b = b.getPrevOr();
				if (!a.eq_inner(b)) {return false;}
			}
			if (a.hasPrev() || b.hasPrev()) {return false;}
			
			a = this;
			b = el;
			while (a.hasNext() && b.hasNext())
			{
				a = a.getNextOr(); b = b.getNextOr();
				if (!a.eq_inner(b)) {return false;}
			}
			if (a.hasNext() || b.hasNext()) {return false;}
		}
		
		return true;
	}
	
	/** Compare the names of this element and the provided element.
	 * @param obj An element for name comparison
	 * @return True if the names match, otherwise false */
	public bool eqName(IDA obj) {return isName(obj.getNameOr());}
	/** <p>Compare the contents of this element and the provided element.</p>
	 * <p>The content type must match, and the contents must match in a type-dependent way
	 * such that each element contains the same <i>information</i>. Numbers must have the
	 * same precision, and scope references must target the same element relative to each
	 * scope. Foreign references are equal by default.</p>
	 * @param obj An element for content comparison
	 * @return True if the contents match, otherwise false */
	public bool eqContent(IDA obj)
	{
		if (cRaw != obj.cRaw) {return false;}
		if (cStr != obj.cStr) {return false;}
		if (cNum != obj.cNum) {return false;}
		//if (cDen != obj.cDen) {return false;} // Assured by testing cRaw
		//if (cUnit != obj.cUnit) {return false;} // Assured by testing cRaw
		if (!IDA.list_equals(cList, obj.cList)) {return false;}
		if (cBool != obj.cBool) {return false;}
		if (!eqContent_ref(cRef, obj.cRef)) return false;
		
		return true;
	}
	private bool eqContent_ref(IDA ref_a, IDA ref_b)
	{
		if ((ref_a != null) ^ (ref_b != null)) return false;
		if (ref_a == ref_b) return true;
		if (ref_a.hasRefName() ^ ref_b.hasRefName()) return false;
		if (!ref_a.hasRefName()) return true;
		return ref_a.getRefName() == ref_b.getRefName();
	}
	/** Compare recursively the parameters of this element and the provided element.
	 * @param obj An element for parameter comparison
	 * @return True if the parameters match, otherwise false */
	public bool eqParams(IDA obj)
	{
		return IDA.list_equals(parameters, obj.parameters);
	}
	
// Object/hash
public override int GetHashCode()
{
	int acc = 655360001;
	
	// Name
	acc = hash_merge(acc, getNameOr());
	
	// Params
	acc = hash_merge(acc, parameters);
	
	// Content
	acc = hash_merge(acc, cRaw);
	acc = hash_merge(acc, cNum);
	acc = hash_merge(acc, cDen);
	acc = hash_merge(acc, cUnit);
	acc = hash_merge(acc, cList);
	acc = hash_merge(acc, cBool);
	
	return acc;
}
private int hash_merge(int acc, object obj)
{
	int hash = 1000000007;
	if (obj != null)
	{
		if (obj is List<IDA>)
		{
			int list_acc = 1777777777;
			foreach (IDA x in ((List<IDA>)obj)) list_acc = hash_merge(list_acc, x);
			hash = list_acc;
		}
		else hash = obj.GetHashCode();
	}
	
	acc = (acc >> 3) | (acc << 29);
	hash = (hash << 5) | (hash >> 27);
	return acc ^ hash ^ 1000000009;
}

	// Object/capture
	private IDAObj repObj = null;
	/** <p>Top-level article encode operation for a full object tree.
	 * This method calls objEncode, so the top-level object must implement idaEncode.</p>
	 * 
	 * <p>After all encoding is complete, any object references declared via setRefObj will be established as element references in the article.</p>
	 * @param obj A top-level object, which must implement idaEncode, to encode and record as this article element.
	 * @return This article element. */
	public IDA objCapture(IDAObj obj)
	{
		this.objEncode(obj);
		
		Dictionary<IDAObj, IDA> m = new Dictionary<IDAObj, IDA>();
		capture_objs(m);
		capture_refs(m);
		
		return this;
	}
	/** <p>Encode one member object via idaEncode, with automatic element name via idaName.
	 * This element will become "representative" of the object, which is used to encode references via setRefObj.</p>
	 * 
	 * <p>This method should be called from within idaEncode itself, to record members which also implement the IDAObj codec interface.
	 * For the top-level operation, user code should call idaCapture instead to ensure that all object references are recorded.</p>
	 * @param obj An object, which must implement idaEncode, to encode and record as this element.
	 * @return This element. */
	public IDA objEncode(IDAObj obj)
	{
		this.setName(obj.idaName());
		repObj = obj;
		obj.idaEncode(this);
		
		return this;
	}
	private void capture_objs(Dictionary<IDAObj, IDA> objs)
	{
		if (repObj != null) {objs[repObj] = this;}
		if (hasParams()) foreach (IDA p in parameters) p.capture_objs(objs);
		if (isList()) foreach (IDA i in cList) i.capture_objs(objs);
	}
	private void capture_refs(Dictionary<IDAObj, IDA> refs)
	{
		if (cRefObj != null) {putRef(refs[cRefObj]);}
		if (hasParams()) foreach (IDA p in parameters) p.capture_refs(refs);
		if (isList()) foreach (IDA i in cList) i.capture_refs(refs);
	}
	
	// Object/restore
	private List<Action<IDAObj>> repObjActions = null;
	private void on_rep_obj(Action<IDAObj> callback)
	{
		if (repObj != null) {callback(repObj);}
		else
		{
			if (repObjActions == null) repObjActions = new List<Action<IDAObj>>();
			repObjActions.Add(callback);
		}
	}
	private void restore_check()
	{
		if (repObjActions != null) throw new IDAOpException("Decode process failed to resolve an object reference");
	}
	/** <p>Top-level article decode operation for a full object tree.
	 * This method calls objDecode, so the top-level object must implement idaDecode.</p>
	 * 
	 * <p>After all decoding is complete, the integrity will be checked to ensure that all object reference requests were resolved.
	 * If any calls to withRefObj did not receive the deferred result, an exception is thrown.
	 * This occurs if the article contained an element reference for which no object was constructed via objDecode.</p>
	 * @param obj A top-level object, which must implement idaDecode, to become the result of decoding this article element.
	 * @return The input object. */
	public T objRestore<T>(T obj) where T : IDAObj
	{
		objDecode(obj);
		this.walk(x => x.restore_check(), true, true);
		return obj;
	}
	/** <p>Decode one member object via idaDecode.
	 * This element will become "representative" of the object, which is used to decode references via getRefObj or withRefObj.</p>
	 * 
	 * <p>This method should be called from within idaDecode itself, to create members which also implement the IDAObj codec interface.
	 * For the top-level operation, user code should call idaRestore instead to verify that all object reference requests were resolved.</p>
	 * @param obj An object, which must implement idaDecode, to become the result of decoding this element.
	 * @return The input object. */
	public T objDecode<T>(T obj) where T : IDAObj
	{
		repObj = obj;
		if (repObjActions != null)
		{
			foreach (Action<IDAObj> c in repObjActions) c(obj);
			repObjActions = null;
		}
		obj.idaDecode(this);
		return obj;
	}
	
	// Deserialisation
	/** Read from the provided input handle, parse it as markup and use the result to build this element.<br>
	 * After the parser is done, this element will become the top-level entry element from the markup stream.
	 * @param input An open input handle for a stream containing markup data
	 * @return This element */
	public IDA inStream(Stream input)
	{
		return IDA_RX.inObj(this, input);
	}
	/** Read the provided byte array, parse it as markup and use the result to build this element.<br>
	 * After the parser is done, this element will become the top-level entry element from the markup stream.
	 * @param input A byte array containing markup data
	 * @return This element */
	public IDA inBytes(byte[] input)
	{
		return inStream(new MemoryStream(input));
	}
	/** Read the provided string, parse it as markup and use the result to build this element.<br>
	 * After the parser is done, this element will become the top-level entry element from the markup stream.
	 * @param input A string containing markup data
	 * @return This element */
	public IDA inString(string input)
	{
		return inBytes(Encoding.UTF8.GetBytes(input));
	}
	/** Read from the file at the provided path, parse it as markup and use the result to build this element.<br>
	 * After the parser is done, this element will become the top-level entry element from the markup stream.
	 * @param input A path to a file containing markup data
	 * @return This element */
	public IDA inFile(string path)
	{
		using (FileStream input = File.OpenRead(path)) {return inStream(input);}
	}
	
	// Serialisation
	/** Encode this element as top-level markup and write it to the provided output handle.<br>
	 * Adjacent elements will not be encoded, as if calling outStream(output, false).<br>
	 * Any write errors will be sent back to the caller.
	 * @param output An open output handle to receive a stream of markup data */
	public void outStream(TextWriter output) {outStream(output, false);}
	/** Encode this element as top-level markup and write it to the provided output handle.<br>
	 * If <i>adjacents</i> is true, all preceding and following elements will also be encoded.<br>
	 * If a preceding element was encoded that way, the markup will contain an entry instruction.<br>
	 * Any write errors will be sent back to the caller.
	 * @param output An open output handle to receive a stream of markup data
	 * @param adjacents Whether to write markup for elements adjacent to this one */
	public void outStream(TextWriter output, bool adjacents)
	{
		new IDA_TX(output).idaWrite(this, adjacents);
	}
	
	/** Encode this element as top-level markup and write it to the provided output handle.<br>
	 * Adjacent elements will not be encoded, as if calling outPrint(output, false).
	 * @param output An open output handle to receive a stream of markup data
	 * @return True on success, or false if a write error occurred */
	public bool outPrint(TextWriter output) {return outPrint(output, false);}
	/** Encode this element as top-level markup and write it to the provided output handle.<br>
	 * If <i>adjacents</i> is true, all preceding and following elements will also be encoded.<br>
	 * If a preceding element was encoded that way, the markup will contain an entry instruction.
	 * @param output An open output handle to receive a stream of markup data
	 * @param adjacents Whether to write markup for elements adjacent to this one
	 * @return True on success, or false if a write error occurred */
	public bool outPrint(TextWriter output, bool adjacents)
	{
		return new IDA_TX(output).idaPrint(this, adjacents);
	}
	
	/** Encode this element as top-level markup and return it as a byte array.<br>
	 * Adjacent elements will not be encoded, as if calling outBytes(false).
	 * @return A byte array containing the markup data */
	public byte[] outBytes() {return outBytes(false);}
	/** Encode this element as top-level markup and return it as a byte array.<br>
	 * If <i>adjacents</i> is true, all preceding and following elements will also be encoded.<br>
	 * If a preceding element was encoded that way, the markup will contain an entry instruction.
	 * @param adjacents Whether to write markup for elements adjacent to this one
	 * @return A byte array containing the markup data */
	public byte[] outBytes(bool adjacents)
	{
		using (MemoryStream mem = new MemoryStream())
		{
			using (StreamWriter buf = new StreamWriter(mem))
			{
				outStream(buf, adjacents);
				mem.Position = 0;
				byte[] output = new byte[mem.Length];
				mem.Read(output, 0, (int)mem.Length);
				return output;
			}
		}
	}
	
	/** Encode this element as top-level markup and return it as a string.<br>
	 * Adjacent elements will not be encoded, as if calling outString(false).
	 * @return A string containing the markup data. */
	public string outString() {return outString(false);}
	/** Encode this element as top-level markup and return it as a byte array.<br>
	 * If <i>adjacents</i> is true, all preceding and following elements will also be encoded.<br>
	 * If a preceding element was encoded that way, the markup will contain an entry instruction.
	 * @param adjacents Whether to write markup for elements adjacent to this one.
	 * @return A string containing the markup data. */
	public string outString(bool adjacents)
	{
		using (StringWriter buf = new StringWriter())
		{
			outStream(buf, adjacents);
			return buf.ToString();
		}
	}
	
	/** Encode this element as top-level markup and write it to the file at the provided path.<br>
	 * Adjacent elements will not be encoded, as if calling outFile(path, false).<br>
	 * Any write errors will be sent back to the caller.
	 * @param path A destination file path for the markup data. */
	public void outFile(string path) {outFile(path, false);}
	/** Encode this element as top-level markup and write it to the file at the provided path.<br>
	 * If <i>adjacents</i> is true, all preceding and following elements will also be encoded.<br>
	 * If a preceding element was encoded that way, the markup will contain an entry instruction.<br>
	 * Any write errors will be sent back to the caller.
	 * @param path A destination file path for the markup data.
	 * @param adjacents Whether to write markup for elements adjacent to this one. */
	public void outFile(string path, bool adjacents)
	{
		using (FileStream res = new FileStream(path, FileMode.Open, FileAccess.Write))
		{
			using (StreamWriter buf = new StreamWriter(res))
			{
				outStream(buf, adjacents);
			}
		}
	}
	
	// Construction
	public IDA(string name = null)
	{
		this.includePwd = IDA.gIncludePwd;
		this.reset();
		this.setName(name);
	}
	
	// Initialisation
	/** <p>Initialisate this element anew.</p>
	 * <p>All existing information comprising this element will be discarded.
	 * The result will be an anonymous, null-content element with no parameters list, no adjacent elements and default input/output options.</p>
	 * @return This element. */
	public IDA reset()
	{
		putNull();
		cRefName = null;
		repObj = null;
		repObjActions = null;
		
		itemClasses = null;
		name = null;
		next = null;
		prev = null;
		clearParams();
		
		opts.copyFrom(IDAOptions.defaultOptions);
		scope = null;
		itemTemplates = null;
		
		inputListDepth = -1;
		inputListLength = -1;
		
		return this;
	}
	
	// Copies
	/** <p>Create a new element as a copy of all aspects of this element, excluding adjacent elements.
	 * Any parameters or list content will be cloned recursively.</p>
	 * @return The copy element */
	public IDA copy() {return copy(false);}
	/** <p>Create a new element as a copy of all aspects of this element.
	 * Any parameters or list content will be cloned recursively.</p>
	 * @param adjacents Whether to include adjacent elements in the cloning process
	 * @return The copy element */
	public IDA copy(bool adjacent) {return copyTo(new IDA(), adjacent);}
	/** <p>Modify the provided element to become a copy of all aspects of this element, excluding adjacent elements.
	 * Any parameters or list content will be cloned recursively.</p>
	 * @param copy An element to become a copy of this element
	 * @return The copy element */
	public IDA copyTo(IDA copy) {return copyTo(copy, false);}
	/** <p>Modify the provided element to become a copy of all aspects of this element.
	 * Any parameters or list content will be cloned recursively.</p>
	 * @param copy An element to become a copy of this element
	 * @param adjacents Whether to include adjacent elements in the cloning process
	 * @return The copy element */
	public IDA copyTo(IDA copy, bool adjacent)
	{
		bool has_ref = txRefLabel(adjacent) > 0;
		copy_impl(this, copy, adjacent, adjacent);
		if (has_ref) copy.rxRefResolve(adjacent);
		return copy;
	}
	
	private IDA copy_inner()
	{
		return copy_impl(this, new IDA(), false, false);
	}
	private static IDA copy_impl(IDA orig, IDA copy, bool scanPrev, bool scanNext)
	{
		copy.copyName(orig);
		copy.copyParams(orig);
		copy.copyContent(orig);
		copy.copyOptions(orig);
		
		IDA a;
		if (scanPrev && (a = orig.getPrevOr()) != null) {copy.linkPrev(copy_impl(a, new IDA(), true, false));}
		if (scanNext && (a = orig.getNextOr()) != null) {copy.linkNext(copy_impl(a, new IDA(), false, true));}
		
		return copy;
	}
	
	// Utility
	/** <p>Call the provided function once for each element within this article.</p>
	 * <p>This element is passed first, then its parameters (if any), then its list content items (if any).
	 * Parameters and list items are treated recursively until the full article has been traversed.</p>
	 * <p>Optionally, the walk may also traverse adjacent elements recursively, requested via the reverse and forward flags.</p>
	 * @param func A callback function which will be supplied with article elements.
	 * @param rev Whether to include elements adjacent to this one in the reverse direction.
	 * @param fwd Whether to include elements adjacent to this one in the forward direction. */
	public void walk(Action<IDA> func, bool rev, bool fwd)
	{
		func(this);
		if (hasParams()) foreach (IDA p in parameters) p.walk(func, false, false);
		if (isList()) foreach (IDA i in cList) i.walk(func, false, false);
		if (fwd && hasNext()) getNextOr().walk(func, false, true);
		if (rev && hasPrev()) getPrevOr().walk(func, true, false);
	}
	
	/** <p>Check the integrity of article element.
	 * If a problem is detected which would lead to errors or corrupted output, an exception is thrown with details in the exception text.</p>
	 * <p>The tests can be performed with adjacent elements enabled or disabled, matching the markup output modes.
	 * Note that disabling adjacents may avoid some problems (such as with linkage), but may also induce others (such as with references).</p>
	 * <p>The top-level element linkage is tested to ensure that there are no loops, and that adjacent elements are doubly-linked correctly.
	 * A bad double-link can disrupt the output flow, and a loop can cause infinite iteration.</p>
	 * <p>Any elements with reference content are tested for foreign references, which become lost/null during output.</p>
	 * @param adjacents Whether to extend the test to include elements adjacent to this one.
	 * @exception IDAOpException if bad integrity is detected. */
	public void validate(bool adjacents)
	{
		validate_loop(adjacents);
		validate_foreign(adjacents);
	}
	private void validate_loop(bool adjacents)
	{
		if (!adjacents) return;
		Func<IDA, IDA> fwd = x => x.getNextOr();
		Func<IDA, IDA> rev = x => x.getPrevOr();
		validate_loop_f(fwd, rev);
		validate_loop_f(rev, fwd);
	}
	private void validate_loop_f(Func<IDA, IDA> step, Func<IDA, IDA> back)
	{
		IDA a = this, b = this;
		
		while (true)
		{
			IDA last = b;
			b = step(b);
			if (b == null) return;
			if (back(b) != last) throw new IDAOpException("Article contains a broken link");
			
			last = b;
			b = step(b);
			if (b == null) return;
			if (back(b) != last) throw new IDAOpException("Article contains a broken link");
			
			a = step(a);
			if (a == b) throw new IDAOpException("Article contains an adjacency loop");
		}
	}
	private void validate_foreign(bool adjacents)
	{
		List<IDA> foreign = new List<IDA>();
		walk(x => {if (x.isRef()) if (!port.list_contains(foreign, x.asRefOr())) foreign.Add(x.asRefOr());}, adjacents, adjacents);
		walk(x => {if (port.list_contains(foreign, x)) port.list_rem(foreign, x);}, adjacents, adjacents);
		if (foreign.Count > 0) throw new IDAOpException("Article contains a foreign reference");
	}
	
	
	private static void list_add(List<IDA> list, IDA el, int pos, string noun)
	{
		int size = list.Count;
		
		if (pos > 0)
		{
			if (pos > size) {throw new IDAOpException("Cannot add "+noun+" beyond position "+size+" (request was "+pos+")");}
		}
		else if (pos < 0)
		{
			if (pos < -size) {throw new IDAOpException("Cannot add "+noun+" before position -"+size+" (request was "+pos+")");}
			pos += size;
		}
		
		if (pos == 0)
		{
			if (list.Count > 0) {el.link(null, list[0]);}
			list.Insert(0, el);
		}
		else if (pos == size)
		{
			el.link(list[size-1], null);
			list.Add(el);
		}
		else // size must be at least 2, pos cannot be at either end
		{
			el.link(list[pos-1], list[pos]);
			list.Insert(pos, el);
		}
	}
	private static IDA list_del(List<IDA> list, int pos, string noun)
	{
		int size = list.Count;
		
		if (pos > 0)
		{
			if (pos > size-1) {throw new IDAOpException("Cannot remove item beyond position "+(size-1)+" (request was "+pos+")");}
		}
		else if (pos < 0)
		{
			if (pos < -size) {throw new IDAOpException("Cannot remove item before position -"+size+" (request was "+pos+")");}
			pos += size;
		}
		
		IDA rem = list[pos].link(null, null);
		list.RemoveAt(pos);
		return rem;
	}
	
	// Syntax
	public static bool id0(char c) {return (c>='a' && c<='z') || (c>='A' && c<='Z') || c == '_' || c == '/';}
	public static bool id1(char c)
	{
		return id0(c) || (c>='0' && c<='9') || c == '-' || c == '.'
			|| c == '!' || c == '@' || c == '^' || c == '&' || c == '?';
	}
	public static bool comment1(char c) {return c == '#';}
	public static bool comment2(char c1, char c2)
	{
		return (c1 == '/' && (c2 == '/' || c2 == '*'))
			|| (c1 == '<' && (c2 == '?' || c2 == '!'));
	}
	
	// Compatibility-port-through
	public static readonly IDA_Port port = new IDA_Port();
	
	public static bool eq_obj(object a, object b)
	{
		return (a is IDA) ? ((IDA)a).Equals(b) : a == b;
	}
	
	private static bool list_rem(List<IDA> l, IDA el)
	{
		if (port.list_rem(l, el)) {el.link(null, null); return true;} else return false;
	}
	
	public static bool list_equals(List<IDA> la, List<IDA> lb)
	{
		if (la == null || lb == null) return la == lb;
		IDA_Iterator<IDA> ia = IDA.port.list_i(la);
		IDA_Iterator<IDA> ib = IDA.port.list_i(lb);
		while (ia.hasNext() && ib.hasNext())
		{
			if (!ia.next().eq(ib.next())) {return false;}
		}
		return !(ia.hasNext() || ib.hasNext());
	}
	
	public static int Main(string[] args) {return (IDA_CLI.cli(args.Length, args));}
}

