//    Copyright 2022 IDA Markup, ida-markup.com
// 
//    Licensed under the Apache License, Version 2.0 (the "License");
//    you may not use this file except in compliance with the License.
//    You may obtain a copy of the License at
// 
//        http://www.apache.org/licenses/LICENSE-2.0
// 
//    Unless required by applicable law or agreed to in writing, software
//    distributed under the License is distributed on an "AS IS" BASIS,
//    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//    See the License for the specific language governing permissions and
//    limitations under the License.

// Namespace

// Options
class IDAOptions
{
	constructor(options = null)
	{
		if (options != null) {this.copyFrom(options);}
		else
		{
			// Default RX
			this.rxXmlReparseStrings = false;
			
			// Default TX
			this.txUnicode = false;
		}
	}
	
	copyFrom(options)
	{
		// Copy RX
		this.rxXmlReparseStrings = options.rxXmlReparseStrings;
		
		// Copy TX
		this.txUnicode = options.txUnicode;
	}
}
IDAOptions.defaultOptions = new IDAOptions();

// Object interface
// Adapted to provide default implementations if called this way
class IDAObj
{
	/** Override to offer a default element name. */
	static idaName(obj)
	{
		if (typeof obj.idaName == "function") {return obj.idaName();}
		else {return null;}
	}
	/** Override to enable capturing of this object into the provided storage element.<br>
	 * Usage note: If applicable, implementing classes can guarantee not to throw.<br>
	 * @return The input element. */
	static idaEncode(obj, el)
	{
		if (typeof obj.idaEncode == "function") {return obj.idaEncode(el);}
		else {throw new IDAOpException("IDA encode not available");}
	}
	/** Override to enable restoration of this object from the provided storage element.<br>
	 * @return This object. */
	static idaDecode(obj, el)
	{
		if (typeof obj.idaDecode == "function") {return obj.idaDecode(el);}
		else {throw new IDAOpException("IDA decode not available");}
	}
	
	// Default implementation
	/*
	idaName() {return null;}
	idaEncode(el)
	{
		throw new IDAOpException("IDA encode not available");
	}
	idaDecode(el)
	{
		throw new IDAOpException("IDA decode not available");
	}
	*/
}

// Port begin
class IDA_Iterator
{
	hasNext() {return false;}
	getNext() {return null;}
	next() {return this.getNext();}
}
class IDA_MutableSeq extends IDA_Iterator
{
	constructor() {super(); this.funcHas = () => false; this.funcGet = () => null;}
	hasNext() {return (this.funcHas)();}
	getNext() {return (this.funcGet)();}
}
class IDA_Port
{
	
	// Port iterators
	// Iterator/MutableSeq in Port begin
	empty_i()
	{
		return new IDA_Iterator();
	}
	list_i(l)
	{
		if (l == null) throw new IDAOpException("Internal list iteration error");
		const i = new IDA_MutableSeq();
		i.n = 0;
		i.funcHas = () => i.n < l.length;
		i.funcGet = () => l[i.n++];
		return i;
	}
	repeat_i(obj)
	{
		const i = new IDA_MutableSeq();
		i.array_item_template = obj;
		i.funcHas = () => true;
		i.funcGet = () => i.array_item_template;
		return i;
	}
	
	// Port arrays
	list_contains(l, item)
	{
		return l.includes(item);
	}
	list_rem(l, item)
	{
		for (var i=0; i<l.length; i+=1)
		{
			if (IDA.eq_obj(item, l[i]))
			{
				l.splice(i, 1);
				return true;
			}
		}
		return false;
	}
	
	// Port streams
	si_allMatch(i, predicate)
	{
		while (i.hasNext()) if (!predicate(i.next())) {return false;}
		return true;
	}
	si_anyMatch(i, predicate)
	{
		while (i.hasNext()) if (predicate(i.next())) {return true;}
		return false;
	}
	si_noneMatch(i, predicate)
	{
		return !this.si_anyMatch(i, predicate);
	}
	
	i_map(i, map)
	{
		const j = new IDA_MutableSeq();
		j.funcHas = () => i.hasNext();
		j.funcGet = () => map(i.getNext());
		return j;
	}
	i_filter(i, filter)
	{
		const j = new IDA_MutableSeq();
		j.buf = null;
		j.more = false;
		j.preLoad = () =>
		{
			j.more = false;
			while (i.hasNext())
			{
				if (j.more = filter(j.buf = i.getNext())) return;
			}
		};
		j.funcHas = () => j.more;
		j.funcGet = () => {var x = j.buf; j.preLoad(); return x;};
		j.preLoad();
		return j;
	}
	i_flatMap(i, map)
	{
		const j = new IDA_MutableSeq();
		j.lbuf = null;
		j.buf = null;
		j.more = false;
		j.preLoad = () =>
		{
			j.more = false;
			while (j.lbuf == null || !j.lbuf.hasNext())
			{
				if (!i.hasNext()) {return;}
				j.lbuf = map(i.getNext());
			}
			
			j.more = true;
			j.buf = j.lbuf.getNext();
		};
		j.funcHas = () => j.more;
		j.funcGet = () => {var y = j.buf; j.preLoad(); return y;};
		j.preLoad();
		return j;
	}
	
	// Port string operation
	replace_all(s, act, dmd)
	{
		if (act.length < 1) return s;
		var out = "";
		
		const n = s.length;
		var i;
		var ptr = 0;
		
		while ((i = s.indexOf(act, ptr)) >= 0)
		{
			out += s.substring(ptr, i);
			out += dmd;
			ptr = i+act.length;
		}
		
		return out + s.substring(ptr);
	}
	
	substring(s, fromChar, toChar=null)
	{
		s = s.substring(this.str_units(s, 0, fromChar));
		return toChar !== null ? s.substring(0, this.str_units(s, 0, toChar-fromChar)) : s;
	}
	
	trim(s)
	{
		var i1 = 0;
		var i2 = s.length;
		while (i1 < i2 && s[i1].charCodeAt(0) <= 0x20) i1++;
		while (i1 < i2 && s[i2-1].charCodeAt(0) <= 0x20) i2--;
		return s.substring(i1, i2);
	}
	rtrim(s)
	{
		var i2 = s.length;
		while (0 < i2 && s[i2-1].charCodeAt(0) <= 0x20) i2--;
		return s.substring(0, i2);
	}
	ltrim(s)
	{
		var i1 = 0;
		var i2 = s.length;
		while (i1 < i2 && s[i1].charCodeAt(0) <= 0x20) i1++;
		return s.substring(i1);
	}
	
	// Port string tests
	ends_with(s, suffix)
	{
		return s.endsWith(suffix);
	}
	starts_with(s, prefix)
	{
		return s.startsWith(prefix);
	}
	str_contains(s, mid)
	{
		return s.indexOf(mid) >= 0;
	}
	
	// Port string info
	/** Character index */
	str_index(s, mid)
	{
		var i = s.indexOf(mid);
		return i < 1 ? i : this.str_chars(s, 0, i);
	}
	/** Character index */
	str_index_from(s, mid, fromChar)
	{
		var fromUnit = this.str_units(s, 0, fromChar);
		var i = s.indexOf(mid, fromUnit);
		return i < 0 ? -1 : (fromChar + this.str_chars(s, fromUnit, i));
	}
	is_high_surrogate(c)
	{
		return (c.charCodeAt(0) & 0xfc00) == 0xd800;
	}
	is_low_surrogate(c)
	{
		return (c.charCodeAt(0) & 0xfc00) == 0xdc00;
	}
	/** The number of characters represented */
	str_len(s)
	{
		return this.str_chars(s, 0, s.length);
	}
	/** The number of characters represented */
	str_chars(s, fromUnit, toUnit)
	{
		var n = toUnit - fromUnit;
		for (var i=toUnit-1; i>=fromUnit; i-=1) if (this.is_low_surrogate(s[i])) n-=1;
		return n;
	}
	/** The number of memory locations */
	str_units(s, fromChar, toChar)
	{
		toChar -= fromChar;
		
		// Skip
		var mark = 0;
		while (fromChar-- > 0) if (this.is_low_surrogate(s[++mark])) ++mark;
		
		// Count
		var ptr = mark;
		while (toChar > 0) if (!this.is_high_surrogate(s[ptr++])) toChar--;
		
		return ptr - mark;
	}
	
	// Port hex
	char_2(c)
	{
		return c.toString(16).padStart(2, "0");
	}
	char_4(c)
	{
		return c.toString(16).padStart(4, "0");
	}
	char_8(c)
	{
		return c.toString(16).padStart(8, "0");
	}
	
// Port end
}

// Format exception
class IDAFormatException extends Error
{
	constructor(source, message)
	{
		if (source instanceof IDA_RX)
		{
			const rx = source;
			super("(line "+rx.ln+") "+message);
			
			this.node = null;
			
			// Extract problem index first
			this.pos = rx.lnBuf.length;
			
			// Read until end of line for context
			var ln = rx.lnBuf;
			try {ln += rx.get_to("\n");}
			catch (e) {}
			this.ln = ln;
		}
		else if (source instanceof IDA)
		{
			const nodeCause = source;
			super((nodeCause.isAnon() ? "(anonymous element) " : "(element '"+nodeCause.getNameOr("null")+"') ") + message);
			
			this.ln = null;
			this.pos = -1;
			this.node = nodeCause;
		}
		else {super("Unknown source type: " + (typeof source));}
		
		this.name = "IDAFormatException";
	}
	
	getMessage() {return this.message;}
}

// Op exception
class IDAOpException extends Error
{
	constructor(msg)
	{
		super(msg);
		this.name = "IDAOpException";
	}
	getMessage() {return this.message;}
}

// Manip begin
class IDA_Manip
{
	static FORWARD() {return true;}
	static REVERSE() {return false;}
	static rev(n) {return typeof n == typeof true ? !n : -n;}
	
	static m_str(el)
	{
		if (el.isNull()) return null;
		try {return el.asStr();}
		catch (ex) {throw new IDAFormatException(el, "String required");}
	}
	
	// Manip/iterate
	/** Return the last node */
	static manip_iterate(manip, node, itemTemplate)
	{
		var last = null;
		while (itemTemplate.hasNext())
		{
			var templateNode = itemTemplate.next();
			
			if (last != null) {node = new IDA(); last.linkNext(node);}
			
			node.copyName(templateNode);
			node.copyParams(templateNode);
			if (templateNode.isList())
			{
				node.putList();
				
				if (templateNode.numItems() > 0)
				{
					var item = new IDA();
					if (IDA_Manip.manip_iterate(manip, item, IDA.port.list_i(templateNode.asListOr())) != null)
					{
						for (var nextEl; item != null; item = nextEl)
						{
							nextEl = item.getNextOr();
							node.addItem(item);
						}
					}
					
				}
			}
			else {node.copyContent(templateNode);}
			last = node;
			
			// Manip/iterate/main
			if (manip.isList())
			manip: for (var m of manip.asList())
			{
				if (m.isAnon() || m.isNull()) {continue;} // Operate on nothing, or no operations
				if (!m.isList()) {throw new IDAFormatException(m, "Manipulation operations must be a list");}
				
				var operateOn = m.getNameOr();
				const n = operateOn.indexOf("n") >= 0; // n: name
				const k = operateOn.indexOf("k") >= 0; // k: param key
				const v = operateOn.indexOf("v") >= 0; // v: param value
				const p = operateOn.indexOf("p") >= 0; // p: parameter(s)
				const c = operateOn.indexOf("c") >= 0; // c: content
				
				// Manip/iterate/match
				// if-match: any of (n node-name, k param-key, v param-val, p(param-key param-val), c node-content)
				var tParamMatch = (x) => true;
				if (m.hasParams()) for (var mp of m.getParamsOr())
				{
					var sel = mp.getNameOr("");
					
					if (sel.replace(/[nkvc!]+/g, "").length > 0) {throw new IDAFormatException(m, "Invalid manipulation match (valid characters are [nkvc!])");}
					
					const flagNot = sel.indexOf("!") >= 0;
					
					if (sel.indexOf("n") >= 0)
					{
						if ((node.isName(IDA_Manip.m_str(mp))) == flagNot) {continue manip;}
					}
					if (sel.indexOf("c") >= 0)
					{
						if (mp.eqContent(node) == flagNot) {continue manip;}
					}
					
					if (sel.indexOf("k") >= 0)
					{
						const tgt = IDA_Manip.m_str(mp);
						const tParamMatchOld = tParamMatch;
						tParamMatch = (x) => tParamMatchOld(x) && (flagNot ^ (x.isName(tgt)));
						break;
					}
					if (sel.indexOf("v") >= 0)
					{
						const tParamMatchOld = tParamMatch;
						tParamMatch = (x) => tParamMatchOld(x) && (flagNot ^ mp.eqContent(x));
						break;
					}
					
				}
				const paramMatch = tParamMatch;
				
				// Manip/iterate/op
				for (var op of m.asList())
				{
					if (op.isAnon()) {throw new IDAFormatException(m, "Missing operation code");}
					var opName = op.getNameOr("");
					
					switch (opName)
					{
					
						// Manip/iterate/op/insert
						case "i": // i: insert (nkvc): i (index) text-string
						{
							if (op.isNull()) {throw new IDAFormatException(m, "Missing insert string");}
							if (op.isList()) {throw new IDAFormatException(m, "Insert string cannot be a list");}
							
							var param = op.getParamFirstOr();
							if (param != null && !param.isNum()) {throw new IDAFormatException(m, "Insert position must be a number");}
							var pos = (param != null) ? param.asIntOr(0) : IDA_Manip.REVERSE();
							
							try
							{
								if (n) {node.setName(IDA_Manip.manip_ins(node.getNameOr(), op.asStr(), pos));}
								if (c) {node.putStr(IDA_Manip.manip_ins(node.asStr(), op.asStr(), pos));}
								if (k || v)
								{
									if (node.hasParams()) for (var np of node.getParams())
									{
										if (!paramMatch(np)) {continue;}
										if (k) {np.setName(IDA_Manip.manip_ins(np.getNameOr(), op.asStr(), pos));}
										if (v) {np.putStr(IDA_Manip.manip_ins(np.asStr(), op.asStr(), pos));}
									}
								}
							}
							catch (opex) {throw opex instanceof IDAOpException ? new IDAFormatException(m, opex.message) : opex;}
							
							break;
						}
						
						// Manip/iterate/op/delete
						case "d": // d: delete (nkvc): d (index) length or d (index) bool
						{
							if (!(op.isNum() || op.isBool() || op.isNull())) {throw new IDAFormatException(m, "Delete amount must be boolean or a number");}
							
							var param = op.getParamFirstOr();
							if (param != null && !param.isNum()) {throw new IDAFormatException(m, "Delete position must be a number");}
							const pos = (param != null) ? param.asIntOr(0) : IDA_Manip.REVERSE();
							var amt = op.isBool() ? (op.isTrue() ? IDA_Manip.FORWARD() : IDA_Manip.REVERSE()) : (op.isNull() ? 1 : op.asIntOr(0));
							if ((pos === IDA_Manip.REVERSE()) && (amt === IDA_Manip.FORWARD() || (typeof amt == typeof 0 && amt > 0))) {amt = IDA_Manip.rev(amt);}
							
							try
							{
								if (n) {node.setName(IDA_Manip.manip_del(node.getNameOr(), pos, amt));}
								if (c) {node.putStr(IDA_Manip.manip_del(node.asStr(), pos, amt));}
								if (k || v)
								{
									if (node.hasParams()) for (var np of node.getParams())
									{
										if (!paramMatch(np)) {continue;}
										if (k) {np.setName(IDA_Manip.manip_del(np.getNameOr(), pos, amt));}
										if (v) {np.putStr(IDA_Manip.manip_del(np.asStr(), pos, amt));}
									}
								}
							}
							catch (opex) {throw opex instanceof IDAOpException ? new IDAFormatException(m, opex.message) : opex;}
							
							break;
						}
						
						// Manip/iterate/op/add
						case "a": // a: add (vpc): a (index) {nodes...}
						{
							// No need to check isList; addItem and insertItem already have exceptions for this
							var param = op.getParamFirstOr();
							if (param != null && !param.isNum()) {throw new IDAFormatException(m, "Add position must be a number");}
							const pos = (param != null) ? param.asIntOr(0) : IDA_Manip.REVERSE();
							
							try
							{
								if (p)
								{
									if (!node.hasParams()) {node.setParams();}
									IDA_Manip.manip_add((item, pos) => node.addParamAt(item, pos), (pos === IDA_Manip.REVERSE()) ? node.numParams() : pos, op);
								}
								if (c)
								{
									IDA_Manip.manip_add((item, pos) => node.addItemAt(item, pos), (pos === IDA_Manip.REVERSE()) ? node.numItems() : pos, op);
								}
								if (v)
								{
									if (node.hasParams()) for (var np of node.getParams())
									{
										if (!paramMatch(np)) {continue;}
										IDA_Manip.manip_add((item, pos) => np.addItemAt(item, pos), (pos === IDA_Manip.REVERSE()) ? np.numItems() : pos, op);
									}
								}
							}
							catch (opex) {throw opex instanceof IDAOpException ? new IDAFormatException(m, opex.message) : opex;}
							
							break;
						}
						
						// Manip/iterate/op/remove
						case "r": // r: remove (vpc): r (index) length or r(index) bool or r {nodes...}
						{
							if (op.isNum() || op.isBool() || op.isNull()) // r (index) length or r(index) bool
							{
								var param = op.getParamFirstOr();
								if (param != null && !param.isNum()) {throw new IDAFormatException(m, "Remove position must be a number");}
								const pos = (param != null) ? param.asIntOr(0) : IDA_Manip.REVERSE();
								var amt = op.isBool() ? (op.isTrue() ? IDA_Manip.FORWARD() : IDA_Manip.REVERSE()) : (op.isNull() ? 1 : op.asIntOr(0));
								if ((pos === IDA_Manip.REVERSE()) && (amt === IDA_Manip.FORWARD() || (typeof amt == typeof 0 && amt > 0))) {amt = IDA_Manip.rev(amt);}
								
								if (p)
								{
									IDA_Manip.manip_rem_pos(m, (pos) => node.remParamAt(pos), () => node.numParams(), pos, amt);
								}
								if (c)
								{
									IDA_Manip.manip_rem_pos(m, (pos) => node.remItemAt(pos), () => node.numItems(), pos, amt);
								}
								if (v)
								{
									if (node.hasParams()) for (var np of node.getParamsOr())
									{
										if (!paramMatch(np)) {continue;}
										IDA_Manip.manip_rem_pos(m, (pos) => np.remItemAt(pos), () => np.numItems(), pos, amt);
									}
								}
							}
							else if (op.isList()) // r {nodes...}
							{
								if (p) {IDA_Manip.manip_rem_item((param) => node.remParam(param), op);}
								if (c) {IDA_Manip.manip_rem_item((item) => node.remItem(item), op);}
								if (v)
								{
									if (node.hasParams()) for (var np of node.getParamsOr())
									{
										if (paramMatch(np)) IDA_Manip.manip_rem_item((item) => np.remItem(item), op);
									}
								}
							}
							else {throw new IDAFormatException(m, "Remove subject must be boolean, a number or a list");}
							
							break;
						}
						
						// Manip/iterate/op/undo
						case "u": // u: unset/unlist (vpc): u text-key or u {text-key...}
						{
							if (op.isList()) for (var un of op.asList()) // u {text-key...}
							{
								const name = IDA_Manip.m_str(un);
								
								if (c) {node.unItem(name);}
								if (p) {node.unParam(name);}
								if (v && node.hasParams()) for (var np of node.getParamsOr())
								{
									if (paramMatch(np)) np.unItem(name);
								}
							}
							else // u text-key
							{
								const name = IDA_Manip.m_str(op);
								
								if (c) {node.unItem(name);}
								if (p) {node.unParam(name);}
								if (v && node.hasParams()) for (var np of node.getParamsOr())
								{
									if (paramMatch(np)) np.unItem(name);
								}
							}
							break;
						}
						
						// Manip/iterate/op/substitute
						case "s": // s: substitute (nkvc): s (search-token replacement-string ...);
						{
							if (op.hasParams()) for (var repl of op.getParamsOr())
							{
								if (repl.isAnon() || repl.isNull()) {continue;}
								const act = repl.getNameOr();
								const dmd = IDA_Manip.m_str(repl);
								
								// nkvc
								if (n && node.hasName()) {node.setName(IDA.port.replace_all(node.getNameOr(), act, dmd));}
								if (c && node.hasContent()) {node.putStr(IDA.port.replace_all(IDA_Manip.m_str(node), act, dmd));}
								if (k || v)
								{
									if (node.hasParams()) for (var np of node.getParamsOr())
									{
										if (!paramMatch(np)) {continue;}
										if (k && np.hasName()) {np.setName(IDA.port.replace_all(np.getNameOr(), act, dmd));}
										if (v && np.hasContent()) {np.putStr(IDA.port.replace_all(IDA_Manip.m_str(np), act, dmd));}
									}
								}
							}
							break;
						}
						
						// Manip/iterate/op/flip
						case "f": // f (vc): flip: f;
						{
							try
							{
								if (c) {IDA_Manip.manip_negate(node);}
								if (v && node.hasParams()) for (var np of node.getParams())
								{
									if (!paramMatch(np)) {continue;}
									IDA_Manip.manip_negate(np);
								}
							}
							catch (opex) {throw opex instanceof IDAOpException ? new IDAFormatException(m, opex.message) : opex;}
							break;
						}
						
						// Manip/iterate/op/write
						case "w": // w (nkvpc): write: w replacement
						{
							if (n) {node.setName(IDA_Manip.m_str(op));}
							if (c) {node.copyContent(op);}
							if (p)
							{
								if (op.isNull()) {node.clearParams();}
								else if (op.isList())
								{
									node.setParams();
									for (var param of op.asListOr()) {node.addParam(param.copy());}
								}
								else {throw new IDAFormatException(m, "Parameter write must be list or null");}
							}
							if (k || v)
							{
								if (node.hasParams()) for (var np of node.getParamsOr())
								{
									if (!paramMatch(np)) {continue;}
									if (k) {np.setName(IDA_Manip.m_str(op));}
									if (v) {np.copyContent(op);}
								}
							} 
							break;
						}
						
						// Manip/iterate/op end
						default: throw new IDAFormatException(m, "Invalid operation code \""+opName+"\"");
					}
				}
				
			// Manip/iterate end
			}
		
		}
		
		return last;
		
	}
	
	// Manip/negate
	static manip_negate(node)
	{
		if (node.isBool())
		{
			node.putBool(!node.asBool());
		}
		else if (node.isNum())
		{
			const num = node.getNumN();
			node.putNum(num.startsWith("-") ? num.substring(1) : "-"+num, node.getNumD());
		}
		else if (node.hasContent()) {throw new IDAOpException("Flip subject must be boolean or a number");}
	}
	
	// Manip/insert
	/** Usage note: the 'ins' string is never null, and the pos can be REVERSE in place of a null to mean the end (never FORWARD though) */
	static manip_ins(orig, ins, pos)
	{
		if (orig == null) {throw new IDAOpException("Insert subject must not be null");}
		if (pos === IDA_Manip.REVERSE()) {return orig + ins;}
		const l = IDA.port.str_len(orig);
		if (pos > 0)
		{
			if (pos > l) {throw new IDAOpException("Cannot insert string beyond position "+l+" (request was "+pos+")");}
			return IDA.port.substring(orig, 0, pos) + ins + IDA.port.substring(orig, pos);
		}
		else if (pos < 0)
		{
			if (pos < -l) {throw new IDAOpException("Cannot insert string before position -"+l+" (request was "+pos+")");}
			return IDA.port.substring(orig, 0, l+pos) + ins + IDA.port.substring(orig, l+pos);
		}
		else {return ins + orig;}
	}
	
	// Manip/delete
	/** Usage note: the pos can be REVERSE in place of a null to mean the end (never FORWARD though) */
	static manip_del(orig, pos, amt)
	{
		if (orig == null) {throw new IDAOpException("Delete subject must not be null");}
		const l = IDA.port.str_len(orig);
		if (pos === IDA_Manip.REVERSE()) {pos = l;}
		if (pos >= 0)
		{
			if (pos > l) {throw new IDAOpException("Cannot delete from string beyond position "+l+" (request was "+pos+")");}
			if (amt === IDA_Manip.REVERSE()) {return IDA.port.substring(orig, pos);}
			if (amt === IDA_Manip.FORWARD()) {return IDA.port.substring(orig, 0, pos);}
			if (amt > 0)
			{
				if (pos + amt > l) {throw new IDAOpException("Cannot delete more than "+(l-pos)+" beyond position "+pos+" (request was "+amt+")");}
				return IDA.port.substring(orig, 0, pos) + IDA.port.substring(orig, pos+amt);
			}
			if (amt < 0)
			{
				if (pos + amt < 0) {throw new IDAOpException("Cannot delete more than "+pos+" before position "+pos+" (request was "+(-amt)+")");}
				return IDA.port.substring(orig, 0, pos+amt) + IDA.port.substring(orig, pos);
			}
			return orig;
		}
		else //if (pos < 0)
		{
			if (pos < -l) {throw new IDAOpException("Cannot delete from string before position -"+l+" (request was "+pos+")");}
			if (amt === IDA_Manip.REVERSE()) {return IDA.port.substring(orig, pos+l);}
			if (amt === IDA_Manip.FORWARD()) {return IDA.port.substring(orig, 0, pos+l);}
			if (amt > 0)
			{
				if (pos+l + amt > l) {throw new IDAOpException("Cannot delete more than "+(-pos)+" beyond position "+pos+" (request was "+amt+")");}
				return IDA.port.substring(orig, 0, pos+l) + IDA.port.substring(orig, pos+l+amt);
			}
			if (amt < 0)
			{
				if (pos+l + amt < 0) {throw new IDAOpException("Cannot delete more than "+(l+pos)+" before position "+pos+" (request was "+(-amt)+")");}
				return IDA.port.substring(orig, 0, pos+l+amt) + IDA.port.substring(orig, pos+l);
			}
			return orig;
		}
		
	}
	
	// Manip/add
	static manip_add(func, pos, nodeFrom)
	{
		for (var ins of nodeFrom.asList())
		{
			func(ins.copy(), pos);
			if (pos >= 0) {pos += 1;}
		}
	}
	
	// Manip/rem
	static manip_rem_pos(opNode, func, size, pos, amt)
	{
		const n = size();
		if (pos === IDA_Manip.REVERSE()) {pos = n;}
		
		if (amt === IDA_Manip.FORWARD()) {amt = n - pos;}
		else if (amt === IDA_Manip.REVERSE()) {amt = -pos;}
		
		try
		{
			if (amt > 0)
			{
				for (var i=pos+amt-1; i>=pos; i-=1)
				{
					func(i);
				}
			}
			else if (amt < 0)
			{
				for (var i=pos-1; i>=pos+amt; i-=1)
				{
					func(i);
				}
			}
			
		}
		catch (opex) {throw opex instanceof IDAOpException ? new IDAFormatException(opNode, opex.message) : opex;}
	}
	static manip_rem_item(func, nodeFrom)
	{
		for (var rem of nodeFrom.asList())
		{
			func(rem);
		}
	}
	
// Manip end
}

// RX begin
class IDA_RX
{
	// Data source
	constructor(inStr)
	{
		this.refs = {};
		this.has_refTarget = false;
		
		this.inStr = inStr;
		this.inStr_leadIn = true;
		
		this.c1 = null;
		this.c2 = null;
		this.has_c1 = false;
		this.has_c2 = false;
		this.eof = false;
		this.ln = 1;
		this.lnBuf = "";
		
	}
	
	// RX basic input
	// JS - variables moved to constructor, see RX begin
	read()
	{
		if (!this.has_c1) this.more();
		return this.has_c1 ? this.read1() : null;
	}
	
	buf(c)
	{
		if (c == "\n") {this.ln+=1; this.lnBuf = "";} else {this.lnBuf += c;}
	}
	/** Assumes has_c1 */
	read1()
	{
		var output = this.c1; this.c1 = this.c2; this.has_c1 = this.has_c2; this.has_c2 = false;
		this.more2();
		this.buf(output);
		return output;
	}
	/** Assumes has_c2 */
	skip2()
	{
		if (this.has_c1) this.buf(this.c1);
		if (this.has_c2) this.buf(this.c2);
		this.has_c1 = this.has_c2 = false;
		if (this.more1()) return;
		this.more2();
	}
	/** Check has_c1 first */
	shift()
	{
		var output = this.c1;
		this.read();
		return output;
	}
	more()
	{
		if (this.has_c2 || this.eof) return;
		if (!this.has_c1 && this.more1()) return;
		if (!this.has_c2 && this.more2()) return;
	}
	/** Returns eof */
	more1()
	{
		if (this.eof) return true;
		var b = this.inStr();
		while ((b === null) && (this.inStr_leadIn)) {b = this.inStr(); this.inStr_leadIn = false;}
		if (b === null || b == 4) return this.eof = true;
		this.c1 = String.fromCharCode(b);
		this.has_c1 = true;
		return false;
	}
	/** Returns eof */
	more2()
	{
		if (this.eof) return true;
		var b = this.inStr();
		while ((b === null) && (this.inStr_leadIn)) {b = this.inStr(); this.inStr_leadIn = false;}
		if (b === null || b == 4) return this.eof = true;
		this.c2 = String.fromCharCode(b);
		this.has_c2 = true;
		return false;
	}
	
	// RX stream advancement
	comment()
	{
		if (!this.has_c1) return false;
		if (IDA.comment1(this.c1)) return true;
		return this.has_c2 && IDA.comment2(this.c1, this.c2);
	}
	adv(checkLn = false)
	{
		this.more();
		while (this.has_c1)
		{
			if (checkLn ? this.whiteTxt(this.c1) : this.white(this.c1)) {this.read1();}
			else if (this.c1 == "#") {this.read1(); this.advPast("\n");}
			else if (this.has_c2)
			{
				if (this.c1 == "/" && this.c2 == "/")
				{
					this.skip2(); this.advPast("\n");
				}
				else if (this.c1 == "/" && this.c2 == "*")
				{
					this.skip2(); this.advPast("*", "/");
				}
				else if (this.c1 == "<" && this.c2 == "?") // XML header compatibility
				{
					this.skip2(); this.advPast("?", ">");
				}
				else if (this.c1 == "<" && this.c2 == "!") // XML comment compatibility
				{
					this.skip2();
					do {this.advPast("-", "-");} while (this.has_c1 && !this.c1e(">")); this.read1();
				}
				else {return;}
			}
			else {return;}
		}
		
	}
	advPast1(c)
	{
		this.more();
		while (this.has_c1 && this.c1 != c) {this.read1();}
		this.read1();
	}
	advPast2(cx, cy)
	{
		this.more();
		while (this.has_c2 && (this.c1 != cx || this.c2 != cy)) {this.read1();}
		this.skip2();
	}
	advPast(c, c2=null)
	{
		if (c2 === null) {this.advPast1(c);} else {this.advPast2(c, c2);}
	}
	
	abrupt(checkLn)
	{
		this.adv(checkLn);
		return !this.has_c1;
	}
	
	// RX scan bytes
	c1e(c) {return this.has_c1 && (this.c1 == c);}
	c1n(c) {return this.has_c1 && (this.c1 != c);}
	c2e(c) {return this.has_c2 && (this.c2 == c);}
	c2n(c) {return this.has_c2 && (this.c2 != c);}
	whiteLn(c) {return c == "\n";}
	whiteTxt(c) {return (c <= " ") && (c != "\n");}
	white(c) {return c <= " ";}
	
	// RX scan numbers
	static num0(c) {return ((c>="0") && (c<="9")) || (c == "-") || (c == "+") || (c == ".");}
	static num1(c) {return IDA_RX.num0(c) || (c == "e") || (c == "E");}
	get_num()
	{
		var output;
		
		if (this.has_c2 && (this.c1=="0") && (this.c2=="x"))
		{
			output = "0x"; this.skip2();
			while (this.has_c1 && IDA_RX.hex(this.c1)) {output += this.shift();}
		}
		else if (this.has_c2 && (this.c1=="0") && (this.c2=="b"))
		{
			output = "0b"; this.skip2();
			while (this.has_c1 && IDA_RX.bin(this.c1)) {output += this.shift();}
		}
		else
		{
			output = "" + this.shift();
			while (this.has_c1 && IDA_RX.num1(this.c1)) {output += this.shift();}
		}
		
		return output;
	}
	static hex(c) {return ((c>="0") && (c<="9")) || ((c>="a") && (c<="f")) || ((c>="A") && (c<="F"));}
	static bin(c) {return (c=="0") || (c=="1");}
	
	// RX scan opts
	opt() {return this.c1e('~');}
	/** Consumes the /, reads the opt, then advances */
	get_opt()
	{
		this.read1();
		var output = this.get_id();
		this.adv();
		return output;
	}
	
	// RX scan unicode
	static is_u8_marker(c) {return (c.charCodeAt(0) & 0xc0) == 0xc0;}
	static is_u8_data(c) {return (c & 0xc0) == 0x80;}
	
	shift_u8_data()
	{
		var data = this.shift().charCodeAt(0);
		if (!IDA_RX.is_u8_data(data)) throw new IDAFormatException(this, "Invalid unicode data");
		return data & 0x3f;
	}
	get_u8()
	{
		var codePoint = 0;
		var marker = this.shift().charCodeAt(0);
		
		if ((marker & 0xe0) == 0xc0) // 2-byte
		{
			codePoint = (marker & 0x1f) << 6;
			codePoint |= this.shift_u8_data();
			if (codePoint < 0x0080) throw new IDAFormatException(this, "Overlong unicode character");
		}
		else if ((marker & 0xf0) == 0xe0) // 3-byte
		{
			codePoint = (marker & 0x0f) << 12;
			codePoint |= this.shift_u8_data() << 6;
			codePoint |= this.shift_u8_data();
			if (codePoint < 0x0800) throw new IDAFormatException(this, "Overlong unicode character");
		}
		else if ((marker & 0xf8) == 0xf0) // 4-byte
		{
			codePoint = (marker & 0x07) << 18;
			codePoint |= this.shift_u8_data() << 12;
			codePoint |= this.shift_u8_data() << 6;
			codePoint |= this.shift_u8_data();
			if (codePoint < 0x10000) throw new IDAFormatException(this, "Overlong unicode character");
		}
		else throw new IDAFormatException(this, "Invalid unicode marker");
		
		return this.to_u8(codePoint);
	}
	to_u8(codePoint)
	{
		if (codePoint >= 0xd800 && codePoint < 0xe000) throw new IDAFormatException(this, "Invalid unicode surrogate");
		try {return String.fromCodePoint(codePoint);}
		catch (ex) {throw new IDAFormatException(this, "Invalid unicode range");}
	}
	
	// RX scan escapes
	static is_esc(c) {return c == "\\";}
	esc() {return this.has_c1 && IDA_RX.is_esc(this.c1);}
	
	shift_hex(ex_txt)
	{
		var h = this.shift().charCodeAt(0);
		if (0x30 <= h && h <= 0x39) return h - 0x30;
		if (0x41 <= h && h <= 0x46) return 10 + h - 0x41;
		if (0x61 <= h && h <= 0x66) return 10 + h - 0x61;
		throw new IDAFormatException(this, "Invalid "+ex_txt+" escape sequence");
	}
	get_esc_byte(ex_txt)
	{
		return (this.shift_hex(ex_txt) << 4) | this.shift_hex(ex_txt);
	}
	get_esc()
	{
		this.read1();
		if (!this.has_c1) {throw new IDAFormatException(this, "Incomplete escape sequence");}
		var esc = this.shift();
		switch (esc)
		{
			case "0": return "\0";
			case "n": return "\n";
			case "r": return "\r";
			case "t": return "\t";
			case "x":
			{
				if (!this.has_c2) {throw new IDAFormatException(this, "Incomplete byte escape sequence");}
				return this.to_u8(this.get_esc_byte("byte"));
			}
			case "u":
			{
				if (!this.has_c2) {throw new IDAFormatException(this, "Incomplete unicode escape sequence");}
				var code = this.get_esc_byte("unicode") << 8;
				if (!this.has_c2) {throw new IDAFormatException(this, "Incomplete unicode escape sequence");}
				code |= this.get_esc_byte("unicode");
				return this.to_u8(code);
			}
			case "U":
			{
				if (!this.has_c2) {throw new IDAFormatException(this, "Incomplete unicode extended escape sequence");}
				var code = this.get_esc_byte("unicode extended") << 24;
				if (!this.has_c2) {throw new IDAFormatException(this, "Incomplete unicode extended escape sequence");}
				code |= this.get_esc_byte("unicode extended") << 16;
				if (!this.has_c2) {throw new IDAFormatException(this, "Incomplete unicode extended escape sequence");}
				code |= this.get_esc_byte("unicode extended") << 8;
				if (!this.has_c2) {throw new IDAFormatException(this, "Incomplete unicode extended escape sequence");}
				code |= this.get_esc_byte("unicode extended");
				return this.to_u8(code);
			}
			
			default: return esc;
		}
	}
	
	// RX scan IDs
	id0() {return IDA.id0(this.c1);}
	id1() {return IDA.id1(this.c1);}
	get_tok()
	{
		if (IDA_RX.is_str_d(this.c1)) {return this.get_str_d(this.c1);}
		else
		{
			var output = "";
			while (!this.comment())
			{
				if (this.esc()) {output += this.get_esc();}
				else if (this.has_c1 && !this.white(this.c1)) {output += this.shift();}
				else break;
			}
			return output;
		}
	}
	
	get_id(untilLn = false)
	{
		var output = "";
		this.id_symbolic = true;
		var holdMin = -1;
		var holdMax = -1;
		
		// Must not end with - or + , that is reserved for boolean false or true
		while (this.has_c1 && !this.comment())
		{
			var ptr = this.id_symbolic ? output.length : holdMin;
			var hold = false;
			
			if (hold = IDA_RX.is_esc(this.c1)) {output += this.get_esc();}
			else if (hold = IDA_RX.is_str_d(this.c1)) {output += this.get_str_d(this.c1);}
			else if (hold = IDA_RX.is_var(this.c1)) {output += this.get_var();}
			else if (hold = IDA_RX.is_u8_marker(this.c1)) {output += this.get_u8();}
			else if ((IDA.id1(this.c1) || (untilLn && (this.c1 != ";") && !this.whiteLn(this.c1))) && !((this.c1 == "-") && (!this.has_c2 || !IDA.id1(this.c2)))) {output += this.shift();}
			else break;
			
			if (hold)
			{
				holdMin = ptr;
				holdMax = output.length;
				this.id_symbolic = false;
			}
			
		}
		
		if (untilLn)
		{
			if (this.id_symbolic) {return IDA.port.trim(output);}
			else return IDA.port.ltrim(output.substring(0,holdMin)) + output.substring(holdMin, holdMax) + IDA.port.rtrim(output.substring(holdMax));
		}
		else return output;
	}
	static is_id_start(c) {return IDA_RX.is_str_d(c) || IDA_RX.is_var(c) || IDA_RX.is_esc(c) || IDA_RX.is_u8_marker(c) || IDA.id0(c);}
	c1_id_start() {return this.has_c1 && IDA_RX.is_id_start(this.c1);}
	
	static is_var(c) {return c == "$";}
	get_var()
	{
		this.read();
		var output = "$";
		
		var endChar = "\0";
		if (this.c1e("{")) {endChar = "}";}
		// TODO: Other common uses of $(...)
		
		if (endChar != "\0") output += this.read1() + this.get_to(endChar) + endChar;
		return output;
	}
	
	xml_att_key()
	{
		return IDA_RX.is_id_start(this.c1) ? this.get_id() : this.get_pr((c) => c != "=" && c != "/" && c != ">" && !this.white(c));
	}
	
	// RX scan strings
	c1_str() {return this.has_c1 && IDA_RX.is_str(this.c1);}
	c1_tok() {return this.has_c1 && !this.white(this.c1);}
	static is_str_d(c) {return c == "\"" || c == "`" || c == "'";}
	static is_str(c) {return IDA_RX.is_str_d(c) || IDA.id1(c);}
	
	get_str()
	{
		return IDA_RX.is_str_d(this.c1) ? this.get_str_d(this.c1) : this.get_id();
	}
	get_str_d(delim)
	{
		var output = "";
		if (this.c1e(delim))
		{
			this.read1();
			while (this.c1n(delim)) output += this.esc() ? this.get_esc() : (IDA_RX.is_u8_marker(this.c1) ? this.get_u8() : this.read1());
			this.read();
		}
		return output;
	}
	
	// RX scan logic
	get_pr(p)
	{
		var output = "";
		while (this.has_c1 && p(this.c1)) {output += this.shift();}
		return output;
	}
	get_prior_to(delim)
	{
		var output = this.get_pr((c) => (c != delim));
		return output;
	}
	get_to(delim)
	{
		var output = this.get_prior_to(delim);
		this.read();
		return output;
	}
	
	// RX class iteration
	// Empty iterator
	static CLASS_NONE()
	{
		if (typeof IDA_RX.CLASS_NONE_OBJ == "undefined") {IDA_RX.CLASS_NONE_OBJ = IDA.port.empty_i();}
		return IDA_RX.CLASS_NONE_OBJ;
	}
	// Infinite iterator of blank nodes
	static CLASS_ARRAY()
	{
		if (typeof IDA_RX.CLASS_ARRAY_OBJ == "undefined") {IDA_RX.CLASS_ARRAY_OBJ = IDA.port.repeat_i(new IDA());}
		return IDA_RX.CLASS_ARRAY_OBJ;
	}
	// Specific class node iterator
	static classOf(itemClassList)
	{
		return (itemClassList != null) ? IDA.port.list_i(itemClassList) : IDA_RX.CLASS_NONE();
	}
	
	// RX entry
	/** Returns the output argument */
	static inObj(output, inStr)
	{
		const rx = new IDA_RX(inStr);
		const globalScope = new IDA();
		var entry = false;
		var entryNode = (new IDA()).rxScope(globalScope).setIncludes(output.rxIncludePwd());
		entryNode.rxInputList(output.inputListDepth, output.inputListLength);
		entryNode.copyOptions(output);
		
		rx.adv();
		while (rx.opt()) switch (rx.get_opt())
		{
			case "": entry = true; break;
		}
		var node = IDA_RX.fill(entryNode, rx, "\0", "\0", null, IDA_RX.CLASS_NONE(), 0);
		
		while (rx.has_c1 && node != null)
		{
			var nextNode = (new IDA()).rxScope(globalScope).setIncludes(output.rxIncludePwd());
			nextNode.rxInputList(output.inputListDepth, output.inputListLength);
			nextNode.copyOptions(output);
			while (rx.opt()) switch (rx.get_opt())
			{
				case "":
				{
					if (!entry) {entry = true; entryNode = nextNode;}
					break;
				}
			}
			
			var last = IDA_RX.fill(nextNode, rx, "\0", "\0", node, IDA_RX.CLASS_NONE(), 0);
			node.linkNext(nextNode);
			node = last;
		}
		
		output.reset();
		rx.transfer_nodes(entryNode, output);
		output.walk((x) => x.rxScope(null), true, true);
		
		if (rx.has_refTarget) output.walk((x) => rx.put_ref(x), true, true);
		
		return output;
	}
	transfer_nodes(src, dst)
	{
		// Name
		dst.copyName(src);
		
		// Parameters
		if (src.hasParams())
		{
			dst.setParams();
			for (var p of src.getParamsOr()) dst.addParam(p);
			src.clearParams();
		}
		else dst.clearParams();
		
		// Content
		if (src.isList())
		{
			dst.putList();
			for (var i of src.asListOr()) dst.addItem(i);
			src.putNull();
		}
		else dst.copyContent(src);
		
		// Other
		dst.copyOptions(src);
		if (src.hasRefName()) {this.refs[src.getRefName()] = dst; dst.setRefName(src.getRefName());}
		
		dst.link(src.getPrevOr(), src.getNextOr());
	}
	put_ref(el)
	{
		var t = el.getRefTarget();
		if (t != null)
		{
			if (!(t in this.refs)) throw new IDAFormatException(el, "Identifier not defined");
			el.putRef(this.refs[t]);
		}
	}
	
	// RX fill
	/** Decodes element nodes read from the RX, using the supplied object as a container for the first such element.
	 * Adjacent elements will be created as needed and linked to the first, accessible using its getNext method.<br>
	 * @return The last consecutive element decoded, or null if no elements were present.
	 * @param node A container into which the first decoded element should be written.
	 * @param rx The serial input to be read.
	 * @param paren_l The character which caused <i>fill</i> to be called recursively, or zero to admit no such structure.
	 * @param paren_r The character which should mark the end of a recursive, structured call to <i>fill</i>.
	 * @param prev An element to be considered prior to the decoded element for the purpose of interpreting <i>as-above</i>
	 * markup, or null if no such element exists.
	 * @param listClass A class element to use for copying implicit element names, or an empty iterator (such as CLASS_NONE).
	 * @param depth The effective recursion level.
	 * @throws IOException If a error occurs while reading input bytes.
	 * @throws IDAFormatException If the parser encounters a syntax error, or if the markup contained an invalid
	 * <i>manipulation</i> which would throw {@link IDAOpException} if called programmatically. */
	static fill(node, rx, paren_l, paren_r, prev, listClass, depth)
	{
		
		// RX fill/state
		var jsonVal = false;
		var forceContent = false; // Used for : and = in an array context
		var needName = true;
		var prelimName = null;
		var nameSet = false;
		var needContent = true;
		var checkDelim = true; // For ; or , between nodes
		var itemClass = null;
		var output = null;
		var lcn = listClass.hasNext();
		
		// RX read
		while (rx.has_c1 && needContent)
		{
			
			// RX read/string implicit
			// This should be first
			// NOTE: By testing this prior to num(), names/strings beginning "e" will not be a number
			if (rx.c1_id_start())
			{
				output = node;
				
				if (!needName)
				{
					if (listClass == IDA_RX.CLASS_ARRAY() && !forceContent)
					{
						nameSet = true;
						needName = false;
						node.putStr(prelimName);
					}
					else
					{
						// String content
						// Potentially also true/false/null if JSON-style following a :
						const s = rx.get_id(depth == 0);
						
						if (jsonVal && rx.id_symbolic)
						{
							if ("true" == s) {node.putTrue();}
							else if ("false" == s) {node.putFalse();}
							else if ("null" == s) {node.putNull();}
							else {node.putStr(s);}
						}
						else {node.putStr(s);}
					}
					
					needContent = false;
				}
				else
				{
					prelimName = rx.get_id();
					if (rx.abrupt(paren_r == "\n")) {node.putStr(prelimName); prelimName = null; needContent = false;}
					else {needName = false;}
				}
			}
			
			// RX read/delim explicit
			else if (rx.c1e(";") || rx.c1e(","))
			{
				if (!((rx.c1 == ",") && (depth == 0)))
				{
					// Consume delimiter unless using CSV compatibility
					rx.read1();
					checkDelim = false;
				}
				
				node.putNull();
				needContent = false;
				output = node;
			}
			
			// RX read/delim implicit
			else if (rx.c1e("}") || rx.c1e("]") || rx.c1e(")"))
			{
				if (paren_l != ':')
				{
					if (rx.c1e("}")) {if (paren_l != "{") throw new IDAFormatException(rx, "List mismatch");}
					else if (rx.c1e("]")) {if (paren_l != "[") throw new IDAFormatException(rx, "Array mismatch");}
					else /*if (rx.c1e(")"))*/ {if (paren_l != "(") throw new IDAFormatException(rx, "Parameters mismatch");}
				}
				
				if (!nameSet)
				{
					if (!needName && lcn) {lcn = false; node.setName(listClass.next().getNameOr()); node.putStr(prelimName);}
					else {node.setName(prelimName);}
					nameSet = true;
				}
				else
				{
					node.putNull();
				}
				
				needContent = false;
				checkDelim = false;
			}
			
			// RX read/as-above
			else if (rx.c1e("^"))
			{
				if (prev == null) {throw new IDAFormatException(rx, "No previous element");}
				
				if (needName)
				{
					prelimName = prev.getNameOr();
					needName = false;
				}
				else
				{
					node.copyContent(prev);
					needContent = false;
				}
				
				if (rx.c2e("^")) {node.copyParams(prev); rx.skip2();} else {rx.read1();}
				output = node;
			}
			
			// RX read/identifier definition
			else if (rx.c1e('*'))
			{
				rx.read1();
				var key = (rx.c1_str()) ? rx.get_str() : "";
				rx.adv();
				
				if (key in rx.refs) throw new IDAFormatException(rx, "Identifier already defined");
				rx.refs[key] = node;
				node.setRefName(key);
				
				output = node;
			}
			
			// RX read/identifier reference
			else if (rx.c1e('&'))
			{
				rx.read1();
				var key = (rx.c1_str()) ? rx.get_str() : "";
				rx.adv();
				
				node.setRefTarget(key);
				rx.has_refTarget = true;
				
				output = node;
				needContent = false;
			}
			
			// RX read/XML
			else if (rx.c1e("<")) // XML compatibility
			{
				output = node;
				
				if (!needName)
				{
					if (listClass == IDA_RX.CLASS_ARRAY() && !forceContent)
					{
						nameSet = true;
						needName = false;
						node.putStr(prelimName);
						needContent = false;
					}
					else if (paren_r == "<") {needContent = false;}
					else throw new IDAFormatException(rx, "Cannot nest XML element header");
				}
				else
				{
					rx.read1();
					rx.adv();
					if (rx.has_c1 && rx.id1())
					{
						node.setName(rx.get_id());
						needName = false;
						nameSet = true;
					}
					else {throw new IDAFormatException(rx, "Invalid XML element name");}
					rx.adv();
					
					while (rx.has_c1 && needContent) // XML inners
					{
						if (rx.c1e(">")) // XML child elements
						{
							rx.read1();
							var l = [];
							var last = null;
							while (rx.has_c1 && needContent)
							{
								if (rx.c1e("<"))
								{
									if (rx.c2e("/")) // End node
									{
										rx.skip2(); rx.adv();
										var end = rx.get_to(">").trim();
										if (!node.isName(end)) {throw new IDAFormatException(rx, "Incorrect XML element footer");}
										needContent = false;
									}
									else if (rx.c2e("!")) // XML comment
									{
										rx.skip2();
										do {rx.advPast("-", "-");} while (rx.c1n(">")); rx.read1();
									}
									else // Child element
									{
										var el = node.rxNew();
										IDA_RX.fill(el, rx, "<", ">", last, IDA_RX.CLASS_NONE(), depth+1);
										last = el;
										l.push(el);
									}
								}
								else
								{
									if (node.opts.rxXmlReparseStrings)
									{
										rx.adv();
										
										var el = node.rxNew();
										
										last = IDA_RX.fill(el, rx, ">", "<", last, IDA_RX.CLASS_NONE(), 0);
										while (el != null && (node.inputListLength < 0 || node.inputListLength > l.length))
										{
											l.push(el);
											el = el.getNextOr();
										}
									}
									else
									{
										var content = rx.get_prior_to("<");
										
										if (content.trim().length > 0)
										{
											var el = (new IDA()).putStr(content);
											l.push(el);
										}
										
									}
								}
							}
							
							node.putList(l);
						}
						else if (rx.c1e("/") && rx.c2e(">")) // Empty XML node
						{
							rx.skip2();
							rx.adv();
							node.putNull();
							needContent = false;
						}
						else // XML attrib
						{
							var attKey = rx.xml_att_key();
							while (IDA_RX.is_id_start(rx.c1) && (rx.c1n("/") || rx.c2n(">"))) attKey += rx.xml_att_key();
							rx.adv();
							var attValue;
							
							if (rx.c1e("="))
							{
								rx.read1();
								attValue = rx.get_str();
								rx.adv();
							}
							else
							{
								attValue = null;
							}
							
							var att = (new IDA(attKey)).putStr(attValue);
							node.addParam(att);
							
						}
					}
				}
			}
			
			// RX read/params
			else if (rx.c1e("("))
			{
				if (node.hasParams())
				{
					needContent = false;
				}
				else
				{
					output = node;
					
					// Parameters also cause content delim
					forceContent = true;
					
					rx.read1();
					rx.adv();
					if (rx.c1e("^"))
					{
						if (prev == null) {throw new IDAFormatException(rx, "No previous element");}
						node.copyParams(prev);
						rx.read1();
						rx.adv();
					}
					else
					{
						node.setParams();
					}
					
					var last = null;
					var cItems = IDA_RX.classOf((itemClass != null) ? itemClass.getParamsOr() : null);
					while (rx.c1n(")"))
					{
						var param = node.rxNew();
						last = IDA_RX.fill(param, rx, "(", ")", last, cItems, depth+1);
						if (last != null) while (param != null)
						{
							// Snapshot the next before addParam breaks it
							var param_next = param.getNextOr();
							node.addParam(param);
							param = param_next;
						}
					}
					while (cItems.hasNext())
					{
						var param = cItems.next().copy().rxScope(node);
						node.addParam(param);
					}
					rx.read();
				}
			}
			
			// RX read/content-delim
			else if ((rx.c1e(":") && !needName) || rx.c1e("="))
			{
				if (rx.read1() == ":") {jsonVal = true;}
				needName = false;
				nameSet = true;
				node.setName(prelimName);
				forceContent = true;
				output = node;
			}
			
			// RX read/null
			else if (rx.c1e("?"))
			{
				output = node;
				
				if (needName)
				{
					rx.read1();
					prelimName = null;
					needName = false;
				}
				else
				{
					if (listClass == IDA_RX.CLASS_ARRAY() && !forceContent)
					{
						nameSet = true;
						needName = false;
						node.putStr(prelimName);
					}
					else
					{
						rx.read1();
						node.putNull();
					}
					needContent = false;
				}
			}
			
			// RX read/number
			else if (IDA_RX.num0(rx.c1))
			{
				if (!needName && listClass == IDA_RX.CLASS_ARRAY() && !forceContent)
				{
					nameSet = true;
					needName = false;
					node.putStr(prelimName);
				}
				else
				{
					var s = rx.get_num();
					var num = s;
					var unit = "";
					var den = "1";
					
					// Check for denominator
					if (rx.c1e("/"))
					{
						rx.read1();
						
						if (!rx.has_c1 || !IDA_RX.num0(rx.c1)) {throw new IDAFormatException(rx, "Missing denominator");}
						den = rx.get_num();
						s += "/"+den;
					}
					
					// Check for unit
					if (rx.has_c1 && rx.id0())
					{
						unit = rx.get_id();
						s += unit;
					}
					
					switch (s)
					{
						// Bools with +, -
						case "+": node.putTrue(); break;
						case "-": node.putFalse(); break;
						
						// Actual number
						default:
						{
							rx.testNumber(num, "numerator");
							if (den != "1") {rx.testNumber(den, "denominator");}
							node.putNum(num, den, unit);
						}
					}
				}
				needContent = false;
				output = node;
			}
			
			// RX read/list array
			else if (rx.c1e("["))
			{
				output = node;
				
				if (!needName && listClass == IDA_RX.CLASS_ARRAY() && !forceContent)
				{
					nameSet = true;
					needName = false;
					node.putStr(prelimName);
				}
				else
				{
					rx.read1();
					rx.adv();
					var listObj = [];
					
					if (node.inputListDepth != 0)
					{
						var last = null;
						while (rx.c1n("]"))
						{
							var el = node.rxNew();
							last = IDA_RX.fill(el, rx, "[", "]", last, IDA_RX.CLASS_ARRAY(), depth+1);
							if (last != null) while (el != null && (node.inputListLength < 0 || node.inputListLength > listObj.length))
							{
								listObj.push(el);
								el = el.getNextOr();
							}
						}
					}
					else IDA_RX.discard(rx, "[", "]");
					
					rx.read();
					node.putList(listObj);
				}
				needContent = false;
			}
			
			// RX read/list object
			else if (rx.c1e("{"))
			{
				output = node;
				
				if (!needName && listClass == IDA_RX.CLASS_ARRAY() && !forceContent)
				{
					nameSet = true;
					needName = false;
					node.putStr(prelimName);
				}
				else
				{
					rx.read1();
					rx.adv();
					var listObj = [];
					
					if (node.inputListDepth != 0)
					{
						var last = null;
						var cItems = IDA_RX.classOf((itemClass != null) ? itemClass.asListOr() : null);
						while (rx.c1n("}"))
						{
							var el = node.rxNew();
							last = IDA_RX.fill(el, rx, "{", "}", last, cItems, depth+1);
							if (last != null) while (el != null && (node.inputListLength < 0 || node.inputListLength > listObj.length))
							{
								listObj.push(el);
								el = el.getNextOr();
							}
						}
						while (cItems.hasNext() && (node.inputListLength < 0 || node.inputListLength > listObj.length))
						{
							var el = cItems.next().copy().rxScope(node);
							listObj.push(el);
						}
					}
					else IDA_RX.discard(rx, "{", "}");
					
					rx.read();
					node.putList(listObj);
				}
				needContent = false;
			}
			
			// RX read/class definition
			else if (rx.c1e(':') && rx.c2e(':'))
			{
				rx.skip2();
				var key = (rx.c1_str()) ? rx.get_str() : "";
				rx.adv();
				
				var el = node.rxNew();
				IDA_RX.fill(el, rx, ':', '\0', null, IDA_RX.CLASS_NONE(), depth+1);
				node.rxAddItemClass(key, el);
			}
			
			// RX read/class instance
			else if (rx.c1e(':'))
			{
				if (paren_l == ':')
				{
					needContent = false;
				}
				else
				{
					rx.read1();
					var key = (rx.c1_str()) ? rx.get_str() : "";
					rx.adv();
					
					itemClass = node.rxGetItemClass(key);
					if (itemClass == null) {throw new IDAFormatException(rx, "Item class \""+key+"\" not found");}
					prelimName = itemClass.getNameOr();
					output = node;
				}
			}
			
			// RX read/template definition
			else if (rx.c1e("!"))
			{
				if (!needName) {throw new IDAFormatException(rx, "Incomplete element prior to template definition");}
				rx.read1();
				var key = (rx.c1_str()) ? rx.get_str() : "";
				rx.adv();
				
				var itemTemplate = node.rxNew();
				IDA_RX.fill(itemTemplate, rx, "\0", "\0", null, IDA_RX.CLASS_NONE(), depth+1);
				if (!itemTemplate.isList()) {throw new IDAFormatException(rx, "Template definition must be a list");}
				node.rxAddTemplate(key, itemTemplate);
			}
			
			// RX read/template instance
			else if (rx.c1e("@"))
			{
				// Hand-over to file include for @? and @@ 
				var inc_opt = rx.c2e("?");
				if (inc_opt || rx.c2e("@"))
				{
					if (!needName) {throw new IDAFormatException(rx, "Incomplete element prior to include");}
					rx.skip2();
					if (!rx.c1_tok()) {throw new IDAFormatException(rx, "Missing include path");}
					
					var path = rx.get_tok();
					rx.adv();
					
					var abort = false;
					try
					{
						// Platform-specific check (NodeJS)
						var FS;
						var PATH;
						if ((typeof require != "function") || (typeof (FS = require("fs")) != "object") || (typeof (PATH = require("path")) != "object"))
						{
							throw new IDAFormatException(rx, "File system is unavailable");
						}
						
						var pwd = node.rxIncludePwd();
						if (abort = (pwd === null)) {throw new IDAOpException("Includes are disabled");}
						
						var chdir = false;
						if (path.startsWith("+")) {chdir = true; path = path.substring(1);}
						
						var inc = path;
						if (!PATH.isAbsolute(inc)) {inc = PATH.resolve(pwd, path);}
						
						if (abort = !FS.existsSync(inc)) {throw new IDAOpException("Include path does not exist");}
						if (abort = !FS.lstatSync(inc).isFile()) {throw new IDAOpException("Include is not a file");}
						try {FS.accessSync(inc, FS.constants.R_OK);}
						catch (e) {abort = true; throw new IDAOpException("Include file has no read access");}
						
						var template_adj = (new IDA()).rxScope(node);
						template_adj.setIncludes(chdir ? PATH.dirname(inc) : node.rxIncludePwd());
						template_adj.inFile(inc);
						
						var last = IDA_RX.manip(rx, node, template_adj.iterateNext());
						
						nameSet = true;
						checkDelim = false;
						needContent = false;
						output = last;
						
					}
					catch (opex)
					{
						if (opex instanceof IDAOpException)
						{
							// Hold off expending the RX line on an exception when the include was optional
							if (!inc_opt) throw new IDAFormatException(rx, opex.message);
							
							// Discard the manipulation node
							if (abort) {IDA_RX.fill(new IDA(), rx, "\0", "\0", null, IDA_RX.CLASS_NONE(), depth+1);}
						}
						else throw opex;
					}
					
				}
				else
				{
					if (!needName) {throw new IDAFormatException(rx, "Incomplete element prior to template instance");}
					rx.read1();
					var key = (rx.c1_str()) ? rx.get_str() : "";
					rx.adv();
					
					var itemTemplate = node.rxGetTemplate(key);
					if (itemTemplate == null) {throw new IDAFormatException(rx, "Template \""+key+"\" not found");}
					
					var last = IDA_RX.manip(rx, node, IDA.port.list_i(itemTemplate.asListOr()));
					
					if (last != null)
					{
						nameSet = true;
						checkDelim = false;
						needContent = false;
						output = last;
					}
					
					
				}
				
			}
			
			// RX read/invalid
			// This should be last
			else
			{
				var c_hex = rx.c1.charCodeAt(0).toString(16);
				if (c_hex.length < 2) {c_hex = "0"+c_hex;}
				throw new IDAFormatException(rx, "Invalid character 0x"+c_hex+" '"+rx.c1+"'");
			}
			
			// RX read delimiter
			if (paren_r != ">")
			{
				var csvLn = paren_r == "\n";
				rx.adv(csvLn);
				csvLn &= rx.c1e("\n");
				if (checkDelim && (rx.c1e(";") || rx.c1e(",") || csvLn) )
				{
					if (needContent && lcn)
					{
						lcn = false;
						node.setName(listClass.next().getNameOr());
						node.putStr(prelimName);
					}
					else if (needName && lcn)
					{
						lcn = false;
						node.setName(listClass.next().getNameOr());
					}
					else if ((rx.c1 == ",") && (depth == 0))
					{
						// CSV compatibility
						rx.read1();
						rx.adv(true);
						
						// Convert to list with new first element
						var first = new IDA();
						first.rxConfig(node);
						if (needContent) {first.putStr(prelimName);} else {node.copyTo(first);}
						
						// Complete the row
						var listObj = [first];
						var last = null;
						var cItems = IDA_RX.CLASS_ARRAY();
						while (rx.c1n("\n"))
						{
							var el = node.rxNew();
							last = IDA_RX.fill(el, rx, "\0", "\n", last, cItems, depth+1);
							if (last != null) while ((el != null) && (node.inputListLength < 0 || node.inputListLength > listObj.length))
							{
								listObj.push(el);
								el = el.getNextOr();
							}
						}
						
						node.putList(listObj);
						node.setName(null);
						
					}
					else if (csvLn)
					{
						// Last CSV item
						if (needContent) {node.putStr(prelimName);}
					}
					else if (!nameSet)
					{
						node.setName(prelimName);
					}
					
					nameSet = true;
					needContent = false;
					
					if (!csvLn) {rx.read1(); rx.adv();}
					
				}
			}
			
			// RX read END
		}
		
		// RX fill/name
		if (!nameSet)
		{
			if (needName && lcn) {lcn = false; node.setName(listClass.next().getNameOr());}
			else {node.setName(prelimName);}
		}
		
		// RX fill END
		if (lcn) listClass.next();
		
		node.rxClearTemplates();
		node.rxClearItemClasses();
		return output;
	}
	
// TODO: RX discard
	// RX number test
	testNumber(num, desc)
	{
		// No need to test hex/bin here; invalid characters will have delimited the number, potentially becoming the unit
		if (IDA.port.starts_with(num, "0x") || IDA.port.starts_with(num, "0b")) {return;}
		
		const test_num = /^[\+\-]?[0-9]*(?:\.[0-9]+)?([eE][\+\-]?[0-9]+)?$/;
		if (!test_num.test(num)) {throw new IDAFormatException(this, "Invalid "+desc+" number");}
	}
	
	// RX manip
	/** Return the last node */
	static manip(rx, node, itemTemplate)
	{
		// Manipulations
		var manip = new IDA();
		IDA_RX.fill(manip, rx, "\0", "\0", null, IDA_RX.CLASS_NONE(), 1);
		if (manip.isList())
		{
			// TODO: any pre-process or validation of manipulations, or simplify logic if none needed
		}
		else if (manip.hasContent()) {throw new IDAFormatException(rx, "Template manipulation must be a list");}
		
		return IDA_Manip.manip_iterate(manip, node, itemTemplate);
	}
	
// RX end
}

// Output interface
class IDA_Output
{
	constructor(out_func) // Function which accepts a complete output string
	{
		this.w = out_func;
		this.queueBuf = "";
	}
	
	send(s) {this.w(s);}
	
	queue(s) {this.queueBuf += s;}
	length() {return this.queueBuf.length;}
	reject() {this.queueBuf = "";}
	accept() {this.send(this.queueBuf); this.queueBuf = "";}
	
	
	/** Element will not be null<br>
	 * This will receive the entry element, which is not necessarily the first to be output<br>
	 * The adjacents-flag is provided if any special state is needed to handle multiple elements to follow */
	ida_header(el, adj) {}
	/** Element will not be null<br>
	 * This is called potentially multiple times if adjacents are enabled */
	ida_element(el, adj) {}
	/** Called once after all elements are done */
	ida_footer() {}
	
	idaWrite(el, adj)
	{
		if (el == null) {return;}
		
		el.txRefLabel(adj);
		
		this.ida_header(el, adj);
		
		if (adj)
		{
			// Rewind first
			while (el.hasPrev()) {el = el.getPrevOr();}
			
			// Output forward
			while (el != null)
			{
				this.ida_element(el, true);
				el = el.getNextOr();
			}
		}
		else {this.ida_element(el, false);}
		
		this.ida_footer();
		
	}
	
	/** Same as idaWrite, but converts exceptions into a return value instead.<br>
	 * True if success, false if a IO problem occurred */
	idaPrint(el, adj)
	{
		try {this.idaWrite(el, adj); return true;}
		catch (ex) {return false;}
	}
	
}

// TX begin
class IDA_TX extends IDA_Output
{
	
	// TX defaults
	static get BREVITY_ALL() {return 3;}
	static get BREVITY_PART() {return 6;}
	static get ONE_LINE_MAX() {return 132;}
	
	// TX instance
	
	setOptsOverride(ov) {this.optsOverride = ov; return this;}
	opts(el) {return this.optsOverride ? IDAOptions.defaultOptions : el.opts;}
	
	constructor(out_func)
	{
		super(out_func);
		this.entry = null;
		this.rootAsAboveAll = false;
		this.rootAsAboveFirst = null;
		this.rootAsAbovePart = false;
		this.rootAsAboveName = null;
		
		this.optsOverride = false;
	}
	
	// TX interface
	ida_header(el, adj)
	{
		this.entry = el;
	}
	ida_element(el, adj)
	{
		// Explicit entry point?
		if (adj && this.entry == el && el.hasPrev()) {this.send("~ ");}
		
		// As-above all?
		if (this.rootAsAboveFirst == null)
		{
			this.rootAsAboveFirst = el;
			this.rootAsAboveAll = IDA_TX.as_above_all_test(el);
		}
		
		// Main output wrapped with name as-above part
		if (this.rootAsAbovePart && !el.isName(this.rootAsAboveName)) {this.rootAsAbovePart = false;}
		this.ida(el, "", false, this.rootAsAbovePart || (adj && this.rootAsAboveAll && el.hasPrev()), false, null);
		this.rootAsAboveName = el.getNameOr();
		if (!this.rootAsAbovePart) this.rootAsAbovePart = IDA_TX.as_above_part_test(el);
	}
	
	// TX structure filter
	has_inner_structure(first)
	{
		for (var outer = first; outer != null; outer = outer.getNextOr())
		{
			if (outer.isList()) for (var inner of outer.asListOr())
			{
				if (inner.hasParams() || inner.isList()) return true;
			}
		}
		return false;
	}
	
	// TX main begin
	ida(node, indent, parameters, nameAsAbove, nameOmit, itemClass)
	{
		var hasOutput = false;
		
		// TX main/indent
		this.send(indent);
		if (node.hasRefName()) this.send("*"+node.getRefName()+" ");
		if (itemClass != null) this.send(": ");
		
		// TX main/name
		if (!nameOmit)
		{
			var name = "";
			if (node.hasName()) {name = nameAsAbove ? "^" : this.id_str(node, node.getNameOr());}
			else if (!node.hasParams() && !node.isList()) {name = "?";}
			
			this.send(name);
			hasOutput = name != "";
		}
		
		// TX main/parameters
		var chain = false;
		if (node.hasParams())
		{
			if (hasOutput) {this.send(" ");}
			hasOutput = true;
			this.send("(");
			
			var pNames = itemClass != null ? itemClass.getParamFirstOr() : null;
			for (var param = node.getParamFirstOr(); param != null; param = param.getNextOr())
			{
				var pNameOmit = pNames != null && pNames.eqName(param);
				if (pNames != null) {pNames = pNames.getNextOr();}
				if (chain && !pNameOmit) {this.send(" ");}
				chain = true;
				this.ida(param, "", true, false, pNameOmit, null);
			}
			this.send(")");
		}
		
		// TX main/content
		if (node.isNull() || IDA_TX.broken_ref(node))
		{
			this.send(";");
			if (!parameters) {this.send("\n");}
		}
		
		// TX main/ref
		else if (node.isRef())
		{
			this.send(" &");
			this.send(node.asRefOr().getRefName());
			this.send(parameters ? (node.hasNext() ? "," : "") : "\n");
		}
		
		// TX main/content/bool
		else if (node.isBool())
		{
			if (nameOmit) this.send(" ");
			this.send(node.isTrue() ? "+" : "-");
			if (!parameters) {this.send("\n");}
		}
		
		// TX main/content/number
		else if (node.isNum())
		{
			if (hasOutput || node.hasPrev()) {this.send(" ");}
			this.send(node.toString());
			this.send(parameters ? (node.hasNext() ? "," : "") : "\n");
		}
		
		// TX main/content/list
		else if (node.isList())
		{
			if (node.numItems() == 0)
			{
				if (hasOutput) {this.send(" ");}
				this.send("{}\n");
			}
			else if (node.numItems() == 1 && !node.getItemFirstOr().isList())
			{
				if (hasOutput) {this.send(" ");}
				var item = node.getItemFirstOr();
				if (item.hasName() || item.hasParams())
				{
					this.send("{ ");
					this.ida(item, "", true, false, false, null);
					this.send(" }\n");
				}
				else
				{
					this.send("[ ");
					this.send(this.value(item, true));
					this.send(" ]\n");
				}
			}
			else
			{
				var oneLine = IDA.port.si_allMatch(node.iterator(), (n) => !n.isList() && !n.hasParams());
				
				if (oneLine)
				{
					if (hasOutput) {this.queue(" ");}
					if (IDA.port.si_anyMatch(node.iterator(), (n) => n.hasName()))
					{
						var pNames = (itemClass != null && itemClass.isList()) ? IDA.port.i_map(itemClass.iterator(), (n)=>n.getNameOr()) : IDA.port.empty_i();
						
						var asAbove = IDA_TX.as_above_all_test(node.getItemFirstOr());
						
						var i = node.iterator();
						this.queue("{");
						{
							var n = i.next();
							var pNameOmit = pNames.hasNext() && (pNames.next() == n.getNameOr());
							if (n.hasRefName()) this.queue(" *"+n.getRefName());
							if (!pNameOmit) this.queue(" "+this.id_str(n, n.getNameOr()));
							this.queue(this.value(n, false));
						}
						while (i.hasNext())
						{
							var n = i.next();
							var pNameOmit = pNames.hasNext() && (pNames.next() == n.getNameOr());
							if (n.hasPrev() && !n.getPrevOr().isBool() && !n.getPrevOr().isNull()) this.queue(",");
							if (n.hasRefName()) this.queue(" *"+n.getRefName());
							if (!pNameOmit) this.queue(" "+(asAbove ? "^" : this.id_str(n, n.getNameOr())));
							this.queue(this.value(n, false));
						}
						this.queue(" }\n");
					}
					else
					{
						var i = node.iterator();
						this.queue("[");
						while (i.hasNext()) {this.queue(" " + this.value(i.next(), true).trim());}
						this.queue(" ]\n");
					}
					
					if (this.length() < 2*IDA_TX.ONE_LINE_MAX) {this.accept();}
					else {this.reject(); oneLine = false;}
					
				}
				
				if (!oneLine)
				{
					// For now, only use name-as-above when all nodes have the same name,
					// and are not lists above size 1
					
					if (hasOutput) {this.send("\n"); this.send(indent);}
					this.send("{\n");
					var indent2 = indent + "\t";
					
					var first = node.getItemFirstOr();
					
					// Check to see if a class could be used for parameters
					var pClass = null;
					var firstName = first.getNameOr();
					var cName = false;
					if (!parameters && node.numItems() > 2)
					{
						cName = true;
						
						// Parameters
						var firstParams = first.getParamsOr();
						var ptrParams = firstParams != null ? firstParams.length : 0;
						
						// Check names separately
						for (var n = first.getNextOr(); n != null; n = n.getNextOr())
						{
							if (!n.isName(firstName)) {cName = false; break;}
						}
						
						// Check parameters
						for (var n = first.getNextOr(); n != null && ptrParams > 0; n = n.getNextOr())
						{
							var nps = n.getParamsOr();
							if (nps == null) {ptrParams = 0; break;}
							
							var i = 0;
							for (var np of nps)
							{
								if (i >= ptrParams) {break;}
								if (!np.isName(firstParams[i].getNameOr())) {break;}
								i+=1;
							}
							
							ptrParams = Math.min(ptrParams, i);
						}
						
						
						// Items (all nodes must be lists)
						var firstItems = first.asListOr();
						var ptrItems = 0;
						var namedItem = false;
						
						// Emergency measure - avoid classing the items if there are ANY parameters or lists inside ANY nodes
						if (firstItems != null && !this.has_inner_structure(first))
						{
							ptrItems = firstItems.length;
							for (var n = first.getNextOr(); n != null && ptrItems > 0; n = n.getNextOr())
							{
								var nis = n.asListOr();
								if (nis == null) {firstItems = null; ptrItems = 0; break;}
								
								var i = 0;
								for (var ni of nis)
								{
									if (i >= ptrItems) {break;}
									var name = firstItems[i].getNameOr();
									if (!ni.isName(name)) {break;}
									if (name != null) namedItem = true;
									i+=1;
								}
								
								ptrItems = Math.min(ptrItems, i);
							}
						}
						
						// Combine
						if (ptrParams > 0 || (namedItem && ptrItems > 0))
						{
							// Class!
							this.send(indent2);
							this.send(":: ");
							if (cName) {this.send(this.id_str(first, firstName));}
							
							pClass = new IDA(firstName);
							
							if (ptrParams > 0)
							{
								pClass.setParams();
								
								this.send("(");
								var np = firstParams[0];
								this.send(this.id_str(np, np.getNameOr()));
								pClass.addParam(new IDA(np.getNameOr()));
								while (np.hasNext() && (--ptrParams > 0))
								{
									np = np.getNextOr();
									this.send(", " + this.id_str(np, np.getNameOr()));
									pClass.addParam(new IDA(np.getNameOr()));
								}
								this.send(")");
							}
							
							if (ptrItems > 0)
							{
								pClass.putList();
								
								this.send("{");
								var ni = firstItems[0];
								this.send(this.id_str(ni, ni.getNameOr()));
								pClass.addItem(new IDA(ni.getNameOr()));
								while (ni.hasNext() && (--ptrItems > 0))
								{
									ni = ni.getNextOr();
									this.send(", " + this.id_str(ni, ni.getNameOr()));
									pClass.addItem(new IDA(ni.getNameOr()));
								}
								this.send("}\n");
							}
							else {this.send(";\n");}
							
						}
						else {cName = false;}
					}
					
					var item = first;
					var asAbove = IDA_TX.as_above_all_test(first);
					
					if (asAbove || cName)
					{
						this.ida(item, indent2, false, false, cName, pClass);
						while ((item = item.getNextOr()) != null)
						{
							this.ida(item, indent2, false, true, cName, pClass);
						}
					}
					else
					{
						var asAbovePart = false;
						var asAboveName = null;
						
						for (; item != null; item = item.getNextOr())
						{
							if (asAbovePart && !item.isName(asAboveName)) {asAbovePart = false;}
							this.ida(item, indent2, false, asAbovePart, false, pClass);
							
							asAboveName = item.getNameOr();
							if (!asAbovePart) asAbovePart = IDA_TX.as_above_part_test(item);
							
						}
					}
					
					
					
					this.send(indent);
					this.send("}\n");
					
				}
				
			}
			
		}
		
		// TX main/content/string
		else
		{
			if (hasOutput || node.hasPrev()) {this.send(" ");}
			this.send(this.id_str(node, node.toString()));
			this.send(parameters ? (node.hasNext() ? "," : "") : "\n");
			
		}
		
	// TX main end
	}
	
	// TX utility
	static broken_ref(el)
	{
		return el.isRef() && !el.asRefOr().hasRefName();
	}
	
	static as_above_all_test(firstItem)
	{
		const itemName = firstItem.getNameOr();
		
		if (itemName === null) return false;
		if (itemName.length < IDA_TX.BREVITY_ALL) return false;
		
		for (var c = firstItem; c != null; c = c.getNextOr())
		{
			if (c.numItems() >= 2) return false;
			if (!c.isName(itemName)) return false;
		}
		return true;
	}
	static as_above_part_test(item)
	{
		var asAboveName = item.getNameOr();
		if ((asAboveName === null) || (asAboveName.length < IDA_TX.BREVITY_ALL)) {return false;}
		var req = IDA_TX.BREVITY_PART;
		for (var n = item.getNextOr(); (--req > 0) && (n != null); n = n.getNextOr())
		{
			if (!n.isName(asAboveName)) {return false;}
		}
		return req <= 0;
	}
	
	value(node, inArray)
	{
		if (node.isNull() || IDA_TX.broken_ref(node)) {return inArray ? "?" : ";";}
		if (node.isBool()) {return node.isTrue() ? "+" : "-";}
		if (node.isRef()) {return (inArray ? "&" : " &") + node.asRefOr().getRefName();}
		return (inArray ? "" : " ") + (node.isNum() ? node.toString() : this.id_str(node, node.toString()));
	}
	
	static replace_ux(s)
	{
		const l = s.length;
		var ptr = 0;
		var output = "";
		for (var i=0; i<l; i+=1)
		{
			var c = s.charAt(i);
			if (c < "\x20" || (c >= "\x7f" && c < "\u0100"))
			{
				output += s.substring(ptr, i);
				output += "\\x";
				output += IDA.port.char_2(c);
				ptr = i+1;
			}
			else if (c >= "\u0100" && c <= "\u9999")
			{
				output += s.substring(ptr, i);
				output += "\\u";
				output += IDA.port.char_4(c);
				ptr = i+1;
			}
		}
		return output + s.substring(ptr);
	}
	id_str(refNode, s)
	{
		if (s == null) {return "?";}
		else if (s.length < 1) {return "\"\"";}
		else if (IDA_TX.id(s)) {return s;}
		else
		{
			var quoted = IDA.port.ends_with(s, "-");
			var buf = "";
			
			var max = s.length;
			for (var i=0; i<max; i+=1)
			{
				var ch = s[i];
				var c = ch.charCodeAt(0);
				if (c < 0x20)
				{
					if (c == 0) {buf += "\\0";}
					else if (c == 0x0a) {buf += "\\n";}
					else if (c == 0x0d) {buf += "\\r";}
					else if (c == 0x09) {buf += "\\t";}
					else {buf += "\\x" + IDA.port.char_2(c);}
				}
				else if (c < 0x7f)
				{
					if (c == 0x5c) {buf += "\\\\";}
					else if (c == 0x22) {buf += "\\\""; quoted = true;}
					else
					{
						if (!quoted)
						{
							if (i<1 ? !IDA.id0(ch) : !IDA.id1(ch)) {quoted = true;}
							else if (i>0 && IDA.comment2(s[i-1], ch)) {quoted = true;}
						}
						buf += ch;
					}
				}
				else if (c < 0x100)
				{
					if (c < 0xa0 || !this.opts(refNode).txUnicode) {buf += "\\x" + IDA.port.char_2(c);}
					else {buf += ch;}
				}
				else if (0xd800 <= c && c < 0xdc00)
				{
					var code = s.codePointAt(i);
					i+=1;
					if (this.opts(refNode).txUnicode)
					{
						buf += String.fromCodePoint(code);
					}
					else
					{
						buf += "\\U";
						buf += IDA.port.char_8(code);
					}
				}
				else if (c >= 0x100 && c < 0x10000)
				{
					if (this.opts(refNode).txUnicode)
					{
						buf += ch;
					}
					else
					{
						buf += "\\u";
						buf += IDA.port.char_4(c);
					}
				}
				
			}
			
			return quoted ? "\"" + buf + "\"" : buf;
		}
		
	}
	static id(s)
	{
		if (s.length < 1 || !IDA.id0(s[0]) || s[s.length-1] == '-') {return false;}
		var c1 = s[0];
		for (var i=1; i<s.length; i+=1)
		{
			var c2 = s[i];
			if (!IDA.id1(c2)) return false;
			if (IDA.comment2(c1, c2)) return false;
			c1 = c2;
		}
		return true;
	}
	
// TX end
}

// CLI begin
class IDA_CLI
{
	
	// CLI
	static cli_err(msg, exitCode = 1)
	{
		process.stderr.write("Error: " + msg + "\n");
		return exitCode;
	}
	static cli_out(ln) {process.stdout.write(ln + "\n");}
	static cli_help()
	{
		//          ".........1.........2.........3.........4.........5.........6.........7.........8"
		IDA.cli_out("IDA markup parser");
		IDA.cli_out("Read markup from standard input and write IDA to standard output.");
		IDA.cli_out("");
		IDA.cli_out("Parser options:");
		IDA.cli_out(" -h, --help         Show usage");
		IDA.cli_out(" -i in-option       Set IDA input option");
		IDA.cli_out(" -o out-option      Set IDA output option");
		IDA.cli_out("");
		IDA.cli_out("IDA input options:");
		IDA.cli_out(" xml-reparse        Parse XML string content as additional inline markup");
		IDA.cli_out("");
		IDA.cli_out("IDA output options:");
		IDA.cli_out(" unicode            Use UTF-8 for string characters beyond U+0080");
		IDA.cli_out("");
		return 0;
	}
	static cli(argc, argv)
	{
		return new Promise((resolve, reject) =>
		{
			var rootEl = new IDA();
			var opts = rootEl.opts;
			
			var i=0;
			while (i<argc)
			{
				var arg = argv[i++];
				
				if (arg == "-h" || arg == "--help")
				{
					return IDA.cli_help();
				}
				else if (arg == "-i")
				{
					if (i >= argc) return IDA.cli_err("No input option specified");
					arg = argv[i++];
					if (arg == "xml-reparse") {opts.rxXmlReparseStrings = true;}
					else return IDA.cli_err("Unknown input option: " + arg);
				}
				else if (arg == "-o")
				{
					if (i >= argc) return IDA.cli_err("No output option specified");
					arg = argv[i++];
					if (arg == "unicode") {opts.txUnicode = true;}
					else return IDA.cli_err("Unknown output option: " + arg);
				}
			}
			
			const tx = str => {process.stdout.write(str);};
			const onComplete = (n) => {n.outPrint(tx, true); resolve(0);};
			const onError = (ex) => {resolve(IDA.cli_err("Parse failed: " + ex.message));}
			rootEl.inStream(process.stdin).then(onComplete).catch(onError);
		});
	}
	
// CLI end
}


/** One data article element. */
class IDA
{
	
	// File includes
	// File includes - Global PWD
	/** Enables file includes globally, as applied to articles created hereafter.<br>
	 * The provided working dir is used for resolving relative paths.<br>
	 * Set null to disable includes by default. */
	static setGlobalIncludes(pwd = "./") {IDA.gIncludePwd = pwd;}
	
	// File includes - this element
	/** Enables file includes for this article, as applied to input decoded hereafter.<br>
	 * The provided working dir is used for resolving relative paths.<br>
	 * Set null to disable includes. */
	setIncludes(pwd = "./") {this.includePwd = pwd; return this;}
	/** May be null, if includes are disabled for this article. */
	rxIncludePwd() {return this.includePwd;}
	
	// Input control
	// See Initialisation for inputListDepth, inputListLength
	rxInputList(depth, length)
	{
		this.inputListDepth = depth;
		this.inputListLength = length;
		return this;
	}
	
	// Options storage
	// See Initialisation for opts
	/** <p>Fetch the input/output options for this element.</p>
	 * @return Input/output options object. */
	getOpts() {return this.opts;}
	/** <p>Set the input/output options of this element to match those of the provided element.</p>
	 * @param src An element from which to copy the options.
	 * @return This element. */
	copyOptions(src)
	{
		this.opts.copyFrom(src.opts);
		return this;
	}
	
	rxConfig(parent)
	{
		this.rxInputList(parent.inputListDepth > 0 ? parent.inputListDepth - 1 : parent.inputListDepth, parent.inputListLength);
		this.copyOptions(parent);
		return this;
	}
	rxNew()
	{
		var el = (new IDA()).rxScope(this).setIncludes(this.rxIncludePwd());
		el.rxConfig(this);
		return el;
	}
	
	// Scope storage
	rxScope(scope) {this.scope = scope; return this;}
	
	// Item classes
	rxClearItemClasses()
	{
		this.itemClasses = null;
	}
	rxGetItemClass(key)
	{
		var c = (this.itemClasses != null) ? this.itemClasses[key] : null;
		return (c != null) ? c : ((this.scope != null) ? this.scope.rxGetItemClass(key) : null);
	}
	/** Actually adds to scope element for use by adjacent elements.<br>
	 * Returns false (and takes no action with the class element) if no scope existed */
	rxAddItemClass(key, itemClass)
	{
		if (this.scope == null) {return false;}
		if (this.scope.itemClasses == null) {this.scope.itemClasses = {};}
		this.scope.itemClasses[key] = itemClass;
		return true;
	}
	
	// Templates
	rxClearTemplates()
	{
		this.itemTemplates = null;
	}
	rxGetTemplate(key)
	{
		var c = (this.itemTemplates != null) ? this.itemTemplates[key] : null;
		return (c != null) ? c : ((this.scope != null) ? this.scope.rxGetTemplate(key) : null);
	}
	/** Actually adds to scope element for use by adjacent elements.<br>
	 * Returns false (and takes no action with the template element) if no scope existed */
	rxAddTemplate(key, itemTemplate)
	{
		if (this.scope == null) {return false;}
		if (this.scope.itemTemplates == null) {this.scope.itemTemplates = {};}
		this.scope.itemTemplates[key] = itemTemplate;
		return true;
	}
	
	// Adjacent nodes
	// next, prev
	/** <p>Test whether an element is adjacent to this element in the forward direction.</p>
	 * @return True if an element exists after this element, otherwise false */
	hasNext() {return this.next != null;}
	/** <p>Test whether an element is adjacent to this element in the reverse direction.</p>
	 * @return True if an element exists before this element, otherwise false */
	hasPrev() {return this.prev != null;}
	/** <p>Test whether no element is adjacent to this element in the reverse direction.</p>
	 * <p>This is the opposite of hasPrev, intended as a convenience for function references.</p>
	 * @return True if nothing exists this element, otherwise false */
	isFirst() {return !this.hasPrev();}
	/** <p>Test whether no element is adjacent to this element in the forward direction.</p>
	 * <p>This is the opposite of hasNext, intended as a convenience for function references.</p>
	 * @return True if nothing exists after this element, otherwise false */
	isLast() {return !this.hasNext();}
	
	/** <p>Fetch the adjacent element after this one.</p>
	 * <p>If this element is last, an exception is thrown.
	 * To avoid that possibility, use getNextOr to provide a default value.</p>
	 * @exception IDAOpException if this element is last.
	 * @return The element after. */
	getNext()
	{
		if (this.isLast()) throw new IDAOpException("Element is last");
		return this.next;
	}
	/** <p>Fetch the adjacent element after this one.</p>
	 * <p>If this element is last, the provided default value is returned.</p>
	 * @param other A default value to return if nothing follows this element.
	 * @return The element after if present, otherwise the default value. */
	getNextOr(other = null)
	{
		return this.hasNext() ? this.next : other;
	}
	/** <p>Fetch the adjacent element before this one.</p>
	 * <p>If this element is first, an exception is thrown.
	 * To avoid that possibility, use getPrevOr to provide a default value.</p>
	 * @exception IDAOpException if this element is first.
	 * @return The element before. */
	getPrev()
	{
		if (this.isFirst()) throw new IDAOpException("Element is first");
		return this.prev;
	}
	/** <p>Fetch the adjacent element before this one.</p>
	 * <p>If this element is first, the provided default value is returned.</p>
	 * @param other A default value to return if nothing precedes this element.
	 * @return The element before if present, otherwise the default value. */
	getPrevOr(other = null)
	{
		return this.hasPrev() ? this.prev : other;
	}
	
	/** This element, followed by everything adjacent after it */
	iterateNext()
	{
		const i = new IDA_MutableSeq();
		i.n = this;
		i.funcHas = () => (i.n != null);
		i.funcGet = () => {var x = i.n; i.n = x.getNextOr(); return x;};
		return i;
	}
	/** <p>Places the provided element adjacent to this one, next in sequence.<br>
	 * If the element is null, this one will become last in sequence.</p>
	 * <p>Any existing forward link will be undone before this operation.</p>
	 * @param next An element to follow this one, or null
	 * @return This element */
	linkNext(next)
	{
		if (this.next != null) {this.next.prev = null;}
		this.next = next; if (next != null) next.prev = this;
		return this;
	}
	/** <p>Places the provided element adjacent to this one, previous in sequence.<br>
	 * If the element is null, this one will become first in sequence.</p>
	 * <p>Any existing reverse link will be undone before this operation.</p>
	 * @param prev An element to precede this one, or null
	 * @return This element */
	linkPrev(prev)
	{
		if (this.prev != null) {this.prev.next = null;}
		this.prev = prev; if (prev != null) prev.next = this;
		return this;
	}
	/** <p>Places the provided elements adjacent to this one, on either side.<br>
	 * If an element is null, this one will become the first/last in sequence.</p>
	 * <p>Before this operation occurs, any existing links will be cross-connected to remove this element from the chain.</p>
	 * @param prev An element to precede this one, or null
	 * @param next An element to follow this one, or null
	 * @return This element */
	link(prev, next)
	{
		if (this.prev != null) {this.prev.next = this.next;}
		if (this.next != null) {this.next.prev = this.prev;}
		this.prev = prev; if (prev != null) prev.next = this;
		this.next = next; if (next != null) next.prev = this;
		return this;
	}
	
	// Node name
	/** <p>Test whether this element has no name.
	 * The name must be null, not merely the empty string.</p>
	 * <p>This is the opposite of hasName, intended as a convenience for function references.</p>  
	 * @return True if this element has no name, otherwise false */
	isAnon() {return this.name === null;}
	/** <p>Test whether this element has a name.
	 * The name can be any non-null string, including the empty string.</p>
	 * @return True if this element has a name, otherwise false */
	hasName() {return this.name !== null;}
	/** <p>Test whether this element has a specific name.
	 * Providing null will perform the same test as with isAnon.</p>
	 * @param name A name for comparison
	 * @return True if this element name matches the provided name, otherwise false */
	isName(name)
	{
		return this.getNameOr() === name;
	}
	
	/** <p>Fetch this element name string.</p>
	 * <p>If this element content is anonymous, an exception is thrown.
	 * To avoid that possibility, use getNameOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no name.
	 * @return Element name. */
	getName()
	{
		if (this.isAnon()) throw new IDAOpException("Element has no name");
		return this.name;
	}
	/** <p>Fetch this element name string.</p>
	 * <p>If this element content is anonymous, the provided default value is returned.</p>
	 * <p>On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
	 * If possible, always call getNameOr() rather than getNameOr(null).</p>
	 * @param other A default value to return if this element has no name.
	 * @return Element name if present, otherwise the default value. */
	getNameOr(other = null)
	{
		return this.hasName() ? this.name : other;
	}
	
	/** <p>Clear the name of this element, leaving it anonymous.</p>
	 * @return This element. */
	clearName() {return this.setName(null);}
	/** <p>Set the name of this element to be the provided string.</p>
	 * <p>If the provided string is null, this element will become anonymous as with calling clearName.</p>
	 * @param name Element name string.
	 * @return This element. */
	setName(name) {this.name = name; return this;}
	/** <p>Set the name of this element to match the name of the provided element.</p>
	 * @param src An element from which to copy the name */
	copyName(src) {this.name = src.name;}
	
	// Parameters
	// parameters
	/** <p>Delete this element's parameters list.</p>
	 * <p>If this element has no parameters list, calling clearParams has no effect.</p>
	 * @return This element. */
	clearParams() {this.parameters = null; return this;}
	/** <p>Set this element's parameters to be an empty list.</p>
	 * <p>Any existing parameter elements will be deleted.</p>
	 * @return This element. */
	setParams() {this.parameters = []; return this;}
	
	/** <p>Fetch this element's parameters list.</p>
	 * <p>If this element has no parameters list, an exception is thrown.
	 * To avoid that possibility, use getParamsOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no parameters list.
	 * @return List of parameters. */
	getParams()
	{
		if (!this.hasParams()) throw new IDAOpException("No parameters list");
		return this.parameters;
	}
	/** <p>Fetch this element's parameters list.</p>
	 * <p>If this element has no parameters list, the provided default value is returned.</p>
	 * <p>On platforms where lists/vectors are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed list, such that code will not be fully portable if null is provided.
	 * If possible, always call getParamsOr() rather than getParamsOr(null).</p>
	 * @param other A default value to return if this element has no parameters list.
	 * @return List of parameters if present, otherwise the default value. */
	getParamsOr(other = null)
	{
		return this.hasParams() ? this.parameters : other;
	}
	
	/** <p>Test whether this element contains a parameters list.</p>
	 * <p>This reports whether a parameters list object exists at all, as opposed to whether at least one parameter is present.
	 * The primary purpose is to ensure the list exists before calling getParams and/or iterating over its elements.</p>
	 * <p>Use numParams to test for an empty list (0) or at least one parameter present (greater than 0).</p>
	 * @return True if a parameters list exists for this element, otherwise false */
	hasParams()
	{
		return this.parameters != null;
	}
	/** <p>Fetch the number of parameters within this element.</p>
	 * <p>If no parameters list is present, the sentinel value -1 is returned.
	 * If 0 is returned, the list is present but empty.</p>
	 * @return The parameters list size if a list is present, otherwise -1 */
	numParams()
	{
		return this.parameters != null ? this.parameters.length : -1;
	}
	/** <p>Test whether this element contains a parameter with the provided name.
	 * If it does, an equivalent call to getParam will return an element.</p>
	 * <p>If false is returned, this can be either due to having no parameter of that name or due to having no parameters list.
	 * Use hasParams to separate those two cases.</p>
	 * @param key A parameter name to be checked
	 * @return True if a parameter exists with the provided name, otherwise false */
	hasParam(key)
	{
		return this.getParamOr(key) != null;
	}
	
	// Parameters/get
	/** <p>Fetch the parameter element at the provided index in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has too few parameters in the list, an exception is thrown.
	 * To avoid that possibility, use getParamAtOr to provide a default value.</p>
	 * @param index A parameter index to be recalled.
	 * @exception IDAOpException if this element has no such parameter.
	 * @return The indexed parameter element. */
	getParamAt(index)
	{
		if (this.numParams() <= index) throw new IDAOpException("Invalid parameter index");
		return this.parameters[index];
	}
	/** <p>Fetch the parameter element at the provided index in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has too few parameters in the list, the provided default value is returned.</p>
	 * @param index A parameter index to be recalled.
	 * @param other A default value to return if this element has no such parameter.
	 * @return The indexed parameter element if present, otherwise the default value. */
	getParamAtOr(index, other = null)
	{
		return this.numParams() > index ? this.parameters[index] : other;
	}
	
	/** <p>Fetch the first parameter element with the provided name in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has no parameter of that name, an exception is thrown.
	 * To avoid that possibility, use getParamOr to provide a default value.</p>
	 * @param key A parameter name to be recalled.
	 * @exception IDAOpException if this element has no such parameter.
	 * @return The named parameter element. */
	getParam(key)
	{
		var p = this.getParamOr(key);
		if (p == null) throw new IDAOpException("Parameter not found");
		return p;
	}
	/** <p>Fetch the first parameter element with the provided name in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has no parameter of that name, the provided default value is returned.</p>
	 * @param key A parameter name to be recalled.
	 * @param other A default value to return if this element has no such parameter.
	 * @return The named parameter element if present, otherwise the default value. */
	getParamOr(key, other=null)
	{
		var out = this.hasParams() ? this.parameters.find((x) => x.isName(key)) : other;
		return out === undefined ? other : out;
	}
	
	/** <p>Cast to a string value the first parameter element with the provided name in this element's parameters list.</p>
	 * <p>If this element has no parameters list, has no parameter of that name or can not be interpreted as a string value, an exception is thrown.
	 * To avoid that possibility, use getParamStrOr to provide a default value.</p>
	 * @param key A parameter name to be recalled.
	 * @exception IDAOpException if this element has no such parameter, or the parameter content cannot cast to a string value.
	 * @return String content interpretation of the named parameter element. */
	getParamStr(key)
	{
		return this.getParam(key).toStr();
	}
	/** <p>Cast to a string value the first parameter element with the provided name in this element's parameters list.</p>
	 * <p>If this element has no parameters list, has no parameter of that name or can not be interpreted as a string value, the provided default value is returned.</p>
	 * <p>On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
	 * If possible, always call getParamStrOr(key) rather than getParamStrOr(key, null).</p>
	 * @param key A parameter name to be recalled.
	 * @param other A default value to return if this element has no such parameter, or the parameter content cannot cast to a string value.
	 * @return String content interpretation of the named parameter element if possible, otherwise the default value. */
	getParamStrOr(key, other = null)
	{
		var param = this.getParamOr(key);
		return param != null ? param.toStrOr(other) : other;
	}
	
	/** <p>Fetch the first parameter element in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has an empty parameters list, an exception is thrown.
	 * To avoid that possibility, use getParamFirstOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no such parameter.
	 * @return The first parameter element. */
	getParamFirst()
	{
		if (this.numParams() < 1) throw new IDAOpException(this.hasParams() ? "Parameters list is empty" : "Element has no parameters list");
		return this.parameters[0];
	}
	/** <p>Fetch the first parameter element in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has an empty parameters list, the provided default value is returned.</p>
	 * @param other A default value to return if this element has no such parameter.
	 * @return The first parameter element if present, otherwise the default value. */
	getParamFirstOr(other = null)
	{
		return this.numParams() > 0 ? this.parameters[0] : other;
	}
	/** <p>Fetch the last parameter element in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has an empty parameters list, an exception is thrown.
	 * To avoid that possibility, use getParamLastOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no such parameter.
	 * @return The last parameter element. */
	getParamLast()
	{
		var n = this.numParams();
		if (n < 1) throw new IDAOpException(this.hasParams() ? "Parameters list is empty" : "Element has no parameters list");
		return this.parameters[n-1];
	}
	/** <p>Fetch the last parameter element in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has an empty parameters list, the provided default value is returned.</p>
	 * @param other A default value to return if this element has no such parameter.
	 * @return The last parameter element if present, otherwise the default value. */
	getParamLastOr(other = null)
	{
		var n = this.numParams();
		return n > 0 ? this.parameters[n-1] : other;
	}
	
	// Parameters/add
	/** Create and insert a new parameter at the end of the list, with the provided element name.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param name Name string for the created parameter element.
	 * @return The created parameter element. */
	newParam(name)
	{
		var el = new IDA(name);
		this.addParam(el);
		return el;
	}
	
	/** Insert a parameter at the end of the list.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param param The parameter element to be stored.
	 * @return This element. */
	addParam(param)
	{
		if (this.parameters == null) {this.setParams();}
		param.link(this.getParamLastOr(), null);
		this.parameters.push(param);
		return this;
	}
	/** Insert a parameter at the specified list index position.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param param The parameter element to be stored.
	 * @param pos The list index position at which the parameter should be inserted.
	 * @exception IDAOpException if the specified position is below zero or above the present list size.
	 * @return This element. */
	addParamAt(param, pos)
	{
		if (this.parameters == null) {this.setParams();}
		IDA.list_add(this.getParams(), param, pos, "param");
		return this;
	}
	/** Replace the first matching parameter name, or insert a new parameter at the end of the list.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param param The parameter element to be stored.
	 * @return This element. */
	setParam(param)
	{
		var n = this.numParams();
		
		try
		{
			for (var i=0; i<n; i+=1)
			{
				var p = this.getParamAt(i);
				if (p.eqName(param))
				{
					this.remParamAt(i);
					return this.addParamAt(param, i);
				}
			}
		}
		catch (ex) {throw new Error("Unexpected parameters state");}
		
		return this.addParam(param);
	}
	
	/** <p>Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.</p>
	 * <p>This is the same as calling setParamNull.</p>
	 * @param key The parameter element name.
	 * @return This element. */
	clearParam(key) {return this.setParamNull(key);}
	/** <p>Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.</p>
	 * <p>This is the same as calling clearParam.</p>
	 * @param key The parameter element name.
	 * @return This element. */
	setParamNull(key) {return this.setParam(new IDA(key).putNull());}
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content String value to be stored.
	 * @return This element. */
	setParamStr(key, content) {return this.setParam(new IDA(key).putStr(content));}
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Boolean value to be stored.
	 * @return This element. */
	setParamBool(key, content) {return this.setParam(new IDA(key).putBool(content));}
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Floating-point value to be stored.
	 * @return This element. */
	setParamDouble(key, content) {return this.setParam(new IDA(key).putDouble(content));}
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Floating-point value to be stored.
	 * @return This element. */
	setParamFloat(key, content) {return this.setParam(new IDA(key).putFloat(content));}
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Integer value to be stored.
	 * @return This element. */
	setParamLong(key, content) {return this.setParam(new IDA(key).putLong(content));}
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Integer value to be stored.
	 * @return This element. */
	setParamInt(key, content) {return this.setParam(new IDA(key).putInt(content));}
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Referenced element to be stored.
	 * @return This element. */
	setParamRef(key, content) {return this.setParam(new IDA(key).putRef(content));}
	
	// Parameters/remove
	/** Remove the parameter at the specified list index position.<br>
	 * If this element has no parameters list, an exception is thrown.<br>
	 * @param pos The list index position of a parameter which should be removed.
	 * @exception IDAOpException if the specified position is below zero or beyond the present list size, or this element has no parameters list. */
	remParamAt(pos)
	{
		if (!this.hasParams()) {throw new IDAOpException("No parameters from which to remove");}
		IDA.list_del(this.parameters, pos, "param");
	}
	/** Remove the provided parameter.<br>
	 * If the parameter is absent or this element has no parameters list, the call has no effect.<br>
	 * @param param The parameter element to be removed.
	 * @return True if a parameter was removed, otherwise false. */
	remParam(param)
	{
		if (this.hasParams() && IDA.list_rem(this.parameters, param)) {param.link(null, null); return true;}
		return false;
	}
	/** Remove the first parameter with the provided element name.<br>
	 * If no such parameter exists or this element has no parameters list, the call has no effect.<br>
	 * @param key Name string of the parameter element to remove.
	 * @return True if a parameter was removed, otherwise false. */
	unParam(key)
	{
		var p = this.getParamOr(key);
		if (p != null) {this.remParam(p); return true;}
		return false;
	}
	
	// Parameters/other
	/** <p>Set the parameters of this element to be clones of the parameters of the provided element.</p>
	 * @param src An element from which to copy the parameters */
	copyParams(src)
	{
		this.clearParams();
		if (src.hasParams())
		{
			this.setParams();
			for (var p of src.parameters) this.addParam(p.copy_inner());
		}
	}
	
	// Content
	// cRaw
	// cStr
	// cNum, cDen, cUnit
	// cList
	// cBool
	// cRef
	// cRefObj
	// cRefTarget
	// cRefName
	
	/** <p>Set the content of this element to match the content of the provided element.
	 * If the content in question is a list, its inner elements will be cloned.</p>
	 * @param src An element from which to copy the content */
	copyContent(src)
	{
		this.cRaw = src.cRaw;
		this.cStr = src.cStr;
		this.cNum = src.cNum;
		this.cDen = src.cDen;
		this.cUnit = src.cUnit;
		this.cBool = src.cBool;
		this.cRef = null;
		this.cRefObj = null;
		this.cRefTarget = src.cRefTarget;
		this.cRefName = src.cRefName;
		this.clearList();
		if (src.isList())
		{
			this.putList();
			for (var x of src.asList()) this.addItem(x.copy_inner());
		}
	}
	
	// Content/ref
	/** Returns this element */
	clearRef() {this.cRef = null; this.cRefObj = null; this.cRefTarget = null; return this;}
	/** <p>Test whether the content type of this element is a reference to another element.</p>
	 * @return True if this element content is a reference, otherwise false */
	isRef() {return this.cRef != null;}
	/** <p>Fetch this element content as a reference (to another element).</p>
	 * <p>If this element content is not a reference, an exception is thrown.
	 * To avoid that possibility, use asRefOr to provide a default value.</p>
	 * @exception IDAOpException if this element does not contain reference content.
	 * @return Reference content. */
	asRef()
	{
		return this.cRef;
	}
	/** <p>Fetch this element content as a reference (to another element).</p>
	 * <p>If this element content is not a reference, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain reference content.
	 * @return Reference content if present, otherwise the default value. */
	asRefOr(other = null)
	{
		return this.isRef() ? this.cRef : other;
	}
	/** Returns this element */
	putRef(ref) {this.putNull(); this.cRaw = "Ref"; this.cRef = ref; this.cRefObj = ref.repObj; this.cRefTarget = ref.cRefName; return this;}
	/** <p>Sets an object for which this element should reference the corresponding element as it is encoded.</p> 
	 * <p>During a restore/decode process, the rebuilt object can be retrieved using getRefObj or withRefObj.
	 * The IDAObj codec interface implementation should ensure that any required type information is captured.</p>
	 * @param refObj A target object which implements the IDAObj codec interface.
	 * @return This element. */
	setRefObj(refObj) {this.putNull(); this.cRaw = "Ref"; this.cRefObj = refObj; return this;}
	/** <p>During a restore/decode process, the object corresponding with the referenced element will be sent to the callback when it becomes available.
	 * If that object is already available, the callback will be resolved before this call returns.</p>
	 * <p>If the object is needed immediately, the article structure must be designed such that the referenced object element is decoded in advance of this call.
	 * Use getRefObj instead if that design should be enforced.</p>
	 * <p>To ensure consistency, a top-level call to objRestore will throw an exception if any unresolved callbacks remain after decoding is complete.</p>
	 * <p>Calling code should use article information to establish the correct object type.</p>
	 * @param callback A function which receives the decoded object associated with the referenced element.
	 * @exception IDAOpException if this element content is not a reference. */
	withRefObj(callback)
	{
		if (this.cRefObj != null) {callback(this.cRefObj);}
		else if (this.cRef == null) throw new IDAOpException("Element is not a reference");
		else {this.cRef.on_rep_obj(callback);}
	}
	/** <p>Fetches the object corresponding with the referenced element, after it has been encoded or decoded.
	 * During a restore/decode process, an exception is thrown if the object is not yet available.</p>
	 * <p>To retrieve the object this way during a restore/decode process, the article structure must be designed such that the referenced object element is decoded in advance of this call.
	 * Use withRefObj instead if that design is not the case.</p>
	 * <p>Calling code should use article information to establish the correct object type.</p>
	 * @exception IDAOpException if this element content is not a reference, or if no object is available immediately.
	 * @return The encoded or decoded object associated with the referenced element. */
	getRefObj()
	{
		if (this.cRefObj != null) return this.cRefObj;
		if (this.cRef == null) throw new IDAOpException("Element is not a reference");
		if (this.cRef.repObj != null) return this.cRef.repObj;
		throw new IDAOpException("Referenced object not available");
	}
	/** <p>Set a transient target for this element.</p>
	 * <p>The targets match the identifiers assigned via setRefName, and are used only for capturing and restoring element pointer information.
	 * The string content itself is not information, insofar as it need not survive a loop test.</p>
	 * @param refTarget A string to store as the transient target for this element. */
	setRefTarget(refTarget) {this.putNull(); this.cRaw = "Ref"; this.cRefTarget = refTarget;}
	/** <p>Fetch the transient target for this element.
	 * The returned value may be <i>stale</i>.</p>
	 * <p>The targets match the identifiers assigned via setRefName, and are used only for capturing and restoring element pointer information.
	 * The string content itself is not information, insofar as it need not survive a loop test.</p>
	 * @return The transient target string for this element if it has a reference target, otherwise null. */
	getRefTarget() {return this.cRefTarget;}
	/** <p>Test whether this element has been assigned a transient target for encoding/decoding an outbound reference.
	 * The returned value may be <i>stale</i>.</p>
	 * <p>If an article has been decoded from markup, any reference target elements will have fresh identifiers from the markup document.
	 * Encoding, copying or testing equal a top-level article element will generate fresh identifiers.</p>
	 * <p>Call txRefLabel at top-level to generate fresh identifiers manually.</p>
	 * @return True if this element has a transient target, otherwise false. */
	hasRefTarget() {return this.cRefTarget !== null;}
	/** <p>Set a transient identifier for this element.</p>
	 * <p>Identifiers are set automatically when encoding/decoding markup, testing equal, or calling txRefLabel directly.
	 * Encoding, copying or testing equal a top-level article element will generate fresh identifiers.</p>
	 * <p>These names are used only for capturing and restoring element pointer information.
	 * The string content itself is not information, insofar as it need not survive a loop test.</p>
	 * @param refName A string to store as the transient identifier for this element. */
	setRefName(refName) {this.cRefName = refName;}
	/** <p>Fetch the transient identifier for this element.
	 * The returned value may be <i>stale</i>.</p>
	 * <p>Identifiers are set automatically when encoding/decoding markup, testing equal, or calling txRefLabel directly.
	 * Encoding, copying or testing equal a top-level article element will generate fresh identifiers.</p>
	 * <p>These names are used only for capturing and restoring element pointer information.
	 * The string content itself is not information, insofar as it need not survive a loop test.</p>
	 * @return The transient identifier string for this element if it is a reference target, otherwise null. */
	getRefName() {return this.cRefName;}
	/** <p>Test whether this element has been assigned a transient identifier for encoding/decoding inbound references.
	 * The returned value may be <i>stale</i>.</p>
	 * <p>If an article has been decoded from markup, any reference target elements will have fresh identifiers from the markup document.
	 * Encoding, copying or testing equal a top-level article element will generate fresh identifiers.</p>
	 * <p>Call txRefLabel at top-level to generate fresh identifiers manually.</p>
	 * @return True if this element has a transient identifier, otherwise false. */
	hasRefName() {return this.cRefName !== null;}
	
	/** Returns the number of reference labels applied */
	txRefLabel(adj)
	{
		// Ref walk
		var rn = [];
		rn.rc = 0;
		this.walk((x) => IDA.ref_label_clear(x, rn), adj, adj);
		this.walk((x) => IDA.ref_label_count(x, rn), adj, adj);
		
		// Null any foreign references at this stage (see IDA.verify)
		for (var r of rn) r.setRefName(null);
		
		return rn.rc;
	}
	static ref_label_clear(el, rn)
	{
		el.setRefName(null);
		if (el.isRef())
		{
			var r = el.asRefOr();
			if (!IDA.port.list_contains(rn, r)) rn.push(r);
		}
	}
	static ref_label_count(el, rn)
	{
		if (IDA.port.list_contains(rn, el)) IDA.port.list_rem(rn, el);
		
		if (el.isRef())
		{
			var r = el.asRefOr();
			if (!r.hasRefName())
			{
				var n = rn.rc++;
				var n0 = n % 52;
				var id = "" + String.fromCharCode(n0 < 26 ? 0x61+n0 : 0x41+n0) + ~~(n/52);
				r.cRefName = id;
				el.cRefTarget = id;
			}
		}
	}
	
	rxRefResolve(adj)
	{
		var rs = {};
		this.walk((x) => IDA.ref_resolve_scan(x, rs), adj, adj);
		this.walk((x) => IDA.ref_resolve_put(x, rs), adj, adj);
	}
	static ref_resolve_scan(el, rs)
	{
		if (el.hasRefName()) rs[el.getRefName()] = el;
	}
	static ref_resolve_put(el, rs)
	{
		if (el.hasRefTarget()) el.putRef(rs[el.getRefTarget()]);
	}
	
	// Content/bool
	/** <p>Test whether the content type of this element is a boolean value.</p>
	 * @return True if this element has boolean content, otherwise false */
	isBool() {return this.cBool != null;}
	/** <p>Fetch this element content as a boolean value.</p>
	 * <p>If this element content is not a boolean value, an exception is thrown.
	 * To avoid that possibility, use asBoolOr to provide a default value.
	 * One may also use isTrue for false by default, or not-isFalse for true by default.</p>
	 * <p>Use toBool/toBoolOr for casting other content types to be a boolean value.</p>
	 * @exception IDAOpException if this element does not contain boolean content.
	 * @return Boolean content value. */
	asBool()
	{
		if (this.cBool == null) throw new IDAOpException("Content is not a boolean value");
		return this.cBool == true;
	}
	/** <p>Fetch this element content as a boolean value.</p>
	 * <p>If this element content is not a boolean value, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain boolean content.
	 * @return Boolean content value if present, otherwise the default value. */
	asBoolOr(other) {return this.cBool != null ? this.cBool == true : other;}
	/** <p>Test whether the content of this element is boolean-true.</p>
	 * <p>If false is returned, this can be either due to being boolean-false or not being a boolean value.
	 * Use isBool and asBool for methods with those effects.</p>
	 * <p>This is intended as a convenience for function references.</p>
	 * @return True if this element has boolean content with value true, otherwise false */
	isTrue() {return (this.cBool != null) && (this.cBool == true);}
	/** <p>Test whether the content of this element is boolean-false.</p>
	 * <p>If false is returned, this can be either due to being boolean-true or not being a boolean value.
	 * Use isBool and (not-)asBool for methods with those effects.</p>
	 * <p>This is intended as a convenience for function references.</p>
	 * @return True if this element has boolean content with value false, otherwise false */
	isFalse() {return (this.cBool != null) && (this.cBool != true);}
	/** <p>Place boolean content into this element with value true, replacing any existing content.</p>
	 * @return This element */
	putTrue() {this.putNull(); this.cRaw = "+"; this.cBool = true; return this;}
	/** <p>Place boolean content into this element with value false, replacing any existing content.</p>
	 * @return This element */
	putFalse() {this.putNull(); this.cRaw = "-"; this.cBool = false; return this;}
	/** <p>Place boolean content into this element with the provided value, replacing any existing content.</p>
	 * @param b Boolean content value.
	 * @return This element */
	putBool(b) {return b ? this.putTrue() : this.putFalse();}
	/** <p>Test whether this element content can be interpreted as a boolean value (see toBool).
	 * This will always be the case for natural boolean content (see isBool).</p>
	 * <p>All numeric content can be cast to boolean, which is true if non-zero.
	 * String content can be "true" or "false".</p>
	 * @return True if this element content can be cast to a boolean value, otherwise false */
	canBool()
	{
		if (this.isBool() || this.isNum()) return true;
		var s = this.toStrOr("");
		return s=="true" || s=="false";
	}
	/** <p>Cast this element content to a boolean value.</p>
	 * <p>If this element content can not be interpreted as a boolean value, an exception is thrown.
	 * To avoid that possibility, use toBoolOr to provide a default value.</p>
	 * @exception IDAOpException if this element content cannot cast to a boolean value.
	 * @return Boolean content interpretation. */
	toBool()
	{
		if (this.isBool()) return this.isTrue();
		if (this.isNum()) return this.asDoubleOr(0.0) != 0.0;
		var s = this.toStrOr("");
		if (s=="true") return true;
		if (s=="false") return false;
		throw new IDAOpException("Content cannot cast to a boolean value");
	}
	/** <p>Cast this element content to a boolean value.</p>
	 * <p>If this element content can not be interpreted as a boolean value, the provided default value is returned.</p>
	 * @param other A default value to return if this element content cannot cast to a boolean value.
	 * @return Boolean content interpretation if able, otherwise the default value. */
	toBoolOr(other)
	{
		try {return this.toBool();}
		catch (opex) {return other;}
	}
	
	// Content/null
	/** <p>Test whether this element contains no content.</p>
	 * <p>This is the opposite of hasContent, intended as a convenience for function references.</p>
	 * @return True if this element has no content, otherwise false. */
	isNull() {return this.cRaw == null;}
	/** <p>Test whether this element contains some type of content.</p>
	 * <p>This is the opposite of isNull, intended as a convenience for function references.</p>
	 * @return True if this element has content present, otherwise false. */
	hasContent() {return this.cRaw != null;}
	/** <p>Erase any existing content from this element.</p>
	 * <p>This is the same as calling clearContent.</p>
	 * @return This element. */
	putNull() {return this.clearContent();}
	/** <p>Erase any existing content from this element.</p>
	 * <p>This is the same as calling putNull.</p>
	 * @return This element. */
	clearContent()
	{
		this.cRaw = null;
		this.cStr = null;
		this.cBool = null;
		this.clearRef();
		this.clearList();
		this.clearNum();
		return this;
	}
	
	// Content/string
	/** <p>Place string content into this element with the provided value, replacing any existing content.</p>
	 * @param s String content value. 
	 * @return This element */
	putStr(s) {this.putNull(); this.cRaw = s; this.cStr = s; return this;}
	/** <p>Test whether the content type of this element is a string value.</p>
	 * @return True if this element has string content, otherwise false */
	isStr() {return this.cStr !== null;}
	toString() {return this.cRaw !== null ? this.cRaw : "?";}
	/** <p>Fetch this element content as a string value.</p>
	 * <p>If this element content is not a string value, an exception is thrown.
	 * To avoid that possibility, use asStrOr to provide a default value.</p>
	 * <p>Use toStr/toStrOr for casting other content types to be a string value.</p>
	 * @exception IDAOpException if this element does not contain string content.
	 * @return String content value. */
	asStr()
	{
		if (this.cStr === null) throw new IDAOpException("Content is not a string value");
		return this.cStr;
	}
	/** <p>Fetch this element content as a string value.</p>
	 * <p>If this element content is not a string value, the provided default value is returned.</p>
	 * <p>On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
	 * If possible, always call asStrOr() rather than asStrOr(null).</p>
	 * @param other A default value to return if this element does not contain string content.
	 * @return String content value if present, otherwise the default value. */
	asStrOr(other=null)
	{
		return this.cStr !== null ? this.cStr : other;
	}
	/** <p>Test whether this element content can be interpreted as a string value (see toStr).
	 * This will always be the case for natural string content (see isStr).</p>
	 * <p>Most content can be cast to string.
	 * Types which cannot be cast are lists, references and nulls.</p>
	 * @return True if this element content can be cast to a string value, otherwise false */
	canStr()
	{
		return this.hasContent() && !this.isList() && !this.isRef();
	}
	/** <p>Cast this element content to a string value.</p>
	 * <p>If this element content can not be interpreted as a string value, an exception is thrown.
	 * To avoid that possibility, use toStrOr to provide a default value.</p>
	 * @exception IDAOpException if this element content cannot cast to a string value.
	 * @return String content interpretation. */
	toStr()
	{
		if (!this.canStr()) throw new IDAOpException("Content cannot cast to a string value");
		return this.cRaw;
	}
	/** <p>Cast this element content to a string value.</p>
	 * <p>If this element content can not be interpreted as a string value, the provided default value is returned.</p>
	 * <p>On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
	 * If possible, always call toStrOr() rather than toStrOr(null).</p>
	 * @param other A default value to return if this element content cannot cast to a string value.
	 * @return String content interpretation if able, otherwise the default value. */
	toStrOr(other=null)
	{
		return this.canStr() ? this.cRaw : other;
	}
	
	// Content/number/other
	clearNum() {this.cNum = ""; this.cDen = "1"; this.cUnit = "";}
	
	// Content/number/set
	refresh_num() {this.cRaw = this.cNum + (this.cDen == "1" ? "" : "/"+this.cDen) + this.cUnit; this.cStr = null;}
	/** <p>Set the unit string appended to the number represented by this element content.</p>
	 * <p>Passing null to setNumUnit has no effect.
	 * To clear the unit, call setNumUnit with an empty string.</p>
	 * <p>If this element content is not numeric, an exception is thrown.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return This element. */
	setNumUnit(unit)
	{
		if (unit != null)
		{
			if (!this.isNum()) throw new IDAOpException("Content is not numeric");
			this.cUnit = unit; this.refresh_num();
		}
		return this;
	}
	
	putNum(num, den="1", unit=null) {if (unit == null) unit = this.cUnit; this.putNull(); this.cNum = ""+num; this.cDen = ""+den; this.cUnit = unit; this.refresh_num(); return this;}
	
	/** <p>Place 64-bit floating-point content into this element with the provided value, replacing any existing content.</p>
	 * <p>If the provided unit string is non-null, also updates the unit string for this number (see setNumUnit).</p>
	 * @param num 64-bit floating-point content value.
	 * @param unit Number unit string.
	 * @return This element */
	putDouble(num, unit=null) {return this.putNum(""+num).setNumUnit(unit);}
	/** <p>Place 32-bit floating-point content into this element with the provided value, replacing any existing content.</p>
	 * <p>If the provided unit string is non-null, also updates the unit string for this number (see setNumUnit).</p>
	 * @param num 32-bit floating-point content value.
	 * @param unit Number unit string.
	 * @return This element */
	putFloat(num, unit=null) {return this.putNum(""+num).setNumUnit(unit);}
	/** <p>Place 64-bit integer content into this element with the provided value, replacing any existing content.</p>
	 * <p>If the provided unit string is non-null, also updates the unit string for this number (see setNumUnit).</p>
	 * @param num 64-bit integer content value.
	 * @param unit Number unit string.
	 * @return This element */
	putLong(num, unit=null) {return this.putNum(""+num).setNumUnit(unit);}
	/** <p>Place 32-bit integer content into this element with the provided value, replacing any existing content.</p>
	 * <p>If the provided unit string is non-null, also updates the unit string for this number (see setNumUnit).</p>
	 * @param num 32-bit integer content value.
	 * @param unit Number unit string.
	 * @return This element */
	putInt(num, unit=null) {return this.putNum(""+num).setNumUnit(unit);}
	
	// Content/number/get
	/** <p>Test whether the content type of this element is a numeric value.</p>
	 * @return True if this element has numeric content, otherwise false */
	isNum()
	{
		return this.cNum.length > 0;
	}
	/** <p>Fetch the unit string appended to the number represented by this element content.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * If numeric content had no unit specified, an empty string is returned.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return Numeric unit string. */
	getNumUnit()
	{
		if (!this.isNum()) throw new IDAOpException("Content is not numeric");
		return this.cUnit;
	}
	/** <p>Fetch the numerator string for the number represented by this element content.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * If the number was not a quotient, calling getNumN will return the same result as calling toStr.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return Numerator as a string. */
	getNumN()
	{
		if (!this.isNum()) throw new IDAOpException("Content is not numeric");
		return this.cNum;
	}
	/** <p>Fetch the denominator string for the number represented by this element content.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * If the number was not a quotient, calling getNumD will return "1".</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return Denominator as a string. */
	getNumD()
	{
		if (!this.isNum()) throw new IDAOpException("Content is not numeric");
		return this.cDen;
	}
	/** <p>Fetch this element content as a 64-bit floating-point value.</p>
	 * <p>If this element content is not numeric, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain numeric content.
	 * @return 64-bit floating-point content value if present, otherwise the default value. */
	asDoubleOr(other)
	{
		return this.isNum() ? this.c_double() : other;
	}
	/** <p>Fetch this element content as a 32-bit floating-point value.</p>
	 * <p>If this element content is not numeric, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain numeric content.
	 * @return 32-bit floating-point content value if present, otherwise the default value. */
	asFloatOr(other)
	{
		return this.isNum() ? this.c_float() : other;
	}
	/** <p>Fetch this element content as a 64-bit integer value.</p>
	 * <p>If this element content is not numeric, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain numeric content.
	 * @return 64-bit integer content value if present, otherwise the default value. */
	asLongOr(other)
	{
		return this.isNum() ? this.c_long() : other;
	}
	/** <p>Fetch this element content as a 32-bit integer value.</p>
	 * <p>If this element content is not numeric, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain numeric content.
	 * @return 32-bit integer content value if present, otherwise the default value. */
	asIntOr(other)
	{
		return this.isNum() ? this.c_int() : other;
	}
	/** <p>Fetch this element content as a 64-bit floating-point value.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * To avoid that possibility, use asDoubleOr to provide a default value.</p>
	 * <p>Use toDouble/toDoubleOr for casting other content types to be a 64-bit floating-point value.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return 64-bit floating-point content value. */
	asDouble()
	{
		if (!this.isNum()) throw new IDAOpException("Content is not numeric");
		return this.c_double();
	}
	/** <p>Fetch this element content as a 32-bit floating-point value.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * To avoid that possibility, use asFloatOr to provide a default value.</p>
	 * <p>Use toFloat/toFloatOr for casting other content types to be a 32-bit floating-point value.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return 32-bit floating-point content value. */
	asFloat()
	{
		if (!this.isNum()) throw new IDAOpException("Content is not numeric");
		return this.c_float();
	}
	/** <p>Fetch this element content as a 64-bit integer value.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * To avoid that possibility, use asLongOr to provide a default value.</p>
	 * <p>Use toLong/toLongOr for casting other content types to be a 64-bit integer value.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return 64-bit integer content value. */
	asLong()
	{
		if (!this.isNum()) throw new IDAOpException("Content is not numeric");
		return this.c_long();
	}
	/** <p>Fetch this element content as a 32-bit integer value.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * To avoid that possibility, use asIntOr to provide a default value.</p>
	 * <p>Use toInt/toIntOr for casting other content types to be a 32-bit integer value.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return 32-bit integer content value. */
	asInt()
	{
		if (!this.isNum()) throw new IDAOpException("Content is not numeric");
		return this.c_int();
	}
	
	c_double()
	{
		return this.cDen == "1" ? this.n_float(this.cNum) : (this.n_float(this.cNum)/this.n_float(this.cDen));
	}
	c_float()
	{
		return this.c_double();
	}
	c_long()
	{
		return this.cDen == "1" ? this.n_int(this.cNum) : parseInt(this.n_float(this.cNum)/this.n_float(this.cDen));
	}
	c_int()
	{
		return this.c_long();
	}
	n_int(str)
	{
		if (str.startsWith("0x")) {return this.n_hex(str);}
		if (str.startsWith("0b")) {return this.n_bin(str);}
		return parseInt(parseFloat(str));
	}
	n_float(str)
	{
		if (str.startsWith("0x")) {return parseFloat(this.n_hex(str));}
		if (str.startsWith("0b")) {return parseFloat(this.n_bin(str));}
		return parseFloat(str);
	}
	n_hex(str) {return parseInt(str.substring(2), 16);}
	n_bin(str) {return parseInt(str.substring(2), 2);}
	
	// Content/list
	clearList() {this.cList = null;}
	refresh_list() {this.cRaw = "List("+this.numItems()+")"; this.cStr = null;}
	
	iterator() {return IDA.port.list_i(this.cList);}
	
	// Content/list/set
	/** Returns this element */
	putList(l = false)
	{
		this.putNull();
		if (l !== false)
		{
			if (l != null)
			{
				this.cList = l; this.refresh_list();
				var last = null;
				for (var item of l)
				{
					item.link(last, null);
					last = item;
				}
			}
		}
		else
		{
			this.cList = [];
			this.refresh_list();
		}
		return this;
	}
	
	// Content/list/get
	/** <p>Test whether the content type of this element is a list.</p>
	 * <p>This reports whether the content is a list or non-list, as opposed to whether at least one list item is present.
	 * The primary purpose is to ensure the list exists before calling asList and/or iterating over its elements.</p>
	 * <p>Use numItems to test for an empty list (0) or at least one item present (greater than 0).</p>
	 * @return True if this element has list content, otherwise false */
	isList() {return this.cList != null;}
	/** <p>Fetch this element content as a list (containing other elements).</p>
	 * <p>If this element content is not a list, an exception is thrown.
	 * To avoid that possibility, use asListOr to provide a default value.</p>
	 * @exception IDAOpException if this element does not contain list content.
	 * @return List content. */
	asList()
	{
		if (this.cList == null) throw new IDAOpException("Content is not a list");
		return this.cList;
	}
	/** <p>Fetch this element content as a list (containing other elements).</p>
	 * <p>If this element content is not a list, the provided default value is returned.</p>
	 * <p>On platforms where lists/vectors are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed list, such that code will not be fully portable if null is provided.
	 * If possible, always call asListOr() rather than asListOr(null).</p>
	 * @param other A default value to return if this element does not contain list content.
	 * @return List content if present, otherwise the default value. */
	asListOr(other = null)
	{
		return this.isList() ? this.cList : other;
	}
	/** <p>Fetch the number of content list items within this element.</p>
	 * <p>If no content list is present, the sentinel value -1 is returned.
	 * If 0 is returned, the list is present but empty.</p>
	 * @return The content list size if a list is present, otherwise -1 */
	numItems()
	{
		return this.isList() ? this.cList.length : -1;
	}
	
	/** <p>Test whether this element contains a list item with the provided name.
	 * If it does, an equivalent call to getItem will return an element.</p>
	 * <p>If false is returned, this can be either due to having no list item of that name or due to being a non-list.
	 * Use isList to separate those two cases.</p>
	 * @param key A list item name to be checked
	 * @return True if a content list item exists with the provided name, otherwise false */
	hasItem(key) {return this.getItemOr(key) !== null;}
	
	/** <p>Fetch the item element at the provided index in this element's content list.</p>
	 * <p>If this element content is not a list or has too few items in the list, an exception is thrown.
	 * To avoid that possibility, use getItemAtOr to provide a default value.</p>
	 * @param index A list item index to be recalled.
	 * @exception IDAOpException if this element has no such list item.
	 * @return The indexed content list item element. */
	getItemAt(index)
	{
		if (this.numItems() <= index) throw new IDAOpException("Invalid list item index");
		return this.cList[index];
	}
	/** <p>Fetch the item element at the provided index in this element's content list.</p>
	 * <p>If this element content is not a list or has too few items in the list, the provided default value is returned.</p>
	 * @param index A list item index to be recalled.
	 * @param other A default value to return if this element has no such list item.
	 * @return The indexed content list item element if present, otherwise the default value. */
	getItemAtOr(index, other = null)
	{
		return this.numItems() > index ? this.cList[index] : other;
	}
	
	/** <p>Fetch the first item element with the provided name in this element's content list.</p>
	 * <p>If this element content is not a list or has no item of that name, an exception is thrown.
	 * To avoid that possibility, use getItemOr to provide a default value.</p>
	 * @param index A list item name to be recalled.
	 * @exception IDAOpException if this element has no such list item.
	 * @return The named content list item element. */
	getItem(key)
	{
		var item = this.getItemOr(key);
		if (item == null) throw new IDAOpException("Item not found");
		return item;
	}
	/** <p>Fetch the first item element with the provided name in this element's content list.</p>
	 * <p>If this element content is not a list or has no item of that name, the provided default value is returned.</p>
	 * @param key A list item name to be recalled.
	 * @param other A default value to return if this element has no such list item.
	 * @return The named content list item element if present, otherwise the default value. */
	getItemOr(key, other = null)
	{
		var out = this.isList() ? this.asList().find((x) => x.isName(key)) : other;
		return out === undefined ? other : out;
	}
	
	/** <p>Cast to a string value the first item element with the provided name in this element's content list.</p>
	 * <p>If this element content is not a list, has no item of that name or can not be interpreted as a string value, an exception is thrown.
	 * To avoid that possibility, use getItemStrOr to provide a default value.</p>
	 * @param key A list item name to be recalled.
	 * @exception IDAOpException if this element has no such list item, or the item content cannot cast to a string value.
	 * @return String content interpretation of the named list item element. */
	getItemStr(key)
	{
		return this.getItem(key).toStr();
	}
	/** <p>Cast to a string value the first item element with the provided name in this element's content list.</p>
	 * <p>If this element content is not a list, has no item of that name or can not be interpreted as a string value, the provided default value is returned.</p>
	 * <p>On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
	 * If possible, always call getItemStrOr(key) rather than getItemStrOr(key, null).</p>
	 * @param key A list item name to be recalled.
	 * @param other A default value to return if this element has no such list item, or the item content cannot cast to a string value.
	 * @return String content interpretation of the named list item element if possible, otherwise the default value. */
	getItemStrOr(key, other=null)
	{
		var item = this.getItemOr(key);
		return item != null ? item.toStrOr(other) : other;
	}
	
	/** <p>Fetch the first item element in this element's content list.</p>
	 * <p>If this element content is not a list or is an empty list, an exception is thrown.
	 * To avoid that possibility, use getItemFirstOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no such list item.
	 * @return The first content list item element. */
	getItemFirst()
	{
		if (this.numItems() < 1) throw new IDAOpException(this.isList() ? "Content list is empty" : "Element content is not a list");
		return this.cList[0];
	}
	/** <p>Fetch the first item element in this element's content list.</p>
	 * <p>If this element content is not a list or is an empty list, the provided default value is returned.</p>
	 * @param other A default value to return if this element has no such list item.
	 * @return The first content list item element if present, otherwise the default value. */
	getItemFirstOr(other = null)
	{
		return this.numItems() > 0 ? this.cList[0] : other;
	}
	/** <p>Fetch the last item element in this element's content list.</p>
	 * <p>If this element content is not a list or is an empty list, an exception is thrown.
	 * To avoid that possibility, use getItemLastOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no such list item.
	 * @return The last content list item element. */
	getItemLast()
	{
		var n = this.numItems();
		if (n < 1) throw new IDAOpException(this.isList() ? "Content list is empty" : "Element content is not a list");
		return this.cList[n-1];
	}
	/** <p>Fetch the last item element in this element's content list.</p>
	 * <p>If this element content is not a list or is an empty list, the provided default value is returned.</p>
	 * @param other A default value to return if this element has no such list item.
	 * @return The last content list item element if present, otherwise the default value. */
	getItemLastOr(other = null)
	{
		var n = this.numItems();
		return n > 0 ? this.cList[n-1] : other;
	}
	
	// Content/list/add
	/** Insert a item element at the end of the content list.<br>
	 * If this element does not have list content, a new list will be created if the makeList flag is set, otherwise an exception is thrown.<br>
	 * @param el The item element to be stored.
	 * @param makeList Whether to overwrite existing non-list content.
	 * @exception IDAOpException if this element content is not a list and the makeList flag is false.
	 * @return This element. */
	addItem(el, makeList=true)
	{
		if (!this.isList() && !makeList) {throw new IDAOpException("Cannot append items onto a non-list");}
		if (!this.isList()) this.putList();
		el.link(this.getItemLastOr(), null);
		this.cList.push(el);
		this.refresh_list();
		return this;
	}
	
	/** Insert a item element at the specified content list index position.<br>
	 * If this element does not have list content, an exception is thrown.<br>
	 * @param el The item element to be stored.
	 * @param pos The list index position at which the item should be inserted.
	 * @exception IDAOpException if the specified position is below zero or above the present list size, or if this element content is not a list.
	 * @return This element. */
	addItemAt(el, pos)
	{
		if (!this.isList()) {throw new IDAOpException("Cannot merge items into a non-list");}
		IDA.list_add(this.cList, el, pos, "item");
		this.refresh_list();
		return this;
	}
	
	/** Create and insert a new item element at the end of the content list, with the provided element name.<br>
	 * If this element does not have list content, a new list will be created if the makeList flag is set, otherwise an exception is thrown.<br>
	 * @param name Name string for the created item element.
	 * @param makeList Whether to overwrite existing non-list content.
	 * @exception IDAOpException if this element content is not a list and the makeList flag is false.
	 * @return The created item element. */
	newItem(name, makeList=true)
	{
		var el = new IDA(name);
		this.addItem(el, makeList);
		return el;
	}
	
	// Content/list/remove
	/** Remove the item at the specified content list index position.<br>
	 * If this element does not have list content, an exception is thrown.<br>
	 * @param pos The list index position of an item which should be removed.
	 * @exception IDAOpException if the specified position is below zero or beyond the present list size, or this element does not have list content. */
	remItemAt(pos)
	{
		if (!this.isList()) {throw new IDAOpException("Cannot remove items from a non-list");}
		IDA.list_del(this.cList, pos, "item");
		this.refresh_list();
	}
	/** Remove the provided content list item.<br>
	 * If the item is absent or this element does not have list content, the call has no effect.<br>
	 * @param el The item element to be removed.
	 * @return True if an item was removed, otherwise false. */
	remItem(el)
	{
		if (this.isList() && IDA.list_rem(this.cList, el)) {this.refresh_list(); return true;}
		return false;
	}
	/** Remove the first content list item with the provided element name.<br>
	 * If no such item exists or this element does not have list content, the call has no effect.<br>
	 * @param name Name string of the item element to remove.
	 * @return True if an item was removed, otherwise false. */
	unItem(name)
	{
		var item = this.getItemOr(name);
		if (item != null) {this.remItem(item); return true;}
		return false;
	}
	
	// Object/equals
	eq_inner(obj)
	{
		if (obj == null || !(obj instanceof IDA)) {return false;}
		if (obj == this) {return true;}
		if (!this.eqName(obj)) {return false;}
		if (!this.eqParams(obj)) {return false;}
		if (!this.eqContent(obj)) {return false;}
		return true;
	}
	/** <p>Perform a top-level comparison of this element and the provided element.</p>
	 * <p>To be equal, both elements must match when interpreted as full articles.
	 * The element names, parameters and content must all match recursively.</p>
	 * <p>Note that content matches consider whether the <i>information</i> is the same.
	 * See eqContent for details.</p>
	 * @param obj An article element for comparison
	 * @param adjacents Whether to include adjacent elements in the comparison
	 * @return True if the elements represent matching articles, otherwise false */
	eq(obj, adjacent=false)
	{
		if (obj == this) return true;
		if (obj == null || !(obj instanceof IDA)) return false;
		if (this.txRefLabel(adjacent) != obj.txRefLabel(adjacent)) return false;
		if (!this.eq_inner(obj)) return false;
		
		if (adjacent)
		{
			var a = this;
			var b = obj;
			var el = b;
			while (a.hasPrev() && b.hasPrev())
			{
				a = a.getPrevOr(); b = b.getPrevOr();
				if (!a.eq_inner(b)) {return false;}
			}
			if (a.hasPrev() || b.hasPrev()) {return false;}
			
			a = this;
			b = el;
			while (a.hasNext() && b.hasNext())
			{
				a = a.getNextOr(); b = b.getNextOr();
				if (!a.eq_inner(b)) {return false;}
			}
			if (a.hasNext() || b.hasNext()) {return false;}
		}
		
		return true;
	}
	
	/** Compare the names of this element and the provided element.
	 * @param obj An element for name comparison
	 * @return True if the names match, otherwise false */
	eqName(obj) {return (obj instanceof IDA) && this.isName(obj.getNameOr());}
	/** <p>Compare the contents of this element and the provided element.</p>
	 * <p>The content type must match, and the contents must match in a type-dependent way
	 * such that each element contains the same <i>information</i>. Numbers must have the
	 * same precision, and scope references must target the same element relative to each
	 * scope. Foreign references are equal by default.</p>
	 * @param obj An element for content comparison
	 * @return True if the contents match, otherwise false */
	eqContent(obj)
	{
		if (!(obj instanceof IDA)) {return false;}
		if (this.cRaw != obj.cRaw) {return false;}
		if (this.cStr != obj.cStr) {return false;}
		if (this.cNum != obj.cNum) {return false;}
		//if (this.cDen != obj.cDen) {return false;} // Assured by testing cRaw
		//if (this.cUnit != obj.cUnit) {return false;} // Assured by testing cRaw
		if (!IDA.list_equals(this.cList, obj.cList)) {return false;}
		if (this.cBool != obj.cBool) {return false;}
		if (!this.eqContent_ref(this.cRef, obj.cRef)) return false;
		
		return true;
	}
	eqContent_ref(ref_a, ref_b)
	{
		if ((ref_a != null) ^ (ref_b != null)) return false;
		if (ref_a == ref_b) return true;
		if (ref_a.hasRefName() ^ ref_b.hasRefName()) return false;
		if (!ref_a.hasRefName()) return true;
		return ref_a.getRefName() == ref_b.getRefName();
	}
	/** Compare recursively the parameters of this element and the provided element.
	 * @param obj An element for parameter comparison
	 * @return True if the parameters match, otherwise false */
	eqParams(obj)
	{
		return (obj instanceof IDA) && IDA.list_equals(this.parameters, obj.parameters);
	}
	

	// Object/capture
	// repObj
	/** <p>Top-level article encode operation for a full object tree.
	 * This method calls objEncode, so the top-level object must implement idaEncode.</p>
	 * 
	 * <p>After all encoding is complete, any object references declared via setRefObj will be established as element references in the article.</p>
	 * @param obj A top-level object, which must implement idaEncode, to encode and record as this article element.
	 * @return This article element. */
	objCapture(obj)
	{
		this.objEncode(obj);
		
		this.capture_objs();
		this.capture_refs();
		
		return this;
	}
	/** <p>Encode one member object via idaEncode, with automatic element name via idaName.
	 * This element will become "representative" of the object, which is used to encode references via setRefObj.</p>
	 * 
	 * <p>This method should be called from within idaEncode itself, to record members which also implement the IDAObj codec interface.
	 * For the top-level operation, user code should call idaCapture instead to ensure that all object references are recorded.</p>
	 * @param obj An object, which must implement idaEncode, to encode and record as this element.
	 * @return This element. */
	objEncode(obj)
	{
		this.setName(IDAObj.idaName(obj));
		this.repObj = obj;
		IDAObj.idaEncode(obj, this);
		
		return this;
	}
	capture_objs()
	{
		if (this.repObj !== null) {this.repObj.idaCaptureObj = this;}
		if (this.hasParams()) for (var p of this.parameters) p.capture_objs();
		if (this.isList()) for (var i of this.cList) i.capture_objs();
	}
	capture_refs()
	{
		if (this.cRefObj !== null) {this.putRef(this.cRefObj.idaCaptureObj);}
		if (this.hasParams()) for (var p of this.parameters) p.capture_refs();
		if (this.isList()) for (var i of this.cList) i.capture_refs();
	}
	
	// Object/restore
	// repObjActions
	on_rep_obj(callback)
	{
		if (this.repObj !== null) {callback(this.repObj);}
		else
		{
			if (this.repObjActions === null) this.repObjActions = [];
			this.repObjActions.push(callback);
		}
	}
	restore_check()
	{
		if (this.repObjActions !== null) throw new IDAOpException("Decode process failed to resolve an object reference");
	}
	/** <p>Top-level article decode operation for a full object tree.
	 * This method calls objDecode, so the top-level object must implement idaDecode.</p>
	 * 
	 * <p>After all decoding is complete, the integrity will be checked to ensure that all object reference requests were resolved.
	 * If any calls to withRefObj did not receive the deferred result, an exception is thrown.
	 * This occurs if the article contained an element reference for which no object was constructed via objDecode.</p>
	 * @param obj A top-level object, which must implement idaDecode, to become the result of decoding this article element.
	 * @return The input object. */
	objRestore(obj)
	{
		this.objDecode(obj);
		this.walk((x) => x.restore_check(), true, true);
		return obj;
	}
	/** <p>Decode one member object via idaDecode.
	 * This element will become "representative" of the object, which is used to decode references via getRefObj or withRefObj.</p>
	 * 
	 * <p>This method should be called from within idaDecode itself, to create members which also implement the IDAObj codec interface.
	 * For the top-level operation, user code should call idaRestore instead to verify that all object reference requests were resolved.</p>
	 * @param obj An object, which must implement idaDecode, to become the result of decoding this element.
	 * @return The input object. */
	objDecode(obj)
	{
		this.repObj = obj;
		if (this.repObjActions !== null)
		{
			for (var c of this.repObjActions) c(obj);
			this.repObjActions = null;
		}
		IDAObj.idaDecode(obj, this);
		return obj;
	}
	
	// Deserialisation
	/** Read bytes from the provided input handle, parse it as markup and use the result to build this element.<br>
	 * After the parser finishes, this element will become the top-level entry element from the markup stream.
	 * @param input NodeJS stream.Readable, such as those created via fs.createReadStream
	 * @return Promise which is resolved with this element when the stream ends */
	inStream(input)
	{
		return new Promise((resolve, reject) =>
		{
			let data = "";
			input.setEncoding("binary");
			input.on("data", buf => {data += buf;});
			input.on("end", () =>
			{
				const lim = data.length;
				const ptr = {"n":0};
				const f = () => ((ptr.n < lim) ? data.charCodeAt(ptr.n++) : null);
				try {resolve(IDA_RX.inObj(this, f));} catch (ex) {reject(ex);}
			});
		});
	}
	/** Read the provided byte array, parse it as markup and use the result to build this element.<br>
	 * After the parser is done, this element will become the top-level entry element from the markup stream.
	 * @param input A byte array containing markup data
	 * @return This element */
	inBytes(input)
	{
		if (typeof input == "object")
		{
			const lim = input.length;
			const ptr = {"n":0};
			const f = () => ((ptr.n < lim) ? input[ptr.n++] : null);
			return IDA_RX.inObj(this, f);
		}
		else if (typeof input == "string")
		{
			return this.inString(input);
		}
		else throw new IDAOpException("Byte input must be Array or Buffer");
	}
	/** Read the provided string, parse it as markup and use the result to build this element.<br>
	 * After the parser is done, this element will become the top-level entry element from the markup stream.
	 * @param input A string containing markup data
	 * @return This element */
	inString(input)
	{
		const lim = input.length;
		const ptr = {"n":0, "t":[]};
		const f = function()
		{
			if (ptr.t.length > 0) return ptr.t.shift();
			if (ptr.n >= lim) return null;
			
			var codePoint = input.codePointAt(ptr.n++);
			if (codePoint < 0x80) return codePoint;
			else if (codePoint < 0x800)
			{
				ptr.t = [0x80 | (codePoint & 0x3f)];
				return 0xc0 | (codePoint >> 6);
			}
			else if (codePoint < 0x10000)
			{
				ptr.t = [0x80 | ((codePoint >> 6) & 0x3f), 0x80 | (codePoint & 0x3f)];
				return 0xe0 | (codePoint >> 12);
			}
			else
			{
				ptr.n++;
				ptr.t = [0x80 | ((codePoint >> 12) & 0x3f), 0x80 | ((codePoint >> 6) & 0x3f), 0x80 | (codePoint & 0x3f)];
				return 0xf0 | (codePoint >> 18);
			}
		};
		
		return IDA_RX.inObj(this, f);
	}
	/** Read from the file at the provided path, parse it as markup and use the result to build this element.<br>
	 * After the parser is done, this element will become the top-level entry element from the markup stream.
	 * @param input A path to a file containing markup data
	 * @return This element */
	inFile(path)
	{
		// Platform-specific check (NodeJS)
		var FS;
		if ((typeof require != "function") || (typeof (FS = require("fs")) != "object"))
		{
			throw new IDAFormatException(this, "File system is unavailable");
		}
		return this.inBytes(FS.readFileSync(path));
	}
	
	// Serialisation
	/** Encode this element as top-level markup and write it to the provided output handle.<br>
	 * If <i>adjacents</i> is true, all preceding and following elements will also be encoded.<br>
	 * If a preceding element was encoded that way, the markup will contain an entry instruction.<br>
	 * Any write errors will be sent back to the caller.
	 * @param output An open output handle to receive a stream of markup data
	 * @param adjacents Whether to write markup for elements adjacent to this one */
	outStream(output, adjacents = false)
	{
		// TODO: change JS outStream to work with actual streams such as process.stdout
		new IDA_TX( function(str) {output(str);}).idaWrite(this, adjacents);
	}
	/** Encode this element as top-level markup and write it to the provided output handle.<br>
	 * If <i>adjacents</i> is true, all preceding and following elements will also be encoded.<br>
	 * If a preceding element was encoded that way, the markup will contain an entry instruction.
	 * @param output An open output handle to receive a stream of markup data
	 * @param adjacents Whether to write markup for elements adjacent to this one
	 * @return True on success, or false if a write error occurred */
	outPrint(output, adjacents = false)
	{
		return new IDA_TX( function(str) {output(str);}).idaPrint(this, adjacents);
	}
	/** Encode this element as top-level markup and return it as a byte array.<br>
	 * If <i>adjacents</i> is true, all preceding and following elements will also be encoded.<br>
	 * If a preceding element was encoded that way, the markup will contain an entry instruction.
	 * @param adjacents Whether to write markup for elements adjacent to this one
	 * @return A byte array containing the markup data */
	outBytes(adjacents = false)
	{
		return this.outString(adjacents);
	}
	/** Encode this element as top-level markup and return it as a byte array.<br>
	 * If <i>adjacents</i> is true, all preceding and following elements will also be encoded.<br>
	 * If a preceding element was encoded that way, the markup will contain an entry instruction.
	 * @param adjacents Whether to write markup for elements adjacent to this one.
	 * @return A string containing the markup data. */
	outString(adjacents = false)
	{
		let buf = {"b":""};
		new IDA_TX( function(str) {buf.b += str;}).idaWrite(this, adjacents);
		return buf.b;
	}
	/** Encode this element as top-level markup and write it to the file at the provided path.<br>
	 * If <i>adjacents</i> is true, all preceding and following elements will also be encoded.<br>
	 * If a preceding element was encoded that way, the markup will contain an entry instruction.<br>
	 * Any write errors will be sent back to the caller.
	 * @param path A destination file path for the markup data.
	 * @param adjacents Whether to write markup for elements adjacent to this one. */
	outFile(path, adjacents = false)
	{
		// Platform-specific check (NodeJS)
		var FS;
		if ((typeof require != "function") || (typeof (FS = require("fs")) != "object"))
		{
			throw new IDAFormatException(this, "File system is unavailable");
		}
		// TODO: Implement file handles for serialisation
		//return this.inString(FS.readFileSync(path).toString());
	}
	
	// Construction
	constructor(name = null)
	{
		this.includePwd = IDA.gIncludePwd;
		this.reset();
		this.setName(name);
	}
	
	// Initialisation
	/** <p>Initialisate this element anew.</p>
	 * <p>All existing information comprising this element will be discarded.
	 * The result will be an anonymous, null-content element with no parameters list, no adjacent elements and default input/output options.</p>
	 * @return This element. */
	reset()
	{
		this.putNull();
		this.cRefName = null;
		this.repObj = null;
		this.repObjActions = null;
		
		this.itemClasses = null;
		this.name = null;
		this.next = null;
		this.prev = null;
		this.clearParams();
		
		this.opts = new IDAOptions(IDAOptions.defaultOptions);
		this.scope = null;
		this.itemTemplates = null;
		
		this.inputListDepth = -1;
		this.inputListLength = -1;
		
		return this;
	}
	
	// Copies
	/** <p>Create a new element as a copy of all aspects of this element.
	 * Any parameters or list content will be cloned recursively.</p>
	 * @param adjacents Whether to include adjacent elements in the cloning process
	 * @return The copy element */
	copy(adjacent = false) {return this.copyTo(new IDA(), adjacent);}
	/** <p>Modify the provided element to become a copy of all aspects of this element.
	 * Any parameters or list content will be cloned recursively.</p>
	 * @param copy An element to become a copy of this element
	 * @param adjacents Whether to include adjacent elements in the cloning process
	 * @return The copy element */
	copyTo(copy, adjacent = false)
	{
		var has_ref = this.txRefLabel(adjacent) > 0;
		IDA.copy_impl(this, copy, adjacent, adjacent);
		if (has_ref) copy.rxRefResolve(adjacent);
		return copy;
	}
	
	copy_inner()
	{
		return IDA.copy_impl(this, new IDA(), false, false);
	}
	static copy_impl(orig, copy, scanPrev, scanNext)
	{
		copy.copyName(orig);
		copy.copyParams(orig);
		copy.copyContent(orig);
		copy.copyOptions(orig);
		
		var a;
		if (scanPrev && (a = orig.getPrevOr()) != null) {copy.linkPrev(IDA.copy_impl(a, new IDA(), true, false));}
		if (scanNext && (a = orig.getNextOr()) != null) {copy.linkNext(IDA.copy_impl(a, new IDA(), false, true));}
		
		return copy;
	}
	
	// Utility
	/** <p>Call the provided function once for each element within this article.</p>
	 * <p>This element is passed first, then its parameters (if any), then its list content items (if any).
	 * Parameters and list items are treated recursively until the full article has been traversed.</p>
	 * <p>Optionally, the walk may also traverse adjacent elements recursively, requested via the reverse and forward flags.</p>
	 * @param func A callback function which will be supplied with article elements.
	 * @param rev Whether to include elements adjacent to this one in the reverse direction.
	 * @param fwd Whether to include elements adjacent to this one in the forward direction. */
	walk(func, rev, fwd)
	{
		func(this);
		if (this.hasParams()) for (var p of this.parameters) p.walk(func, false, false);
		if (this.isList()) for (var i of this.cList) i.walk(func, false, false);
		if (fwd && this.hasNext()) this.getNextOr().walk(func, false, true);
		if (rev && this.hasPrev()) this.getPrevOr().walk(func, true, false);
	}
	
	/** <p>Check the integrity of article element.
	 * If a problem is detected which would lead to errors or corrupted output, an exception is thrown with details in the exception text.</p>
	 * <p>The tests can be performed with adjacent elements enabled or disabled, matching the markup output modes.
	 * Note that disabling adjacents may avoid some problems (such as with linkage), but may also induce others (such as with references).</p>
	 * <p>The top-level element linkage is tested to ensure that there are no loops, and that adjacent elements are doubly-linked correctly.
	 * A bad double-link can disrupt the output flow, and a loop can cause infinite iteration.</p>
	 * <p>Any elements with reference content are tested for foreign references, which become lost/null during output.</p>
	 * @param adjacents Whether to extend the test to include elements adjacent to this one.
	 * @exception IDAOpException if bad integrity is detected. */
	validate(adjacents)
	{
		this.validate_loop(adjacents);
		this.validate_foreign(adjacents);
	}
	validate_loop(adjacents)
	{
		if (!adjacents) return;
		var fwd = (x) => x.getNextOr();
		var rev = (x) => x.getPrevOr();
		this.validate_loop_f(fwd, rev);
		this.validate_loop_f(rev, fwd);
	}
	validate_loop_f(step, back)
	{
		var a = this;
		var b = this;
		
		while (true)
		{
			var last = b;
			b = step(b);
			if (b == null) return;
			if (back(b) != last) throw new IDAOpException("Article contains a broken link");
			
			last = b;
			b = step(b);
			if (b == null) return;
			if (back(b) != last) throw new IDAOpException("Article contains a broken link");
			
			a = step(a);
			if (a == b) throw new IDAOpException("Article contains an adjacency loop");
		}
	}
	validate_foreign(adjacents)
	{
		var foreign = [];
		this.walk(function(x) {if (x.isRef()) if (!IDA.port.list_contains(foreign, x.asRefOr())) foreign.push(x.asRefOr());}, adjacents, adjacents);
		this.walk(function(x) {if (IDA.port.list_contains(foreign, x)) IDA.port.list_rem(foreign, x);}, adjacents, adjacents);
		if (foreign.length > 0) throw new IDAOpException("Article contains a foreign reference");
	}
	
	
	static list_add(list, el, pos, noun)
	{
		const size = list.length;
		
		if (pos > 0)
		{
			if (pos > size) {throw new IDAOpException("Cannot add "+noun+" beyond position "+size+" (request was "+pos+")");}
		}
		else if (pos < 0)
		{
			if (pos < -size) {throw new IDAOpException("Cannot add "+noun+" before position -"+size+" (request was "+pos+")");}
			pos += size;
		}
		
		if (pos == 0)
		{
			if (size > 0) {el.link(null, list[0]);}
			list.splice(0, 0, el);
		}
		else if (pos == size)
		{
			el.link(list[size-1], null);
			list.push(el);
		}
		else // size must be at least 2, pos cannot be at either end
		{
			el.link(list[pos-1], list[pos]);
			list.splice(pos, 0, el);
		}
	}
	static list_del(list, pos, noun)
	{
		const size = list.length;
		
		if (pos > 0)
		{
			if (pos > size-1) {throw new IDAOpException("Cannot remove item beyond position "+(size-1)+" (request was "+pos+")");}
		}
		else if (pos < 0)
		{
			if (pos < -size) {throw new IDAOpException("Cannot remove item before position -"+size+" (request was "+pos+")");}
			pos += size;
		}
		
		return list.splice(pos, 1)[0].link(null, null);
	}
	
	// Syntax
	static id0(c) {return (c>='a' && c<='z') || (c>='A' && c<='Z') || c == '_' || c == '/';}
	static id1(c)
	{
		return IDA.id0(c) || (c>='0' && c<='9') || c == '-' || c == '.'
			|| c == '!' || c == '@' || c == '^' || c == '&' || c == '?';
	}
	static comment1(c) {return c == '#';}
	static comment2(c1, c2)
	{
		return (c1 == '/' && (c2 == '/' || c2 == '*'))
			|| (c1 == '<' && (c2 == '?' || c2 == '!'));
	}
	
	// Compatibility/port through
	// (END)
	
	// Compatibility
	static eq_obj(a, b)
	{
		return (a instanceof IDA) ? a.eq(b) : (a === b);
	}
	
	static list_rem(l, el)
	{
		if (IDA.port.list_rem(l, el)) {el.link(null, null); return true;} else return false;
	}
	
	static list_equals(la, lb)
	{
		if (la == null || lb == null) return la == lb;
		var ia = IDA.port.list_i(la);
		var ib = IDA.port.list_i(lb);
		while (ia.hasNext() && ib.hasNext())
		{
			if (!ia.next().eq(ib.next())) {return false;}
		}
		return !(ia.hasNext() || ib.hasNext());
	}
	
}

IDA.gIncludePwd = null;

// From Compatibility/port
IDA.port = new IDA_Port();

// Unit testing
IDA.IDA_Output = IDA_Output;
IDA.IDA_TX = IDA_TX;

// nodejs hook
if (typeof module == "object") module.exports = IDA;
// CLI
if (typeof require == "function" && require.main === module) IDA_CLI.cli(process.argv.length, process.argv).then(process.exit);

