<?php
//    Copyright 2022 IDA Markup, ida-markup.com
// 
//    Licensed under the Apache License, Version 2.0 (the "License");
//    you may not use this file except in compliance with the License.
//    You may obtain a copy of the License at
// 
//        http://www.apache.org/licenses/LICENSE-2.0
// 
//    Unless required by applicable law or agreed to in writing, software
//    distributed under the License is distributed on an "AS IS" BASIS,
//    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//    See the License for the specific language governing permissions and
//    limitations under the License.

// Namespace

// Options
class IDAOptions
{
	// RX
	public $rxXmlReparseStrings = false;
	
	// TX
	public $txUnicode = false;
	
	function __construct($options = null)
	{
		if ($options !== null) $this->copyFrom($options);
	}
	
	public function copyFrom($options)
	{
		// Copy RX
		$this->rxXmlReparseStrings = $options->rxXmlReparseStrings;
		
		// Copy TX
		$this->txUnicode = $options->txUnicode;
	}
	
	static $defaultOptions;
}
IDAOptions::$defaultOptions = new IDAOptions();

// Object interface
trait IDAObj
{
	/** Override to offer a default element name. */
	public function idaName() {return null;}
	/** Override to enable capturing of this object into the provided storage element.<br>
	 * Usage note: If applicable, implementing classes can guarantee not to throw.<br>
	 * @return The input element. */
	public function idaEncode(IDA $el)
	{
		throw new IDAOpException("IDA encode not available");
	}
	/** Override to enable restoration of this object from the provided storage element.<br>
	 * @return This object. */
	public function idaDecode(IDA $el)
	{
		throw new IDAOpException("IDA decode not available");
	}
	
}

// Port begin
class IDA_Iterator
{
	public function hasNext() {return false;}
	public function getNext() {return null;}
	public function next() {return $this->getNext();}
}
class IDA_MutableSeq extends IDA_Iterator
{
	public $funcHas;
	public $funcGet;
	function __construct() {$this->funcHas = function() {return false;}; $this->funcGet = function() {return null;};}
	public function hasNext() {return ($this->funcHas)();}
	public function getNext() {return ($this->funcGet)();}
}
class IDA_Port
{
	
	// Port iterators
	// Iterator/MutableSeq in Port begin
	public function empty_i()
	{
		return new IDA_Iterator();
	}
	public function list_i($l)
	{
		if ($l === null) throw new IDAOpException("Internal list iteration error");
		$i = new IDA_MutableSeq();
		$i->n = 0;
		$i->lim = count($l);
		$i->funcHas = function() use($i) {return $i->n < $i->lim;};
		$i->funcGet = function() use($i, $l) {return $l[$i->n++];};
		return $i;
	}
	public function repeat_i($obj)
	{
		$i = new IDA_MutableSeq();
		$i->array_item_template = $obj;
		$i->funcHas = function() {return true;};
		$i->funcGet = function() use($i) {return $i->array_item_template;};
		return $i;
	}
	
	// Port arrays
	public function list_contains($l, $item)
	{
		return in_array($item, $l, true);
	}
	public function list_rem(array &$l, IDA &$item)
	{
		for ($i=0; $i<count($l); $i+=1)
		{
			if (IDA::eq_obj($item, $l[$i]))
			{
				array_splice($l, $i, 1);
				return true;
			}
		}
		return false;
	}
	
	// Port streams
	public function si_allMatch($i, $predicate)
	{
		while ($i->hasNext()) if (!$predicate($i->next())) {return false;}
		return true;
	}
	public function si_anyMatch($i, $predicate)
	{
		while ($i->hasNext()) if ($predicate($i->next())) {return true;}
		return false;
	}
	public function si_noneMatch($i, $predicate)
	{
		return !$this->si_anyMatch($i, $predicate);
	}
	
	public function i_map($i, $map)
	{
		$j = new IDA_MutableSeq();
		$j->funcHas = function() use ($i) {return $i->hasNext();};
		$j->funcGet = function() use ($i, $map) {return $map($i->getNext());};
		return $j;
	}
	public function i_filter($i, $filter)
	{
		$j = new IDA_MutableSeq();
		$j->buf = null;
		$j->more = false;
		$j->preLoad = function() use($j, $i, $filter)
		{
			$j->more = false;
			while ($i->hasNext())
			{
				if ($j->more = $filter($j->buf = $i->next())) return;
			}
		};
		$j->funcHas = function() use($j) {return $j->more;};
		$j->funcGet = function() use($j) {$x = $j->buf; ($j->preLoad)(); return $x;};
		($j->preLoad)();
		return $j;
	}
	public function i_flatMap($i, $map)
	{
		$j = new IDA_MutableSeq();
		$j->lbuf = null;
		$j->buf = null;
		$j->more = false;
		$j->preLoad = function() use ($j, $i, $map)
		{
			$j->more = false;
			while ($j->lbuf == null || !$j->lbuf->hasNext())
			{
				if (!$i->hasNext()) {return;}
				$j->lbuf = $map($i->next());
			}
			
			$j->more = true;
			$j->buf = $j->lbuf->next();
		};
		$j->funcHas = function() use($j) {return $j->more;};
		$j->funcGet = function() use($j) {$y = $j->buf; ($j->preLoad)(); return $y;};
		($j->preLoad)();
		return $j;
	}
	
	// Port string operation
	public function replace_all($s, $act, $dmd)
	{
		return str_replace(act, dmd, s);
	}
	
	public function substring($s, $fromChar, $toChar=null)
	{
		$s = substr($s, $this->str_units($s, 0, $fromChar));
		return $toChar !== null ? substr($s, 0, $this->str_units($s, 0, $toChar-$fromChar)) : $s;
	}
	
	public function trim($s)
	{
		$i1 = 0;
		$i2 = strlen($s);
		while ($i1 < $i2 && ord($s[$i1]) <= 0x20) $i1++;
		while ($i1 < $i2 && ord($s[$i2-1]) <= 0x20) $i2--;
		return substr($s, $i1, $i2-$i1);
	}
	public function rtrim($s)
	{
		$i2 = strlen($s);
		while (0 < $i2 && ord($s[$i2-1]) <= 0x20) $i2--;
		return substr($s, 0, $i2);
	}
	public function ltrim($s)
	{
		$i1 = 0;
		$i2 = strlen($s);
		while ($i1 < $i2 && ord($s[$i1]) <= 0x20) $i1++;
		return substr($s, $i1);
	}
	
	// Port string tests
	function ends_with($s, $suffix)
	{
		return strpos(strrev($s), strrev($suffix)) === 0;
	}
	function starts_with($s, $prefix)
	{
		return strpos($s, $prefix) === 0;
	}
	function str_contains($s, $mid)
	{
		return strpos($s, $mid) !== false;
	}
	
	// Port string info
	/** Character index */
	function str_index($s, $mid)
	{
		$i = strpos($s, $mid);
		return ($i === false) ? -1 : $this->str_chars($s, 0, $i);
	}
	/** Character index */
	function str_index_from($s, $mid, $fromChar)
	{
		$fromUnit = $this->str_units($s, 0, $fromChar);
		$i = strpos($s, $mid, $fromUnit);
		return ($i === false) ? -1 : ($fromChar + $this->str_chars($s, $fromUnit, $i));
	}
	function is_continuation($c)
	{
		return (ord($c) & 0xc0) == 0x80;
	}
	/** The number of characters represented */
	function str_len($s)
	{
		return $this->str_chars($s, 0, strlen($s));
	}
	/** The number of characters represented */
	function str_chars($s, $fromUnit, $toUnit)
	{
		$n = $toUnit - $fromUnit;
		for ($i=$toUnit-1; $i>=$fromUnit; $i-=1) if ($this->is_continuation($s[$i])) $n-=1;
		return $n;
	}
	/** The number of memory locations */
	function str_units($s, $fromChar, $toChar)
	{
		$toChar -= $fromChar;
		$top = strlen($s);
		
		// Skip
		$mark = 0;
		while ($fromChar-- > 0) while ((++$mark < $top) && $this->is_continuation($s[$mark])) ;
		
		// Count
		$ptr = $mark;
		while ($toChar-- > 0) while ((++$ptr < $top) && $this->is_continuation($s[$ptr])) ;
		
		return $ptr - $mark;
	}
	
	// Port hex
	public function char_2($c)
	{
		return str_pad(dechex($c), 2, "0", STR_PAD_LEFT);
	}
	public function char_4($c)
	{
		return str_pad(dechex($c), 4, "0", STR_PAD_LEFT);
	}
	public function char_8($c)
	{
		return str_pad(dechex($c), 8, "0", STR_PAD_LEFT);
	}
	
// Port end
}

// Format exception
class IDAFormatException extends Exception
{
	// Syntax
	public $ln;
	public $pos;
	
	// Structure
	public $node;
	
	function __construct($source, $message)
	{
		if ($source instanceof IDA_RX)
		{
			$rx = $source;
			$this->message = "(line ".$rx->ln.") ".$message;
			
			$this->node = null;
			
			// Extract problem index first
			$this->pos = strlen($rx->lnBuf);
			
			// Read until end of line for context
			$ln = $rx->lnBuf;
			try {$ln .= $rx->get_to("\n");}
			catch (Exception $e) {}
			$this->ln = $ln;
		}
		else if ($source instanceof IDA)
		{
			$nodeCause = $source;
			$this->message = ($nodeCause->isAnon() ? "(anonymous element) " : "(node '".$nodeCause->getNameOr("null")."') ") . $message;
			
			$this->ln = null;
			$this->pos = -1;
			$this->node = $nodeCause;
		}
		
		$this->name = "IDAFormatException";
	}
	
}

// Op exception
class IDAOpException extends Exception
{
	public function __construct($msg)
	{
		parent::__construct($msg);
	}
}

// Manip begin
class IDA_Manip
{
	static $FORWARD = true;
	static $REVERSE = false;
	static function rev($n) {return is_bool($n) ? !$n : -$n;}
	
	static function m_str($el)
	{
		if ($el->isNull()) return null;
		try {return $el->asStr();}
		catch (IDAOpException $ex) {throw new IDAFormatException($el, "String required");}
	}
	
	// Manip/iterate
	/** Return the last node */
	static function manip_iterate(IDA $manip, IDA $node, $itemTemplate)
	{
		$last = null;
		while ($itemTemplate->hasNext())
		{
			$templateNode = $itemTemplate->next();
			
			if ($last !== null) {$node = new IDA(); $last->linkNext($node);}
			
			$node->copyName($templateNode);
			$node->copyParams($templateNode);
			if ($templateNode->isList())
			{
				$node->putList();
				
				if ($templateNode->numItems() > 0)
				{
					$item = new IDA();
					if (IDA_Manip::manip_iterate($manip, $item, IDA::$port->list_i($templateNode->asListOr())) !== null)
					{
						for ($nextEl = null; $item != null; $item = $nextEl)
						{
							$nextEl = $item->getNextOr();
							$node->addItem($item);
						}
					}
					
				}
			}
			else {$node->copyContent($templateNode);}
			$last = $node;
			
			// Manip/iterate/main
			if ($manip->isList())
			// manip:
			foreach ($manip->asList() as $m)
			{
				if ($m->isAnon() || $m->isNull()) {continue;} // Operate on nothing, or no operations
				if (!$m->isList()) {throw new IDAFormatException($m, "Manipulation operations must be a list");}
				
				$operateOn = $m->getNameOr();
				$n = strpos($operateOn, "n") !== false; // n: name
				$k = strpos($operateOn, "k") !== false; // k: param key
				$v = strpos($operateOn, "v") !== false; // v: param value
				$p = strpos($operateOn, "p") !== false; // p: parameter(s)
				$c = strpos($operateOn, "c") !== false; // c: content
				
				// Manip/iterate/match
				// if-match: any of (n node-name, k param-key, v param-val, p(param-key param-val), c node-content)
				$tParamMatch = function($x) {return true;};
				if ($m->hasParams()) foreach ($m->getParamsOr() as $mp)
				{
					$sel = $mp->getNameOr("");
					
					if (strlen(preg_replace("/[nkvc!]+/", "", $sel)) > 0) {throw new IDAFormatException($m, "Invalid manipulation match (valid characters are [nkvc!])");}
					
					$flagNot = strpos($sel, "!") !== false;
					
					if (strpos($sel, "n") !== false)
					{
						if (($node->isName(IDA_Manip::m_str($mp))) == $flagNot) {continue 2;}
					}
					if (strpos($sel, "c") !== false)
					{
						if ($mp->eqContent($node) == $flagNot) {continue 2;}
					}
					
					if (strpos($sel, "k") !== false)
					{
						$tgt = IDA_Manip::m_str($mp);
						$tParamMatchOld = $tParamMatch;
						$tParamMatch = function($x) use ($tParamMatchOld, $flagNot, $tgt) {return $tParamMatchOld($x) && ($flagNot ^ ($x->isName($tgt)));};
						break;
					}
					if (strpos($sel, "v") !== false)
					{
						$tParamMatchOld = $tParamMatch;
						$tParamMatch = function($x) use ($tParamMatchOld, $flagNot, $mp) {return $tParamMatchOld($x) && ($flagNot ^ $mp->eqContent($x));};
						break;
					}
					
				}
				$paramMatch = $tParamMatch;
				
				// Manip/iterate/op
				foreach ($m->asList() as $op)
				{
					if ($op->isAnon()) {throw new IDAFormatException($m, "Missing operation code");}
					$opName = $op->getNameOr("");
					
					switch ($opName)
					{
					
						// Manip/iterate/op/insert
						case "i": // i: insert (nkvc): i (index) text-string
						{
							if ($op->isNull()) {throw new IDAFormatException($m, "Missing insert string");}
							if ($op->isList()) {throw new IDAFormatException($m, "Insert string cannot be a list");}
							
							$param = $op->getParamFirstOr();
							if ($param !== null && !$param->isNum()) {throw new IDAFormatException($m, "Insert position must be a number");}
							$pos = ($param !== null) ? $param->asIntOr(0) : IDA_Manip::$REVERSE;
							
							try
							{
								if ($n) {$node->setName(IDA_Manip::manip_ins($node->getNameOr(), $op->asStr(), $pos));}
								if ($c) {$node->putStr(IDA_Manip::manip_ins($node->asStr(), $op->asStr(), $pos));}
								if ($k || $v)
								{
									if ($node->hasParams()) foreach ($node->getParams() as $np)
									{
										if (!$paramMatch($np)) {continue;}
										if ($k) {$np->setName(IDA_Manip::manip_ins($np->getNameOr(), $op->asStr(), $pos));}
										if ($v) {$np->putStr(IDA_Manip::manip_ins($np->asStr(), $op->asStr(), $pos));}
									}
								}
							}
							catch (IDAOpException $opex) {throw new IDAFormatException($m, $opex->getMessage());}
							
							break;
						}
						
						// Manip/iterate/op/delete
						case "d": // d: delete (nkvc): d (index) length or d (index) bool
						{
							if (!($op->isNum() || $op->isBool() || $op->isNull())) {throw new IDAFormatException($m, "Delete amount must be boolean or a number");}
							
							$param = $op->getParamFirstOr();
							if ($param !== null && !$param->isNum()) {throw new IDAFormatException($m, "Delete position must be a number");}
							$pos = ($param !== null) ? $param->asIntOr(0) : IDA_Manip::$REVERSE;
							$amt = $op->isBool() ? ($op->isTrue() ? IDA_Manip::$FORWARD : IDA_Manip::$REVERSE) : ($op->isNull() ? 1 : $op->asIntOr(0));
							if (($pos === IDA_Manip::$REVERSE) && ($amt === IDA_Manip::$FORWARD || (is_int($amt) && $amt > 0))) {$amt = IDA_Manip::rev($amt);}
							
							try
							{
								if ($n) {$node->setName(IDA_Manip::manip_del($node->getNameOr(), $pos, $amt));}
								if ($c) {$node->putStr(IDA_Manip::manip_del($node->asStr(), $pos, $amt));}
								if ($k || $v)
								{
									if ($node->hasParams()) foreach ($node->getParams() as $np)
									{
										if (!$paramMatch($np)) {continue;}
										if ($k) {$np->setName(IDA_Manip::manip_del($np->getNameOr(), $pos, $amt));}
										if ($v) {$np->putStr(IDA_Manip::manip_del($np->asStr(), $pos, $amt));}
									}
								}
							}
							catch (IDAOpException $opex) {throw new IDAFormatException($m, $opex->getMessage());}
							
							break;
						}
						
						// Manip/iterate/op/add
						case "a": // a: add (vpc): a (index) {nodes...}
						{
							// No need to check isList; addItem and insertItem already have exceptions for this
							$param = $op->getParamFirstOr();
							if ($param !== null && !$param->isNum()) {throw new IDAFormatException($m, "Add position must be a number");}
							$pos = ($param !== null) ? $param->asIntOr(0) : IDA_Manip::$REVERSE;
							
							try
							{
								if ($p)
								{
									if (!$node->hasParams()) {$node->setParams();}
									IDA_Manip::manip_add(function($item, $pos) use ($node) {return $node->addParamAt($item, $pos);}, ($pos === IDA_Manip::$REVERSE) ? $node->numParams() : $pos, $op);
								}
								if ($c)
								{
									IDA_Manip::manip_add(function($item, $pos) use ($node) {return $node->addItemAt($item, $pos);}, ($pos === IDA_Manip::$REVERSE) ? $node->numItems() : $pos, $op);
								}
								if ($v)
								{
									if ($node->hasParams()) foreach ($node->getParams() as $np)
									{
										if (!$paramMatch($np)) {continue;}
										IDA_Manip::manip_add(function($item, $pos) use ($np) {return $np->addItemAt($item, $pos);}, ($pos === IDA_Manip::$REVERSE) ? $np->numItems() : $pos, $op);
									}
								}
							}
							catch (IDAOpException $opex) {throw new IDAFormatException($m, $opex->getMessage());}
							
							break;
						}
						
						// Manip/iterate/op/remove
						case "r": // r: remove (vpc): r (index) length or r(index) bool or r {nodes...}
						{
							if ($op->isNum() || $op->isBool() || $op->isNull()) // r (index) length or r(index) bool
							{
								$param = $op->getParamFirstOr();
								if ($param !== null && !$param->isNum()) {throw new IDAFormatException($m, "Remove position must be a number");}
								$pos = ($param !== null) ? $param->asIntOr(0) : IDA_Manip::$REVERSE;
								$amt = $op->isBool() ? ($op->isTrue() ? IDA_Manip::$FORWARD : IDA_Manip::$REVERSE) : ($op->isNull() ? 1 : $op->asIntOr(0));
								if (($pos === IDA_Manip::$REVERSE) && ($amt === IDA_Manip::$FORWARD || (is_int($amt) && $amt > 0))) {$amt = IDA_Manip::rev($amt);}
								
								if ($p)
								{
									IDA_Manip::manip_rem_pos($m, function($pos) use ($node) {return $node->remParamAt($pos);}, function() use ($node) {return $node->numParams();}, $pos, $amt);
								}
								if ($c)
								{
									IDA_Manip::manip_rem_pos($m, function($pos) use ($node) {return $node->remItemAt($pos);}, function() use ($node) {return $node->numItems();}, $pos, $amt);
								}
								if ($v)
								{
									if ($node->hasParams()) foreach ($node->getParamsOr() as $np)
									{
										if (!$paramMatch($np)) {continue;}
										IDA_Manip::manip_rem_pos($m, function($pos) use ($np) {return $np->remItemAt($pos);}, function() use ($np) {return $np->numItems();}, $pos, $amt);
									}
								}
							}
							else if ($op->isList()) // r {nodes...}
							{
								if ($p) {IDA_Manip::manip_rem_item(function($param) use ($node) {return $node->remParam($param);}, $op);}
								if ($c) {IDA_Manip::manip_rem_item(function($item) use ($node) {return $node->remItem($item);}, $op);}
								if ($v)
								{
									if ($node->hasParams()) foreach ($node->getParamsOr() as $np)
									{
										if ($paramMatch($np)) IDA_Manip::manip_rem_item(function($item) use ($np) {return $np->remItem($item);}, $op);
									}
								}
							}
							else {throw new IDAFormatException($m, "Remove subject must be boolean, a number or a list");}
							
							break;
						}
						
						// Manip/iterate/op/undo
						case "u": // u: unset/unlist (vpc): u text-key or u {text-key...}
						{
							if ($op->isList()) foreach ($op->asList() as $un) // u {text-key...}
							{
								$name = IDA_Manip::m_str($un);
								
								if ($c) {$node->unItem($name);}
								if ($p) {$node->unParam($name);}
								if ($v && $node->hasParams()) foreach ($node->getParamsOr() as $np)
								{
									if ($paramMatch($np)) $np->unItem($name);
								}
							}
							else // u text-key
							{
								$name = IDA_Manip::m_str($op);
								
								if ($c) {$node->unItem($name);}
								if ($p) {$node->unParam($name);}
								if ($v && $node->hasParams()) foreach ($node->getParamsOr() as $np)
								{
									if ($paramMatch($np)) $np->unItem($name);
								}
							}
							break;
						}
						
						// Manip/iterate/op/substitute
						case "s": // s: substitute (nkvc): s (search-token replacement-string ...);
						{
							if ($op->hasParams()) foreach ($op->getParamsOr() as $repl)
							{
								if ($repl->isAnon() || $repl->isNull()) {continue;}
								$act = $repl->getNameOr();
								$dmd = IDA_Manip::m_str($repl);
								
								// nkvc
								if ($n && $node->hasName()) {$node->setName(str_replace($act, $dmd, $node->getNameOr()));}
								if ($c && $node->hasContent()) {$node->putStr(str_replace($act, $dmd, IDA_Manip::m_str($node)));}
								if ($k || $v)
								{
									if ($node->hasParams()) foreach ($node->getParamsOr() as $np)
									{
										if (!$paramMatch($np)) {continue;}
										if ($k && $np->hasName()) {$np->setName(str_replace($act, $dmd, $np->getNameOr()));}
										if ($v && $np->hasContent()) {$np->putStr(str_replace($act, $dmd, IDA_Manip::m_str($np)));}
									}
								}
							}
							break;
						}
						
						// Manip/iterate/op/flip
						case "f": // f (vc): flip: f;
						{
							try
							{
								if ($c) {IDA_Manip::manip_negate($node);}
								if ($v && $node->hasParams()) foreach ($node->getParams() as $np)
								{
									if (!$paramMatch($np)) {continue;}
									IDA_Manip::manip_negate($np);
								}
							}
							catch (IDAOpException $opex) {throw new IDAFormatException($m, $opex->getMessage());}
							break;
						}
						
						// Manip/iterate/op/write
						case "w": // w (nkvpc): write: w replacement
						{
							if ($n) {$node->setName(IDA_Manip::m_str($op));}
							if ($c) {$node->copyContent($op);}
							if ($p)
							{
								if ($op->isNull()) {$node->clearParams();}
								else if ($op->isList())
								{
									$node->setParams();
									foreach ($op->asListOr() as $param) {$node->addParam($param->copy());}
								}
								else {throw new IDAFormatException($m, "Parameter write must be list or null");}
							}
							if ($k || $v)
							{
								if ($node->hasParams()) foreach ($node->getParamsOr() as $np)
								{
									if (!$paramMatch($np)) {continue;}
									if ($k) {$np->setName(IDA_Manip::m_str($op));}
									if ($v) {$np->copyContent($op);}
								}
							} 
							break;
						}
						
						// Manip/iterate/op end
						default: throw new IDAFormatException($m, "Invalid operation code \"".$opName."\"");
					}
				}
				
			// Manip/iterate end
			}
		
		}
		
		return $last;
		
	}
	
	// Manip/negate
	static function manip_negate($node)
	{
		if ($node->isBool())
		{
			$node->putBool(!$node->asBool());
		}
		else if ($node->isNum())
		{
			$num = $node->getNumN();
			$node->putNum((strpos($num, "-") === 0) ? substr($num, 1) : "-".$num, $node->getNumD());
		}
		else if ($node->hasContent()) {throw new IDAOpException("Flip subject must be boolean or a number");}
	}
	
	// Manip/insert
	/** Usage note: the 'ins' string is never null, and the pos can be REVERSE in place of a null to mean the end (never FORWARD though) */
	static function manip_ins($orig, $ins, $pos)
	{
		if ($orig === null) {throw new IDAOpException("Insert subject must not be null");}
		if ($pos === IDA_Manip::$REVERSE) {return $orig . $ins;}
		$l = IDA::$port->str_len($orig);
		if ($pos > 0)
		{
			if ($pos > $l) {throw new IDAOpException("Cannot insert string beyond position ".$l." (request was ".$pos.")");}
			return IDA::$port->substring($orig, 0, $pos) . $ins . IDA::$port->substring($orig, $pos);
		}
		else if ($pos < 0)
		{
			if ($pos < -$l) {throw new IDAOpException("Cannot insert string before position -".$l." (request was ".$pos.")");}
			return IDA::$port->substring($orig, 0, $l+$pos) . $ins . IDA::$port->substring($orig, $l+$pos);
		}
		else {return $ins . $orig;}
	}
	
	// Manip/delete
	/** Usage note: the pos can be REVERSE in place of a null to mean the end (never FORWARD though) */
	static function manip_del($orig, $pos, $amt)
	{
		if ($orig === null) {throw new IDAOpException("Delete subject must not be null");}
		$l = IDA::$port->str_len($orig);
		if ($pos === IDA_Manip::$REVERSE) {$pos = $l;}
		if ($pos >= 0)
		{
			if ($pos > $l) {throw new IDAOpException("Cannot delete from string beyond position ".$l." (request was ".$pos.")");}
			if ($amt === IDA_Manip::$REVERSE) {return IDA::$port->substring($orig, $pos);}
			if ($amt === IDA_Manip::$FORWARD) {return IDA::$port->substring($orig, 0, $pos);}
			if ($amt > 0)
			{
				if (($pos + $amt) > $l) {throw new IDAOpException("Cannot delete more than ".($l-$pos)." beyond position ".$pos." (request was ".$amt.")");}
				return IDA::$port->substring($orig, 0, $pos) . IDA::$port->substring($orig, $pos+$amt);
			}
			if ($amt < 0)
			{
				if (($pos + $amt) < 0) {throw new IDAOpException("Cannot delete more than ".$pos." before position ".$pos." (request was ".(-$amt).")");}
				return IDA::$port->substring($orig, 0, $pos+$amt) . IDA::$port->substring($orig, $pos);
			}
			return $orig;
		}
		else //if ($pos < 0)
		{
			if ($pos < -$l) {throw new IDAOpException("Cannot delete from string before position -".$l." (request was ".$pos.")");}
			if ($amt === IDA_Manip::$REVERSE) {return IDA::$port->substring($orig, $pos+$l);}
			if ($amt === IDA_Manip::$FORWARD) {return IDA::$port->substring($orig, 0, $pos+$l);}
			if ($amt > 0)
			{
				if (($pos+$l + $amt) > $l) {throw new IDAOpException("Cannot delete more than ".(-$pos)." beyond position ".$pos." (request was ".$amt.")");}
				return IDA::$port->substring($orig, 0, $pos+$l) . IDA::$port->substring($orig, $pos+$l+$amt);
			}
			if ($amt < 0)
			{
				if (($pos+$l + $amt) < 0) {throw new IDAOpException("Cannot delete more than ".($l+$pos)." before position ".$pos." (request was ".(-$amt).")");}
				return IDA::$port->substring($orig, 0, $pos+$l+$amt) . IDA::$port->substring($orig, $pos+$l);
			}
			return $orig;
		}
		
	}
	
	// Manip/add
	static function manip_add($func, $pos, $nodeFrom)
	{
		foreach ($nodeFrom->asList() as $ins)
		{
			$func($ins->copy(), $pos);
			if ($pos >= 0) {$pos += 1;}
		}
	}
	
	// Manip/rem
	static function manip_rem_pos($opNode, $func, $size, $pos, $amt)
	{
		$n = $size();
		if ($pos === IDA_Manip::$REVERSE) {$pos = $n;}
		
		if ($amt === IDA_Manip::$FORWARD) {$amt = $n - $pos;}
		else if ($amt === IDA_Manip::$REVERSE) {$amt = -$pos;}
		
		try
		{
			if ($amt > 0)
			{
				for ($i=$pos+$amt-1; $i>=$pos; $i-=1)
				{
					$func($i);
				}
			}
			else if ($amt < 0)
			{
				for ($i=$pos-1; $i>=$pos+$amt; $i-=1)
				{
					$func($i);
				}
			}
			
		}
		catch (IDAOpException $opex) {throw new IDAFormatException($opNode, $opex->getMessage());}
	}
	static function manip_rem_item($func, $nodeFrom)
	{
		foreach ($nodeFrom->asList() as $rem)
		{
			$func($rem);
		}
	}
	
// Manip end
}

// RX begin
class IDA_RX
{
	protected $refs = array();
	protected $has_refTarget = false;
	
	// Data source
	private $inStr;
	function __construct($inStr)
	{
		$this->inStr = $inStr;
	}
	
	// RX basic input
	private $c1, $c2;
	private $has_c1 = false, $has_c2 = false, $eof = false;
	public $ln = 1;
	public $lnBuf = "";
	public function read()
	{
		if (!$this->has_c1) $this->more();
		return $this->has_c1 ? $this->read1() : null;
	}
	
	private function buf($c)
	{
		if ($c == "\n") {$this->ln+=1; $this->lnBuf = "";} else {$this->lnBuf .= $c;}
	}
	/** Assumes has_c1 */
	private function read1()
	{
		$output = $this->c1; $this->c1 = $this->c2; $this->has_c1 = $this->has_c2; $this->has_c2 = false;
		$this->more2();
		$this->buf($output);
		return $output;
	}
	/** Assumes has_c2 */
	private function skip2()
	{
		if ($this->has_c1) $this->buf($this->c1);
		if ($this->has_c2) $this->buf($this->c2);
		$this->has_c1 = $this->has_c2 = false;
		if ($this->more1()) return;
		$this->more2();
	}	
	/** Check has_c1 first */
	public function shift()
	{
		$output = $this->c1;
		$this->read();
		return $output;
	}
	private function more()
	{
		if ($this->has_c2 || $this->eof) return;
		if (!$this->has_c1 && $this->more1()) return;
		if (!$this->has_c2 && $this->more2()) return;
	}
	/** Returns eof */
	private function more1()
	{
		if ($this->eof) return true;
		$b = fgetc($this->inStr);
		if ($b === false || ord($b) == 4) return $this->eof = true;
		$this->c1 = $b;
		$this->has_c1 = true;
		return false;
	}
	/** Returns eof */
	private function more2()
	{
		if ($this->eof) return true;
		$b = fgetc($this->inStr);
		if ($b === false || ord($b) == 4) return $this->eof = true;
		$this->c2 = $b;
		$this->has_c2 = true;
		return false;
	}
	
	// RX stream advancement
	function comment()
	{
		if (!$this->has_c1) return false;
		if (IDA::comment1($this->c1)) return true;
		return $this->has_c2 && IDA::comment2($this->c1, $this->c2);
	}
	function adv($checkLn = false)
	{
		$this->more();
		while ($this->has_c1)
		{
			if ($checkLn ? $this->whiteTxt($this->c1) : $this->white($this->c1)) {$this->read1();}
			else if ($this->c1 == "#") {$this->read1(); $this->advPast("\n");}
			else if ($this->has_c2)
			{
				if ($this->c1 == "/" && $this->c2 == "/")
				{
					$this->skip2(); $this->advPast("\n");
				}
				else if ($this->c1 == "/" && $this->c2 == "*")
				{
					$this->skip2(); $this->advPast("*", "/");
				}
				else if ($this->c1 == "<" && $this->c2 == "?") // XML header compatibility
				{
					$this->skip2(); $this->advPast("?", ">");
				}
				else if ($this->c1 == "<" && $this->c2 == "!") // XML comment compatibility
				{
					$this->skip2();
					do {$this->advPast("-", "-");} while ($this->has_c1 && !$this->c1e(">")); $this->read1();
				}
				else {return;}
			}
			else {return;}
		}
		
	}
	function advPast1($c)
	{
		$this->more();
		while ($this->has_c1 && $this->c1 != $c) {$this->read1();}
		$this->read1();
	}
	function advPast2($cx, $cy)
	{
		$this->more();
		while ($this->has_c2 && ($this->c1 != $cx || $this->c2 != $cy)) {$this->read1();}
		$this->skip2();
	}
	function advPast($c, $c2=null)
	{
		if ($c2 === null) {$this->advPast1($c);} else {$this->advPast2($c, $c2);}
	}
	
	function abrupt($checkLn)
	{
		$this->adv($checkLn);
		return !$this->has_c1;
	}
	
	// RX scan bytes
	function c1e($c) {return $this->has_c1 && ($this->c1 == $c);}
	function c1n($c) {return $this->has_c1 && ($this->c1 != $c);}
	function c2e($c) {return $this->has_c2 && ($this->c2 == $c);}
	function c2n($c) {return $this->has_c2 && ($this->c2 != $c);}
	function whiteLn($c) {return $c == "\n";}
	function whiteTxt($c) {return ($c <= " ") && ($c != "\n");}
	function white($c) {return $c <= " ";}
	
	// RX scan numbers
	static function num0($c) {return (($c>="0") && ($c<="9")) || ($c == "-") || ($c == "+") || ($c == ".");}
	static function num1($c) {return IDA_RX::num0($c) || ($c == "e") || ($c == "E");}
	function get_num()
	{
		$output = "";
		
		if ($this->has_c2 && ($this->c1=="0") && ($this->c2=="x"))
		{
			$output = "0x"; $this->skip2();
			while ($this->has_c1 && IDA_RX::hex($this->c1)) {$output .= $this->shift();}
		}
		else if ($this->has_c2 && ($this->c1=="0") && ($this->c2=="b"))
		{
			$output = "0b"; $this->skip2();
			while ($this->has_c1 && IDA_RX::bin($this->c1)) {$output .= $this->shift();}
		}
		else
		{
			$output = "" . $this->shift();
			while ($this->has_c1 && IDA_RX::num1($this->c1)) {$output .= $this->shift();}
		}
		
		return $output;
	}
	static function hex($c) {return (($c>="0") && ($c<="9")) || (($c>="a") && ($c<="f")) || (($c>="A") && ($c<="F"));}
	static function bin($c) {return ($c=="0") || ($c=="1");}
	
	// RX scan opts
	function opt() {return $this->c1e('~');}
	/** Consumes the /, reads the opt, then advances */
	function get_opt()
	{
		$this->read1();
		$output = $this->get_id();
		$this->adv();
		return $output;
	}
	
	// RX scan unicode
	public static function is_u8_marker($c) {return (ord($c) & 0xc0) == 0xc0;}
	public static function is_u8_data($c) {return ($c & 0xc0) == 0x80;}
	
	public function shift_u8_data()
	{
		$data = ord($this->shift());
		if (!IDA_RX::is_u8_data($data)) throw new IDAFormatException($this, "Invalid unicode data");
		return $data & 0x3f;
	}
	public function get_u8()
	{
		$codePoint = 0;
		$marker = ord($this->shift());
		
		if (($marker & 0xe0) == 0xc0) // 2-byte
		{
			$codePoint = ($marker & 0x1f) << 6;
			$codePoint |= $this->shift_u8_data();
			if ($codePoint < 0x0080) throw new IDAFormatException($this, "Overlong unicode character");
		}
		else if (($marker & 0xf0) == 0xe0) // 3-byte
		{
			$codePoint = ($marker & 0x0f) << 12;
			$codePoint |= $this->shift_u8_data() << 6;
			$codePoint |= $this->shift_u8_data();
			if ($codePoint < 0x0800) throw new IDAFormatException($this, "Overlong unicode character");
		}
		else if (($marker & 0xf8) == 0xf0) // 4-byte
		{
			$codePoint = ($marker & 0x07) << 18;
			$codePoint |= $this->shift_u8_data() << 12;
			$codePoint |= $this->shift_u8_data() << 6;
			$codePoint |= $this->shift_u8_data();
			if ($codePoint < 0x10000) throw new IDAFormatException($this, "Overlong unicode character");
		}
		else throw new IDAFormatException($this, "Invalid unicode marker");
		
		return $this->to_u8($codePoint);
	}
	public function to_u8($codePoint)
	{
		if ($codePoint < 0x80)
		{
			$output = "_";
			$output[0] = chr($codePoint);
			return $output;
		}
		else if ($codePoint < 0x800)
		{
			$output = "__";
			$output[0] = chr(0xc0 | (($codePoint >> 6) & 0x1f));
			$output[1] = chr(0x80 | ($codePoint & 0x3f));
			return $output;
		}
		else if ($codePoint < 0x10000)
		{
			if ($codePoint >= 0xd800 && $codePoint < 0xe000) throw new IDAFormatException($this, "Invalid unicode surrogate");
			$output = "___";
			$output[0] = chr(0xe0 | (($codePoint >> 12) & 0x0f));
			$output[1] = chr(0x80 | (($codePoint >> 6) & 0x3f));
			$output[2] = chr(0x80 | ($codePoint & 0x3f));
			return $output;
		}
		else if ($codePoint < 0x110000)
		{
			$output = "____";
			$output[0] = chr(0xf0 | (($codePoint >> 18) & 0x07));
			$output[1] = chr(0x80 | (($codePoint >> 12) & 0x3f));
			$output[2] = chr(0x80 | (($codePoint >> 6) & 0x3f));
			$output[3] = chr(0x80 | ($codePoint & 0x3f));
			return $output;
		}
		else throw new IDAFormatException($this, "Invalid unicode range");
	}
	
	// RX scan escapes
	static function is_esc($c) {return $c == "\\";}
	function esc() {return $this->has_c1 && IDA_RX::is_esc($this->c1);}
	
	function shift_hex($ex_txt)
	{
		$h = ord($this->shift());
		if (0x30 <= $h && $h <= 0x39) return $h - 0x30;
		if (0x41 <= $h && $h <= 0x46) return 10 + $h - 0x41;
		if (0x61 <= $h && $h <= 0x66) return 10 + $h - 0x61;
		throw new IDAFormatException($this, "Invalid ".$ex_txt." escape sequence");
	}
	function get_esc_byte($ex_txt)
	{
		return ($this->shift_hex($ex_txt) << 4) | $this->shift_hex($ex_txt);
	}
	function get_esc()
	{
		$this->read1();
		if (!$this->has_c1) {throw new IDAFormatException($this, "Incomplete escape sequence");}
		$esc = $this->shift();
		switch ($esc)
		{
			case "0": return "\0";
			case "n": return "\n";
			case "r": return "\r";
			case "t": return "\t";
			case "x":
			{
				if (!$this->has_c2) {throw new IDAFormatException($this, "Incomplete byte escape sequence");}
				return $this->to_u8($this->get_esc_byte("byte"));
			}
			case "u":
			{
				//if (!function_exists("mb_chr")) {throw new IDAFormatException($this, "The mbstring extension is required for reading unicode sequences");}
				if (!$this->has_c2) {throw new IDAFormatException($this, "Incomplete unicode escape sequence");}
				$code = $this->get_esc_byte("unicode") << 8;
				if (!$this->has_c2) {throw new IDAFormatException($this, "Incomplete unicode escape sequence");}
				$code |= $this->get_esc_byte("unicode");
				return $this->to_u8($code);
			}
			case "U":
			{
				//if (!function_exists("mb_chr")) {throw new IDAFormatException($this, "The mbstring extension is required for reading unicode sequences");}
				if (!$this->has_c2) {throw new IDAFormatException($this, "Incomplete unicode extended escape sequence");}
				$code = $this->get_esc_byte("unicode extended") << 24;
				if (!$this->has_c2) {throw new IDAFormatException($this, "Incomplete unicode extended escape sequence");}
				$code |= $this->get_esc_byte("unicode extended") << 16;
				if (!$this->has_c2) {throw new IDAFormatException($this, "Incomplete unicode extended escape sequence");}
				$code |= $this->get_esc_byte("unicode extended") << 8;
				if (!$this->has_c2) {throw new IDAFormatException($this, "Incomplete unicode extended escape sequence");}
				$code |= $this->get_esc_byte("unicode extended");
				return $this->to_u8($code);
			}
			
			default: return $esc;
		}
	}
	
	// RX scan IDs
	function id0() {return IDA::id0($this->c1);}
	function id1() {return IDA::id1($this->c1);}
	function get_tok()
	{
		if (IDA_RX::is_str_d($this->c1)) {return $this->get_str_d($this->c1);}
		else
		{
			$output = "";
			while (!$this->comment())
			{
				if ($this->esc()) {$output .= $this->get_esc();}
				else if ($this->has_c1 && !$this->white($this->c1)) {$output .= $this->shift();}
				else break;
			}
			return $output;
		}
	}
	
	public $id_symbolic = false;
	function get_id($untilLn = false)
	{
		$output = "";
		$this->id_symbolic = true;
		$holdMin = -1;
		$holdMax = -1;
		
		// Must not end with - or + , that is reserved for boolean false or true
		while ($this->has_c1 && !$this->comment())
		{
			$ptr = $this->id_symbolic ? strlen($output) : $holdMin;
			$hold = false;
			
			if ($hold = IDA_RX::is_esc($this->c1)) {$output .= $this->get_esc();}
			else if ($hold = IDA_RX::is_str_d($this->c1)) {$output .= $this->get_str_d($this->c1);}
			else if ($hold = IDA_RX::is_var($this->c1)) {$output .= $this->get_var();}
			else if ($hold = IDA_RX::is_u8_marker($this->c1)) {$output .= $this->get_u8();}
			else if ((IDA::id1($this->c1) || ($untilLn && ($this->c1 != ";") && !$this->whiteLn($this->c1))) && !(($this->c1 == "-") && (!$this->has_c2 || !IDA::id1($this->c2)))) {$output .= $this->shift();}
			else break;
			
			if ($hold)
			{
				$holdMin = $ptr;
				$holdMax = strlen($output);
				$this->id_symbolic = false;
			}
			
		}
		
		if ($untilLn)
		{
			if ($this->id_symbolic) {return IDA::$port->trim($output);}
			else return IDA::$port->ltrim(substr($output,0,$holdMin)) . substr($output, $holdMin, $holdMax) . IDA::$port->rtrim(substr($output,$holdMax));
		}
		else return $output;
	}
	static function is_id_start($c) {return IDA_RX::is_str_d($c) || IDA_RX::is_var($c) || IDA_RX::is_esc($c) || IDA_RX::is_u8_marker($c) || IDA::id0($c);}
	function c1_id_start() {return $this->has_c1 && IDA_RX::is_id_start($this->c1);}
	
	static function is_var($c) {return $c == "$";}
	function get_var()
	{
		$this->read();
		$output = "$";
		
		$endChar = "\0";
		if ($this->c1e("{")) {$endChar = "}";}
		// TODO: Other common uses of $(...)
		
		if ($endChar != "\0") $output .= $this->read1() . $this->get_to($endChar) . $endChar;
		return $output;
	}
	
	function xml_att_key()
	{
		return IDA_RX::is_id_start($this->c1) ? $this->get_id() : $this->get_pr(function($c) {return $c != "=" && $c != "/" && $c != ">" && !$this->white($c);});
	}
	
	// RX scan strings
	function c1_str() {return $this->has_c1 && IDA_RX::is_str($this->c1);}
	function c1_tok() {return $this->has_c1 && !$this->white($this->c1);}
	static function is_str_d($c) {return $c == "\"" || $c == "`" || $c == "'";}
	static function is_str($c) {return IDA_RX::is_str_d($c) || IDA::id1($c);}
	
	function get_str()
	{
		return IDA_RX::is_str_d($this->c1) ? $this->get_str_d($this->c1) : $this->get_id();
	}
	function get_str_d($delim)
	{
		$output = "";
		if ($this->c1e($delim))
		{
			$this->read1();
			while ($this->c1n($delim)) $output .= $this->esc() ? $this->get_esc() : (IDA_RX::is_u8_marker($this->c1) ? $this->get_u8() : $this->read1());
			$this->read();
		}
		return $output;
	}
	
	// RX scan logic
	function get_pr($p)
	{
		$output = "";
		while ($this->has_c1 && $p($this->c1)) {$output .= $this->shift();}
		return $output;
	}
	function get_prior_to($delim)
	{
		$output = $this->get_pr(function($c) use ($delim) {return ($c != $delim);});
		return $output;
	}
	function get_to($delim)
	{
		$output = $this->get_prior_to($delim);
		$this->read();
		return $output;
	}
	
	// RX class iteration
	// Empty iterator
	static $CLASS_NONE_OBJ = null;
	static function CLASS_NONE()
	{
		if (IDA_RX::$CLASS_NONE_OBJ === null) {IDA_RX::$CLASS_NONE_OBJ = IDA::$port->empty_i();}
		return IDA_RX::$CLASS_NONE_OBJ;
	}
	// Infinite iterator of blank nodes
	static $CLASS_ARRAY_OBJ = null;
	static function CLASS_ARRAY()
	{
		if (IDA_RX::$CLASS_ARRAY_OBJ === null) {IDA_RX::$CLASS_ARRAY_OBJ = IDA::$port->repeat_i(new IDA());}
		return IDA_RX::$CLASS_ARRAY_OBJ;
	}
	// Specific class node iterator
	static function classOf($itemClassList)
	{
		return ($itemClassList != null) ? IDA::$port->list_i($itemClassList) : IDA_RX::CLASS_NONE();
	}
	
	// RX entry
	/** Returns the output argument */
	static function inObj($output, $inStr)
	{
		$rx = new IDA_RX($inStr);
		$globalScope = new IDA();
		$entry = false;
		$entryNode = (new IDA())->rxScope($globalScope)->setIncludes($output->rxIncludePwd());
		$entryNode->rxInputList($output->inputListDepth, $output->inputListLength);
		$entryNode->copyOptions($output);
		
		$rx->adv();
		while ($rx->opt()) switch ($rx->get_opt())
		{
			case "": $entry = true; break;
		}
		$node = IDA_RX::fill($entryNode, $rx, "\0", "\0", null, IDA_RX::CLASS_NONE(), 0);
		
		while ($rx->has_c1 && $node != null)
		{
			$nextNode = (new IDA())->rxScope($globalScope)->setIncludes($output->rxIncludePwd());
			$nextNode->rxInputList($output->inputListDepth, $output->inputListLength);
			$nextNode->copyOptions($output);
			while ($rx->opt()) switch ($rx->get_opt())
			{
				case "":
				{
					if (!$entry) {$entry = true; $entryNode = $nextNode;}
					break;
				}
			}
			
			$last = IDA_RX::fill($nextNode, $rx, "\0", "\0", $node, IDA_RX::CLASS_NONE(), 0);
			$node->linkNext($nextNode);
			$node = $last;
		}
		
		$output->reset();
		$rx->transfer_nodes($entryNode, $output);
		$output->walk(function($x) {$x->rxScope(null);}, true, true);
		
		if ($rx->has_refTarget) $output->walk(function($x) use(&$rx) {$rx->put_ref($x);}, true, true);
		
		return $output;
	}
	function transfer_nodes(IDA $src, IDA $dst)
	{
		// Name
		$dst->copyName($src);
		
		// Parameters
		if ($src->hasParams())
		{
			$dst->setParams();
			foreach ($src->getParamsOr() as $p) $dst->addParam($p);
			$src->clearParams();
		}
		else $dst->clearParams();
		
		// Content
		if ($src->isList())
		{
			$dst->putList();
			foreach ($src->asListOr() as $i) $dst->addItem($i);
			$src->putNull();
		}
		else $dst->copyContent($src);
		
		// Other
		$dst->copyOptions($src);
		if ($src->hasRefName()) {$this->refs[$src->getRefName()] = $dst; $dst->setRefName($src->getRefName());}
		
		$dst->link($src->getPrevOr(), $src->getNextOr());
	}
	function put_ref($el)
	{
		$t = $el->getRefTarget();
		if ($t !== null)
		{
			if (!array_key_exists($t, $this->refs)) throw new IDAFormatException($el, "Identifier not defined");
			$el->putRef($this->refs[$t]);
		}
	}
	
	// RX fill
	/** Decodes element nodes read from the RX, using the supplied object as a container for the first such element.
	 * Adjacent elements will be created as needed and linked to the first, accessible using its getNext method.<br>
	 * @return The last consecutive element decoded, or null if no elements were present.
	 * @param node A container into which the first decoded element should be written.
	 * @param rx The serial input to be read.
	 * @param paren_l The character which caused <i>fill</i> to be called recursively, or zero to admit no such structure.
	 * @param paren_r The character which should mark the end of a recursive, structured call to <i>fill</i>.
	 * @param prev An element to be considered prior to the decoded element for the purpose of interpreting <i>as-above</i>
	 * markup, or null if no such element exists.
	 * @param listClass A class element to use for copying implicit element names, or an empty iterator (such as CLASS_NONE).
	 * @param depth The effective recursion level.
	 * @throws IOException If a error occurs while reading input bytes.
	 * @throws IDAFormatException If the parser encounters a syntax error, or if the markup contained an invalid
	 * <i>manipulation</i> which would throw {@link IDAOpException} if called programmatically. */
	static function fill($node, $rx, $paren_l, $paren_r, $prev, $listClass, $depth)
	{
		
		// RX fill/state
		$jsonVal = false;
		$forceContent = false; // Used for : and = in an array context
		$needName = true;
		$prelimName = null;
		$nameSet = false;
		$needContent = true;
		$checkDelim = true; // For ; or , between nodes
		$itemClass = null;
		$output = null;
		$lcn = $listClass->hasNext();
		
		// RX read
		while ($rx->has_c1 && $needContent)
		{
			
			// RX read/string implicit
			// This should be first
			// NOTE: By testing this prior to num(), names/strings beginning "e" will not be a number
			if ($rx->c1_id_start())
			{
				$output = $node;
				
				if (!$needName)
				{
					if (($listClass == IDA_RX::CLASS_ARRAY()) && !$forceContent)
					{
						$nameSet = true;
						$needName = false;
						$node->putStr($prelimName);
					}
					else
					{
						// String content
						// Potentially also true/false/null if JSON-style following a :
						$s = $rx->get_id($depth == 0);
						
						if ($jsonVal && $rx->id_symbolic)
						{
							if ("true" == $s) {$node->putTrue();}
							else if ("false" == $s) {$node->putFalse();}
							else if ("null" == $s) {$node->putNull();}
							else {$node->putStr($s);}
						}
						else {$node->putStr($s);}
					}
					
					$needContent = false;
				}
				else
				{
					$prelimName = $rx->get_id();
					if ($rx->abrupt($paren_r == "\n")) {$node->putStr($prelimName); $prelimName = null; $needContent = false;}
					else {$needName = false;}
				}
			}
			
			// RX read/delim explicit
			else if ($rx->c1e(";") || $rx->c1e(","))
			{
				if (!(($rx->c1 == ",") && ($depth == 0)))
				{
					// Consume delimiter unless using CSV compatibility
					$rx->read1();
					$checkDelim = false;
				}
				
				$node->putNull();
				$needContent = false;
				$output = $node;
			}
			
			// RX read/delim implicit
			else if ($rx->c1e("}") || $rx->c1e("]") || $rx->c1e(")"))
			{
				if ($paren_l != ':')
				{
					if ($rx->c1e("}")) {if ($paren_l != "{") throw new IDAFormatException($rx, "List mismatch");}
					else if ($rx->c1e("]")) {if ($paren_l != "[") throw new IDAFormatException($rx, "Array mismatch");}
					else /*if ($rx->c1e(")"))*/ {if ($paren_l != "(") throw new IDAFormatException($rx, "Parameters mismatch");}
				}
				
				if (!$nameSet)
				{
					if (!$needName && $lcn) {$lcn = false; $node->setName($listClass->next()->getNameOr()); $node->putStr($prelimName);}
					else {$node->setName($prelimName);}
					$nameSet = true;
				}
				else
				{
					$node->putNull();
				}
				
				$needContent = false;
				$checkDelim = false;
			}
			
			// RX read/as-above
			else if ($rx->c1e("^"))
			{
				if ($prev === null) {throw new IDAFormatException($rx, "No previous element");}
				
				if ($needName)
				{
					$prelimName = $prev->getNameOr();
					$needName = false;
				}
				else
				{
					$node->copyContent($prev);
					$needContent = false;
				}
				
				if ($rx->c2e("^")) {$node->copyParams($prev); $rx->skip2();} else {$rx->read1();}
				$output = $node;
			}
			
			// RX read/identifier definition
			else if ($rx->c1e('*'))
			{
				$rx->read1();
				$key = ($rx->c1_str()) ? $rx->get_str() : "";
				$rx->adv();
				
				if (array_key_exists($key, $rx->refs)) throw new IDAFormatException($rx, "Identifier already defined");
				$rx->refs[$key] = $node;
				$node->setRefName($key);
				
				$output = $node;
			}
			
			// RX read/identifier reference
			else if ($rx->c1e('&'))
			{
				$rx->read1();
				$key = ($rx->c1_str()) ? $rx->get_str() : "";
				$rx->adv();
				
				$node->setRefTarget($key);
				$rx->has_refTarget = true;
				
				$output = $node;
				$needContent = false;
			}
			
			// RX read/XML
			else if ($rx->c1e("<")) // XML compatibility
			{
				$output = $node;
				
				if (!$needName)
				{
					if ($listClass == IDA_RX::CLASS_ARRAY() && !$forceContent)
					{
						$nameSet = true;
						$needName = false;
						$node->putStr($prelimName);
						$needContent = false;
					}
					else if ($paren_r == "<") {$needContent = false;}
					else throw new IDAFormatException($rx, "Cannot nest XML element header");
				}
				else
				{
					$rx->read1();
					$rx->adv();
					if ($rx->has_c1 && $rx->id1())
					{
						$node->setName($rx->get_id());
						$needName = false;
						$nameSet = true;
					}
					else {throw new IDAFormatException($rx, "Invalid XML element name");}
					$rx->adv();
					
					while ($rx->has_c1 && $needContent) // XML inners
					{
						if ($rx->c1e(">")) // XML child elements
						{
							$rx->read1();
							$l = [];
							$last = null;
							while ($rx->has_c1 && $needContent)
							{
								if ($rx->c1e("<"))
								{
									if ($rx->c2e("/")) // End node
									{
										$rx->skip2(); $rx->adv();
										$end = trim($rx->get_to(">"));
										if (!$node->isName($end)) {throw new IDAFormatException($rx, "Incorrect XML element footer");}
										$needContent = false;
									}
									else if ($rx->c2e("!")) // XML comment
									{
										$rx->skip2();
										do {$rx->advPast("-", "-");} while ($rx->c1n(">")); $rx->read1();
									}
									else // Child element
									{
										$el = $node->rxNew();
										IDA_RX::fill($el, $rx, "<", ">", $last, IDA_RX::CLASS_NONE(), $depth+1);
										$last = $el;
										$l[] = $el;
									}
								}
								else
								{
									if ($node->opts->rxXmlReparseStrings)
									{
										$rx->adv();
										
										$el = $node->rxNew();
										
										$last = IDA_RX::fill($el, $rx, ">", "<", $last, IDA_RX::CLASS_NONE(), 0);
										while ($el !== null && ($node->inputListLength < 0 || $node->inputListLength > count($l)))
										{
											$l[] = $el;
											$el = $el->getNextOr();
										}
									}
									else
									{
										$content = $rx->get_prior_to("<");
										
										if (strlen(trim($content)) > 0)
										{
											$el = (new IDA())->putStr($content);
											$l[] = $el;
										}
										
									}
								}
								
							}
							
							$node->putList($l);
						}
						else if ($rx->c1e("/") && $rx->c2e(">")) // Empty XML node
						{
							$rx->skip2();
							$rx->adv();
							$node->putNull();
							$needContent = false;
						}
						else // XML attrib
						{
							$attKey = $rx->xml_att_key();
							while (IDA_RX::is_id_start($rx->c1) && ($rx->c1n("/") || $rx->c2n(">"))) $attKey .= $rx->xml_att_key();
							$rx->adv();
							$attValue = null;
							
							if ($rx->c1e("="))
							{
								$rx->read1();
								$attValue = $rx->get_str();
								$rx->adv();
							}
							else
							{
								$attValue = null;
							}
							
							$att = (new IDA($attKey))->putStr($attValue);
							$node->addParam($att);
							
						}
					}
				}
			}
			
			// RX read/params
			else if ($rx->c1e("("))
			{
				if ($node->hasParams())
				{
					$needContent = false;
				}
				else
				{
					$output = $node;
					
					// Parameters also cause content delim
					$forceContent = true;
					
					$rx->read1();
					$rx->adv();
					if ($rx->c1e("^"))
					{
						if ($prev === null) {throw new IDAFormatException($rx, "No previous element");}
						$node->copyParams($prev);
						$rx->read1();
						$rx->adv();
					}
					else
					{
						$node->setParams();
					}
					
					$last = null;
					$cItems = IDA_RX::classOf(($itemClass !== null) ? $itemClass->getParamsOr() : null);
					while ($rx->c1n(")"))
					{
						$param = $node->rxNew();
						$last = IDA_RX::fill($param, $rx, "(", ")", $last, $cItems, $depth+1);
						if ($last != null) while ($param !== null)
						{
							// Snapshot the next before addParam breaks it
							$param_next = $param->getNextOr();
							$node->addParam($param);
							$param = $param_next;
						}
					}
					while ($cItems->hasNext())
					{
						$param = $cItems->next()->copy()->rxScope($node);
						$node->addParam($param);
					}
					$rx->read();
				}
			}
			
			// RX read/content-delim
			else if (($rx->c1e(":") && !$needName) || $rx->c1e("="))
			{
				if ($rx->read1() == ":") {$jsonVal = true;}
				$needName = false;
				$nameSet = true;
				$node->setName($prelimName);
				$forceContent = true;
				$output = $node;
			}
			
			// RX read/null
			else if ($rx->c1e("?"))
			{
				$output = $node;
				
				if ($needName)
				{
					$rx->read1();
					$prelimName = null;
					$needName = false;
				}
				else
				{
					if (($listClass == IDA_RX::CLASS_ARRAY()) && !$forceContent)
					{
						$nameSet = true;
						$needName = false;
						$node->putStr($prelimName);
					}
					else
					{
						$rx->read1();
						$node->putNull();
					}
					$needContent = false;
				}
			}
			
			// RX read/number
			else if (IDA_RX::num0($rx->c1))
			{
				if (!$needName && ($listClass == IDA_RX::CLASS_ARRAY()) && !$forceContent)
				{
					$nameSet = true;
					$needName = false;
					$node->putStr($prelimName);
				}
				else
				{
					$s = $rx->get_num();
					$num = $s;
					$unit = "";
					$den = "1";
					
					// Check for denominator
					if ($rx->c1e("/"))
					{
						$rx->read1();
						
						if (!$rx->has_c1 || !IDA_RX::num0($rx->c1)) {throw new IDAFormatException($rx, "Missing denominator");}
						$den = $rx->get_num();
						$s .= "/".$den;
					}
					
					// Check for unit
					if ($rx->has_c1 && $rx->id0())
					{
						$unit = $rx->get_id();
						$s .= $unit;
					}
					
					switch ($s)
					{
						// Bools with +, -
						case "+": $node->putTrue(); break;
						case "-": $node->putFalse(); break;
						
						// Actual number
						default:
						{
							$rx->testNumber($num, "numerator");
							if ($den != "1") {$rx->testNumber($den, "denominator");}
							$node->putNum($num, $den, $unit);
						}
					}
				}
				$needContent = false;
				$output = $node;
			}
			
			// RX read/list array
			else if ($rx->c1e("["))
			{
				$output = $node;
				
				if (!$needName && ($listClass == IDA_RX::CLASS_ARRAY()) && !$forceContent)
				{
					$nameSet = true;
					$needName = false;
					$node->putStr($prelimName);
				}
				else
				{
					$rx->read1();
					$rx->adv();
					$listObj = array();
					
					if ($node->inputListDepth != 0)
					{
						$last = null;
						while ($rx->c1n("]"))
						{
							$el = $node->rxNew();
							$last = IDA_RX::fill($el, $rx, "[", "]", $last, IDA_RX::CLASS_ARRAY(), $depth+1);
							if ($last != null) while ($el !== null && ($node->inputListLength < 0 || $node->inputListLength > count($listObj)))
							{
								$listObj[] = $el;
								$el = $el->getNextOr();
							}
						}
					}
					else {IDA_RX::discard($rx, "[", "]");}
					
					$rx->read();
					$node->putList($listObj);
				}
				$needContent = false;
			}
			
			// RX read/list object
			else if ($rx->c1e("{"))
			{
				$output = $node;
				
				if (!$needName && ($listClass == IDA_RX::CLASS_ARRAY()) && !$forceContent)
				{
					$nameSet = true;
					$needName = false;
					$node->putStr($prelimName);
				}
				else
				{
					$rx->read1();
					$rx->adv();
					$listObj = array();
					
					if ($node->inputListDepth != 0)
					{
						$last = null;
						$cItems = IDA_RX::classOf(($itemClass !== null) ? $itemClass->asListOr() : null);
						while ($rx->c1n("}"))
						{
							$el = $node->rxNew();
							$last = IDA_RX::fill($el, $rx, "{", "}", $last, $cItems, $depth+1);
							if ($last != null) while ($el !== null && ($node->inputListLength < 0 || $node->inputListLength > count($listObj)))
							{
								$listObj[] = $el;
								$el = $el->getNextOr();
							}
						}
						while ($cItems->hasNext() && ($node->inputListLength < 0 || $node->inputListLength > count($listObj)))
						{
							$el = $cItems->next()->copy()->rxScope($node);
							$listObj[] = $el;
						}
					}
					else {IDA_RX::discard($rx, "{", "}");}
					
					$rx->read();
					$node->putList($listObj);
				}
				$needContent = false;
			}
			
			// RX read/class definition
			else if ($rx->c1e(':') && $rx->c2e(':'))
			{
				$rx->skip2();
				$key = ($rx->c1_str()) ? $rx->get_str() : "";
				$rx->adv();
				
				$el = $node->rxNew();
				IDA_RX::fill($el, $rx, ':', '\0', null, IDA_RX::CLASS_NONE(), $depth+1);
				$node->rxAddItemClass($key, $el);
			}
			
			// RX read/class instance
			else if ($rx->c1e(':'))
			{
				if ($paren_l == ':')
				{
					$needContent = false;
				}
				else
				{
					$rx->read1();
					$key = ($rx->c1_str()) ? $rx->get_str() : "";
					$rx->adv();
					
					$itemClass = $node->rxGetItemClass($key);
					if ($itemClass === null) {throw new IDAFormatException($rx, "Item class \"".$key."\" not found");}
					$prelimName = $itemClass->getNameOr();
					$output = $node;
				}
			}
			
			// RX read/template definition
			else if ($rx->c1e("!"))
			{
				if (!$needName) {throw new IDAFormatException($rx, "Incomplete element prior to template definition");}
				$rx->read1();
				$key = ($rx->c1_str()) ? $rx->get_str() : "";
				$rx->adv();
				
				$itemTemplate = $node->rxNew();
				IDA_RX::fill($itemTemplate, $rx, "\0", "\0", null, IDA_RX::CLASS_NONE(), $depth+1);
				if (!$itemTemplate->isList()) {throw new IDAFormatException($rx, "Template definition must be a list");}
				$node->rxAddTemplate($key, $itemTemplate);
			}
			
			// RX read/template instance
			else if ($rx->c1e("@"))
			{
				// Hand-over to file include for @? and @@ 
				$inc_opt = $rx->c2e("?");
				if ($inc_opt || $rx->c2e("@"))
				{
					if (!$needName) {throw new IDAFormatException($rx, "Incomplete element prior to include");}
					$rx->skip2();
					if (!$rx->c1_tok()) {throw new IDAFormatException($rx, "Missing include path");}
					
					$path = $rx->get_tok();
					$rx->adv();
					
					$abort = false;
					try
					{
						// Platform-specific check (PHP)
						$pwd = $node->rxIncludePwd();
						if ($abort = ($pwd === null)) {throw new IDAOpException("Includes are disabled");}
						
						$chdir = false;
						if (IDA::$port->starts_with($path, "+")) {$chdir = true; $path = substr($path, 1);}
						
						$inc = $path;
						// TODO: Multi-platform solution for handling absolute/relative paths
						// The intention of the following test is:
						// if (!is_absolute_path($inc))
						// The subsequent formulation of an equivalent absolute path also needs attention
						if (!IDA::$port->starts_with($inc, "/")) {$inc = $pwd."/".$path;}
						
						if ($abort = !file_exists($inc)) {throw new IDAOpException("Include path does not exist");}
						if ($abort = !is_file($inc)) {throw new IDAOpException("Include is not a file");}
						if ($abort = !is_readable($inc)) {throw new IDAOpException("Include file has no read access");}
						
						$template_adj = (new IDA())->rxScope($node);
						// TODO: Use actual path system to get path parent
						if ($chdir) throw new IDAFormatException($rx, "File include chdir is NYI");
						$template_adj->setIncludes($chdir ? "Parent-file-NYI" : $node->rxIncludePwd());
						$template_adj->inFile($inc);
						
						$last = IDA_RX::manip($rx, $node, $template_adj->iterateNext());
						
						$nameSet = true;
						$checkDelim = false;
						$needContent = false;
						$output = $last;
						
					}
					catch (IDAOpException $opex)
					{
						// Hold off expending the RX line on an exception when the include was optional
						if (!$inc_opt) throw new IDAFormatException($rx, $opex->getMessage());
						
						// Discard the manipulation node
						if ($abort) {IDA_RX::fill(new IDA(), $rx, "\0", "\0", null, IDA_RX::CLASS_NONE(), $depth+1);}
						
					}
					
				}
				else
				{
					if (!$needName) {throw new IDAFormatException($rx, "Incomplete element prior to template instance");}
					$rx->read1();
					$key = ($rx->c1_str()) ? $rx->get_str() : "";
					$rx->adv();
					
					$itemTemplate = $node->rxGetTemplate($key);
					if ($itemTemplate === null) {throw new IDAFormatException($rx, "Template \"".$key."\" not found");}
					
					$last = IDA_RX::manip($rx, $node, IDA::$port->list_i($itemTemplate->asListOr()));
					
					if ($last != null)
					{
						$nameSet = true;
						$checkDelim = false;
						$needContent = false;
						$output = $last;
					}
					
					
				}
				
			}
			
			// RX read/invalid
			// This should be last
			else
			{
				$c_hex = dechex(ord($rx->c1));
				if (strlen($c_hex) < 2) {$c_hex = "0".$c_hex;}
				throw new IDAFormatException($rx, "Invalid character 0x".$c_hex." '".$rx->c1."'");
			}
			
			// RX read delimiter
			if ($paren_r != ">")
			{
				$csvLn = $paren_r == "\n";
				$rx->adv($csvLn);
				$csvLn &= $rx->c1e("\n");
				if ($checkDelim && ($rx->c1e(";") || $rx->c1e(",") || $csvLn) )
				{
					if ($needContent && $lcn)
					{
						$lcn = false;
						$node->setName($listClass->next()->getNameOr());
						$node->putStr($prelimName);
					}
					else if ($needName && $lcn)
					{
						$lcn = false;
						$node->setName($listClass->next()->getNameOr());
					}
					else if (($rx->c1 == ",") && ($depth == 0))
					{
						// CSV compatibility
						$rx->read1();
						$rx->adv(true);
						
						// Convert to list with new first element
						$first = new IDA();
						$first->rxConfig($node);
						if ($needContent) {$first->putStr($prelimName);} else {$node->copyTo($first);}
						
						// Complete the row
						$listObj = array($first);
						$last = null;
						$cItems = IDA_RX::CLASS_ARRAY();
						while ($rx->c1n("\n"))
						{
							$el = $node->rxNew();
							$last = IDA_RX::fill($el, $rx, "\0", "\n", $last, $cItems, $depth+1);
							if ($last != null) while (($el !== null) && ($node->inputListLength < 0 || $node->inputListLength > count($listObj)))
							{
								$listObj[] = $el;
								$el = $el->getNextOr();
							}
						}
						
						$node->putList($listObj);
						$node->setName(null);
						
					}
					else if ($csvLn)
					{
						// Last CSV item
						if ($needContent) {$node->putStr($prelimName);}
					}
					else if (!$nameSet)
					{
						$node->setName($prelimName);
					}
					
					$nameSet = true;
					$needContent = false;
					
					if (!$csvLn) {$rx->read1(); $rx->adv();}
					
				}
			}
			
			// RX read END
		}
		
		// RX fill/name
		if (!$nameSet)
		{
			if ($needName && $lcn) {$lcn = false; $node->setName($listClass->next()->getNameOr());}
			else {$node->setName($prelimName);}
		}
		
		// RX fill END
		if ($lcn) $listClass->next();
		
		$node->rxClearTemplates();
		$node->rxClearItemClasses();
		return $output;
	}
	
// TODO: RX discard
	// RX number test
	function testNumber($num, $desc)
	{
		// No need to test hex/bin here; invalid characters will have delimited the number, potentially becoming the unit
		if (IDA::$port->starts_with($num, "0x") || IDA::$port->starts_with($num, "0b")) {return;}
		
		$test_num = "/^[\\+\\-]?[0-9]*(?:\\.[0-9]+)?([eE][\\+\\-]?[0-9]+)?$/";
		if (!preg_match($test_num, $num)) {throw new IDAFormatException($this, "Invalid ".$desc." number");}
	}
	
	// RX manip
	/** Return the last node */
	static function manip($rx, $node, $itemTemplate)
	{
		// Manipulations
		$manip = new IDA();
		IDA_RX::fill($manip, $rx, "\0", "\0", null, IDA_RX::CLASS_NONE(), 1);
		if ($manip->isList())
		{
			// TODO: any pre-process or validation of manipulations, or simplify logic if none needed
		}
		else if ($manip->hasContent()) {throw new IDAFormatException($rx, "Template manipulation must be a list");}
		
		return IDA_Manip::manip_iterate($manip, $node, $itemTemplate);
	}
	
// RX end
}

// Output interface
class IDA_Output
{
	private $w;
	
	function __construct($output)
	{
		$this->w = $output;
		$this->queueBuf = "";
	}
	
	protected function send(string $s)
	{
		// PHP 8.1 throws a TypeError when fwrite is given a closed stream
		// This was not documented and may not be intended, though either way it can be caught
		try {if (@fwrite($this->w, $s) !== false) return;}
		catch (Error $ex) {}
		throw new Exception("Stream closed");
	}
	
	private $queueBuf;
	protected function queue($s) {$this->queueBuf .= $s;}
	protected function length() {return strlen($this->queueBuf);}
	protected function reject() {$this->queueBuf = "";}
	protected function accept() {$this->send($this->queueBuf); $this->queueBuf = "";}
	
	
	/** Element will not be null<br>
	 * This will receive the entry element, which is not necessarily the first to be output<br>
	 * The adjacents-flag is provided if any special state is needed to handle multiple elements to follow */
	protected function ida_header(IDA $el, bool $adj) {}
	/** Element will not be null<br>
	 * This is called potentially multiple times if adjacents are enabled */
	protected function ida_element(IDA $el, bool $adj) {}
	/** Called once after all elements are done */
	protected function ida_footer() {}
	
	public function idaWrite(?IDA $el, bool $adj)
	{
		if ($el === null) {return;}
		
		$el->txRefLabel($adj);
		
		$this->ida_header($el, $adj);
		
		if ($adj)
		{
			// Rewind first
			while ($el->hasPrev()) {$el = $el->getPrevOr();}
			
			// Output forward
			while ($el !== null)
			{
				$this->ida_element($el, true);
				$el = $el->getNextOr();
			}
		}
		else {$this->ida_element($el, false);}
		
		$this->ida_footer();
		
		if (@fflush($this->w) === false) throw new Exception("Stream closed");
	}
	
	/** Same as idaWrite, but converts exceptions into a return value instead.<br>
	 * True if success, false if a IO problem occurred */
	public function idaPrint(IDA $el, bool $adj)
	{
		try {$this->idaWrite($el, $adj); return true;}
		catch (Exception $ex) {return false;}
	}
}

// TX begin
class IDA_TX extends IDA_Output
{
	
	// TX defaults
	private static $BREVITY_ALL = 3;
	private static $BREVITY_PART = 6;
	private static $ONE_LINE_MAX = 132;
	
	// TX instance
	private $entry;
	private $rootAsAboveAll;
	private $rootAsAboveFirst;
	private $rootAsAbovePart;
	private $rootAsAboveName;
	
	private $optsOverride = false;
	public function setOptsOverride($ov) {$this->optsOverride = $ov; return $this;}
	private function opts($el) {return $this->optsOverride ? IDAOptions::$defaultOptions : $el->opts;}
	
	function __construct($out_func)
	{
		parent::__construct($out_func);
		$this->entry = null;
		$this->rootAsAboveAll = false;
		$this->rootAsAboveFirst = null;
		$this->rootAsAbovePart = false;
		$this->rootAsAboveName = null;
	}
	
	// TX interface
	protected function ida_header(IDA $el, bool $adj)
	{
		$this->entry = $el;
	}
	protected function ida_element(IDA $el, bool $adj)
	{
		// Explicit entry point?
		if ($adj && $this->entry == $el && $el->hasPrev()) {$this->send("~ ");}
		
		// As-above all?
		if ($this->rootAsAboveFirst == null)
		{
			$this->rootAsAboveFirst = $el;
			$this->rootAsAboveAll = IDA_TX::as_above_all_test($el);
		}
		
		// Main output wrapped with name as-above part
		if ($this->rootAsAbovePart && !$el->isName($this->rootAsAboveName)) {$this->rootAsAbovePart = false;}
		$this->ida($el, "", false, $this->rootAsAbovePart || ($adj && $this->rootAsAboveAll && $el->hasPrev()), false, null);
		$this->rootAsAboveName = $el->getNameOr();
		if (!$this->rootAsAbovePart) $this->rootAsAbovePart = IDA_TX::as_above_part_test($el);
	}
	
	// TX structure filter
	function has_inner_structure($first)
	{
		for ($outer = $first; $outer != null; $outer = $outer->getNextOr())
		{
			if ($outer->isList()) foreach ($outer->asListOr() as $inner)
			{
				if ($inner->hasParams() || $inner->isList()) return true;
			}
		}
		return false;
	}
	
	// TX main begin
	private function ida(IDA $node, string $indent, bool $parameters, bool $nameAsAbove, bool $nameOmit, ?IDA $itemClass)
	{
		$hasOutput = false;
		
		// TX main/indent
		$this->send($indent);
		if ($node->hasRefName()) $this->send("*".$node->getRefName()." ");
		if ($itemClass !== null) $this->send(": ");
		
		// TX main/name
		if (!$nameOmit)
		{
			$name = "";
			if ($node->hasName()) {$name = $nameAsAbove ? "^" : $this->id_str($node, $node->getNameOr());}
			elseif (!$node->hasParams() && !$node->isList()) {$name = "?";}
			
			$this->send($name);
			$hasOutput = $name != "";
		}
		
		// TX main/parameters
		$chain = false;
		if ($node->hasParams())
		{
			if ($hasOutput) {$this->send(" ");}
			$hasOutput = true;
			$this->send("(");
			
			$pNames = $itemClass !== null ? $itemClass->getParamFirstOr() : null;
			for ($param = $node->getParamFirstOr(); $param !== null; $param = $param->getNextOr())
			{
				$pNameOmit = $pNames !== null && $pNames->eqName($param);
				if ($pNames !== null) {$pNames = $pNames->getNextOr();}
				if ($chain && !$pNameOmit) {$this->send(" ");}
				$chain = true;
				$this->ida($param, "", true, false, $pNameOmit, null);
			}
			$this->send(")");
		}
		
		// TX main/content
		if ($node->isNull() || IDA_TX::broken_ref($node))
		{
			$this->send(";");
			if (!$parameters) {$this->send("\n");}
		}
		
		// TX main/ref
		else if ($node->isRef())
		{
			$this->send(" &");
			$this->send($node->asRefOr()->getRefName());
			$this->send($parameters ? ($node->hasNext() ? "," : "") : "\n");
		}
		
		// TX main/content/bool
		elseif ($node->isBool())
		{
			if ($nameOmit) $this->send(" ");
			$this->send($node->isTrue() ? "+" : "-");
			if (!$parameters) {$this->send("\n");}
		}
		
		// TX main/content/number
		elseif ($node->isNum())
		{
			if ($hasOutput || $node->hasPrev()) {$this->send(" ");}
			$this->send($node->toString());
			$this->send($parameters ? ($node->hasNext() ? "," : "") : "\n");
		}
		
		// TX main/content/list
		else if ($node->isList())
		{
			if ($node->numItems() == 0)
			{
				if ($hasOutput) {$this->send(" ");}
				$this->send("{}\n");
			}
			else if ($node->numItems() == 1 && !$node->getItemFirstOr()->isList())
			{
				if ($hasOutput) {$this->send(" ");}
				$item = $node->getItemFirstOr();
				if ($item->hasName() || $item->hasParams())
				{
					$this->send("{ ");
					$this->ida($item, "", true, false, false, null);
					$this->send(" }\n");
				}
				else
				{
					$this->send("[ ");
					$this->send($this->value($item, true));
					$this->send(" ]\n");
				}
			}
			else
			{
				$oneLine = IDA::$port->si_allMatch($node->iterator(), function($n) {return !$n->isList() && !$n->hasParams();});
				
				if ($oneLine)
				{
					if ($hasOutput) {$this->queue(" ");}
					if (IDA::$port->si_anyMatch($node->iterator(), function($n) {return $n->hasName();}))
					{
						$pNames = ($itemClass !== null && $itemClass->isList()) ? IDA::$port->i_map($itemClass->iterator(), function($n) {return $n->getNameOr();}) : IDA::$port->empty_i();
						
						$asAbove = IDA_TX::as_above_all_test($node->getItemFirstOr());
						
						$i = $node->iterator();
						$this->queue("{");
						{
							$n = $i->next();
							$pNameOmit = $pNames->hasNext() && ($pNames->next() == $n->getNameOr());
							if ($n->hasRefName()) $this->queue(" *".$n->getRefName());
							if (!$pNameOmit) $this->queue(" ".$this->id_str($n, $n->getNameOr()));
							$this->queue($this->value($n, false));
						}
						while ($i->hasNext())
						{
							$n = $i->next();
							$pNameOmit = $pNames->hasNext() && ($pNames->next() == $n->getNameOr());
							if ($n->hasPrev() && !$n->getPrevOr()->isBool() && !$n->getPrevOr()->isNull()) $this->queue(",");
							if ($n->hasRefName()) $this->queue(" *".$n->getRefName());
							if (!$pNameOmit) $this->queue(" ".($asAbove ? "^" : $this->id_str($n, $n->getNameOr())));
							$this->queue($this->value($n, false));
						}
						$this->queue(" }\n");
					}
					else
					{
						$i = $node->iterator();
						$this->queue("[");
						while ($i->hasNext()) {$this->queue(" " . trim($this->value($i->next(), true)));}
						$this->queue(" ]\n");
					}
					
					if ($this->length() < 2*IDA_TX::$ONE_LINE_MAX) {$this->accept();}
					else {$this->reject(); $oneLine = false;}
					
				}
				
				if (!$oneLine)
				{
					// For now, only use name-as-above when all nodes have the same name,
					// and are not lists above size 1
					
					if ($hasOutput) {$this->send("\n"); $this->send($indent);}
					$this->send("{\n");
					$indent2 = $indent . "\t";
					
					$first = $node->getItemFirstOr();
					
					// Check to see if a class could be used for parameters
					$pClass = null;
					$firstName = $first->getNameOr();
					$cName = false;
					if (!$parameters && $node->numItems() > 2)
					{
						$cName = true;
						
						// Parameters
						$firstParams = $first->getParamsOr();
						$ptrParams = $firstParams !== null ? count($firstParams) : 0;
						
						// Check names separately
						for ($n = $first->getNextOr(); $n !== null; $n = $n->getNextOr())
						{
							if (!$n->isName($firstName)) {$cName = false; break;}
						}
						
						// Check parameters
						for ($n = $first->getNextOr(); $n !== null && $ptrParams > 0; $n = $n->getNextOr())
						{
							$nps = $n->getParamsOr();
							if ($nps === null) {$ptrParams = 0; break;}
							
							$i = 0;
							foreach ($nps as $np)
							{
								if ($i >= $ptrParams) {break;}
								if (!$np->isName($firstParams[$i]->getNameOr())) {break;}
								$i+=1;
							}
							
							$ptrParams = min($ptrParams, $i);
						}
						
						
						// Items (all nodes must be lists)
						$firstItems = $first->asListOr();
						$ptrItems = 0;
						$namedItem = false;
						
						// Emergency measure - avoid classing the items if there are ANY parameters or lists inside ANY nodes
						if ($firstItems !== null && !$this->has_inner_structure($first))
						{
							$ptrItems = count($firstItems);
							for ($n = $first->getNextOr(); $n != null && $ptrItems > 0; $n = $n->getNextOr())
							{
								$nis = $n->asListOr();
								if ($nis === null) {$firstItems = null; $ptrItems = 0; break;}
								
								$i = 0;
								foreach ($nis as $ni)
								{
									if ($i >= $ptrItems) {break;}
									$name = $firstItems[$i]->getNameOr();
									if (!$ni->isName($name)) {break;}
									if ($name !== null) $namedItem = true;
									$i+=1;
								}
								
								$ptrItems = min($ptrItems, $i);
							}
						}
						
						// Combine
						if ($ptrParams > 0 || ($namedItem && ($ptrItems > 0)))
						{
							// Class!
							$this->send($indent2);
							$this->send(":: ");
							if ($cName) {$this->send($this->id_str($first, $firstName));}
							
							$pClass = new IDA($firstName);
							
							if ($ptrParams > 0)
							{
								$pClass->setParams();
								
								$this->send("(");
								$np = $firstParams[0];
								$this->send($this->id_str($np, $np->getNameOr()));
								$pClass->addParam(new IDA($np->getNameOr()));
								while ($np->hasNext() && (--$ptrParams > 0))
								{
									$np = $np->getNextOr();
									$this->send(", " . $this->id_str($np, $np->getNameOr()));
									$pClass->addParam(new IDA($np->getNameOr()));
								}
								$this->send(")");
							}
							
							if ($ptrItems > 0)
							{
								$pClass->putList();
								
								$this->send("{");
								$ni = $firstItems[0];
								$this->send($this->id_str($ni, $ni->getNameOr()));
								$pClass->addItem(new IDA($ni->getNameOr()));
								while ($ni->hasNext() && (--$ptrItems > 0))
								{
									$ni = $ni->getNextOr();
									$this->send(", " . $this->id_str($ni, $ni->getNameOr()));
									$pClass->addItem(new IDA($ni->getNameOr()));
								}
								$this->send("}\n");
							}
							else {$this->send(";\n");}
							
						}
						else {$cName = false;}
					}
					
					$item = $first;
					$asAbove = IDA_TX::as_above_all_test($first);
					
					if ($asAbove || $cName)
					{
						$this->ida($item, $indent2, false, false, $cName, $pClass);
						while (($item = $item->getNextOr()) !== null)
						{
							$this->ida($item, $indent2, false, true, $cName, $pClass);
						}
					}
					else
					{
						$asAbovePart = false;
						$asAboveName = null;
						
						for (; $item !== null; $item = $item->getNextOr())
						{
							if ($asAbovePart && !$item->isName($asAboveName)) {$asAbovePart = false;}
							$this->ida($item, $indent2, false, $asAbovePart, false, $pClass);
							
							$asAboveName = $item->getNameOr();
							if (!$asAbovePart) $asAbovePart = IDA_TX::as_above_part_test($item);
							
						}
					}
					
					
					
					$this->send($indent);
					$this->send("}\n");
					
				}
				
			}
			
		}
		
		// TX main/content/string
		else
		{
			if ($hasOutput || $node->hasPrev()) {$this->send(" ");}
			$this->send($this->id_str($node, $node->toString()));
			$this->send($parameters ? ($node->hasNext() ? "," : "") : "\n");
			
		}
				
	// TX main end
	}
	
	// TX utility
	private static function broken_ref($el)
	{
		return $el->isRef() && !$el->asRefOr()->hasRefName();
	}
	
	private static function as_above_all_test(IDA $firstItem)
	{
		$itemName = $firstItem->getNameOr();
		
		if ($itemName === null) return false;
		if (strlen($itemName) < IDA_TX::$BREVITY_ALL) return false;
		
		for ($c = $firstItem; $c != null; $c = $c->getNextOr())
		{
			if ($c->numItems() >= 2) return false;
			if (!$c->isName($itemName)) return false;
		}
		return true;
	}
	private static function as_above_part_test(IDA $item)
	{
		$asAboveName = $item->getNameOr();
		if (($asAboveName === null) || (strlen($asAboveName) < IDA_TX::$BREVITY_ALL)) {return false;}
		$req = IDA_TX::$BREVITY_PART;
		for ($n = $item->getNextOr(); (--$req > 0) && ($n !== null); $n = $n->getNextOr())
		{
			if (!$n->isName($asAboveName)) {return false;}
		}
		return $req <= 0;
	}
	
	private function value(IDA $node, bool $inArray)
	{
		if ($node->isNull() || IDA_TX::broken_ref($node)) {return $inArray ? "?" : ";";}
		if ($node->isBool()) {return $node->isTrue() ? "+" : "-";}
		if ($node->isRef()) {return ($inArray ? "&" : " &") . $node->asRefOr()->getRefName();}
		return ($inArray ? "" : " ") . ($node->isNum() ? $node->toString() : $this->id_str($node, $node->toString()));
	}
	
	private function id_str(IDA $refNode, ?string $s)
	{
		if ($s === null) {return "?";}
		else if (strlen($s) < 1) {return "\"\"";}
		else if (IDA_TX::id($s)) {return $s;}
		else
		{
			$quoted = IDA::$port->ends_with($s, "-");
			$buf = "";
			
			$max = strlen($s);
			for ($i=0; $i<$max; $i+=1)
			{
				$ch = $s[$i];
				$c = ord($ch);
				if ($c < 0x20)
				{
					if ($c == 0) {$buf .= "\\0";}
					else if ($c == 0x0a) {$buf .= "\\n";}
					else if ($c == 0x0d) {$buf .= "\\r";}
					else if ($c == 0x09) {$buf .= "\\t";}
					else {$buf .= "\\x" . IDA::$port->char_2($c);}
				}
				else if ($c < 0x7f)
				{
					if ($c == 0x5c) {$buf .= "\\\\";}
					else if ($c == 0x22) {$buf .= "\\\""; $quoted = true;}
					else
					{
						if (!$quoted)
						{
							if ($i<1 ? !IDA::id0($ch) : !IDA::id1($ch)) {$quoted = true;}
							else if ($i>0 && IDA::comment2($s[$i-1], $ch)) {$quoted = true;}
						}
						$buf .= $ch;
					}
				}
				else
				{
					if ($c == 0xc2 && ord($s[$i+1]) < 0xa0)
					{
						$buf .= "\\x" . IDA::$port->char_2(ord($s[++$i]));
					}
					elseif ($this->opts($refNode)->txUnicode)
					{
						$buf .= $ch . $s[++$i];
						if ($c & 0x20) {$buf .= $s[++$i];}
						if ($c & 0x10) {$buf .= $s[++$i];}
					}
					else
					{
						$u = 0;
						if (($c & 0xe0) == 0xc0) {$u = (($c & 0x1f) << 6) | (ord($s[++$i]) & 0x3f);}
						else if (($c & 0xf0) == 0xe0) {$u = (($c & 0x0f) << 12) | ((ord($s[++$i]) & 0x3f) << 6) | (ord($s[++$i]) & 0x3f);}
						else if (($c & 0xf8) == 0xf0) {$u = (($c & 0x07) << 18) | ((ord($s[++$i]) & 0x3f) << 12) | ((ord($s[++$i]) & 0x3f) << 6) | (ord($s[++$i]) & 0x3f);}
						if ($u < 0x100) {$buf .= "\\x" . IDA::$port->char_2($u);}
						else if ($u < 0x10000) {$buf .= "\\u" . IDA::$port->char_4($u);}
						else if ($u < 0x110000) {$buf .= "\\U" . IDA::$port->char_8($u);}
					}
				}
				
			}
			
			return $quoted ? "\"" . $buf . "\"" : $buf;
		}
		
	}
	private static function id(string $s)
	{
		$len = strlen($s);
		if ($len < 1 || !IDA::id0($s[0]) || $s[-1] == '-') {return false;}
		$c1 = $s[0];
		for ($i=1; $i<$len; $i+=1)
		{
			$c2 = $s[$i];
			if (!IDA::id1($c2)) return false;
			if (IDA::comment2($c1, $c2)) return false;
			$c1 = $c2;
		}
		return true;
	}
	
// TX end
}

// CLI begin
class IDA_CLI
{
	
	// CLI
	static function cli_err($msg, $exitCode = 1)
	{
		$stderr = fopen("php://stderr", "w");
		fwrite($stderr, "Error: " . $msg . "\n");
		fclose($stderr);
		return $exitCode;
	}
	static function cli_out($ln) {echo $ln . "\n";}
	static function cli_help()
	{
		//           ".........1.........2.........3.........4.........5.........6.........7.........8"
		IDA::cli_out("IDA markup parser");
		IDA::cli_out("Read markup from standard input and write IDA to standard output.");
		IDA::cli_out("");
		IDA::cli_out("Parser options:");
		IDA::cli_out(" -h, --help         Show usage");
		IDA::cli_out(" -i in-option       Set IDA input option");
		IDA::cli_out(" -o out-option      Set IDA output option");
		IDA::cli_out("");
		IDA::cli_out("IDA input options:");
		IDA::cli_out(" xml-reparse        Parse XML string content as additional inline markup");
		IDA::cli_out("");
		IDA::cli_out("IDA output options:");
		IDA::cli_out(" unicode            Use UTF-8 for string characters beyond U+0080");
		IDA::cli_out("");
		return 0;
	}
	public static function cli($argc = 0, $argv = null)
	{
		$rootEl = new IDA();
		$opts = $rootEl->opts;
		
		$i=0;
		if ($argv !== null) while ($i<$argc)
		{
			$arg = $argv[$i++];
			
			if ($arg == "-h" || $arg == "--help")
			{
				return IDA::cli_help();
			}
			else if ($arg == "-i")
			{
				if ($i >= $argc) return IDA::cli_err("No input option specified");
				$arg = $argv[$i++];
				if ($arg == "xml-reparse") {$opts->rxXmlReparseStrings = true;}
				else return IDA::cli_err("Unknown input option: " . $arg);
			}
			else if ($arg == "-o")
			{
				if ($i >= $argc) return IDA::cli_err("No output option specified");
				$arg = $argv[$i++];
				if ($arg == "unicode") {$opts->txUnicode = true;}
				else return IDA::cli_err("Unknown output option: " . $arg);
			}
		}
		
		try {$rootEl->inFile(php_sapi_name() == "cli" ? "php://stdin" : "php://input");}
		catch (Exception $ex) {return IDA::cli_err("Parse failed: " . $ex->getMessage());}
		
		$rootEl->outFile(php_sapi_name() == "cli" ? "php://stdout" : "php://output", true);
		return 0;
	}
	
// CLI end
}


/** One data article element. */
class IDA
{
	
	// File includes
	// File includes - Global PWD
	private static $gIncludePwd = null;
	/** Enables file includes globally, as applied to articles created hereafter.<br>
	 * The provided working dir is used for resolving relative paths.<br>
	 * Set null to disable includes by default. */
	public static function setGlobalIncludes($pwd = "./") {IDA::$gIncludePwd = $pwd;}
	
	// File includes - this element
	private $includePwd;
	/** Enables file includes for this article, as applied to input decoded hereafter.<br>
	 * The provided working dir is used for resolving relative paths.<br>
	 * Set null to disable includes. */
	public function setIncludes($pwd = "./") {$this->includePwd = $pwd; return $this;}
	/** May be null, if includes are disabled for this article. */
	public function rxIncludePwd() {return $this->includePwd;}
	
	// Input control
	/** Negative for unlimited */
	public $inputListDepth = -1;
	/** Negative for unlimited */
	public $inputListLength = -1;
	public function rxInputList($depth, $length)
	{
		$this->inputListDepth = $depth;
		$this->inputListLength = $length;
		return $this;
	}
	
	// Options storage
	// See Initialisation for opts
	/** <p>Fetch the input/output options for this element.</p>
	 * @return Input/output options object. */
	public function getOpts() {return $this->opts;}
	/** <p>Set the input/output options of this element to match those of the provided element.</p>
	 * @param src An element from which to copy the options.
	 * @return This element. */
	public function copyOptions(IDA $src)
	{
		$this->opts->copyFrom($src->opts);
		return $this;
	}
	
	public function rxConfig($parent)
	{
		$this->rxInputList($parent->inputListDepth > 0 ? $parent->inputListDepth - 1 : $parent->inputListDepth, $parent->inputListLength);
		$this->copyOptions($parent);
		return $this;
	}
	public function rxNew()
	{
		$el = (new IDA())->rxScope($this)->setIncludes($this->rxIncludePwd());
		$el->rxConfig($this);
		return $el;
	}
	
	// Scope storage
	private $scope = null;
	public function rxScope(?IDA $scope) {$this->scope = $scope; return $this;}
	
	// Item classes
	// array: String -> IDA
	private $itemClasses = null;
	public function rxClearItemClasses()
	{
		$this->itemClasses = null;
	}
	public function rxGetItemClass(String $key)
	{
		$c = ($this->itemClasses !== null) ? $this->itemClasses[$key] : null;
		return ($c !== null) ? $c : (($this->scope !== null) ? $this->scope->rxGetItemClass($key) : null);
	}
	/** Actually adds to scope element for use by adjacent elements.<br>
	 * Returns false (and takes no action with the class element) if no scope existed */
	public function rxAddItemClass(String $key, IDA $itemClass)
	{
		if ($this->scope === null) {return false;}
		if ($this->scope->itemClasses === null) {$this->scope->itemClasses = array();}
		$this->scope->itemClasses[$key] = $itemClass;
		return true;
	}
	
	// Templates
	// array: String -> IDA
	private $itemTemplates = null;
	public function rxClearTemplates()
	{
		$this->itemTemplates = null;
	}
	public function rxGetTemplate(String $key)
	{
		$c = ($this->itemTemplates !== null) ? $this->itemTemplates[$key] : null;
		return ($c !== null) ? $c : (($this->scope !== null) ? $this->scope->rxGetTemplate($key) : null);
	}
	/** Actually adds to scope element for use by adjacent elements.<br>
	 * Returns false (and takes no action with the template element) if no scope existed */
	public function rxAddTemplate(String $key, IDA $itemTemplate)
	{
		if ($this->scope === null) {return false;}
		if ($this->scope->itemTemplates === null) {$this->scope->itemTemplates = array();}
		$this->scope->itemTemplates[$key] = $itemTemplate;
		return true;
	}
	
	// Adjacent nodes
	private $next = null;
	private $prev = null;
	/** <p>Test whether an element is adjacent to this element in the forward direction.</p>
	 * @return True if an element exists after this element, otherwise false */
	public function hasNext() {return $this->next !== null;}
	/** <p>Test whether an element is adjacent to this element in the reverse direction.</p>
	 * @return True if an element exists before this element, otherwise false */
	public function hasPrev() {return $this->prev !== null;}
	/** <p>Test whether no element is adjacent to this element in the reverse direction.</p>
	 * <p>This is the opposite of hasPrev, intended as a convenience for function references.</p>
	 * @return True if nothing exists this element, otherwise false */
	public function isFirst() {return !$this->hasPrev();}
	/** <p>Test whether no element is adjacent to this element in the forward direction.</p>
	 * <p>This is the opposite of hasNext, intended as a convenience for function references.</p>
	 * @return True if nothing exists after this element, otherwise false */
	
	/** <p>Fetch the adjacent element after this one.</p>
	 * <p>If this element is last, an exception is thrown.
	 * To avoid that possibility, use getNextOr to provide a default value.</p>
	 * @exception IDAOpException if this element is last.
	 * @return The element after. */
	public function getNext()
	{
		if ($this->isLast()) throw new IDAOpException("Element is last");
		return $this->next;
	}
	/** <p>Fetch the adjacent element after this one.</p>
	 * <p>If this element is last, the provided default value is returned.</p>
	 * @param other A default value to return if nothing follows this element.
	 * @return The element after if present, otherwise the default value. */
	public function getNextOr($other = null)
	{
		return $this->hasNext() ? $this->next : $other;
	}
	/** <p>Fetch the adjacent element before this one.</p>
	 * <p>If this element is first, an exception is thrown.
	 * To avoid that possibility, use getPrevOr to provide a default value.</p>
	 * @exception IDAOpException if this element is first.
	 * @return The element before. */
	public function getPrev()
	{
		if ($this->isFirst()) throw new IDAOpException("Element is first");
		return $this->prev;
	}
	/** <p>Fetch the adjacent element before this one.</p>
	 * <p>If this element is first, the provided default value is returned.</p>
	 * @param other A default value to return if nothing precedes this element.
	 * @return The element before if present, otherwise the default value. */
	public function getPrevOr($other = null)
	{
		return $this->hasPrev() ? $this->prev : $other;
	}
	
	public function isLast() {return !$this->hasNext();}
	/** This element, followed by everything adjacent after it */
	public function iterateNext()
	{
		$i = new IDA_MutableSeq();
		$i->n = $this;
		$i->funcHas = function() use($i) {return ($i->n !== null);};
		$i->funcGet = function() use($i) {$x = $i->n; $i->n = $x->getNextOr(); return $x;};
		return $i;
	}
	/** <p>Places the provided element adjacent to this one, next in sequence.<br>
	 * If the element is null, this one will become last in sequence.</p>
	 * <p>Any existing forward link will be undone before this operation.</p>
	 * @param next An element to follow this one, or null
	 * @return This element */
	public function linkNext(?IDA $next)
	{
		if ($this->next !== null) {$this->next->prev = null;}
		$this->next = $next; if ($next !== null) $next->prev = $this;
		return $this;
	}
	/** <p>Places the provided element adjacent to this one, previous in sequence.<br>
	 * If the element is null, this one will become first in sequence.</p>
	 * <p>Any existing reverse link will be undone before this operation.</p>
	 * @param prev An element to precede this one, or null
	 * @return This element */
	public function linkPrev(?IDA $prev)
	{
		if ($this->prev !== null) {$this->prev->next = null;}
		$this->prev = $prev; if ($prev !== null) $prev->next = $this;
		return $this;
	}
	/** <p>Places the provided elements adjacent to this one, on either side.<br>
	 * If an element is null, this one will become the first/last in sequence.</p>
	 * <p>Before this operation occurs, any existing links will be cross-connected to remove this element from the chain.</p>
	 * @param prev An element to precede this one, or null
	 * @param next An element to follow this one, or null
	 * @return This element */
	public function link(?IDA $prev, ?IDA $next)
	{
		if ($this->prev !== null) {$this->prev->next = $this->next;}
		if ($this->next !== null) {$this->next->prev = $this->prev;}
		$this->prev = $prev; if ($prev !== null) $prev->next = $this;
		$this->next = $next; if ($next !== null) $next->prev = $this;
		return $this;
	}
	
	// Node name
	protected $name = null;
	/** <p>Test whether this element has no name.
	 * The name must be null, not merely the empty string.</p>
	 * <p>This is the opposite of hasName, intended as a convenience for function references.</p>  
	 * @return True if this element has no name, otherwise false */
	public function isAnon() {return $this->name === null;}
	/** <p>Test whether this element has a name.
	 * The name can be any non-null string, including the empty string.</p>
	 * @return True if this element has a name, otherwise false */
	public function hasName() {return $this->name !== null;}
	/** <p>Test whether this element has a specific name.
	 * Providing null will perform the same test as with isAnon.</p>
	 * @param name A name for comparison
	 * @return True if this element name matches the provided name, otherwise false */
	public function isName($name)
	{
		return $this->getNameOr() === $name;
	}
	
	/** <p>Fetch this element name string.</p>
	 * <p>If this element content is anonymous, an exception is thrown.
	 * To avoid that possibility, use getNameOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no name.
	 * @return Element name. */
	public function getName()
	{
		if ($this->isAnon()) throw new IDAOpException("Element has no name");
		return $this->name;
	}
	/** <p>Fetch this element name string.</p>
	 * <p>If this element content is anonymous, the provided default value is returned.</p>
	 * <p>On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
	 * If possible, always call getNameOr() rather than getNameOr(null).</p>
	 * @param other A default value to return if this element has no name.
	 * @return Element name if present, otherwise the default value. */
	public function getNameOr($other = null)
	{
		return $this->hasName() ? $this->name : $other;
	}
	
	/** <p>Clear the name of this element, leaving it anonymous.</p>
	 * @return This element. */
	public function clearName() {return $this->setName(null);}
	/** <p>Set the name of this element to be the provided string.</p>
	 * <p>If the provided string is null, this element will become anonymous as with calling clearName.</p>
	 * @param name Element name string.
	 * @return This element. */
	public function setName($name) {$this->name = $name; return $this;}
	/** <p>Set the name of this element to match the name of the provided element.</p>
	 * @param src An element from which to copy the name */
	public function copyName($src) {$this->name = $src->name;}
	
	// Parameters
	protected $parameters = null;
	/** <p>Delete this element's parameters list.</p>
	 * <p>If this element has no parameters list, calling clearParams has no effect.</p>
	 * @return This element. */
	public function clearParams() {$this->parameters = null; return $this;}
	/** <p>Set this element's parameters to be an empty list.</p>
	 * <p>Any existing parameter elements will be deleted.</p>
	 * @return This element. */
	public function setParams() {$this->parameters = array(); return $this;}
	
	/** <p>Fetch this element's parameters list.</p>
	 * <p>If this element has no parameters list, an exception is thrown.
	 * To avoid that possibility, use getParamsOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no parameters list.
	 * @return List of parameters. */
	public function getParams()
	{
		if (!$this->hasParams()) throw new IDAOpException("No parameters list");
		return $this->parameters;
	}
	/** <p>Fetch this element's parameters list.</p>
	 * <p>If this element has no parameters list, the provided default value is returned.</p>
	 * <p>On platforms where lists/vectors are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed list, such that code will not be fully portable if null is provided.
	 * If possible, always call getParamsOr() rather than getParamsOr(null).</p>
	 * @param other A default value to return if this element has no parameters list.
	 * @return List of parameters if present, otherwise the default value. */
	public function getParamsOr($other = null)
	{
		return $this->hasParams() ? $this->parameters : $other;
	}
	
	/** <p>Test whether this element contains a parameters list.</p>
	 * <p>This reports whether a parameters list object exists at all, as opposed to whether at least one parameter is present.
	 * The primary purpose is to ensure the list exists before calling getParams and/or iterating over its elements.</p>
	 * <p>Use numParams to test for an empty list (0) or at least one parameter present (greater than 0).</p>
	 * @return True if a parameters list exists for this element, otherwise false */
	public function hasParams()
	{
		return $this->parameters !== null;
	}
	/** <p>Fetch the number of parameters within this element.</p>
	 * <p>If no parameters list is present, the sentinel value -1 is returned.
	 * If 0 is returned, the list is present but empty.</p>
	 * @return The parameters list size if a list is present, otherwise -1 */
	public function numParams()
	{
		return $this->parameters !== null ? count($this->parameters) : -1;
	}
	/** <p>Test whether this element contains a parameter with the provided name.
	 * If it does, an equivalent call to getParam will return an element.</p>
	 * <p>If false is returned, this can be either due to having no parameter of that name or due to having no parameters list.
	 * Use hasParams to separate those two cases.</p>
	 * @param key A parameter name to be checked
	 * @return True if a parameter exists with the provided name, otherwise false */
	public function hasParam($key)
	{
		return $this->getParamOr($key) !== null;
	}
	
	// Parameters/get
	/** <p>Fetch the parameter element at the provided index in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has too few parameters in the list, an exception is thrown.
	 * To avoid that possibility, use getParamAtOr to provide a default value.</p>
	 * @param index A parameter index to be recalled.
	 * @exception IDAOpException if this element has no such parameter.
	 * @return The indexed parameter element. */
	public function getParamAt($index)
	{
		if ($this->numParams() <= $index) throw new IDAOpException("Invalid parameter index");
		return $this->parameters[$index];
	}
	/** <p>Fetch the parameter element at the provided index in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has too few parameters in the list, the provided default value is returned.</p>
	 * @param index A parameter index to be recalled.
	 * @param other A default value to return if this element has no such parameter.
	 * @return The indexed parameter element if present, otherwise the default value. */
	public function getParamAtOr($index, $other = null)
	{
		return $this->numParams() > $index ? $this->parameters[$index] : $other;
	}
	
	/** <p>Fetch the first parameter element with the provided name in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has no parameter of that name, an exception is thrown.
	 * To avoid that possibility, use getParamOr to provide a default value.</p>
	 * @param key A parameter name to be recalled.
	 * @exception IDAOpException if this element has no such parameter.
	 * @return The named parameter element. */
	public function getParam($key)
	{
		$p = $this->getParamOr($key);
		if ($p === null) throw new IDAOpException("Parameter not found");
		return $p;
	}
	/** <p>Fetch the first parameter element with the provided name in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has no parameter of that name, the provided default value is returned.</p>
	 * @param key A parameter name to be recalled.
	 * @param other A default value to return if this element has no such parameter.
	 * @return The named parameter element if present, otherwise the default value. */
	public function getParamOr($key, $other = null)
	{
		if ($this->hasParams()) foreach ($this->parameters as $p) if ($p->isName($key)) return $p;
		return $other;
	}
	
	/** <p>Cast to a string value the first parameter element with the provided name in this element's parameters list.</p>
	 * <p>If this element has no parameters list, has no parameter of that name or can not be interpreted as a string value, an exception is thrown.
	 * To avoid that possibility, use getParamStrOr to provide a default value.</p>
	 * @param key A parameter name to be recalled.
	 * @exception IDAOpException if this element has no such parameter, or the parameter content cannot cast to a string value.
	 * @return String content interpretation of the named parameter element. */
	public function getParamStr(?string $key)
	{
		return $this->getParam($key)->toStr();
	}
	/** <p>Cast to a string value the first parameter element with the provided name in this element's parameters list.</p>
	 * <p>If this element has no parameters list, has no parameter of that name or can not be interpreted as a string value, the provided default value is returned.</p>
	 * <p>On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
	 * If possible, always call getParamStrOr(key) rather than getParamStrOr(key, null).</p>
	 * @param key A parameter name to be recalled.
	 * @param other A default value to return if this element has no such parameter, or the parameter content cannot cast to a string value.
	 * @return String content interpretation of the named parameter element if possible, otherwise the default value. */
	public function getParamStrOr(?string $key, $other = null)
	{
		$param = $this->getParamOr($key);
		return $param !== null ? $param->toStrOr($other) : $other;
	}
	
	/** <p>Fetch the first parameter element in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has an empty parameters list, an exception is thrown.
	 * To avoid that possibility, use getParamFirstOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no such parameter.
	 * @return The first parameter element. */
	public function getParamFirst()
	{
		if ($this->numParams() < 1) throw new IDAOpException($this->hasParams() ? "Parameters list is empty" : "Element has no parameters list");
		return $this->parameters[0];
	}
	/** <p>Fetch the first parameter element in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has an empty parameters list, the provided default value is returned.</p>
	 * @param other A default value to return if this element has no such parameter.
	 * @return The first parameter element if present, otherwise the default value. */
	public function getParamFirstOr($other = null)
	{
		return $this->numParams() > 0 ? $this->parameters[0] : $other;
	}
	/** <p>Fetch the last parameter element in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has an empty parameters list, an exception is thrown.
	 * To avoid that possibility, use getParamLastOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no such parameter.
	 * @return The last parameter element. */
	public function getParamLast()
	{
		$n = $this->numParams();
		if ($n < 1) throw new IDAOpException($this->hasParams() ? "Parameters list is empty" : "Element has no parameters list");
		return $this->parameters[$n-1];
	}
	/** <p>Fetch the last parameter element in this element's parameters list.</p>
	 * <p>If this element has no parameters list or has an empty parameters list, the provided default value is returned.</p>
	 * @param other A default value to return if this element has no such parameter.
	 * @return The last parameter element if present, otherwise the default value. */
	public function getParamLastOr($other = null)
	{
		$n = $this->numParams();
		return $n > 0 ? $this->parameters[$n-1] : $other;
	}
	
	// Parameters/add
	/** Create and insert a new parameter at the end of the list, with the provided element name.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param name Name string for the created parameter element.
	 * @return The created parameter element. */
	public function newParam($name)
	{
		$el = new IDA($name);
		$this->addParam($el);
		return $el;
	}
	
	/** Insert a parameter at the end of the list.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param param The parameter element to be stored.
	 * @return This element. */
	public function addParam(IDA $param)
	{
		if ($this->parameters === null) {$this->setParams();}
		$param->link($this->getParamLastOr(), null);
		$this->parameters[] = $param;
		return $this;
	}
	/** Insert a parameter at the specified list index position.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param param The parameter element to be stored.
	 * @param pos The list index position at which the parameter should be inserted.
	 * @exception IDAOpException if the specified position is below zero or above the present list size.
	 * @return This element. */
	public function addParamAt(IDA $param, $pos)
	{
		if ($this->parameters === null) {$this->setParams();}
		IDA::list_add($this->parameters, $param, $pos, "param");
		return $this;
	}
	/** Replace the first matching parameter name, or insert a new parameter at the end of the list.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param param The parameter element to be stored.
	 * @return This element. */
	public function setParam($param)
	{
		$n = $this->numParams();
		
		try
		{
			for ($i=0; $i<$n; $i+=1)
			{
				$p = $this->getParamAt($i);
				if ($p->eqName($param))
				{
					$this->remParamAt($i);
					return $this->addParamAt($param, $i);
				}
			}
		}
		catch (IDAOpException $ex) {throw new RuntimeException("Unexpected parameters state", 1, $ex);}
		
		return $this->addParam($param);
	}
	
	/** <p>Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.</p>
	 * <p>This is the same as calling setParamNull.</p>
	 * @param key The parameter element name.
	 * @return This element. */
	public function clearParam($key) {return $this->setParamNull($key);}
	/** <p>Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.</p>
	 * <p>This is the same as calling clearParam.</p>
	 * @param key The parameter element name.
	 * @return This element. */
	public function setParamNull($key) {return $this->setParam((new IDA($key))->putNull());}
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content String value to be stored.
	 * @return This element. */
	public function setParamStr($key, $content) {return $this->setParam((new IDA($key))->putStr($content));}
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Boolean value to be stored.
	 * @return This element. */
	public function setParamBool($key, $content) {return $this->setParam((new IDA($key))->putBool($content));}
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Floating-point value to be stored.
	 * @return This element. */
	public function setParamDouble($key, $content) {return $this->setParam((new IDA($key))->putDouble($content));}
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Floating-point value to be stored.
	 * @return This element. */
	public function setParamFloat($key, $content) {return $this->setParam((new IDA($key))->putFloat($content));}
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Integer value to be stored.
	 * @return This element. */
	public function setParamLong($key, $content) {return $this->setParam((new IDA($key))->putLong($content));}
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Integer value to be stored.
	 * @return This element. */
	public function setParamInt($key, $content) {return $this->setParam((new IDA($key))->putInt($content));}
	/** Replace the first matching parameter name, or insert a new parameter.<br>
	 * If this element has no parameters list, a new list will be created.
	 * @param key The parameter element name.
	 * @param content Referenced element to be stored.
	 * @return This element. */
	public function setParamRef($key, $content) {return $this->setParam((new IDA($key))->putRef($content));}
	
	// Parameters/remove
	/** Remove the parameter at the specified list index position.<br>
	 * If this element has no parameters list, an exception is thrown.<br>
	 * @param pos The list index position of a parameter which should be removed.
	 * @exception IDAOpException if the specified position is below zero or beyond the present list size, or this element has no parameters list. */
	public function remParamAt($pos)
	{
		if (!$this->hasParams()) {throw new IDAOpException("No parameters from which to remove");}
		IDA::list_del($this->parameters, $pos, "param");
	}
	/** Remove the provided parameter.<br>
	 * If the parameter is absent or this element has no parameters list, the call has no effect.<br>
	 * @param param The parameter element to be removed.
	 * @return True if a parameter was removed, otherwise false. */
	public function remParam(IDA $param)
	{
		if ($this->hasParams() && IDA::list_rem($this->parameters, $param)) {$param->link(null, null); return true;}
		return false;
	}
	/** Remove the first parameter with the provided element name.<br>
	 * If no such parameter exists or this element has no parameters list, the call has no effect.<br>
	 * @param key Name string of the parameter element to remove.
	 * @return True if a parameter was removed, otherwise false. */
	public function unParam(?String $key)
	{
		$p = $this->getParamOr($key);
		if ($p !== null) {$this->remParam($p); return true;}
		return false;
	}
	
	// Parameters/other
	/** <p>Set the parameters of this element to be clones of the parameters of the provided element.</p>
	 * @param src An element from which to copy the parameters */
	public function copyParams(IDA $src)
	{
		$this->clearParams();
		if ($src->hasParams())
		{
			$this->setParams();
			foreach ($src->parameters as $p) $this->addParam($p->copy_inner());
		}
	}
	
	// Content
	private $cRaw = null;
	private $cStr = null;
	private $cNum = "", $cDen = "1", $cUnit = "";
	/** array: IDA */
	private $cList = null;
	private $cBool = null;
	private $cRef = null;
	private $cRefObj = null;
	private $cRefTarget = null;
	private $cRefName = null;
	
	/** <p>Set the content of this element to match the content of the provided element.
	 * If the content in question is a list, its inner elements will be cloned.</p>
	 * @param src An element from which to copy the content */
	public function copyContent(IDA $src)
	{
		$this->cRaw = $src->cRaw;
		$this->cStr = $src->cStr;
		$this->cNum = $src->cNum;
		$this->cDen = $src->cDen;
		$this->cUnit = $src->cUnit;
		$this->cBool = $src->cBool;
		$this->cRef = null;
		$this->cRefObj = null;
		$this->cRefTarget = $src->cRefTarget;
		$this->cRefName = $src->cRefName;
		$this->clearList();
		if ($src->isList())
		{
			$this->putList();
			foreach ($src->asList() as $item) $this->addItem($item->copy_inner());
		}
	}
	
	// Content/ref
	/** Returns this element */
	public function clearRef() {$this->cRef = null; $this->cRefObj = null; $this->cRefTarget = null; return $this;}
	/** <p>Test whether the content type of this element is a reference to another element.</p>
	 * @return True if this element content is a reference, otherwise false */
	public function isRef() {return $this->cRef !== null;}
	/** <p>Fetch this element content as a reference (to another element).</p>
	 * <p>If this element content is not a reference, an exception is thrown.
	 * To avoid that possibility, use asRefOr to provide a default value.</p>
	 * @exception IDAOpException if this element does not contain reference content.
	 * @return Reference content. */
	public function asRef()
	{
		return $this->cRef;
	}
	/** <p>Fetch this element content as a reference (to another element).</p>
	 * <p>If this element content is not a reference, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain reference content.
	 * @return Reference content if present, otherwise the default value. */
	public function asRefOr($other = null)
	{
		return $this->isRef() ? $this->cRef : $other;
	}
	/** Returns this element */
	public function putRef($ref) {$this->putNull(); $this->cRaw = "Ref"; $this->cRef = $ref; $this->cRefObj = $ref->repObj; $this->cRefTarget = $ref->cRefName; return $this;}
	/** <p>Sets an object for which this element should reference the corresponding element as it is encoded.</p> 
	 * <p>During a restore/decode process, the rebuilt object can be retrieved using getRefObj or withRefObj.
	 * The IDAObj codec interface implementation should ensure that any required type information is captured.</p>
	 * @param refObj A target object which implements the IDAObj codec interface.
	 * @return This element. */
	public function setRefObj($refObj) {$this->putNull(); $this->cRaw = "Ref"; $this->cRefObj = $refObj; return $this;}
	/** <p>During a restore/decode process, the object corresponding with the referenced element will be sent to the callback when it becomes available.
	 * If that object is already available, the callback will be resolved before this call returns.</p>
	 * <p>If the object is needed immediately, the article structure must be designed such that the referenced object element is decoded in advance of this call.
	 * Use getRefObj instead if that design should be enforced.</p>
	 * <p>To ensure consistency, a top-level call to objRestore will throw an exception if any unresolved callbacks remain after decoding is complete.</p>
	 * <p>Calling code should use article information to establish the correct object type.</p>
	 * @param callback A function which receives the decoded object associated with the referenced element.
	 * @exception IDAOpException if this element content is not a reference. */
	public function withRefObj($callback)
	{
		if ($this->cRefObj != null) {$callback($this->cRefObj);}
		else if ($this->cRef == null) throw new IDAOpException("Element is not a reference");
		else {$this->cRef->on_rep_obj($callback);}
	}
	/** <p>Fetches the object corresponding with the referenced element, after it has been encoded or decoded.
	 * During a restore/decode process, an exception is thrown if the object is not yet available.</p>
	 * <p>To retrieve the object this way during a restore/decode process, the article structure must be designed such that the referenced object element is decoded in advance of this call.
	 * Use withRefObj instead if that design is not the case.</p>
	 * <p>Calling code should use article information to establish the correct object type.</p>
	 * @exception IDAOpException if this element content is not a reference, or if no object is available immediately.
	 * @return The encoded or decoded object associated with the referenced element. */
	public function getRefObj()
	{
		if ($this->cRefObj != null) return $this->cRefObj;
		if ($this->cRef == null) throw new IDAOpException("Element is not a reference");
		if ($this->cRef->repObj != null) return $this->cRef->repObj;
		throw new IDAOpException("Referenced object not available");
	}
	/** <p>Set a transient target for this element.</p>
	 * <p>The targets match the identifiers assigned via setRefName, and are used only for capturing and restoring element pointer information.
	 * The string content itself is not information, insofar as it need not survive a loop test.</p>
	 * @param refTarget A string to store as the transient target for this element. */
	public function setRefTarget($refTarget) {$this->putNull(); $this->cRaw = "Ref"; $this->cRefTarget = $refTarget;}
	/** <p>Fetch the transient target for this element.
	 * The returned value may be <i>stale</i>.</p>
	 * <p>The targets match the identifiers assigned via setRefName, and are used only for capturing and restoring element pointer information.
	 * The string content itself is not information, insofar as it need not survive a loop test.</p>
	 * @return The transient target string for this element if it has a reference target, otherwise null. */
	public function getRefTarget() {return $this->cRefTarget;}
	/** <p>Test whether this element has been assigned a transient target for encoding/decoding an outbound reference.
	 * The returned value may be <i>stale</i>.</p>
	 * <p>If an article has been decoded from markup, any reference target elements will have fresh identifiers from the markup document.
	 * Encoding, copying or testing equal a top-level article element will generate fresh identifiers.</p>
	 * <p>Call txRefLabel at top-level to generate fresh identifiers manually.</p>
	 * @return True if this element has a transient target, otherwise false. */
	public function hasRefTarget() {return $this->cRefTarget !== null;}
	/** <p>Set a transient identifier for this element.</p>
	 * <p>Identifiers are set automatically when encoding/decoding markup, testing equal, or calling txRefLabel directly.
	 * Encoding, copying or testing equal a top-level article element will generate fresh identifiers.</p>
	 * <p>These names are used only for capturing and restoring element pointer information.
	 * The string content itself is not information, insofar as it need not survive a loop test.</p>
	 * @param refName A string to store as the transient identifier for this element. */
	public function setRefName($refName) {$this->cRefName = $refName;}
	/** <p>Fetch the transient identifier for this element.
	 * The returned value may be <i>stale</i>.</p>
	 * <p>Identifiers are set automatically when encoding/decoding markup, testing equal, or calling txRefLabel directly.
	 * Encoding, copying or testing equal a top-level article element will generate fresh identifiers.</p>
	 * <p>These names are used only for capturing and restoring element pointer information.
	 * The string content itself is not information, insofar as it need not survive a loop test.</p>
	 * @return The transient identifier string for this element if it is a reference target, otherwise null. */
	public function getRefName() {return $this->cRefName;}
	/** <p>Test whether this element has been assigned a transient identifier for encoding/decoding inbound references.
	 * The returned value may be <i>stale</i>.</p>
	 * <p>If an article has been decoded from markup, any reference target elements will have fresh identifiers from the markup document.
	 * Encoding, copying or testing equal a top-level article element will generate fresh identifiers.</p>
	 * <p>Call txRefLabel at top-level to generate fresh identifiers manually.</p>
	 * @return True if this element has a transient identifier, otherwise false. */
	public function hasRefName() {return $this->cRefName !== null;}
	
	/** Returns the number of reference labels applied */
	public function txRefLabel($adj)
	{
		// Ref walk
		$rc = 0;
		$rn = [];
		$this->walk(function($x) use(&$rn) {IDA::ref_label_clear($x, $rn);}, $adj, $adj);
		$this->walk(function($x) use(&$rn, &$rc) {IDA::ref_label_count($x, $rn, $rc);}, $adj, $adj);
		
		// Null any foreign references at this stage (see IDA.verify)
		foreach ($rn as $r) $r->setRefName(null);
		
		return $rc;
	}
	private static function ref_label_clear($el, &$rn)
	{
		$el->setRefName(null);
		if ($el->isRef())
		{
			$r = $el->asRefOr();
			if (!IDA::$port->list_contains($rn, $r)) $rn[] = $r;
		}
	}
	private static function ref_label_count($el, &$rn, &$rc)
	{
		if (IDA::$port->list_contains($rn, $el)) IDA::$port->list_rem($rn, $el);
		
		if ($el->isRef())
		{
			$r = $el->asRefOr();
			if (!$r->hasRefName())
			{
				$n = $rc++;
				$n0 = $n % 52;
				$id = "" . chr($n0 < 26 ? 0x61+$n0 : 0x41+$n0) . intdiv($n, 52);
				$r->cRefName = $id;
				$el->cRefTarget = $id;
			}
		}
	}
	
	public function rxRefResolve($adj)
	{
		$rs = array();
		$this->walk(function($x) use(&$rs) {IDA::ref_resolve_scan($x, $rs);}, $adj, $adj);
		$this->walk(function($x) use(&$rs) {IDA::ref_resolve_put($x, $rs);}, $adj, $adj);
	}
	private static function ref_resolve_scan($el, &$rs)
	{
		if ($el->hasRefName()) $rs[$el->getRefName()] = $el;
	}
	private static function ref_resolve_put($el, &$rs)
	{
		if ($el->hasRefTarget()) $el->putRef($rs[$el->getRefTarget()]);
	}
	
	// Content/bool
	/** <p>Test whether the content type of this element is a boolean value.</p>
	 * @return True if this element has boolean content, otherwise false */
	public function isBool() {return $this->cBool !== null;}
	/** <p>Fetch this element content as a boolean value.</p>
	 * <p>If this element content is not a boolean value, an exception is thrown.
	 * To avoid that possibility, use asBoolOr to provide a default value.
	 * One may also use isTrue for false by default, or not-isFalse for true by default.</p>
	 * <p>Use toBool/toBoolOr for casting other content types to be a boolean value.</p>
	 * @exception IDAOpException if this element does not contain boolean content.
	 * @return Boolean content value. */
	public function asBool()
	{
		if ($this->cBool === null) throw new IDAOpException("Content is not a boolean value");
		return $this->cBool == true;
	}
	/** <p>Fetch this element content as a boolean value.</p>
	 * <p>If this element content is not a boolean value, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain boolean content.
	 * @return Boolean content value if present, otherwise the default value. */
	public function asBoolOr($other) {return ($this->cBool !== null) ? ($this->cBool == true) : $other;}
	/** <p>Test whether the content of this element is boolean-true.</p>
	 * <p>If false is returned, this can be either due to being boolean-false or not being a boolean value.
	 * Use isBool and asBool for methods with those effects.</p>
	 * <p>This is intended as a convenience for function references.</p>
	 * @return True if this element has boolean content with value true, otherwise false */
	public function isTrue() {return ($this->cBool !== null) && ($this->cBool == true);}
	/** <p>Test whether the content of this element is boolean-false.</p>
	 * <p>If false is returned, this can be either due to being boolean-true or not being a boolean value.
	 * Use isBool and (not-)asBool for methods with those effects.</p>
	 * <p>This is intended as a convenience for function references.</p>
	 * @return True if this element has boolean content with value false, otherwise false */
	public function isFalse() {return ($this->cBool !== null) && ($this->cBool != true);}
	/** <p>Place boolean content into this element with value true, replacing any existing content.</p>
	 * @return This element */
	public function putTrue() {$this->putNull(); $this->cRaw = "+"; $this->cBool = true; return $this;}
	/** <p>Place boolean content into this element with value false, replacing any existing content.</p>
	 * @return This element */
	public function putFalse() {$this->putNull(); $this->cRaw = "-"; $this->cBool = false; return $this;}
	/** <p>Place boolean content into this element with the provided value, replacing any existing content.</p>
	 * @param b Boolean content value.
	 * @return This element */
	public function putBool($b) {return $b ? $this->putTrue() : $this->putFalse();}
	/** <p>Test whether this element content can be interpreted as a boolean value (see toBool).
	 * This will always be the case for natural boolean content (see isBool).</p>
	 * <p>All numeric content can be cast to boolean, which is true if non-zero.
	 * String content can be "true" or "false".</p>
	 * @return True if this element content can be cast to a boolean value, otherwise false */
	public function canBool()
	{
		if ($this->isBool() || $this->isNum()) return true;
		$s = $this->toStrOr("");
		return $s=="true" || $s=="false";
	}
	/** <p>Cast this element content to a boolean value.</p>
	 * <p>If this element content can not be interpreted as a boolean value, an exception is thrown.
	 * To avoid that possibility, use toBoolOr to provide a default value.</p>
	 * @exception IDAOpException if this element content cannot cast to a boolean value.
	 * @return Boolean content interpretation. */
	public function toBool()
	{
		if ($this->isBool()) return $this->isTrue();
		if ($this->isNum()) return $this->asDoubleOr(0.0) != 0.0;
		$s = $this->toStrOr("");
		if ($s=="true") return true;
		if ($s=="false") return false;
		throw new IDAOpException("Content cannot cast to a boolean value");
	}
	/** <p>Cast this element content to a boolean value.</p>
	 * <p>If this element content can not be interpreted as a boolean value, the provided default value is returned.</p>
	 * @param other A default value to return if this element content cannot cast to a boolean value.
	 * @return Boolean content interpretation if able, otherwise the default value. */
	public function toBoolOr($other)
	{
		try {return $this->toBool();}
		catch (IDAOpException $opex) {return $other;}
	}
	
	// Content/null
	/** <p>Test whether this element contains no content.</p>
	 * <p>This is the opposite of hasContent, intended as a convenience for function references.</p>
	 * @return True if this element has no content, otherwise false. */
	public function isNull() {return $this->cRaw === null;}
	/** <p>Test whether this element contains some type of content.</p>
	 * <p>This is the opposite of isNull, intended as a convenience for function references.</p>
	 * @return True if this element has content present, otherwise false. */
	public function hasContent() {return $this->cRaw !== null;}
	/** <p>Erase any existing content from this element.</p>
	 * <p>This is the same as calling clearContent.</p>
	 * @return This element. */
	public function putNull() {return $this->clearContent();}
	/** <p>Erase any existing content from this element.</p>
	 * <p>This is the same as calling putNull.</p>
	 * @return This element. */
	public function clearContent()
	{
		$this->cRaw = null;
		$this->cStr = null;
		$this->cBool = null;
		$this->clearRef();
		$this->clearList();
		$this->clearNum();
		return $this;
	}
	
	// Content/string
	/** <p>Place string content into this element with the provided value, replacing any existing content.</p>
	 * @param s String content value. 
	 * @return This element */
	public function putStr(?String $s) {$this->putNull(); $this->cRaw = $s; $this->cStr = $s; return $this;}
	/** <p>Test whether the content type of this element is a string value.</p>
	 * @return True if this element has string content, otherwise false */
	public function isStr() {return $this->cStr !== null;}
	public function toString() {return $this->cRaw !== null ? $this->cRaw : "?";}
	/** <p>Fetch this element content as a string value.</p>
	 * <p>If this element content is not a string value, an exception is thrown.
	 * To avoid that possibility, use asStrOr to provide a default value.</p>
	 * <p>Use toStr/toStrOr for casting other content types to be a string value.</p>
	 * @exception IDAOpException if this element does not contain string content.
	 * @return String content value. */
	public function asStr()
	{
		if ($this->cStr === null) throw new IDAOpException("Content is not a string value");
		return $this->cStr;
	}
	/** <p>Fetch this element content as a string value.</p>
	 * <p>If this element content is not a string value, the provided default value is returned.</p>
	 * <p>On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
	 * If possible, always call asStrOr() rather than asStrOr(null).</p>
	 * @param other A default value to return if this element does not contain string content.
	 * @return String content value if present, otherwise the default value. */
	public function asStrOr($other=null)
	{
		return $this->cStr !== null ? $this->cStr : $other;
	}
	/** <p>Test whether this element content can be interpreted as a string value (see toStr).
	 * This will always be the case for natural string content (see isStr).</p>
	 * <p>Most content can be cast to string.
	 * Types which cannot be cast are lists, references and nulls.</p>
	 * @return True if this element content can be cast to a string value, otherwise false */
	public function canStr()
	{
		return $this->hasContent() && !$this->isList() && !$this->isRef();
	}
	/** <p>Cast this element content to a string value.</p>
	 * <p>If this element content can not be interpreted as a string value, an exception is thrown.
	 * To avoid that possibility, use toStrOr to provide a default value.</p>
	 * @exception IDAOpException if this element content cannot cast to a string value.
	 * @return String content interpretation. */
	public function toStr()
	{
		if (!$this->canStr()) throw new IDAOpException("Content cannot cast to a string value");
		return $this->cRaw;
	}
	/** <p>Cast this element content to a string value.</p>
	 * <p>If this element content can not be interpreted as a string value, the provided default value is returned.</p>
	 * <p>On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
	 * If possible, always call toStrOr() rather than toStrOr(null).</p>
	 * @param other A default value to return if this element content cannot cast to a string value.
	 * @return String content interpretation if able, otherwise the default value. */
	public function toStrOr($other=null)
	{
		return $this->canStr() ? $this->cRaw : $other;
	}
	
	// Content/number/other
	private function clearNum() {$this->cNum = ""; $this->cDen = "1"; $this->cUnit = "";}
	
	// Content/number/set
	private function refresh_num() {$this->cRaw = $this->cNum . (($this->cDen == "1") ? "" : "/".$this->cDen) . $this->cUnit; $this->cStr = null;}
	/** <p>Set the unit string appended to the number represented by this element content.</p>
	 * <p>Passing null to setNumUnit has no effect.
	 * To clear the unit, call setNumUnit with an empty string.</p>
	 * <p>If this element content is not numeric, an exception is thrown.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return This element. */
	public function setNumUnit($unit)
	{
		if ($unit !== null)
		{
			if (!$this->isNum()) throw new IDAOpException("Content is not numeric");
			$this->cUnit = $unit; $this->refresh_num();
		}
		return $this;
	}
	
	public function putNum($num, $den="1", $unit=null) {if ($unit === null) $unit = $this->cUnit; $this->putNull(); $this->cNum = $num; $this->cDen = $den; $this->cUnit = $unit; $this->refresh_num(); return $this;}
	
	/** <p>Place 64-bit floating-point content into this element with the provided value, replacing any existing content.</p>
	 * <p>If the provided unit string is non-null, also updates the unit string for this number (see setNumUnit).</p>
	 * @param num 64-bit floating-point content value.
	 * @param unit Number unit string.
	 * @return This element */
	public function putDouble(float $value, ?string $unit=null) {return $this->putNum("".$value)->setNumUnit($unit);}
	/** <p>Place 32-bit floating-point content into this element with the provided value, replacing any existing content.</p>
	 * <p>If the provided unit string is non-null, also updates the unit string for this number (see setNumUnit).</p>
	 * @param num 32-bit floating-point content value.
	 * @param unit Number unit string.
	 * @return This element */
	public function putFloat(float $value, ?string $unit=null) {return $this->putNum("".$value)->setNumUnit($unit);}
	/** <p>Place 64-bit integer content into this element with the provided value, replacing any existing content.</p>
	 * <p>If the provided unit string is non-null, also updates the unit string for this number (see setNumUnit).</p>
	 * @param num 64-bit integer content value.
	 * @param unit Number unit string.
	 * @return This element */
	public function putLong(int $value, ?string $unit=null) {return $this->putNum("".$value)->setNumUnit($unit);}
	/** <p>Place 32-bit integer content into this element with the provided value, replacing any existing content.</p>
	 * <p>If the provided unit string is non-null, also updates the unit string for this number (see setNumUnit).</p>
	 * @param num 32-bit integer content value.
	 * @param unit Number unit string.
	 * @return This element */
	public function putInt(int $value, ?string $unit=null) {return $this->putNum("".$value)->setNumUnit($unit);}
	
	// Content/number/get
	/** <p>Test whether the content type of this element is a numeric value.</p>
	 * @return True if this element has numeric content, otherwise false */
	public function isNum()
	{
		return $this->cNum != "";
	}
	/** <p>Fetch the unit string appended to the number represented by this element content.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * If numeric content had no unit specified, an empty string is returned.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return Numeric unit string. */
	public function getNumUnit()
	{
		if (!$this->isNum()) throw new IDAOpException("Content is not numeric");
		return $this->cUnit;
	}
	/** <p>Fetch the numerator string for the number represented by this element content.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * If the number was not a quotient, calling getNumN will return the same result as calling toStr.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return Numerator as a string. */
	public function getNumN()
	{
		if (!$this->isNum()) throw new IDAOpException("Content is not numeric");
		return $this->cNum;
	}
	/** <p>Fetch the denominator string for the number represented by this element content.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * If the number was not a quotient, calling getNumD will return "1".</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return Denominator as a string. */
	public function getNumD()
	{
		if (!$this->isNum()) throw new IDAOpException("Content is not numeric");
		return $this->cDen;
	}
	/** <p>Fetch this element content as a 64-bit floating-point value.</p>
	 * <p>If this element content is not numeric, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain numeric content.
	 * @return 64-bit floating-point content value if present, otherwise the default value. */
	public function asDoubleOr($other)
	{
		return $this->isNum() ? $this->c_double() : $other;
	}
	/** <p>Fetch this element content as a 32-bit floating-point value.</p>
	 * <p>If this element content is not numeric, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain numeric content.
	 * @return 32-bit floating-point content value if present, otherwise the default value. */
	public function asFloatOr($other)
	{
		return $this->isNum() ? $this->c_float() : $other;
	}
	/** <p>Fetch this element content as a 64-bit integer value.</p>
	 * <p>If this element content is not numeric, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain numeric content.
	 * @return 64-bit integer content value if present, otherwise the default value. */
	public function asLongOr($other)
	{
		return $this->isNum() ? $this->c_long() : $other;
	}
	/** <p>Fetch this element content as a 32-bit integer value.</p>
	 * <p>If this element content is not numeric, the provided default value is returned.</p>
	 * @param other A default value to return if this element does not contain numeric content.
	 * @return 32-bit integer content value if present, otherwise the default value. */
	public function asIntOr($other)
	{
		return $this->isNum() ? $this->c_int() : $other;
	}
	/** <p>Fetch this element content as a 64-bit floating-point value.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * To avoid that possibility, use asDoubleOr to provide a default value.</p>
	 * <p>Use toDouble/toDoubleOr for casting other content types to be a 64-bit floating-point value.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return 64-bit floating-point content value. */
	public function asDouble()
	{
		if (!$this->isNum()) throw new IDAOpException("Content is not numeric");
		return $this->c_double();
	}
	/** <p>Fetch this element content as a 32-bit floating-point value.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * To avoid that possibility, use asFloatOr to provide a default value.</p>
	 * <p>Use toFloat/toFloatOr for casting other content types to be a 32-bit floating-point value.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return 32-bit floating-point content value. */
	public function asFloat()
	{
		if (!$this->isNum()) throw new IDAOpException("Content is not numeric");
		return $this->c_float();
	}
	/** <p>Fetch this element content as a 64-bit integer value.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * To avoid that possibility, use asLongOr to provide a default value.</p>
	 * <p>Use toLong/toLongOr for casting other content types to be a 64-bit integer value.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return 64-bit integer content value. */
	public function asLong()
	{
		if (!$this->isNum()) throw new IDAOpException("Content is not numeric");
		return $this->c_long();
	}
	/** <p>Fetch this element content as a 32-bit integer value.</p>
	 * <p>If this element content is not numeric, an exception is thrown.
	 * To avoid that possibility, use asIntOr to provide a default value.</p>
	 * <p>Use toInt/toIntOr for casting other content types to be a 32-bit integer value.</p>
	 * @exception IDAOpException if this element does not contain numeric content.
	 * @return 32-bit integer content value. */
	public function asInt()
	{
		if (!$this->isNum()) throw new IDAOpException("Content is not numeric");
		return $this->c_int();
	}
	
	private function c_double()
	{
		return ($this->cDen == "1") ? $this->n_float($this->cNum) : ($this->n_float($this->cNum)/$this->n_float($this->cDen));
	}
	private function c_float()
	{
		return $this->c_double();
	}
	private function c_long()
	{
		return ($this->cDen == "1") ? $this->n_int($this->cNum) : intval($this->n_float($this->cNum)/$this->n_float($this->cDen));
	}
	private function c_int()
	{
		return $this->c_long();
	}
	private function n_int(String $str)
	{
		if (strpos($str, "0x") === 0) {return $this->n_hex($str);}
		if (strpos($str, "0b") === 0) {return $this->n_bin($str);}
		return intval($str);
	}
	private function n_float(String $str)
	{
		if (strpos($str, "0x") === 0) {return floatval($this->n_hex($str));}
		if (strpos($str, "0b") === 0) {return floatval($this->n_bin($str));}
		return floatval($str);
	}
	private function n_hex(String $str) {return hexdec(substr($str, 2));}
	private function n_bin(String $str) {return bindec(substr($str, 2));}
	
	// Content/list
	private function clearList() {$this->cList = null;}
	private function refresh_list() {$this->cRaw = "List(".$this->numItems().")"; $this->cStr = null;}
	
	public function iterator() {return IDA::$port->list_i($this->cList);}
	
	// Content/list/set
	/** Returns this element */
	public function putList($l = false)
	{
		$this->putNull();
		if ($l !== false)
		{
			if ($l !== null)
			{
				$this->cList = $l; $this->refresh_list();
				$last = null;
				foreach ($l as $item)
				{
					$item->link($last, null);
					$last = $item;
				}
			}
		}
		else
		{
			$this->cList = array();
			$this->refresh_list();
		}
		return $this;
	}
	
	// Content/list/get
	/** <p>Test whether the content type of this element is a list.</p>
	 * <p>This reports whether the content is a list or non-list, as opposed to whether at least one list item is present.
	 * The primary purpose is to ensure the list exists before calling asList and/or iterating over its elements.</p>
	 * <p>Use numItems to test for an empty list (0) or at least one item present (greater than 0).</p>
	 * @return True if this element has list content, otherwise false */
	public function isList() {return $this->cList !== null;}
	/** <p>Fetch this element content as a list (containing other elements).</p>
	 * <p>If this element content is not a list, an exception is thrown.
	 * To avoid that possibility, use asListOr to provide a default value.</p>
	 * @exception IDAOpException if this element does not contain list content.
	 * @return List content. */
	public function asList()
	{
		if ($this->cList === null) throw new IDAOpException("Content is not a list");
		return $this->cList;
	}
	/** <p>Fetch this element content as a list (containing other elements).</p>
	 * <p>If this element content is not a list, the provided default value is returned.</p>
	 * <p>On platforms where lists/vectors are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed list, such that code will not be fully portable if null is provided.
	 * If possible, always call asListOr() rather than asListOr(null).</p>
	 * @param other A default value to return if this element does not contain list content.
	 * @return List content if present, otherwise the default value. */
	public function asListOr($other = null)
	{
		return $this->isList() ? $this->cList : $other;
	}
	/** <p>Fetch the number of content list items within this element.</p>
	 * <p>If no content list is present, the sentinel value -1 is returned.
	 * If 0 is returned, the list is present but empty.</p>
	 * @return The content list size if a list is present, otherwise -1 */
	public function numItems()
	{
		return $this->isList() ? count($this->cList) : -1;
	}
	
	/** <p>Test whether this element contains a list item with the provided name.
	 * If it does, an equivalent call to getItem will return an element.</p>
	 * <p>If false is returned, this can be either due to having no list item of that name or due to being a non-list.
	 * Use isList to separate those two cases.</p>
	 * @param key A list item name to be checked
	 * @return True if a content list item exists with the provided name, otherwise false */
	public function hasItem($key) {return $this->getItemOr($key) !== null;}
	
	/** <p>Fetch the item element at the provided index in this element's content list.</p>
	 * <p>If this element content is not a list or has too few items in the list, an exception is thrown.
	 * To avoid that possibility, use getItemAtOr to provide a default value.</p>
	 * @param index A list item index to be recalled.
	 * @exception IDAOpException if this element has no such list item.
	 * @return The indexed content list item element. */
	public function getItemAt($index)
	{
		if ($this->numItems() <= $index) throw new IDAOpException("Invalid list item index");
		return $this->cList[$index];
	}
	/** <p>Fetch the item element at the provided index in this element's content list.</p>
	 * <p>If this element content is not a list or has too few items in the list, the provided default value is returned.</p>
	 * @param index A list item index to be recalled.
	 * @param other A default value to return if this element has no such list item.
	 * @return The indexed content list item element if present, otherwise the default value. */
	public function getItemAtOr($index, $other = null)
	{
		return $this->numItems() > $index ? $this->cList[$index] : $other;
	}
	
	/** <p>Fetch the first item element with the provided name in this element's content list.</p>
	 * <p>If this element content is not a list or has no item of that name, an exception is thrown.
	 * To avoid that possibility, use getItemOr to provide a default value.</p>
	 * @param index A list item name to be recalled.
	 * @exception IDAOpException if this element has no such list item.
	 * @return The named content list item element. */
	public function getItem($key)
	{
		$item = $this->getItemOr($key);
		if ($item === null) throw new IDAOpException("Item not found");
		return $item;
	}
	/** <p>Fetch the first item element with the provided name in this element's content list.</p>
	 * <p>If this element content is not a list or has no item of that name, the provided default value is returned.</p>
	 * @param key A list item name to be recalled.
	 * @param other A default value to return if this element has no such list item.
	 * @return The named content list item element if present, otherwise the default value. */
	public function getItemOr($key, $other = null)
	{
		if ($this->isList()) foreach ($this->asList() as $x) if ($x->isName($key)) {return $x;}
		return $other;
	}
	
	/** <p>Cast to a string value the first item element with the provided name in this element's content list.</p>
	 * <p>If this element content is not a list, has no item of that name or can not be interpreted as a string value, an exception is thrown.
	 * To avoid that possibility, use getItemStrOr to provide a default value.</p>
	 * @param key A list item name to be recalled.
	 * @exception IDAOpException if this element has no such list item, or the item content cannot cast to a string value.
	 * @return String content interpretation of the named list item element. */
	public function getItemStr($key)
	{
		return $this->getItem($key)->toStr();
	}
	/** <p>Cast to a string value the first item element with the provided name in this element's content list.</p>
	 * <p>If this element content is not a list, has no item of that name or can not be interpreted as a string value, the provided default value is returned.</p>
	 * <p>On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
	 * The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
	 * If possible, always call getItemStrOr(key) rather than getItemStrOr(key, null).</p>
	 * @param key A list item name to be recalled.
	 * @param other A default value to return if this element has no such list item, or the item content cannot cast to a string value.
	 * @return String content interpretation of the named list item element if possible, otherwise the default value. */
	public function getItemStrOr($key, $other=null)
	{
		$item = $this->getItemOr($key);
		return ($item !== null) ? $item->toStrOr($other) : $other;
	}
	
	/** <p>Fetch the first item element in this element's content list.</p>
	 * <p>If this element content is not a list or is an empty list, an exception is thrown.
	 * To avoid that possibility, use getItemFirstOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no such list item.
	 * @return The first content list item element. */
	public function getItemFirst()
	{
		if ($this->numItems() < 1) throw new IDAOpException($this->isList() ? "Content list is empty" : "Element content is not a list");
		return $this->cList[0];
	}
	/** <p>Fetch the first item element in this element's content list.</p>
	 * <p>If this element content is not a list or is an empty list, the provided default value is returned.</p>
	 * @param other A default value to return if this element has no such list item.
	 * @return The first content list item element if present, otherwise the default value. */
	public function getItemFirstOr($other = null)
	{
		return $this->numItems() > 0 ? $this->cList[0] : $other;
	}
	/** <p>Fetch the last item element in this element's content list.</p>
	 * <p>If this element content is not a list or is an empty list, an exception is thrown.
	 * To avoid that possibility, use getItemLastOr to provide a default value.</p>
	 * @exception IDAOpException if this element has no such list item.
	 * @return The last content list item element. */
	public function getItemLast()
	{
		$n = $this->numItems();
		if ($n < 1) throw new IDAOpException($this->isList() ? "Content list is empty" : "Element content is not a list");
		return $this->cList[$n-1];
	}
	/** <p>Fetch the last item element in this element's content list.</p>
	 * <p>If this element content is not a list or is an empty list, the provided default value is returned.</p>
	 * @param other A default value to return if this element has no such list item.
	 * @return The last content list item element if present, otherwise the default value. */
	public function getItemLastOr($other = null)
	{
		$n = $this->numItems();
		return $n > 0 ? $this->cList[$n-1] : $other;
	}
	
	// Content/list/add
	/** Insert a item element at the end of the content list.<br>
	 * If this element does not have list content, a new list will be created if the makeList flag is set, otherwise an exception is thrown.<br>
	 * @param el The item element to be stored.
	 * @param makeList Whether to overwrite existing non-list content.
	 * @exception IDAOpException if this element content is not a list and the makeList flag is false.
	 * @return This element. */
	public function addItem(IDA $el, bool $makeList=true)
	{
		if (!$this->isList() && !$makeList) {throw new IDAOpException("Cannot append items onto a non-list");}
		if (!$this->isList()) $this->putList();
		$el->link($this->getItemLastOr(), null);
		$this->cList[] = $el;
		$this->refresh_list();
		return $this;
	}
	
	/** Insert a item element at the specified content list index position.<br>
	 * If this element does not have list content, an exception is thrown.<br>
	 * @param el The item element to be stored.
	 * @param pos The list index position at which the item should be inserted.
	 * @exception IDAOpException if the specified position is below zero or above the present list size, or if this element content is not a list.
	 * @return This element. */
	public function addItemAt(IDA $el, int $pos)
	{
		if (!$this->isList()) {throw new IDAOpException("Cannot merge items into a non-list");}
		IDA::list_add($this->cList, $el, $pos, "item");
		$this->refresh_list();
		return $this;
	}
	
	/** Create and insert a new item element at the end of the content list, with the provided element name.<br>
	 * If this element does not have list content, a new list will be created if the makeList flag is set, otherwise an exception is thrown.<br>
	 * @param name Name string for the created item element.
	 * @param makeList Whether to overwrite existing non-list content.
	 * @exception IDAOpException if this element content is not a list and the makeList flag is false.
	 * @return The created item element. */
	public function newItem(?string $name, $makeList=true)
	{
		$el = new IDA($name);
		$this->addItem($el, $makeList);
		return $el;
	}
	
	// Content/list/remove
	/** Remove the item at the specified content list index position.<br>
	 * If this element does not have list content, an exception is thrown.<br>
	 * @param pos The list index position of an item which should be removed.
	 * @exception IDAOpException if the specified position is below zero or beyond the present list size, or this element does not have list content. */
	public function remItemAt(int $pos)
	{
		if (!$this->isList()) {throw new IDAOpException("Cannot remove items from a non-list");}
		IDA::list_del($this->cList, $pos, "item");
		$this->refresh_list();
	}
	/** Remove the provided content list item.<br>
	 * If the item is absent or this element does not have list content, the call has no effect.<br>
	 * @param el The item element to be removed.
	 * @return True if an item was removed, otherwise false. */
	public function remItem(IDA $el)
	{
		if ($this->isList() && IDA::list_rem($this->cList, $el)) {$this->refresh_list(); return true;}
		return false;
	}
	/** Remove the first content list item with the provided element name.<br>
	 * If no such item exists or this element does not have list content, the call has no effect.<br>
	 * @param name Name string of the item element to remove.
	 * @return True if an item was removed, otherwise false. */
	public function unItem(string $name)
	{
		$item = $this->getItemOr($name);
		if ($item !== null) {$this->remItem($item); return true;}
		return false;
	}
	
	// Object/equals
	private function eq_inner($obj)
	{
		if ($obj === null || !($obj instanceof IDA)) {return false;}
		if ($obj === $this) {return true;}
		if (!$this->eqName($obj)) {return false;}
		if (!$this->eqParams($obj)) {return false;}
		if (!$this->eqContent($obj)) {return false;}
		return true;
	}
	/** <p>Perform a top-level comparison of this element and the provided element.</p>
	 * <p>To be equal, both elements must match when interpreted as full articles.
	 * The element names, parameters and content must all match recursively.</p>
	 * <p>Note that content matches consider whether the <i>information</i> is the same.
	 * See eqContent for details.</p>
	 * @param obj An article element for comparison
	 * @param adjacents Whether to include adjacent elements in the comparison
	 * @return True if the elements represent matching articles, otherwise false */
	public function eq($obj, $adjacent=false)
	{
		if ($obj === $this) return true;
		if ($obj === null || !($obj instanceof IDA)) return false;
		if ($this->txRefLabel($adjacent) != $obj->txRefLabel($adjacent)) return false;
		if (!$this->eq_inner($obj)) return false;
		
		if ($adjacent)
		{
			$a = $this;
			$b = $obj;
			$el = $b;
			while ($a->hasPrev() && $b->hasPrev())
			{
				$a = $a->getPrevOr(); $b = $b->getPrevOr();
				if (!$a->eq_inner($b)) {return false;}
			}
			if ($a->hasPrev() || $b->hasPrev()) {return false;}
			
			$a = $this;
			$b = $el;
			while ($a->hasNext() && $b->hasNext())
			{
				$a = $a->getNextOr(); $b = $b->getNextOr();
				if (!$a->eq_inner($b)) {return false;}
			}
			if ($a->hasNext() || $b->hasNext()) {return false;}
		}
		
		return true;
	}
	
	/** Compare the names of this element and the provided element.
	 * @param obj An element for name comparison
	 * @return True if the names match, otherwise false */
	public function eqName($obj) {return ($obj instanceof IDA) && $this->isName($obj->getNameOr());}
	/** <p>Compare the contents of this element and the provided element.</p>
	 * <p>The content type must match, and the contents must match in a type-dependent way
	 * such that each element contains the same <i>information</i>. Numbers must have the
	 * same precision, and scope references must target the same element relative to each
	 * scope. Foreign references are equal by default.</p>
	 * @param obj An element for content comparison
	 * @return True if the contents match, otherwise false */
	public function eqContent($obj)
	{
		if (!($obj instanceof IDA)) {return false;}
		if ($this->cRaw != $obj->cRaw) {return false;}
		if ($this->cStr != $obj->cStr) {return false;}
		if ($this->cNum != $obj->cNum) {return false;}
		//if ($this->cDen != $obj->cDen) {return false;} // Assured by testing cRaw
		//if ($this->cUnit != $obj->cUnit) {return false;} // Assured by testing cRaw
		if (!IDA::list_equals($this->cList, $obj->cList)) {return false;}
		if ($this->cBool != $obj->cBool) {return false;}
		if (!$this->eqContent_ref($this->cRef, $obj->cRef)) return false;
		
		return true;
	}
	private function eqContent_ref($ref_a, $ref_b)
	{
		if (($ref_a !== null) ^ ($ref_b !== null)) return false;
		if ($ref_a === $ref_b) return true;
		if ($ref_a->hasRefName() ^ $ref_b->hasRefName()) return false;
		if (!$ref_a->hasRefName()) return true;
		return $ref_a->getRefName() == $ref_b->getRefName();
	}
	/** Compare recursively the parameters of this element and the provided element.
	 * @param obj An element for parameter comparison
	 * @return True if the parameters match, otherwise false */
	public function eqParams($obj)
	{
		return ($obj instanceof IDA) && IDA::list_equals($this->parameters, $obj->parameters);
	}
	

	// Object/capture
	private $repObj = null;
	/** <p>Top-level article encode operation for a full object tree.
	 * This method calls objEncode, so the top-level object must implement idaEncode.</p>
	 * 
	 * <p>After all encoding is complete, any object references declared via setRefObj will be established as element references in the article.</p>
	 * @param obj A top-level object, which must implement idaEncode, to encode and record as this article element.
	 * @return This article element. */
	public function objCapture($obj)
	{
		$this->objEncode($obj);
		
		$this->capture_objs();
		$this->capture_refs();
		
		return $this;
	}
	/** <p>Encode one member object via idaEncode, with automatic element name via idaName.
	 * This element will become "representative" of the object, which is used to encode references via setRefObj.</p>
	 * 
	 * <p>This method should be called from within idaEncode itself, to record members which also implement the IDAObj codec interface.
	 * For the top-level operation, user code should call idaCapture instead to ensure that all object references are recorded.</p>
	 * @param obj An object, which must implement idaEncode, to encode and record as this element.
	 * @return This element. */
	public function objEncode($obj)
	{
		$this->setName($obj->idaName());
		$this->repObj = $obj;
		$obj->idaEncode($this);
		
		return $this;
	}
	private function capture_objs()
	{
		if ($this->repObj !== null) {$this->repObj->idaCaptureObj = $this;}
		if ($this->hasParams()) foreach ($this->parameters as $p) $p->capture_objs();
		if ($this->isList()) foreach ($this->cList as $i) $i->capture_objs();
	}
	private function capture_refs()
	{
		if ($this->cRefObj !== null) {$this->putRef($this->cRefObj->idaCaptureObj);}
		if ($this->hasParams()) foreach ($this->parameters as $p) $p->capture_refs();
		if ($this->isList()) foreach ($this->cList as $i) $i->capture_refs();
	}
	
	// Object/restore
	private $repObjActions = null;
	private function on_rep_obj($callback)
	{
		if ($this->repObj !== null) {$callback($this->repObj);}
		else
		{
			if ($this->repObjActions === null) $this->repObjActions = array();
			$this->repObjActions[] = $callback;
		}
	}
	private function restore_check()
	{
		if ($this->repObjActions !== null) throw new IDAOpException("Decode process failed to resolve an object reference");
	}
	/** <p>Top-level article decode operation for a full object tree.
	 * This method calls objDecode, so the top-level object must implement idaDecode.</p>
	 * 
	 * <p>After all decoding is complete, the integrity will be checked to ensure that all object reference requests were resolved.
	 * If any calls to withRefObj did not receive the deferred result, an exception is thrown.
	 * This occurs if the article contained an element reference for which no object was constructed via objDecode.</p>
	 * @param obj A top-level object, which must implement idaDecode, to become the result of decoding this article element.
	 * @return The input object. */
	public function objRestore($obj)
	{
		$this->objDecode($obj);
		$this->walk(function($x) {$x->restore_check();}, true, true);
		return $obj;
	}
	/** <p>Decode one member object via idaDecode.
	 * This element will become "representative" of the object, which is used to decode references via getRefObj or withRefObj.</p>
	 * 
	 * <p>This method should be called from within idaDecode itself, to create members which also implement the IDAObj codec interface.
	 * For the top-level operation, user code should call idaRestore instead to verify that all object reference requests were resolved.</p>
	 * @param obj An object, which must implement idaDecode, to become the result of decoding this element.
	 * @return The input object. */
	public function objDecode($obj)
	{
		$this->repObj = $obj;
		if ($this->repObjActions !== null)
		{
			foreach ($this->repObjActions as $c) $c($obj);
			$this->repObjActions = null;
		}
		$obj->idaDecode($this);
		return $obj;
	}
	
	// Deserialisation
	/** Read from the provided input handle, parse it as markup and use the result to build this element.<br>
	 * After the parser is done, this element will become the top-level entry element from the markup stream.
	 * @param input An open input handle for a stream containing markup data
	 * @return This element */
	public function inStream($input) {return IDA_RX::inObj($this, $input);}
	/** Read the provided string, parse it as markup and use the result to build this element.<br>
	 * After the parser is done, this element will become the top-level entry element from the markup stream.
	 * @param input A string containing markup data
	 * @return This element */
	public function inBytes($input)
	{
		$buf = false;
		try
		{
			$buf = fopen("php://temp", "w+");
			@fwrite($buf, $input);
			rewind($buf);
			return $this->inStream($buf);
		}
		finally
		{
			if ($buf !== false) fclose($buf);
		}
	}
	/** Read the provided string, parse it as markup and use the result to build this element.<br>
	 * After the parser is done, this element will become the top-level entry element from the markup stream.
	 * @param input A string containing markup data
	 * @return This element */
	public function inString($input) {return $this->inBytes($input);}
	/** Read from the file at the provided path, parse it as markup and use the result to build this element.<br>
	 * After the parser is done, this element will become the top-level entry element from the markup stream.
	 * @param input A path to a file containing markup data
	 * @return This element */
	public function inFile($path)
	{
		$res = false;
		try
		{
			$res = fopen($path, "r");
			return $this->inStream($res);
		}
		finally
		{
			if ($res !== false) fclose($res);
		}
	}
	
	// Serialisation
	/** Encode this element as top-level markup and write it to the provided output handle.<br>
	 * If <i>adjacents</i> is true, all preceding and following elements will also be encoded.<br>
	 * If a preceding element was encoded that way, the markup will contain an entry instruction.<br>
	 * Any write errors will be sent back to the caller.
	 * @param output An open output handle to receive a stream of markup data
	 * @param adjacents Whether to write markup for elements adjacent to this one */
	public function outStream($output, bool $adjacents = false)
	{
		$tx = new IDA_TX($output);
		$tx->idaWrite($this, $adjacents);
	}
	/** Encode this element as top-level markup and write it to the provided output handle.<br>
	 * If <i>adjacents</i> is true, all preceding and following elements will also be encoded.<br>
	 * If a preceding element was encoded that way, the markup will contain an entry instruction.
	 * @param output An open output handle to receive a stream of markup data
	 * @param adjacents Whether to write markup for elements adjacent to this one
	 * @return True on success, or false if a write error occurred */
	public function outPrint($output, bool $adjacents = false)
	{
		$tx = new IDA_TX($output);
		return $tx->idaPrint($this, $adjacents);
	}
	/** Encode this element as top-level markup and return it as a byte array.<br>
	 * If <i>adjacents</i> is true, all preceding and following elements will also be encoded.<br>
	 * If a preceding element was encoded that way, the markup will contain an entry instruction.
	 * @param adjacents Whether to write markup for elements adjacent to this one
	 * @return A byte array containing the markup data */
	public function outBytes(bool $adjacents = false)
	{
		return $this->outString($adjacents);
	}
	/** Encode this element as top-level markup and return it as a byte array.<br>
	 * If <i>adjacents</i> is true, all preceding and following elements will also be encoded.<br>
	 * If a preceding element was encoded that way, the markup will contain an entry instruction.
	 * @param adjacents Whether to write markup for elements adjacent to this one.
	 * @return A string containing the markup data. */
	public function outString(bool $adjacents = false)
	{
		$buf = fopen("php://temp", "w+");
		$tx = new IDA_TX($buf);
		$tx->idaPrint($this, $adjacents);
		$ptr = ftell($buf);
		rewind($buf);
		$output = $ptr > 0 ? fread($buf, $ptr) : "";
		fclose($buf);
		return $output;
	}
	/** Encode this element as top-level markup and write it to the file at the provided path.<br>
	 * If <i>adjacents</i> is true, all preceding and following elements will also be encoded.<br>
	 * If a preceding element was encoded that way, the markup will contain an entry instruction.<br>
	 * Any write errors will be sent back to the caller.
	 * @param path A destination file path for the markup data.
	 * @param adjacents Whether to write markup for elements adjacent to this one. */
	public function outFile($path, $adjacents = false)
	{
		$res = false;
		try
		{
			$res = fopen($path, "w");
			$this->outStream($res, $adjacents);
		}
		finally
		{
			fclose($res);
		}
	}
	
	// Construction
	function __construct($name = null)
	{
		$this->includePwd = IDA::$gIncludePwd;
		$this->reset();
		if ($name !== null) {$this->setName($name);}
	}
	
	// Initialisation
	/** <p>Initialisate this element anew.</p>
	 * <p>All existing information comprising this element will be discarded.
	 * The result will be an anonymous, null-content element with no parameters list, no adjacent elements and default input/output options.</p>
	 * @return This element. */
	public function reset()
	{
		$this->putNull();
		$this->cRefName = null;
		$this->repObj = null;
		$this->repObjActions = null;
		
		$this->itemClasses = null;
		$this->name = null;
		$this->next = null;
		$this->prev = null;
		$this->clearParams();
		
		$this->opts = new IDAOptions(IDAOptions::$defaultOptions);
		$this->scope = null;
		$this->itemTemplates = null;
		
		$this->inputListDepth = -1;
		$this->inputListLength = -1;
		
		return $this;
	}
	
	// Copies
	/** <p>Create a new element as a copy of all aspects of this element.
	 * Any parameters or list content will be cloned recursively.</p>
	 * @param adjacents Whether to include adjacent elements in the cloning process
	 * @return The copy element */
	public function copy($adjacent = false) {return $this->copyTo(new IDA(), $adjacent);}
	/** <p>Modify the provided element to become a copy of all aspects of this element.
	 * Any parameters or list content will be cloned recursively.</p>
	 * @param copy An element to become a copy of this element
	 * @param adjacents Whether to include adjacent elements in the cloning process
	 * @return The copy element */
	public function copyTo(IDA $copy, $adjacent = false)
	{
		$has_ref = $this->txRefLabel($adjacent) > 0;
		IDA::copy_impl($this, $copy, $adjacent, $adjacent);
		if ($has_ref) $copy->rxRefResolve($adjacent);
		return $copy;
	}
	
	private function copy_inner()
	{
		return IDA::copy_impl($this, new IDA(), false, false);
	}
	private static function copy_impl(IDA $orig, IDA $copy, $scanPrev, $scanNext)
	{
		$copy->copyName($orig);
		$copy->copyParams($orig);
		$copy->copyContent($orig);
		$copy->copyOptions($orig);
		
		if ($scanPrev && ($a = $orig->getPrevOr()) !== null) {$copy->linkPrev(IDA::copy_impl($a, new IDA(), true, false));}
		if ($scanNext && ($a = $orig->getNextOr()) !== null) {$copy->linkNext(IDA::copy_impl($a, new IDA(), false, true));}
		
		return $copy;
	}
	
	// Utility
	/** <p>Call the provided function once for each element within this article.</p>
	 * <p>This element is passed first, then its parameters (if any), then its list content items (if any).
	 * Parameters and list items are treated recursively until the full article has been traversed.</p>
	 * <p>Optionally, the walk may also traverse adjacent elements recursively, requested via the reverse and forward flags.</p>
	 * @param func A callback function which will be supplied with article elements.
	 * @param rev Whether to include elements adjacent to this one in the reverse direction.
	 * @param fwd Whether to include elements adjacent to this one in the forward direction. */
	public function walk($func, $rev, $fwd)
	{
		$func($this);
		if ($this->hasParams()) foreach ($this->parameters as $p) $p->walk($func, false, false);
		if ($this->isList()) foreach ($this->cList as $i) $i->walk($func, false, false);
		if ($fwd && $this->hasNext()) $this->getNextOr()->walk($func, false, true);
		if ($rev && $this->hasPrev()) $this->getPrevOr()->walk($func, true, false);
	}
	
	/** <p>Check the integrity of article element.
	 * If a problem is detected which would lead to errors or corrupted output, an exception is thrown with details in the exception text.</p>
	 * <p>The tests can be performed with adjacent elements enabled or disabled, matching the markup output modes.
	 * Note that disabling adjacents may avoid some problems (such as with linkage), but may also induce others (such as with references).</p>
	 * <p>The top-level element linkage is tested to ensure that there are no loops, and that adjacent elements are doubly-linked correctly.
	 * A bad double-link can disrupt the output flow, and a loop can cause infinite iteration.</p>
	 * <p>Any elements with reference content are tested for foreign references, which become lost/null during output.</p>
	 * @param adjacents Whether to extend the test to include elements adjacent to this one.
	 * @exception IDAOpException if bad integrity is detected. */
	public function validate($adjacents)
	{
		$this->validate_loop($adjacents);
		$this->validate_foreign($adjacents);
	}
	private function validate_loop($adjacents)
	{
		if (!$adjacents) return;
		$fwd = function($x) {return $x->getNextOr();};
		$rev = function($x) {return $x->getPrevOr();};
		$this->validate_loop_f($fwd, $rev);
		$this->validate_loop_f($rev, $fwd);
	}
	private function validate_loop_f($step, $back)
	{
		$a = $this;
		$b = $this;
		
		while (true)
		{
			$last = $b;
			$b = $step($b);
			if ($b === null) return;
			if ($back($b) !== $last) throw new IDAOpException("Article contains a broken link");
			
			$last = $b;
			$b = $step($b);
			if ($b === null) return;
			if ($back($b) !== $last) throw new IDAOpException("Article contains a broken link");
			
			$a = $step($a);
			if ($a === $b) throw new IDAOpException("Article contains an adjacency loop");
		}
	}
	private function validate_foreign($adjacents)
	{
		$foreign = [];
		$this->walk(function($x) use(&$foreign) {if ($x->isRef()) if (!IDA::$port->list_contains($foreign, $x->asRefOr())) $foreign[] = $x->asRefOr();}, $adjacents, $adjacents);
		$this->walk(function($x) use(&$foreign) {if (IDA::$port->list_contains($foreign, $x)) IDA::$port->list_rem($foreign, $x);}, $adjacents, $adjacents);
		if (count($foreign) > 0) throw new IDAOpException("Article contains a foreign reference");
	}
	
	
	public static function list_add(array &$list, IDA $el, $pos, String $noun)
	{
		$size = count($list);
		
		if ($pos > 0)
		{
			if ($pos > $size) {throw new IDAOpException("Cannot add ".$noun." beyond position ".$size." (request was ".$pos.")");}
		}
		else if ($pos < 0)
		{
			if ($pos < -$size) {throw new IDAOpException("Cannot add ".$noun." before position -".$size." (request was ".$pos.")");}
			$pos += $size;
		}
		
		if ($pos == 0)
		{
			if ($size > 0) {$el->link(null, $list[0]);}
			array_unshift($list, $el);
		}
		else if ($pos == $size)
		{
			$el->link($list[$size-1], null);
			$list[] = $el;
		}
		else // size must be at least 2, pos cannot be at either end
		{
			$el->link($list[$pos-1], $list[$pos]);
			array_splice($list, $pos, 0, [$el]);
		}
	}
	public static function list_del(array &$list, $pos, String $noun)
	{
		$size = count($list);
		
		if ($pos > 0)
		{
			if ($pos > $size-1) {throw new IDAOpException("Cannot remove item beyond position ".($size-1)." (request was ".$pos.")");}
		}
		else if ($pos < 0)
		{
			if ($pos < -$size) {throw new IDAOpException("Cannot remove item before position -".$size." (request was ".$pos.")");}
			$pos += $size;
		}
		
		$rem = array_splice($list, $pos, 1);
		$rem[0]->link(null, null);
		return $rem[0];
	}
	
	// Syntax
	static function id0($c) {return ($c>='a' && $c<='z') || ($c>='A' && $c<='Z') || $c == '_' || $c == '/';}
	static function id1($c)
	{
		return IDA::id0($c) || ($c>='0' && $c<='9') || $c == '-' || $c == '.'
			|| $c == '!' || $c == '@' || $c == '^' || $c == '&' || $c == '?';
	}
	static function comment1($c) {return $c == '#';}
	static function comment2($c1, $c2)
	{
		return ($c1 == '/' && ($c2 == '/' || $c2 == '*'))
			|| ($c1 == '<' && ($c2 == '?' || $c2 == '!'));
	}
	
	// Compatibility/port through
	static $port = null;
	// (END)
	
	// Compatibility
	public static function eq_obj($a, $b)
	{
		return ($a instanceof IDA) ? $a->eq($b) : ($a === $b);
	}
	
	public static function list_rem(array &$l, IDA &$el)
	{
		if (IDA::$port->list_rem($l, $el)) {$el->link(null, null); return true;} else return false;
	}
	
	public static function list_equals($la, $lb)
	{
		if ($la === null || $lb === null) return $la == $lb;
		$ia = IDA::$port->list_i($la);
		$ib = IDA::$port->list_i($lb);
		while ($ia->hasNext() && $ib->hasNext())
		{
			if (!$ia->next()->eq($ib->next())) {return false;}
		}
		return !($ia->hasNext() || $ib->hasNext());
	}
	
}

// From Compatibility/port
IDA::$port = new IDA_Port();

// CLI
if (realpath($_SERVER["SCRIPT_FILENAME"]) == realpath(__FILE__))
{
	header("Content-Type: text/plain");
	exit((@isset($argc) && @isset($argv)) ? IDA_CLI::cli($argc, $argv) : IDA_CLI::cli());
}

