#    Copyright 2022 IDA Markup, ida-markup.com
# 
#    Licensed under the Apache License, Version 2.0 (the "License");
#    you may not use this file except in compliance with the License.
#    You may obtain a copy of the License at
# 
#        http://www.apache.org/licenses/LICENSE-2.0
# 
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS,
#    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#    See the License for the specific language governing permissions and
#    limitations under the License.

# Namespace

import sys
import io
import types
import re
import os

# Options
class IDAOptions(object):
	def __init__(self, options = None):
		if options != None:
			self.copyFrom(options)
		else:
			# Default RX
			self.rxXmlReparseStrings = False
			
			# Default TX
			self.txUnicode = False
	
	def copyFrom(self, options):
		# Copy RX
		self.rxXmlReparseStrings = options.rxXmlReparseStrings
		
		# Copy TX
		self.txUnicode = options.txUnicode
		

IDAOptions.defaultOptions = IDAOptions()

# Object interface
# Intended use via multiple inheritance
class IDAObj:
	# Override to offer a default element name.
	def idaName(self):
		return None
	# Override to enable capturing of this object into the provided storage element.<br>
	# Usage note: If applicable, implementing classes can guarantee not to throw.<br>
	# @return The input element.
	def idaEncode(self, el):
		raise IDAOpException("IDA encode not available")
	# Override to enable restoration of this object from the provided storage element.<br>
	# @return This object.
	def idaDecode(self, el):
		raise IDAOpException("IDA decode not available")
	
# Port begin
class IDA_Iterator(object):
	def hasNext(self):
		return False
	def getNext(self):
		return None
class IDA_MutableSeq(IDA_Iterator):
	def __init__(self):
		self.funcHas = lambda: False
		self.funcGet = lambda: None
	def hasNext(self):
		return (self.funcHas)()
	def getNext(self):
		return (self.funcGet)()
class IDA_Port(object):
	
	# Port iterators
	# Iterator/MutableSeq in Port begin
	def empty_i(self):
		return IDA_Iterator()
	@staticmethod
	def list_i__next(l, i):
		x = l[i.n]
		i.n += 1
		return x
	def list_i(self, l):
		if l == None:
			raise IDAOpException("Internal list iteration error")
		i = IDA_MutableSeq()
		i.n = 0
		i.lim = len(l)
		i.funcHas = lambda: i.n < i.lim
		i.funcGet = lambda: IDA_Port.list_i__next(l, i)
		return i
	def repeat_i(self, obj):
		i = IDA_MutableSeq()
		i.array_item_template = obj
		i.funcHas = lambda: True
		i.funcGet = lambda: i.array_item_template
		return i
	
	# Port arrays
	def list_contains(self, l, item):
		return item in l
	def list_rem(self, l, item):
		for i in range(0, len(l)):
			if IDA.eq_obj(item, l[i]):
				l[:] = l[:i] + l[i+1:]
				return True
		return False
	
	# Port streams
	def si_allMatch(self, i, predicate):
		while i.hasNext():
			if not predicate(i.getNext()):
				return False
		return True
	def si_anyMatch(self, i, predicate):
		while i.hasNext():
			if predicate(i.getNext()):
				return True
		return False
	def si_noneMatch(self, i, predicate):
		return not self.si_anyMatch(i, predicate)
	
	def i_map(self, i, m):
		j = IDA_MutableSeq()
		j.funcHas = lambda: i.hasNext()
		j.funcGet = lambda: m(i.getNext())
		return j
	@staticmethod
	def i_filter__preLoad(i, j, fil):
		j.more = False
		while i.hasNext():
			j.buf = i.getNext()
			j.more = fil(j.buf)
			if j.more:
				return
	@staticmethod
	def i__next(j):
		x = j.buf
		j.preLoad()
		return x
	def i_filter(self, i, fil):
		j = IDA_MutableSeq()
		j.buf = None
		j.more = False
		j.preLoad = lambda: IDA_Port.i_filter__preLoad(i, j, fil)
		j.funcHas = lambda: j.more
		j.funcGet = lambda: IDA_Port.i__next(j)
		j.preLoad()
		return j
	@staticmethod
	def i_flatMap__preLoad(i, j, m):
		j.more = False
		while (j.lbuf == None) or (not j.lbuf.hasNext()):
			if not i.hasNext():
				return
			j.lbuf = m(i.getNext())
		j.more = True
		j.buf = j.lbuf.getNext()
	def i_flatMap(self, i, m):
		j = IDA_MutableSeq()
		j.lbuf = None
		j.buf = None
		j.more = False
		j.preLoad = lambda: IDA_Port.i_flatMap__preLoad(i, j, m)
		j.funcHas = lambda: j.more
		j.funcGet = lambda: IDA_Port.i__next(j)
		j.preLoad()
		return j
	
	# Port string operation
	def replace_all(self, s, act, dmd):
		return s.replace(act, dmd)
	
	def substring(self, s, fromChar, toChar):
		s = s[self.str_units(s, 0, fromChar):]
		return s[:self.str_units(s, 0, toChar-fromChar)]
	def substring(self, s, fromChar):
		return s[self.str_units(s, 0, fromChar):]
	
	def trim(self, s):
		i1 = 0
		i2 = len(s)
		while (i1 < i2) and (ord(s[i1]) <= 0x20): i1+=1
		while (i1 < i2) and (ord(s[i2-1]) <= 0x20): i2-=1
		return s[i1:i2]
	def rtrim(self, s):
		i2 = len(s)
		while (0 < i2) and (ord(s[i2-1]) <= 0x20): i2-=1
		return s[:i2]
	def ltrim(self, s):
		i1 = 0
		i2 = len(s)
		while (i1 < i2) and (ord(s[i1]) <= 0x20): i1+=1
		return s[i1:]
	
	# Port string tests
	def ends_with(self, s, suffix):
		return s.endswith(suffix)
	def starts_with(self, s, prefix):
		return s.startswith(prefix)
	def str_contains(self, s, mid):
		return mid in s
	
	# Port string info
	# Character index
	def str_index(self, s, mid):
		return s.index(mid) if mid in s else -1
	# Character index
	def str_index_from(self, s, mid, fromIndex):
		return s.index(mid, fromIndex) if mid in s[fromIndex:] else -1
	# The number of characters represented
	def str_len(self, s):
		return len(s)
	# The number of characters represented
	def str_chars(self, s, fromUnit, toUnit):
		return toUnit - fromUnit
	# The number of memory locations
	def str_units(self, s, fromChar, toChar):
		return toChar - fromChar
	
	# Port hex
	def char_2(self, c):
		return hex(c)[2:].rjust(2, "0")
	def char_4(self, c):
		return hex(c)[2:].rjust(4, "0")
	def char_8(self, c):
		return hex(c)[2:].rjust(8, "0")
	
# Port end
	pass
# Format exception
class IDAFormatException(Exception):
	
	def __init__(self, source, message):
		if isinstance(source, IDA_RX):
			rx = source
			super(IDAFormatException, self).__init__("(line "+str(rx.ln)+") "+message)
			
			self.node = None
			
			# Extract problem index first
			self.pos = len(rx.lnBuf)
			
			# Read until end of line for context
			ln = rx.lnBuf
			try:
				ln += rx.get_to("\n")
			except:
				pass
			self.ln = ln
			
		elif isinstance(source, IDA):
			nodeCause = source
			super(IDAFormatException, self).__init__(("(anonymous element) " if nodeCause.isAnon() else "(element '"+nodeCause.getNameOr("null")+"') ") + message)
			
			self.ln = None
			self.pos = -1
			self.node = nodeCause
	
# Op exception
class IDAOpException(Exception):
	pass

# Manip begin
class IDA_Manip:
	FORWARD = True
	REVERSE = False
	@staticmethod
	def rev(n):
		return not n if n.__class__ == True.__class__ else -n
	
	@staticmethod
	def m_str(el):
		if el.isNull(): return None
		try:
			return el.asStr()
		except:
			raise IDAFormatException(el, "String required")
	
	# Manip/iterate
	
	# Return the last node
	@staticmethod
	def manip_iterate(manip, node, itemTemplate):
		last = None
		while (itemTemplate.hasNext()):
			templateNode = itemTemplate.getNext()
			
			if last != None:
				node = IDA()
				last.linkNext(node)
			
			node.copyName(templateNode)
			node.copyParams(templateNode)
			if templateNode.isList():
				node.putList()
				
				if templateNode.numItems() > 0:
					item = IDA()
					if IDA_Manip.manip_iterate(manip, item, IDA.port.list_i(templateNode.asListOr())) != None:
						nextEl = None
						while item != None:
							nextEl = item.getNextOr()
							node.addItem(item)
							item = nextEl
			
			else:
				node.copyContent(templateNode)
			last = node
			
			# Manip/iterate/main
			if manip.isList():
				# manip:
				for m in manip.asList():
					if m.isAnon() or m.isNull():
						continue # Operate on nothing, or no operations
					if not m.isList():
						raise IDAFormatException(m, "Manipulation operations must be a list")
					
					operateOn = m.getNameOr()
					n = "n" in operateOn # n: name
					k = "k" in operateOn # k: param key
					v = "v" in operateOn # v: param value
					p = "p" in operateOn # p: parameter(s)
					c = "c" in operateOn # c: content
					
					# Manip/iterate/match
					# if-match: any of (n node-name, k param-key, v param-val, p(param-key param-val), c node-content)
					tParamMatch = lambda x: True
					if m.hasParams():
						continueManip = False
						for mp in m.getParamsOr():
							sel = mp.getNameOr("")
							
							if (len(re.sub("[nkvc!]+", "", sel)) > 0):
								raise IDAFormatException(m, "Invalid manipulation match (valid characters are [nkvc!])")
							
							flagNot = "!" in sel
							
							if "n" in sel:
								if (node.isName(IDA_Manip.m_str(mp))) == flagNot:
									continueManip = True
									break
							
							if "c" in sel:
								if mp.eqContent(node) == flagNot:
									continueManip = True
									break
							
							if "k" in sel:
								tgt = IDA_Manip.m_str(mp)
								tParamMatchOld = tParamMatch
								tParamMatch = lambda x: (tParamMatchOld(x) and (flagNot ^ (x.isName(tgt))))
								break
							
							if "v" in sel:
								tParamMatchOld = tParamMatch
								tParamMatch = lambda x: (tParamMatchOld(x) and (flagNot ^ mp.eqContent(x)))
								break
						
						if continueManip:
							continue
					
					paramMatch = tParamMatch
					
					# Manip/iterate/op
					for op in m.asList():
						if op.isAnon(): raise IDAFormatException(m, "Missing operation code")
						opName = op.getName()
						
						if False:
							pass
						
						# Manip/iterate/op/insert
						elif opName == "i": # i: insert (nkvc): i (index) text-string
							if op.isNull():
								raise IDAFormatException(m, "Missing insert string")
							if op.isList():
								raise IDAFormatException(m, "Insert string cannot be a list")
							
							param = op.getParamFirstOr()
							if (param != None) and (not param.isNum()):
								raise IDAFormatException(m, "Insert position must be a number")
							pos = param.asIntOr(0) if (param != None) else IDA_Manip.REVERSE
							
							try:
								if n:
									node.setName(IDA_Manip.manip_ins(node.getNameOr(), op.asStr(), pos))
								if c:
									node.putStr(IDA_Manip.manip_ins(node.asStr(), op.asStr(), pos))
								if k or v:
									if node.hasParams():
										for np in node.getParams():
											if not paramMatch(np):
												continue
											if k:
												np.setName(IDA_Manip.manip_ins(np.getNameOr(), op.asStr(), pos))
											if v:
												np.putStr(IDA_Manip.manip_ins(np.asStr(), op.asStr(), pos))
							except IDAOpException as opex:
								raise IDAFormatException(m, str(opex))
							
							pass
							
						# Manip/iterate/op/delete
						elif opName == "d": # d: delete (nkvc): d (index) length or d (index) bool
							if not (op.isNum() or op.isBool() or op.isNull()):
								raise IDAFormatException(m, "Delete amount must be boolean or a number")
							
							param = op.getParamFirstOr()
							if (param != None) and not param.isNum():
								raise IDAFormatException(m, "Delete position must be a number")
							pos = param.asIntOr(0) if (param != None) else IDA_Manip.REVERSE
							amt = (IDA_Manip.FORWARD if op.isTrue() else IDA_Manip.REVERSE) if op.isBool() else (1 if op.isNull() else op.asIntOr(0))
							if IDA.eq_obj(pos, IDA_Manip.REVERSE) and (IDA.eq_obj(amt, IDA_Manip.FORWARD) or (amt.__class__ == (1).__class__ and amt > 0)):
								amt = IDA_Manip.rev(amt)
							
							try:
								if n:
									node.setName(IDA_Manip.manip_del(node.getNameOr(), pos, amt))
								if c:
									node.putStr(IDA_Manip.manip_del(node.asStr(), pos, amt))
								if k or v:
									if node.hasParams():
										for np in node.getParams():
											if not paramMatch(np):
												continue
											if k:
												np.setName(IDA_Manip.manip_del(np.getNameOr(), pos, amt))
											if v:
												np.putStr(IDA_Manip.manip_del(np.asStr(), pos, amt))
							except IDAOpException as opex:
								raise IDAFormatException(m, str(opex))
							
							pass
							
						# Manip/iterate/op/add
						elif opName == "a": # a: add (vpc): a (index) {nodes...}
							# No need to check isList; addItem and insertItem already have exceptions for this
							param = op.getParamFirstOr()
							if (param != None) and not param.isNum():
								raise IDAFormatException(m, "Add position must be a number")
							pos = param.asIntOr(0) if (param != None) else IDA_Manip.REVERSE
							
							try:
								if p:
									if not node.hasParams():
										node.setParams()
									IDA_Manip.manip_add((lambda item, pos: node.addParamAt(item, pos)), (node.numParams() if IDA.eq_obj(pos, IDA_Manip.REVERSE) else pos), op)
								if c:
									IDA_Manip.manip_add((lambda item, pos: node.addItemAt(item, pos)), (node.numItems() if IDA.eq_obj(pos, IDA_Manip.REVERSE) else pos), op)
								if v:
									if node.hasParams():
										for np in node.getParams():
											if not paramMatch(np):
												continue
											IDA_Manip.manip_add((lambda item, pos: np.addItemAt(item, pos)), (np.numItems() if IDA.eq_obj(pos, IDA_Manip.REVERSE) else pos), op)
							except IDAOpException as opex:
								raise IDAFormatException(m, str(opex))
							
							pass
							
						# Manip/iterate/op/remove
						elif opName == "r": # r: remove (vpc): r (index) length or r(index) bool or r {nodes...}
							if op.isNum() or op.isBool() or op.isNull(): # r (index) length or r(index) bool
								
								param = op.getParamFirstOr()
								if (param != None) and not param.isNum():
									raise IDAFormatException(m, "Remove position must be a number")
								pos = param.asIntOr(0) if (param != None) else IDA_Manip.REVERSE
								amt = (IDA_Manip.FORWARD if op.isTrue() else IDA_Manip.REVERSE) if op.isBool() else (1 if op.isNull() else op.asIntOr(0))
								if IDA.eq_obj(pos, IDA_Manip.REVERSE) and (IDA.eq_obj(amt, IDA_Manip.FORWARD) or (amt.__class__ == (1).__class__ and amt > 0)):
									amt = IDA_Manip.rev(amt)
								
								if p:
									IDA_Manip.manip_rem_pos(m, (lambda pos: node.remParamAt(pos)), (lambda : node.numParams()), pos, amt)
								if c:
									IDA_Manip.manip_rem_pos(m, (lambda pos: node.remItemAt(pos)), (lambda : node.numItems()), pos, amt)
								if v:
									if node.hasParams():
										for np in node.getParamsOr():
											if not paramMatch(np):
												continue
											IDA_Manip.manip_rem_pos(m, (lambda pos: np.remItemAt(pos)), (lambda : np.numItems()), pos, amt)
							elif op.isList(): # r {nodes...}
								if p:
									IDA_Manip.manip_rem_item((lambda param: node.remParam(param)), op)
								if c:
									IDA_Manip.manip_rem_item((lambda item: node.remItem(item)), op)
								if v:
									if node.hasParams():
										for np in node.getParamsOr():
											if paramMatch(np):
												IDA_Manip.manip_rem_item((lambda item: np.remItem(item)), op)
							else:
								raise IDAFormatException(m, "Remove subject must be boolean, a number or a list")
							
							pass
							
						# Manip/iterate/op/undo
						elif opName == "u": # u: unset/unlist (vpc): u text-key or u {text-key...}
							if op.isList():
								for un in op.asList(): # u {text-key...}
									name = IDA_Manip.m_str(un)
									
									if c:
										node.unItem(name)
									if p:
										node.unParam(name)
									if v and node.hasParams():
										for np in node.getParamsOr():
											if paramMatch(np):
												np.unItem(name)
							else: # u text-key
								name = IDA_Manip.m_str(op)
								
								if c:
									node.unItem(name)
								if p:
									node.unParam(name)
								if v and node.hasParams():
									for np in node.getParamsOr():
										if paramMatch(np):
											np.unItem(name)
							pass
							
						# Manip/iterate/op/substitute
						elif opName == "s": # s: substitute (nkvc): s (search-token replacement-string ...);
							if op.hasParams():
								for repl in op.getParamsOr():
									if repl.isAnon() or repl.isNull():
										continue
									act = repl.getNameOr()
									dmd = IDA_Manip.m_str(repl)
									
									# nkvc
									if n and node.hasName():
										node.setName(node.getNameOr().replace(act, dmd))
									if c and node.hasContent():
										node.putStr(IDA_Manip.m_str(node).replace(act, dmd))
									if k or v:
										if node.hasParams():
											for np in node.getParamsOr():
												if not paramMatch(np):
													continue
												if k and np.hasName():
													np.setName(np.getNameOr().replace(act, dmd))
												if v and np.hasContent():
													np.putStr(IDA_Manip.m_str(np).replace(act, dmd))
							pass
							
						# Manip/iterate/op/flip
						elif opName == "f": # f (vc): flip: f;
							try:
								if c:
									IDA_Manip.manip_negate(node)
								if v and node.hasParams():
									for np in node.getParams():
										if not paramMatch(np):
											continue
										IDA_Manip.manip_negate(np)
							except IDAOpException as opex:
								raise IDAFormatException(m, str(opex))
							pass
							
						# Manip/iterate/op/write
						elif opName == "w": # w (nkvpc): write: w replacement
							if n:
								node.setName(IDA_Manip.m_str(op))
							if c:
								node.copyContent(op)
							if p:
								if op.isNull():
									node.clearParams()
								elif op.isList():
									node.setParams()
									for param in op.asListOr():
										node.addParam(param.copy())
								else:
									raise IDAFormatException(m, "Parameter write must be list or null")
							if k or v:
								if node.hasParams():
									for np in node.getParamsOr():
										if not paramMatch(np):
											continue
										if k:
											np.setName(IDA_Manip.m_str(op))
										if v:
											np.copyContent(op)
							pass
							
						# Manip/iterate/op end
						else:
							raise IDAFormatException(m, "Invalid operation code \""+opName+"\"")
						
				# Manip/iterate end
				pass
		
		return last
		
	# Manip/negate
	@staticmethod
	def manip_negate(node):
		if node.isBool():
			node.putBool(not node.asBool())
		elif node.isNum():
			num = node.getNumN()
			node.putNum(num[1:] if num.startswith("-") else "-"+num, node.getNumD())
		elif node.hasContent():
			raise IDAOpException("Flip subject must be boolean or a number")
	
	# Manip/insert
	## Usage note: the 'ins' string is never null, and the pos can be REVERSE in place of a null to mean the end (never FORWARD though)
	@staticmethod
	def manip_ins(orig, ins, pos):
		if orig == None:
			raise IDAOpException("Insert subject must not be null")
		if IDA.eq_obj(pos, IDA_Manip.REVERSE):
			return orig + ins
		l = len(orig)
		if pos > 0:
			if pos > l:
				raise IDAOpException("Cannot insert string beyond position "+str(l)+" (request was "+str(pos)+")")
			return orig[0:pos] + ins + orig[pos:]
		elif pos < 0:
			if pos < -l:
				raise IDAOpException("Cannot insert string before position -"+str(l)+" (request was "+str(pos)+")")
			return orig[0:l+pos] + ins + orig[l+pos:]
		else:
			return ins + orig
	
	# Manip/delete
	## Usage note: the pos can be REVERSE in place of a null to mean the end (never FORWARD though)
	@staticmethod
	def manip_del(orig, pos, amt):
		if orig == None:
			raise IDAOpException("Delete subject must not be null")
		l = len(orig)
		if IDA.eq_obj(pos, IDA_Manip.REVERSE):
			pos = l
		if pos >= 0:
			if pos > l:
				raise IDAOpException("Cannot delete from string beyond position "+str(l)+" (request was "+str(pos)+")")
			if IDA.eq_obj(amt, IDA_Manip.REVERSE):
				return orig[pos:]
			if IDA.eq_obj(amt, IDA_Manip.FORWARD):
				return orig[0:pos]
			if amt > 0:
				if (pos + amt) > l:
					raise IDAOpException("Cannot delete more than "+str(l-pos)+" beyond position "+str(pos)+" (request was "+str(amt)+")")
				return orig[0:pos] + orig[pos+amt:]
			if amt < 0:
				if (pos + amt) < 0:
					raise IDAOpException("Cannot delete more than "+str(pos)+" before position "+str(pos)+" (request was "+str(-amt)+")")
				return orig[0:pos+amt] + orig[pos:]
			return orig
		else: #if pos < 0
			if pos < -l:
				raise IDAOpException("Cannot delete from string before position -"+str(l)+" (request was "+str(pos)+")")
			if IDA.eq_obj(amt, IDA_Manip.REVERSE):
				return orig[pos+l:]
			if IDA.eq_obj(amt, IDA_Manip.FORWARD):
				return orig[0:pos+l]
			if amt > 0:
				if (pos+l + amt) > l:
					raise IDAOpException("Cannot delete more than "+str(-pos)+" beyond position "+str(pos)+" (request was "+str(amt)+")")
				return orig[0:pos+l] + orig[pos+l+amt:]
			if amt < 0:
				if (pos+l + amt) < 0:
					raise IDAOpException("Cannot delete more than "+str(l+pos)+" before position "+str(pos)+" (request was "+str(-amt)+")")
				return orig[0:pos+l+amt] + orig[pos+l:]
			return orig
	
	# Manip/add
	@staticmethod
	def manip_add(func, pos, nodeFrom):
		for ins in nodeFrom.asList():
			func(ins.copy(), pos)
			if pos >= 0:
				pos += 1
	
	# Manip/rem
	@staticmethod
	def manip_rem_pos(opNode, func, size, pos, amt):
		n = size()
		if IDA.eq_obj(pos, IDA_Manip.REVERSE):
			pos = n
		
		if IDA.eq_obj(amt, IDA_Manip.FORWARD):
			amt = n - pos
		elif IDA.eq_obj(amt, IDA_Manip.REVERSE):
			amt = -pos
		
		try:
			if amt > 0:
				i=pos+amt-1
				while i>=pos:
					func(i)
					i-=1
			elif amt < 0:
				i=pos-1
				while i>=pos+amt:
					func(i)
					i-=1
		except IDAOpException as opex:
			raise IDAFormatException(opNode, str(opex))
	@staticmethod
	def manip_rem_item(func, nodeFrom):
		for rem in nodeFrom.asList():
			func(rem)
	
# Manip end
	pass
# RX begin
class IDA_RX:
	def __init__(self, inStr):
		self.refs = {}
		self.has_refTarget = False
		
		self.inStr = inStr
		
		self.c1 = None
		self.c2 = None
		self.has_c1 = False
		self.has_c2 = False
		self.eof = False
		self.ln = 1
		self.lnBuf = ""
		
	# RX basic input
	# Py - variables moved to __init__, see RX begin
	def read(self):
		if not self.has_c1:
			self.more()
		return self.read1() if self.has_c1 else None
	
	def buf(self, c):
		if c == "\n":
			self.ln+=1
			self.lnBuf = ""
		else:
			self.lnBuf += c
	# Assumes has_c1
	def read1(self):
		output = self.c1
		self.c1 = self.c2
		self.has_c1 = self.has_c2
		self.has_c2 = False
		self.more2()
		self.buf(output)
		return output
	# Assumes has_c2
	def skip2(self):
		if self.has_c1:
			self.buf(self.c1)
		if self.has_c2:
			self.buf(self.c2)
		self.has_c1 = False
		self.has_c2 = False
		if self.more1():
			return
		self.more2()
	# Check has_c1 first
	def shift(self):
		output = self.c1
		self.read()
		return output
	def more(self):
		if self.has_c2 or self.eof:
			return
		if (not self.has_c1) and self.more1():
			return
		if (not self.has_c2) and self.more2():
			return
	# Returns eof
	def more1(self):
		if self.eof:
			return True
		b = self.inStr.read(1)
		if b == b"" or b == b"\x04":
			self.eof = True
			return True
		self.c1 = chr(ord(b))
		self.has_c1 = True
		return False
	# Returns eof
	def more2(self):
		if self.eof:
			return True
		b = self.inStr.read(1)
		if b == b"" or b == b"\x04":
			self.eof = True
			return True
		self.c2 = chr(ord(b))
		self.has_c2 = True
		return False
		
	
	# RX stream advancement
	def comment(self):
		if not self.has_c1:
			return False
		if IDA.comment1(self.c1):
			return True
		return self.has_c2 and IDA.comment2(self.c1, self.c2)
	def adv(self, checkLn = False):
		self.more()
		while self.has_c1:
			if (self.whiteTxt(self.c1) if checkLn else self.white(self.c1)):
				self.read1()
			elif self.c1 == "#":
				self.read1()
				self.advPast("\n")
			elif self.has_c2:
				if (self.c1 == "/") and (self.c2 == "/"):
					self.skip2()
					self.advPast("\n")
				elif (self.c1 == "/") and (self.c2 == "*"):
					self.skip2()
					self.advPast("*", "/")
				elif (self.c1 == "<") and (self.c2 == "?"): # XML header compatibility
					self.skip2()
					self.advPast("?", ">")
				elif (self.c1 == "<") and (self.c2 == "!"): # XML comment compatibility
					self.skip2()
					self.advPast("-", "-")
					while self.has_c1 and (not self.c1e(">")):
						self.advPast("-", "-")
					self.read1()
				else:
					return
			else:
				return
		
	def advPast1(self, c):
		self.more()
		while (self.has_c1) and (self.c1 != c):
			self.read1()
		self.read1()
	def advPast2(self, cx, cy):
		self.more()
		while self.has_c2 and ((self.c1 != cx) or (self.c2 != cy)):
			self.read1()
		self.skip2()
	def advPast(self, c, c2=None):
		if c2 == None:
			self.advPast1(c)
		else:
			self.advPast2(c, c2)
	
	def abrupt(self, checkLn):
		self.adv(checkLn)
		return not self.has_c1
	
	# RX scan bytes
	def c1e(self, c):
		return self.has_c1 and (self.c1 == c)
	def c1n(self, c):
		return self.has_c1 and (self.c1 != c)
	def c2e(self, c):
		return self.has_c2 and (self.c2 == c)
	def c2n(self, c):
		return self.has_c2 and (self.c2 != c)
	def whiteLn(self, c):
		return c == "\n"
	def whiteTxt(self, c):
		return (c <= " ") and (c != "\n")
	def white(self, c):
		return c <= " "
	
	# RX scan numbers
	@staticmethod
	def num0(c):
		return ((c>="0") and (c<="9")) or (c == "-") or (c == "+") or (c == ".")
	@staticmethod
	def num1(c):
		return IDA_RX.num0(c) or (c == "e") or (c == "E")
	def get_num(self):
		output = ""
		
		if self.has_c2 and (self.c1=="0") and (self.c2=="x"):
			output = "0x"
			self.skip2()
			while self.has_c1 and IDA_RX.hex(self.c1):
				output += self.shift()
		elif self.has_c2 and (self.c1=="0") and (self.c2=="b"):
			output = "0b"
			self.skip2()
			while self.has_c1 and IDA_RX.bin(self.c1):
				output += self.shift()
		else:
			output = "" + self.shift()
			while self.has_c1 and IDA_RX.num1(self.c1):
				output += self.shift()
		
		return output
	
	@staticmethod
	def hex(c):
		return ((c>="0") and (c<="9")) or ((c>="a") and (c<="f")) or ((c>="A") and (c<="F"))
	@staticmethod
	def bin(c):
		return (c=="0") or (c=="1")
	
	# RX scan opts
	def opt(self):
		return self.c1e('~')
	## Consumes the /, reads the opt, then advances
	def get_opt(self):
		self.read1()
		output = self.get_id()
		self.adv()
		return output
	
	# RX scan unicode
	@staticmethod
	def is_u8_marker(c):
		return (ord(c) & 0xc0) == 0xc0
	@staticmethod
	def is_u8_data(c):
		return (c & 0xc0) == 0x80
	
	def shift_u8_data(self):
		data = ord(self.shift())
		if not IDA_RX.is_u8_data(data): raise IDAFormatException(self, "Invalid unicode data")
		return data & 0x3f
	def get_u8(self):
		codePoint = 0
		marker = ord(self.shift())
		
		if (marker & 0xe0) == 0xc0: # 2-byte
			codePoint = (marker & 0x1f) << 6
			codePoint |= self.shift_u8_data()
			if codePoint < 0x0080: raise IDAFormatException(self, "Overlong unicode character")
		elif (marker & 0xf0) == 0xe0: # 3-byte
			codePoint = (marker & 0x0f) << 12
			codePoint |= self.shift_u8_data() << 6
			codePoint |= self.shift_u8_data()
			if codePoint < 0x0800: raise IDAFormatException(self, "Overlong unicode character")
		elif (marker & 0xf8) == 0xf0: # 4-byte
			codePoint = (marker & 0x07) << 18
			codePoint |= self.shift_u8_data() << 12
			codePoint |= self.shift_u8_data() << 6
			codePoint |= self.shift_u8_data()
			if codePoint < 0x10000: raise IDAFormatException(self, "Overlong unicode character")
		else:
			raise IDAFormatException(self, "Invalid unicode marker")
		
		return self.to_u8(codePoint)
	def to_u8(self, codePoint):
		if codePoint >= 0xd800 and codePoint < 0xe000: raise IDAFormatException(self, "Invalid unicode surrogate")
		try: return unichr(codePoint).encode("UTF-8") if IDA.v2 else chr(codePoint)
		except: raise IDAFormatException(self, "Invalid unicode range")
	
	# RX scan escapes
	@staticmethod
	def is_esc(c):
		return c == "\\"
	def esc(self):
		return self.has_c1 and IDA_RX.is_esc(self.c1)
	
	def shift_hex(self, ex_txt):
		h = ord(self.shift())
		if (0x30 <= h) and (h <= 0x39):
			return h - 0x30
		if (0x41 <= h) and (h <= 0x46):
			return 10 + h - 0x41
		if (0x61 <= h) and (h <= 0x66):
			return 10 + h - 0x61
		raise IDAFormatException(self, "Invalid "+ex_txt+" escape sequence")
	def get_esc_byte(self, ex_txt):
		return (self.shift_hex(ex_txt) << 4) | self.shift_hex(ex_txt)
	def get_esc(self):
		self.read1()
		if not self.has_c1:
			raise IDAFormatException(self, "Incomplete escape sequence")
		esc = self.shift()
		if esc == "0":
			return "\0"
		elif esc == "n":
			return "\n"
		elif esc == "r":
			return "\r"
		elif esc == "t":
			return "\t"
		elif esc == "x":
			if not self.has_c2: raise IDAFormatException(self, "Incomplete byte escape sequence")
			return chr(self.get_esc_byte("byte"))
		elif esc == "u":
			if not self.has_c2: raise IDAFormatException(self, "Incomplete unicode escape sequence")
			code = self.get_esc_byte("unicode") << 8
			if not self.has_c2: raise IDAFormatException(self, "Incomplete unicode escape sequence")
			code |= self.get_esc_byte("unicode")
			return self.to_u8(code)
		elif esc == "U":
			if not self.has_c2: raise IDAFormatException(self, "Incomplete unicode extended escape sequence")
			code = self.get_esc_byte("unicode") << 24
			if not self.has_c2: raise IDAFormatException(self, "Incomplete unicode extended escape sequence")
			code |= self.get_esc_byte("unicode") << 16
			if not self.has_c2: raise IDAFormatException(self, "Incomplete unicode extended escape sequence")
			code |= self.get_esc_byte("unicode") << 8
			if not self.has_c2: raise IDAFormatException(self, "Incomplete unicode extended escape sequence")
			code |= self.get_esc_byte("unicode")
			return self.to_u8(code)
		else:
			return esc
	
	# RX scan IDs
	def id0(self):
		return IDA.id0(self.c1)
	def id1(self):
		return IDA.id1(self.c1)
	def get_tok(self):
		if IDA_RX.is_str_d(self.c1):
			return self.get_str_d(self.c1)
		else:
			output = ""
			while not self.comment():
				if self.esc():
					output += self.get_esc()
				elif self.has_c1 and (not self.white(self.c1)):
					output += self.shift()
				else:
					break
			return output
	
	def get_id(self, untilLn = False):
		output = ""
		self.id_symbolic = True
		holdMin = -1
		holdMax = -1
		
		# Must not end with - or + , that is reserved for boolean false or true
		while self.has_c1 and not self.comment():
			ptr = len(output) if self.id_symbolic else holdMin
			hold = False
			
			if IDA_RX.is_esc(self.c1):
				hold = True
				output += self.get_esc()
			elif IDA_RX.is_str_d(self.c1):
				hold = True
				output += self.get_str_d(self.c1)
			elif IDA_RX.is_var(self.c1):
				hold = True
				output += self.get_var()
			elif IDA_RX.is_u8_marker(self.c1):
				hold = True
				output += self.get_u8()
			elif (IDA.id1(self.c1) or (untilLn and (self.c1 != ";") and not self.whiteLn(self.c1))) and (not ((self.c1 == "-") and ((not self.has_c2) or (not IDA.id1(self.c2)) ))):
				output += self.shift()
			else:
				break
			
			if hold:
				holdMin = ptr
				holdMax = len(output)
				self.id_symbolic = False
			
		
		if untilLn:
			if self.id_symbolic: return IDA.port.trim(output)
			else: return IDA.port.ltrim(output[:holdMin]) + output[holdMin:holdMax] + IDA.port.rtrim(output[holdMax:])
		else: return output
	@staticmethod
	def is_id_start(c):
		return IDA_RX.is_str_d(c) or IDA_RX.is_var(c) or IDA_RX.is_esc(c) or IDA_RX.is_u8_marker(c) or IDA.id0(c)
	def c1_id_start(self):
		return self.has_c1 and IDA_RX.is_id_start(self.c1)
	
	@staticmethod
	def is_var(c):
		return c == "$"
	def get_var(self):
		self.read()
		output = "$"
		
		endChar = "\0"
		if self.c1e("{"):
			endChar = "}"
		# TODO: Other common uses of $(...)
		
		if endChar != "\0":
			output += self.read1() + self.get_to(endChar) + endChar
		return output
	
	def xml_att_key(self):
		return self.get_id() if IDA_RX.is_id_start(self.c1) else self.get_pr(lambda c: c != "=" and c != "/" and c != ">" and not self.white(c))
	
	# RX scan strings
	def c1_str(self):
		return self.has_c1 and IDA_RX.is_str(self.c1)
	def c1_tok(self):
		return self.has_c1 and (not self.white(self.c1))
	@staticmethod
	def is_str_d(c):
		return (c == "\"") or (c == "`") or (c == "'")
	@staticmethod
	def is_str(c):
		return IDA_RX.is_str_d(c) or IDA.id1(c)
	
	def get_str(self):
		return self.get_str_d(self.c1) if IDA_RX.is_str_d(self.c1) else self.get_id()
	def get_str_d(self, delim):
		output = ""
		if self.c1e(delim):
			self.read1()
			while self.c1n(delim):
				output += self.get_esc() if self.esc() else (self.get_u8() if IDA_RX.is_u8_marker(self.c1) else self.read1())
			self.read()
		return output
	
	# RX scan logic
	def get_pr(self, p):
		output = ""
		while self.has_c1 and p(self.c1):
			output += self.shift()
		return output
	def get_prior_to(self, delim):
		output = self.get_pr((lambda c: (c != delim)))
		return output
	def get_to(self, delim):
		output = self.get_prior_to(delim)
		self.read()
		return output
	
	#  RX class iteration
	# Empty iterator
	CLASS_NONE_OBJ = None
	@staticmethod
	def CLASS_NONE():
		if IDA_RX.CLASS_NONE_OBJ == None:
			IDA_RX.CLASS_NONE_OBJ = IDA.port.empty_i()
		return IDA_RX.CLASS_NONE_OBJ
	# Infinite iterator of blank nodes
	CLASS_ARRAY_OBJ = None
	@staticmethod
	def CLASS_ARRAY():
		if IDA_RX.CLASS_ARRAY_OBJ == None:
			IDA_RX.CLASS_ARRAY_OBJ = IDA.port.repeat_i(IDA())
		return IDA_RX.CLASS_ARRAY_OBJ
	# Specific class node iterator
	@staticmethod
	def classOf(itemClassList):
		return IDA.port.list_i(itemClassList) if (itemClassList != None) else IDA_RX.CLASS_NONE()
	
	# RX entry
	## Returns the output argument
	@staticmethod
	def inObj(output, inStr):
		rx = IDA_RX(inStr)
		globalScope = IDA()
		entry = False
		entryNode = (IDA()).rxScope(globalScope).setIncludes(output.rxIncludePwd())
		entryNode.rxInputList(output.inputListDepth, output.inputListLength)
		entryNode.copyOptions(output)
		
		rx.adv()
		while rx.opt():
			opt = rx.get_opt()
			if opt == "":
				entry = True
		
		node = IDA_RX.fill(entryNode, rx, "\0", "\0", None, IDA_RX.CLASS_NONE(), 0)
		
		while rx.has_c1 and node != None:
			nextNode = (IDA()).rxScope(globalScope).setIncludes(output.rxIncludePwd())
			nextNode.rxInputList(output.inputListDepth, output.inputListLength)
			nextNode.copyOptions(output)
			while rx.opt():
				opt = rx.get_opt()
				if opt == "":
					if not entry:
						entry = True
						entryNode = nextNode
			
			last = IDA_RX.fill(nextNode, rx, "\0", "\0", node, IDA_RX.CLASS_NONE(), 0)
			node.linkNext(nextNode)
			node = last
		
		output.reset()
		rx.transfer_nodes(entryNode, output)
		output.walk(lambda x: x.rxScope(None), True, True)
		
		if rx.has_refTarget: output.walk(rx.put_ref, True, True)
		
		return output
	def transfer_nodes(self, src, dst):
		# Name
		dst.copyName(src)
		
		# Parameters
		if src.hasParams():
			dst.setParams()
			for p in src.getParamsOr(): dst.addParam(p)
			src.clearParams()
		else:
			dst.clearParams()
		
		# Content
		if src.isList():
			dst.putList()
			for i in src.asListOr(): dst.addItem(i)
			src.putNull()
		else:
			dst.copyContent(src)
		
		# Other
		dst.copyOptions(src)
		if src.hasRefName():
			self.refs[src.getRefName()] = dst
			dst.setRefName(src.getRefName())
		
		dst.link(src.getPrevOr(), src.getNextOr())
	def put_ref(self, el):
		t = el.getRefTarget()
		if t != None:
			if t not in self.refs: raise IDAFormatException(el, "Identifier not defined")
			el.putRef(self.refs[t])
	
	# RX fill
	@staticmethod
	def fill(node, rx, paren_l, paren_r, prev, listClass, depth):
		"""! Decodes element nodes read from the RX, using the supplied object as a container for the first such element.
		Adjacent elements will be created as needed and linked to the first, accessible using its getNext method.<br>
		@return The last consecutive element decoded, or null if no elements were present.
		@param node A container into which the first decoded element should be written.
		@param rx The serial input to be read.
		@param paren_l The character which caused <i>fill</i> to be called recursively, or zero to admit no such structure.
		@param paren_r The character which should mark the end of a recursive, structured call to <i>fill</i>.
		@param prev An element to be considered prior to the decoded element for the purpose of interpreting <i>as-above</i>
		markup, or null if no such element exists.
		@param listClass A class element to use for copying implicit element names, or an empty iterator (such as CLASS_NONE).
		@param depth The effective recursion level.
		@throws IOException If a error occurs while reading input bytes.
		@throws IDAFormatException If the parser encounters a syntax error, or if the markup contained an invalid
		<i>manipulation</i> which would throw {@link IDAOpException} if called programmatically.
		"""
		

		# RX fill/state
		jsonVal = False
		forceContent = False # Used for : and = in an array context
		needName = True
		prelimName = None
		nameSet = False
		needContent = True
		checkDelim = True # For ; or , between nodes
		itemClass = None
		output = None
		lcn = listClass.hasNext()
		
		# RX read
		while rx.has_c1 and needContent:
			
			# RX read/string implicit
			# This should be first
			# NOTE: By testing this prior to num(), names/strings beginning "e" will not be a number
			if rx.c1_id_start():
				output = node
				
				if not needName:
					if (listClass == IDA_RX.CLASS_ARRAY()) and (not forceContent):
						nameSet = True
						needName = False
						node.putStr(prelimName)
					else:
						# String content
						# Potentially also true/false/null if JSON-style following a :
						s = rx.get_id(depth == 0)
						
						if jsonVal and rx.id_symbolic:
							if "true" == s:
								node.putTrue()
							elif "false" == s:
								node.putFalse()
							elif "null" == s:
								node.putNull()
							else:
								node.putStr(s)
						else:
							node.putStr(s)
					
					needContent = False
				else:
					prelimName = rx.get_id()
					if rx.abrupt(paren_r == "\n"):
						node.putStr(prelimName)
						prelimName = None
						needContent = False
					else:
						needName = False
			
			# RX read/delim explicit
			elif rx.c1e(";") or rx.c1e(","):
				if not ((rx.c1 == ",") and (depth == 0)):
					# Consume delimiter unless using CSV compatibility
					rx.read1()
					checkDelim = False
				
				node.putNull()
				needContent = False
				output = node
			
			# RX read/delim implicit
			elif rx.c1e("}") or rx.c1e("]") or rx.c1e(")"):
				if paren_l != ':':
					if rx.c1e("}"):
						if paren_l != "{":
							raise IDAFormatException(rx, "List mismatch")
					elif rx.c1e("]"):
						if paren_l != "[":
							raise IDAFormatException(rx, "Array mismatch")
					else:
						if paren_l != "(":
							raise IDAFormatException(rx, "Parameters mismatch")
				
				if not nameSet:
					if (not needName) and lcn:
						lcn = False
						node.setName(listClass.getNext().getNameOr())
						node.putStr(prelimName)
					else:
						node.setName(prelimName)
					nameSet = True
				else:
					node.putNull()
				
				needContent = False
				checkDelim = False
			
			# RX read/as-above
			elif rx.c1e("^"):
				if prev == None:
					raise IDAFormatException(rx, "No previous element")
				
				if needName:
					prelimName = prev.getNameOr()
					needName = False
				else:
					node.copyContent(prev)
					needContent = False
				
				if rx.c2e("^"):
					node.copyParams(prev)
					rx.skip2()
				else:
					rx.read1()
				output = node
				
			# RX read/identifier definition
			elif rx.c1e('*'):
				rx.read1()
				key = rx.get_str() if (rx.c1_str()) else ""
				rx.adv()
				
				if key in rx.refs:
					raise IDAFormatException(rx, "Identifier already defined")
				rx.refs[key] = node
				node.setRefName(key)
				
				output = node
			
			# RX read/identifier reference
			elif rx.c1e('&'):
				rx.read1()
				key = rx.get_str() if (rx.c1_str()) else ""
				rx.adv()
				
				node.setRefTarget(key)
				rx.has_refTarget = True
				
				output = node
				needContent = False
			
			# RX read/XML
			elif rx.c1e("<"): # XML compatibility
				output = node
				if not needName:
					if (listClass == IDA_RX.CLASS_ARRAY()) and (not forceContent):
						nameSet = True
						needName = False
						node.putStr(prelimName)
						needContent = False
					elif paren_r == "<":
						needContent = False
					else:
						raise IDAFormatException(rx, "Cannot nest XML element header")
				else:
					rx.read1()
					rx.adv()
					if rx.has_c1 and rx.id1():
						node.setName(rx.get_id())
						needName = False
						nameSet = True
					else:
						raise IDAFormatException(rx, "Invalid XML element name")
					rx.adv()
					
					while rx.has_c1 and needContent: # XML inners
						if rx.c1e(">"): # XML child elements
							rx.read1()
							l = []
							last = None
							while rx.has_c1 and needContent:
								if rx.c1e("<"):
									if rx.c2e("/"): # End node
										rx.skip2()
										rx.adv()
										end = rx.get_to(">").strip()
										if not node.isName(end):
											raise IDAFormatException(rx, "Incorrect XML element footer")
										needContent = False
									elif rx.c2e("!"): # XML comment
										rx.skip2()
										rx.advPast("-", "-")
										while rx.c1n(">"):
											rx.advPast("-", "-")
										rx.read1()
									else: # Child element
										el = node.rxNew()
										IDA_RX.fill(el, rx, "<", ">", last, IDA_RX.CLASS_NONE(), depth+1)
										last = el
										l.append(el)
								else:
									if node.opts.rxXmlReparseStrings:
										rx.adv()
										
										el = node.rxNew()
										
										last = IDA_RX.fill(el, rx, ">", "<", last, IDA_RX.CLASS_NONE(), 0)
										while (el != None) and (node.inputListLength < 0 or node.inputListLength > len(l)):
											l.append(el)
											el = el.getNextOr()
									else:
										content = rx.get_prior_to("<")
										
										if len(content.strip()) > 0:
											el = (IDA()).putStr(content)
											l.append(el)
										
								
							node.putList(l)
							
						elif rx.c1e("/") and rx.c2e(">"): # Empty XML node
							rx.skip2()
							rx.adv()
							node.putNull()
							needContent = False
						else: # XML attrib
							attKey = rx.xml_att_key()
							while IDA_RX.is_id_start(rx.c1) and (rx.c1n("/") or rx.c2n(">")):
								attKey += rx.xml_att_key()
							rx.adv()
							attValue = None
							
							if rx.c1e("="):
								rx.read1()
								attValue = rx.get_str()
								rx.adv()
							else:
								attValue = None
							
							att = (IDA(attKey)).putStr(attValue)
							node.addParam(att)
			
			# RX read/params
			elif rx.c1e("("):
				if node.hasParams():
					needContent = False
				else:
					output = node
					
					# Parameters also cause content delim
					forceContent = True
					
					rx.read1()
					rx.adv()
					if rx.c1e("^"):
						if prev == None:
							raise IDAFormatException(rx, "No previous element")
						node.copyParams(prev)
						rx.read1()
						rx.adv()
					else:
						node.setParams()
					
					last = None
					cItems = IDA_RX.classOf(itemClass.getParamsOr() if (itemClass != None) else None)
					while rx.c1n(")"):
						param = node.rxNew()
						last = IDA_RX.fill(param, rx, "(", ")", last, cItems, depth+1)
						if last != None:
							while param != None:
								# Snapshot the next before addParam breaks it
								param_next = param.getNextOr()
								node.addParam(param)
								param = param_next
					while cItems.hasNext():
						param = cItems.getNext().copy().rxScope(node)
						node.addParam(param)
					rx.read()
			
			# RX read/content-delim
			elif (rx.c1e(":") and not needName) or rx.c1e("="):
				if rx.read1() == ":":
					jsonVal = True
				needName = False
				nameSet = True
				node.setName(prelimName)
				forceContent = True
				output = node
			
			# RX read/null
			elif rx.c1e("?"):
				output = node
				
				if needName:
					rx.read1()
					prelimName = None
					needName = False
				else:
					if (listClass == IDA_RX.CLASS_ARRAY()) and (not forceContent):
						nameSet = True
						needName = False
						node.putStr(prelimName)
					else:
						rx.read1()
						node.putNull()
					needContent = False
				
			# RX read/number
			elif IDA_RX.num0(rx.c1):
				if (not needName) and (listClass == IDA_RX.CLASS_ARRAY()) and (not forceContent):
					nameSet = True
					needName = False
					node.putStr(prelimName)
				else:
					s = rx.get_num()
					num = s
					unit = ""
					den = "1"
					
					# Check for denominator
					if rx.c1e("/"):
						rx.read1()
						
						if (not rx.has_c1) or (not IDA_RX.num0(rx.c1)):
							raise IDAFormatException(rx, "Missing denominator")
						den = rx.get_num()
						s += "/"+den
					
					# Check for unit
					if rx.has_c1 and rx.id0():
						unit = rx.get_id()
						s += unit
					
					if s == "+":
						node.putTrue()
					elif s == "-":
						node.putFalse()
					else:
						rx.testNumber(num, "numerator")
						if den != "1":
							rx.testNumber(den, "denominator")
						node.putNum(num, den, unit)
				needContent = False
				output = node
			
			# RX read/list array
			elif rx.c1e("["):
				output = node
				if (not needName) and (listClass == IDA_RX.CLASS_ARRAY()) and (not forceContent):
					nameSet = True
					needName = False
					node.putStr(prelimName)
				else:
					rx.read1()
					rx.adv()
					listObj = []
					
					if node.inputListDepth != 0:
						last = None
						while rx.c1n("]"):
							el = node.rxNew()
							last = IDA_RX.fill(el, rx, "[", "]", last, IDA_RX.CLASS_ARRAY(), depth+1)
							if last != None:
								while el != None and (node.inputListLength < 0 or node.inputListLength > len(listObj)):
									listObj.append(el)
									el = el.getNextOr()
					else:
						IDA_RX.discard(rx, "[", "]")
					
					rx.read()
					node.putList(listObj)
				needContent = False
			
			# RX read/list object
			elif rx.c1e("{"):
				output = node
				if (not needName) and (listClass == IDA_RX.CLASS_ARRAY()) and (not forceContent):
					nameSet = True
					needName = False
					node.putStr(prelimName)
				else:
					rx.read1()
					rx.adv()
					listObj = []
					
					if node.inputListDepth != 0:
						last = None
						cItems = IDA_RX.classOf(itemClass.asListOr() if (itemClass != None) else None)
						while rx.c1n("}"):
							el = node.rxNew()
							last = IDA_RX.fill(el, rx, "{", "}", last, cItems, depth+1)
							if last != None:
								while el != None and (node.inputListLength < 0 or node.inputListLength > len(listObj)):
									listObj.append(el)
									el = el.getNextOr()
						while cItems.hasNext() and (node.inputListLength < 0 or node.inputListLength > len(listObj)):
							el = cItems.getNext().copy().rxScope(node)
							listObj.append(el)
					else:
						IDA_RX.discard(rx, "{", "}")
					
					rx.read()
					node.putList(listObj)
				needContent = False
			
			# RX read/class definition
			elif rx.c1e(':') and rx.c2e(':'):
				rx.skip2()
				key = rx.get_str() if (rx.c1_str()) else ""
				rx.adv()
				
				el = node.rxNew()
				IDA_RX.fill(el, rx, ':', '\0', None, IDA_RX.CLASS_NONE(), depth+1)
				node.rxAddItemClass(key, el)
			
			# RX read/class instance
			elif rx.c1e(':'):
				if paren_l == ':':
					needContent = False
				else:
					rx.read1()
					key = rx.get_str() if (rx.c1_str()) else ""
					rx.adv()
					
					itemClass = node.rxGetItemClass(key)
					if itemClass == None:
						raise IDAFormatException(rx, "Item class \""+key+"\" not found")
					prelimName = itemClass.getNameOr()
					output = node
			
			# RX read/template definition
			elif rx.c1e("!"):
				if not needName:
					raise IDAFormatException(rx, "Incomplete element prior to template definition")
				rx.read1()
				key = rx.get_str() if (rx.c1_str()) else ""
				rx.adv()
				
				itemTemplate = node.rxNew()
				IDA_RX.fill(itemTemplate, rx, "\0", "\0", None, IDA_RX.CLASS_NONE(), depth+1)
				if not itemTemplate.isList():
					raise IDAFormatException(rx, "Template definition must be a list")
				node.rxAddTemplate(key, itemTemplate)
			
			# RX read/template instance
			elif rx.c1e("@"):
				# Hand-over to file include for @? and @@ 
				inc_opt = rx.c2e("?")
				if inc_opt or rx.c2e("@"):
					if not needName:
						raise IDAFormatException(rx, "Incomplete element prior to include")
					rx.skip2()
					if not rx.c1_tok():
						raise IDAFormatException(rx, "Missing include path")
					
					path = rx.get_tok()
					rx.adv()
					
					abort = False
					try:
						# Platform-specific check (Python)
						pwd = node.rxIncludePwd()
						abort = (pwd == None)
						if abort:
							raise IDAOpException("Includes are disabled")
						
						chdir = False
						if path.startswith("+"):
							chdir = True
							path = path[1:]
						
						inc = path
						# TODO: Cross-platform formulation of an equivalent absolute path (or test cross-platform)
						if not os.path.isabs(inc):
							inc = os.path.abspath(pwd+"/"+path)
						
						abort = not os.path.exists(inc)
						if abort:
							raise IDAOpException("Include path does not exist")
						abort = not os.path.isfile(inc)
						if abort:
							raise IDAOpException("Include is not a file")
						abort = not os.access(inc, os.R_OK)
						if abort:
							raise IDAOpException("Include file has no read access")
						
						template_adj = (IDA()).rxScope(node)
						if chdir:
							# TODO: Use actual path system to get path parent
							raise IDAFormatException(rx, "File include chdir is NYI")
							#template_adj.setIncludes(inc.getParentFile())
						else:
							template_adj.setIncludes(node.rxIncludePwd())
						template_adj.inFile(inc)
						
						last = IDA_RX.manip(rx, node, template_adj.iterateNext())
						
						nameSet = True
						checkDelim = False
						needContent = False
						output = last
					except IDAOpException as opex:
						# Hold off expending the RX line on an exception when the include was optional
						if not inc_opt:
							raise IDAFormatException(rx, str(opex))
						
						# Discard the manipulation node
						if abort:
							IDA_RX.fill(IDA(), rx, "\0", "\0", None, IDA_RX.CLASS_NONE(), depth+1)
				else:
					if not needName:
						raise IDAFormatException(rx, "Incomplete element prior to template instance")
					rx.read1()
					key = rx.get_str() if (rx.c1_str()) else ""
					rx.adv()
					
					itemTemplate = node.rxGetTemplate(key)
					if itemTemplate == None:
						raise IDAFormatException(rx, "Template \""+key+"\" not found")
					
					last = IDA_RX.manip(rx, node, IDA.port.list_i(itemTemplate.asListOr()))
					
					if last != None:
						nameSet = True
						checkDelim = False
						needContent = False
						output = last
					
					
				
			
			# RX read/invalid
			# This should be last
			else:
				c_hex = hex(ord(rx.c1))[2:]
				if len(c_hex) < 2:
					c_hex = "0"+c_hex
				raise IDAFormatException(rx, "Invalid character 0x"+c_hex+" '"+rx.c1+"'")
			
			# RX read delimiter
			if paren_r != ">":
				csvLn = paren_r == "\n"
				rx.adv(csvLn)
				csvLn = csvLn and rx.c1e("\n")
				if checkDelim and (rx.c1e(";") or rx.c1e(",") or csvLn):
					if needContent and lcn:
						lcn = False
						node.setName(listClass.getNext().getNameOr())
						node.putStr(prelimName)
					elif needName and lcn:
						lcn = False
						node.setName(listClass.getNext().getNameOr())
					elif (rx.c1 == ",") and (depth == 0):
						# CSV compatibility
						rx.read1()
						rx.adv(True)
						
						# Convert to list with new first element
						first = IDA()
						first.rxConfig(node)
						if needContent:
							first.putStr(prelimName)
						else:
							node.copyTo(first)
						
						# Complete the row
						listObj = [first]
						last = None
						cItems = IDA_RX.CLASS_ARRAY()
						while rx.c1n("\n"):
							el = node.rxNew()
							last = IDA_RX.fill(el, rx, "\0", "\n", last, cItems, depth+1)
							if last != None:
								while (el != None) and (node.inputListLength < 0 or node.inputListLength > len(listObj)):
									listObj.append(el)
									el = el.getNextOr()
						
						node.putList(listObj)
						node.setName(None)
						
					elif csvLn:
						# Last CSV item
						if needContent:
							node.putStr(prelimName)
					elif not nameSet:
						node.setName(prelimName)
					
					nameSet = True
					needContent = False
					
					if not csvLn:
						rx.read1()
						rx.adv()
					
				
			
			# RX read END
			pass
		
		# RX fill/name
		if not nameSet:
			if needName and lcn:
				lcn = False
				node.setName(listClass.getNext().getNameOr())
			else:
				node.setName(prelimName)
		
		# RX fill END
		if lcn:
			listClass.getNext()
		
		node.rxClearTemplates()
		node.rxClearItemClasses()
		return output
	
# TODO: RX discard
	# RX number test
	def testNumber(self, num, desc):
		# No need to test hex/bin here; invalid characters will have delimited the number, potentially becoming the unit
		if IDA.port.starts_with(num, "0x") or IDA.port.starts_with(num, "0b"):
			return
		
		test_num = "^[\\+\\-]?[0-9]*(?:\\.[0-9]+)?([eE][\\+\\-]?[0-9]+)?$"
		if not re.search(test_num, num):
			raise IDAFormatException(self, "Invalid "+desc+" number")
	
	# RX manip
	## Return the last node
	@staticmethod
	def manip(rx, node, itemTemplate):
		# Manipulations
		manip = IDA()
		IDA_RX.fill(manip, rx, "\0", "\0", None, IDA_RX.CLASS_NONE(), 1)
		if manip.isList():
			# TODO: any pre-process or validation of manipulations, or simplify logic if none needed
			pass
		elif manip.hasContent():
			raise IDAFormatException(rx, "Template manipulation must be a list")
		
		return IDA_Manip.manip_iterate(manip, node, itemTemplate)
	
# RX end
	pass

# Output interface
class IDA_Output(object):
	def __init__(self, output): # Resource pointer
		self.w = output
		self.queueBuf = ""
		pass
	
	def send(self, s):
		self.w.write(s.encode("UTF-8"))
		pass
	
	def queue(self, s):
		self.queueBuf += s
	def length(self):
		return len(self.queueBuf)
	def reject(self):
		self.queueBuf = ""
	def accept(self):
		self.send(self.queueBuf)
		self.queueBuf = ""
	
	
	# Element will not be null<br>
	# This will receive the entry element, which is not necessarily the first to be output<br>
	# The adjacents-flag is provided if any special state is needed to handle multiple elements to follow
	def ida_header(self, el, adj):
		pass
	# Element will not be null<br>
	# This is called potentially multiple times if adjacents are enabled
	def ida_element(self, el, adj):
		pass
	# Called once after all elements are done
	def ida_footer(self):
		pass
	
	def idaWrite(self, el, adj):
		if el == None:
			return
		
		el.txRefLabel(adj)
		
		self.ida_header(el, adj)
		
		if adj:
			# Rewind first
			while el.hasPrev():
				el = el.getPrevOr()
			
			# Output forward
			while el != None:
				self.ida_element(el, True)
				el = el.getNextOr()
		else:
			self.ida_element(el, False)
		
		self.ida_footer()
		
		self.w.flush()
		pass
	
	# Same as idaWrite, but converts exceptions into a return value instead.<br>
	# True if success, false if a IO problem occurred
	def idaPrint(self, el, adj):
		try:
			self.idaWrite(el, adj)
			return True
		except Exception as ex:
			return False
		pass
	
	pass

# TX begin
class IDA_TX(IDA_Output):
	# TX defaults
	BREVITY_ALL = 3
	BREVITY_PART = 6
	ONE_LINE_MAX = 132
	
	# TX instance
	# entry
	
	def setOptsOverride(self, ov):
		self.optsOverride = ov
		return self
	def opts(self, el):
		return IDAOptions.defaultOptions if self.optsOverride else el.opts
	
	def __init__(self, output):
		super(IDA_TX, self).__init__(output)
		self.entry = None
		self.rootAsAboveAll = False
		self.rootAsAboveFirst = None
		self.rootAsAbovePart = False
		self.rootAsAboveName = None
		
		self.optsOverride = False
	
	# TX interface
	def ida_header(self, el, adj):
		self.entry = el
	def ida_element(self, el, adj):
		# Explicit entry point?
		if adj and (self.entry == el) and el.hasPrev():
			self.send("~ ")
		
		# As-above all?
		if self.rootAsAboveFirst == None:
			self.rootAsAboveFirst = el
			self.rootAsAboveAll = IDA_TX.as_above_all_test(el)
		
		# Main output wrapped with name as-above part
		if self.rootAsAbovePart and (not el.isName(self.rootAsAboveName)):
			self.rootAsAbovePart = False
		self.ida(el, "", False, self.rootAsAbovePart or (adj and self.rootAsAboveAll and el.hasPrev()), False, None)
		self.rootAsAboveName = el.getNameOr()
		if not self.rootAsAbovePart:
			self.rootAsAbovePart = IDA_TX.as_above_part_test(el)
	
	# TX structure filter
	def has_inner_structure(self, first):
		outer = first
		while outer != None:
			if outer.isList():
				for inner in outer.asListOr():
					if inner.hasParams() or inner.isList():
						return True
			outer = outer.getNextOr()
		return False
	
	# TX main begin
	def ida(self, node, indent, parameters, nameAsAbove, nameOmit, itemClass):
		hasOutput = False
		
		# TX main/indent
		self.send(indent)
		if node.hasRefName(): self.send("*"+node.getRefName()+" ")
		if itemClass != None: self.send(": ")
		
		# TX main/name
		if not nameOmit:
			name = ""
			if node.hasName():
				name = "^" if nameAsAbove else self.id_str(node, node.getNameOr())
			elif (not node.hasParams()) and (not node.isList()):
				name = "?"
			
			self.send(name)
			hasOutput = name != ""
		
		# TX main/parameters
		chain = False
		if node.hasParams():
			if hasOutput:
				self.send(" ")
			hasOutput = True
			self.send("(")
			
			pNames = itemClass.getParamFirstOr() if itemClass != None else None
			param = node.getParamFirstOr()
			while param != None:
				pNameOmit = (pNames != None) and pNames.eqName(param)
				if pNames != None:
					pNames = pNames.getNextOr()
				if chain and not pNameOmit:
					self.send(" ")
				chain = True
				self.ida(param, "", True, False, pNameOmit, None)
				param = param.getNextOr()
			
			self.send(")")
		
		# TX main/content
		if node.isNull() or IDA_TX.broken_ref(node):
			self.send(";")
			if not parameters:
				self.send("\n")
		
		# TX main/ref
		elif node.isRef():
			self.send(" &")
			self.send(node.asRefOr().getRefName())
			self.send(("," if node.hasNext() else "") if parameters else "\n")
		
		# TX main/content/bool
		elif node.isBool():
			if nameOmit: self.send(" ")
			self.send("+" if node.isTrue() else "-")
			if not parameters: self.send("\n")
		
		# TX main/content/number
		elif (node.isNum()):
			if hasOutput or node.hasPrev():
				self.send(" ")
			self.send(node.toString())
			self.send(("," if node.hasNext() else "") if parameters else "\n")
		
		# TX main/content/list
		elif node.isList():
			if node.numItems() == 0:
				if hasOutput:
					self.send(" ")
				self.send("{}\n")
			elif (node.numItems() == 1) and (not node.getItemFirstOr().isList()):
				if hasOutput:
					self.send(" ")
				item = node.getItemFirstOr()
				if item.hasName() or item.hasParams():
					self.send("{ ")
					self.ida(item, "", True, False, False, None)
					self.send(" }\n")
				else:
					self.send("[ ")
					self.send(self.value(item, True))
					self.send(" ]\n")
			else:
				oneLine = IDA.port.si_allMatch(node.iterator(), lambda n: (not n.isList()) and (not n.hasParams()))
				
				if oneLine:
					if hasOutput:
						self.queue(" ")
					if IDA.port.si_anyMatch(node.iterator(), lambda n: n.hasName()):
						pNames = IDA.port.i_map(itemClass.iterator(), lambda n: n.getNameOr()) if ((itemClass != None) and itemClass.isList()) else IDA.port.empty_i()
						
						asAbove = IDA_TX.as_above_all_test(node.getItemFirstOr())
						
						i = node.iterator()
						self.queue("{")
						
						n = i.getNext()
						pNameOmit = pNames.hasNext() and (pNames.getNext() == n.getNameOr())
						if n.hasRefName(): self.queue(" *"+n.getRefName())
						if not pNameOmit: self.queue(" "+self.id_str(n, n.getNameOr()))
						self.queue(self.value(n, False))
						
						while i.hasNext():
							n = i.getNext()
							pNameOmit = pNames.hasNext() and (pNames.getNext() == n.getNameOr())
							if n.hasPrev() and (not n.getPrevOr().isBool()) and (not n.getPrevOr().isNull()):
								self.queue(",")
							if n.hasRefName(): self.queue(" *"+n.getRefName())
							if not pNameOmit: self.queue(" "+("^" if asAbove else self.id_str(n, n.getNameOr())))
							self.queue(self.value(n, False))
						self.queue(" }\n")
					else:
						i = node.iterator()
						self.queue("[")
						while i.hasNext():
							self.queue(" " + self.value(i.getNext(), True).strip())
						self.queue(" ]\n")
					
					if self.length() < 2*IDA_TX.ONE_LINE_MAX:
						self.accept()
					else:
						self.reject()
						oneLine = False
				
				if not oneLine:
					# For now, only use name-as-above when all nodes have the same name,
					# and are not lists above size 1
					
					if hasOutput:
						self.send("\n")
						self.send(indent)
					self.send("{\n")
					indent2 = indent + "\t"
					
					first = node.getItemFirstOr()
					
					# Check to see if a class could be used for parameters
					pClass = None
					firstName = first.getNameOr()
					cName = False
					if (not parameters) and (node.numItems() > 2):
						cName = True
						
						# Parameters
						firstParams = first.getParamsOr()
						ptrParams = len(firstParams) if (firstParams != None) else 0
						
						# Check names separately
						n = first.getNextOr()
						while (n != None):
							if not n.isName(firstName):
								cName = False
								break
							n = n.getNextOr()
						
						# Check parameters
						n = first.getNextOr()
						while (n != None) and (ptrParams > 0):
							nps = n.getParamsOr()
							if nps == None:
								ptrParams = 0
								break
							
							i = 0
							for np in nps:
								if i >= ptrParams:
									break
								if not np.isName(firstParams[i].getNameOr()):
									break
								i+=1
							
							ptrParams = min(ptrParams, i)
							
							n = n.getNextOr()
						
						
						# Items (all nodes must be lists)
						firstItems = first.asListOr()
						ptrItems = 0
						namedItem = False
						
						# Emergency measure - avoid classing the items if there are ANY parameters or lists inside ANY nodes
						if (firstItems != None) and not self.has_inner_structure(first):
							ptrItems = len(firstItems)
							n = first.getNextOr()
							while (n != None) and (ptrItems > 0):
								nis = n.asListOr()
								if nis == None:
									firstItems = None
									ptrItems = 0
									break
								
								i = 0
								for ni in nis:
									if i >= ptrItems:
										break
									name = firstItems[i].getNameOr()
									if not ni.isName(name):
										break
									if name != None:
										namedItem = True
									i+=1
								
								ptrItems = min(ptrItems, i)
								
								n = n.getNextOr()
						
						# Combine
						if (ptrParams > 0) or (namedItem and (ptrItems > 0)):
							# Class!
							self.send(indent2)
							self.send(":: ")
							if cName:
								self.send(self.id_str(first, firstName))
							
							pClass = IDA(firstName)
							
							if ptrParams > 0:
								pClass.setParams()
								
								self.send("(")
								np = firstParams[0]
								self.send(self.id_str(np, np.getNameOr()))
								pClass.addParam(IDA(np.getNameOr()))
								
								while np.hasNext():
									ptrParams -= 1
									if ptrParams <= 0:
										break
									np = np.getNextOr()
									self.send(", " + self.id_str(np, np.getNameOr()))
									pClass.addParam(IDA(np.getNameOr()))
								self.send(")")
							
							if (ptrItems > 0):
								pClass.putList()
								
								self.send("{")
								ni = firstItems[0]
								self.send(self.id_str(ni, ni.getNameOr()))
								pClass.addItem(IDA(ni.getNameOr()))
								while ni.hasNext():
									ptrItems -= 1
									if ptrItems <= 0:
										break
									ni = ni.getNextOr()
									self.send(", " + self.id_str(ni, ni.getNameOr()))
									pClass.addItem(IDA(ni.getNameOr()))
								self.send("}\n")
							else:
								self.send(";\n")
						else:
							cName = False
					
					item = first
					asAbove = IDA_TX.as_above_all_test(first)
					
					if asAbove or cName:
						self.ida(item, indent2, False, False, cName, pClass)
						item = item.getNextOr()
						while item != None:
							self.ida(item, indent2, False, True, cName, pClass)
							item = item.getNextOr()
					else:
						asAbovePart = False
						asAboveName = None
						
						while item != None:
							if asAbovePart and (not item.isName(asAboveName)):
								asAbovePart = False
							self.ida(item, indent2, False, asAbovePart, False, pClass)
							
							asAboveName = item.getNameOr()
							if not asAbovePart:
								asAbovePart = IDA_TX.as_above_part_test(item)
							
							item = item.getNextOr()
						pass
					
					self.send(indent)
					self.send("}\n")
			pass
		
		# TX main/content/string
		else:
			if hasOutput or node.hasPrev():
				self.send(" ")
			self.send(self.id_str(node, node.toString()))
			self.send(("," if node.hasNext() else "") if parameters else "\n")
		
		# TX main end
		pass
	
	# TX utility
	@staticmethod
	def broken_ref(el):
		return el.isRef() and not el.asRefOr().hasRefName()
	
	@staticmethod
	def as_above_all_test(firstItem):
		itemName = firstItem.getNameOr()
		
		if itemName == None:
			return False
		if len(itemName) < IDA_TX.BREVITY_ALL:
			return False
		
		c = firstItem
		while c != None:
			if c.numItems() >= 2:
				return False
			if not c.isName(itemName):
				return False
			c = c.getNextOr()
		return True
	@staticmethod
	def as_above_part_test(item):
		asAboveName = item.getNameOr()
		if (asAboveName == None) or (len(asAboveName) < IDA_TX.BREVITY_ALL):
			return False
		req = IDA_TX.BREVITY_PART - 1
		n = item.getNextOr()
		while (req > 0) and (n != None):
			req -= 1
			if not n.isName(asAboveName):
				return False
			n = n.getNextOr()
		return req <= 0
	
	def value(self, node, inArray):
		if node.isNull() or IDA_TX.broken_ref(node): return "?" if inArray else ";"
		if node.isBool(): return "+" if node.isTrue() else "-"
		if node.isRef(): return ("&" if inArray else " &") + node.asRefOr().getRefName()
		return ("" if inArray else " ") + (node.toString() if node.isNum() else self.id_str(node, node.toString()))
	
	def id_str(self, refNode, s):
		if s == None:
			return "?"
		elif len(s) < 1:
			return "\"\""
		elif IDA_TX.id(s):
			return s
		else:
			quoted = IDA.port.ends_with(s, "-")
			buf = ""
			
			i = 0
			n = len(s)
			while i < n:
				ch = s[i]
				c = ord(ch)
				if c < 0x20:
					if c == 0: buf += "\\0"
					elif c == 0x0a: buf += "\\n"
					elif c == 0x0d: buf += "\\r"
					elif c == 0x09: buf += "\\t"
					else: buf += "\\x" + IDA.port.char_2(c)
				elif c < 0x7f:
					if c == 0x5c: buf += "\\\\"
					elif c == 0x22:
						buf += "\\\""
						quoted = True
					else:
						if not quoted:
							if (not IDA.id0(ch) if i<1 else not IDA.id1(ch)):
								quoted = True
							elif i>0 and IDA.comment2(s[i-1], ch):
								quoted = True
						buf += ch
				else:
					u = c
					
					if IDA.v2:
						if ((c & 0xe0) == 0xc0):
							u = ((c & 0x1f) << 6) | (ord(s[i+1]) & 0x3f)
							i+=1
						elif ((c & 0xf0) == 0xe0):
							u = ((c & 0x0f) << 12) | ((ord(s[i+1]) & 0x3f) << 6) | (ord(s[i+2]) & 0x3f)
							i+=2
						elif ((c & 0xf8) == 0xf0):
							u = ((c & 0x07) << 18) | ((ord(s[i+1]) & 0x3f) << 12) | ((ord(s[i+2]) & 0x3f) << 6) | (ord(s[i+3]) & 0x3f)
							i+=3
					
					if u < 0xa0:
						buf += "\\x" + IDA.port.char_2(u)
					elif self.opts(refNode).txUnicode:
						buf += unichr(u) if IDA.v2 else ch
					else:
						if u < 0x100: buf += "\\x" + IDA.port.char_2(u)
						elif u < 0x10000: buf += "\\u" + IDA.port.char_4(u)
						elif u < 0x110000: buf += "\\U" + IDA.port.char_8(u)
				i+=1
			
			return "\"" + buf + "\"" if quoted else buf
		pass
	
	@staticmethod
	def id(s):
		if (len(s) < 1) or not IDA.id0(s[0]) or (s[-1] == '-'):
			return False
		c1 = s[0]
		for i in range(1, len(s)):
			c2 = s[i]
			if not IDA.id1(c2): return False
			if IDA.comment2(c1, c2): return False
			c1 = c2
		return True
	
# TX end
	pass

# CLI begin
class IDA_CLI:
	
	# CLI
	@staticmethod
	def cli_err(msg, exitCode = 1):
		sys.stderr.write("Error: " + str(msg) + "\n")
		return exitCode
	@staticmethod
	def cli_out(ln):
		sys.stdout.write(str(ln) + "\n")
	@staticmethod
	def cli_help():
		#           ".........1.........2.........3.........4.........5.........6.........7.........8"
		IDA.cli_out("IDA markup parser")
		IDA.cli_out("Read markup from standard input and write IDA to standard output.")
		IDA.cli_out("")
		IDA.cli_out("Parser options:")
		IDA.cli_out(" -h, --help         Show usage")
		IDA.cli_out(" -i in-option       Set IDA input option")
		IDA.cli_out(" -o out-option      Set IDA output option")
		IDA.cli_out("")
		IDA.cli_out("IDA input options:")
		IDA.cli_out(" xml-reparse        Parse XML string content as additional inline markup")
		IDA.cli_out("")
		IDA.cli_out("IDA output options:")
		IDA.cli_out(" unicode            Use UTF-8 for string characters beyond U+0080")
		IDA.cli_out("")
		return 0
	@staticmethod
	def cli(argc, argv):
		rootEl = IDA()
		opts = rootEl.opts
		debug = False
		
		i=0
		while i<argc:
			arg = argv[i]
			i+=1
			
			if arg == "-h" or arg == "--help":
				return IDA.cli_help()
			elif arg == "-d":
				debug = True
			elif arg == "-i":
				if i >= argc:
					return IDA.cli_err("No input option specified")
				arg = argv[i]
				i+=1
				if arg == "xml-reparse":
					opts.rxXmlReparseStrings = True
				else:
					return IDA.cli_err("Unknown input option: " + arg)
			elif arg == "-o":
				if i >= argc:
					return IDA.cli_err("No output option specified")
				arg = argv[i]
				i+=1
				if arg == "unicode":
					opts.txUnicode = True
				else:
					return IDA.cli_err("Unknown output option: " + arg)
		
		try:
			rootEl.inStream(sys.stdin if IDA.v2 else sys.stdin.buffer)
		except Exception as ex:
			if debug:
				import traceback
				traceback.print_exc()
			return IDA.cli_err("Parse failed: " + str(ex))
		
		rootEl.outPrint(sys.stdout if IDA.v2 else sys.stdout.buffer, True)
		return 0
	
# CLI end
	pass
	

class IDA:
	""" One data article element. """
	
	# File includes
	# File includes - Global PWD
	gIncludePwd = None
	## Enables file includes globally, as applied to articles created hereafter.<br>
	## The provided working dir is used for resolving relative paths.<br>
	## Set None to disable includes by default.
	@staticmethod
	def setGlobalIncludes(pwd = "./"):
		IDA.gIncludePwd = pwd
	
	# File includes - this element
	def setIncludes(self, pwd = "./"):
		""" Enables file includes for this article, as applied to input decoded hereafter.
		The provided working dir is used for resolving relative paths.
		Set null to disable includes. """
		self.includePwd = pwd
		return self
	## May be null, if includes are disabled for this article.
	def rxIncludePwd(self):
		return self.includePwd
	
	# Input control
	# See Initialisation for inputListDepth, inputListLength
	def rxInputList(self, depth, length):
		self.inputListDepth = depth
		self.inputListLength = length
		return self
	
	# Options storage
	# See Initialisation for opts
	def getOpts(self):
		""" Fetch the input/output options for this element.
		
		Returns:
		Input/output options object. """
		return self.opts
	def copyOptions(self, src):
		""" Set the input/output options of this element to match those of the provided element.
		
		Arguments:
		src, An element from which to copy the options.
		Returns:
		This element. """
		self.opts.copyFrom(src.opts)
		return self
	
	def rxConfig(self, parent):
		self.rxInputList(parent.inputListDepth - 1 if parent.inputListDepth > 0 else parent.inputListDepth, parent.inputListLength)
		self.copyOptions(parent)
		return self
	def rxNew(self):
		el = (IDA()).rxScope(self).setIncludes(self.rxIncludePwd())
		el.rxConfig(self)
		return el
	
	# Scope storage
	def rxScope(self, scope):
		self.scope = scope
		return self
	
	# Item classes
	def rxClearItemClasses(self):
		self.itemClasses = None
	def rxGetItemClass(self, key):
		c = None
		try:
			c = self.itemClasses[key] if (self.itemClasses != None) else None
		except:
			pass
		return c if (c != None) else (self.scope.rxGetItemClass(key) if (self.scope != None) else None)
	## Actually adds to scope element for use by adjacent elements.<br>
	## Returns false (and takes no action with the class element) if no scope existed
	def rxAddItemClass(self, key, itemClass):
		if self.scope == None:
			return False
		if self.scope.itemClasses == None:
			self.scope.itemClasses = {}
		self.scope.itemClasses[key] = itemClass
		return True
	
	# Templates
	def rxClearTemplates(self):
		self.itemTemplates = None
	def rxGetTemplate(self, key):
		c = None
		try:
			c = self.itemTemplates[key] if (self.itemTemplates != None) else None
		except:
			pass
		return c if (c != None) else (self.scope.rxGetTemplate(key) if (self.scope != None) else None)
	## Actually adds to scope element for use by adjacent elements.<br>
	## Returns false (and takes no action with the template element) if no scope existed
	def rxAddTemplate(self, key, itemTemplate):
		if self.scope == None:
			return False
		if self.scope.itemTemplates == None:
			self.scope.itemTemplates = {}
		self.scope.itemTemplates[key] = itemTemplate
		return True
	
	# Adjacent nodes
	
	# nextNode, prevNode
	def hasNext(self):
		""" Test whether an element is adjacent to this element in the forward direction.
		
		Returns:
		True if an element exists after this element, otherwise false """
		return self.nextNode != None
	def hasPrev(self):
		""" Test whether an element is adjacent to this element in the reverse direction.
		
		Returns:
		True if an element exists before this element, otherwise false """
		return self.prevNode != None
	def isFirst(self):
		""" Test whether no element is adjacent to this element in the reverse direction.
		
		This is the opposite of hasPrev, intended as a convenience for function references.
		
		Returns:
		True if nothing exists this element, otherwise false """
		return not self.hasPrev()
	def isLast(self):
		""" Test whether no element is adjacent to this element in the forward direction.
		
		This is the opposite of hasNext, intended as a convenience for function references.
		
		Returns:
		True if nothing exists after this element, otherwise false """
		return not self.hasNext()
	
	def getNext(self):
		""" Fetch the adjacent element after this one.
		
		If this element is last, an exception is thrown.
		To avoid that possibility, use getNextOr to provide a default value.
		
		Throws:
		IDAOpException, if this element is last.
		Returns:
		The element after. """
		if self.isLast(): raise IDAOpException("Element is last")
		return self.nextNode
	def getNextOr(self, other = None):
		""" Fetch the adjacent element after this one.
		
		If this element is last, the provided default value is returned.
		
		Arguments:
		other, A default value to return if nothing follows this element.
		Returns:
		The element after if present, otherwise the default value. """
		return self.nextNode if self.hasNext() else other
	def getPrev(self):
		""" Fetch the adjacent element before this one.
		
		If this element is first, an exception is thrown.
		To avoid that possibility, use getPrevOr to provide a default value.
		
		Throws:
		IDAOpException, if this element is first.
		Returns:
		The element before. """
		if self.isFirst(): raise IDAOpException("Element is first")
		return self.prevNode
	def getPrevOr(self, other = None):
		""" Fetch the adjacent element before this one.
		
		If this element is first, the provided default value is returned.
		
		Arguments:
		other, A default value to return if nothing precedes this element.
		Returns:
		The element before if present, otherwise the default value. """
		return self.prevNode if self.hasPrev() else other
	
	def __iterateNext_chain(self, i):
		obj = i.data_item
		i.data_item = obj.getNextOr()
		return obj
	## This element, followed by everything adjacent after it
	def iterateNext(self):
		i = IDA_MutableSeq()
		i.data_item = self
		i.funcHas = lambda: (i.data_item != None)
		i.funcGet = lambda: self.__iterateNext_chain(i)
		return i
	
	def linkNext(self, nextNode):
		""" Places the provided element adjacent to this one, next in sequence.
		If the element is null, this one will become last in sequence.
		
		Any existing forward link will be undone before this operation.
		
		Arguments:
		next, An element to follow this one, or null
		Returns:
		This element """
		if self.nextNode != None:
			self.nextNode.prevNode = None
		self.nextNode = nextNode
		if nextNode != None:
			nextNode.prevNode = self
		return self
	
	def linkPrev(self, prevNode):
		""" Places the provided element adjacent to this one, previous in sequence.
		If the element is null, this one will become first in sequence.
		
		Any existing reverse link will be undone before this operation.
		
		Arguments:
		prev, An element to precede this one, or null
		Returns:
		This element """
		if self.prevNode != None:
			self.prevNode.nextNode = None
		self.prevNode = prevNode
		if prevNode != None:
			prevNode.nextNode = self
		return self
	
	def link(self, prevNode, nextNode):
		""" Places the provided elements adjacent to this one, on either side.
		If an element is null, this one will become the first/last in sequence.
		
		Before this operation occurs, any existing links will be cross-connected to remove this element from the chain.
		
		Arguments:
		prev, An element to precede this one, or null
		next, An element to follow this one, or null
		Returns:
		This element """
		if self.prevNode != None:
			self.prevNode.nextNode = self.nextNode
		if self.nextNode != None:
			self.nextNode.prevNode = self.prevNode
		self.prevNode = prevNode
		if prevNode != None:
			prevNode.nextNode = self
		self.nextNode = nextNode
		if nextNode != None:
			nextNode.prevNode = self
		return self
	
	# Node name
	# Inverse of hasName, intended as a convenience for function references.
	def isAnon(self):
		""" Test whether this element has no name.
		The name must be null, not merely the empty string.
		
		This is the opposite of hasName, intended as a convenience for function references.
		
		Returns:
		True if this element has no name, otherwise false """
		return self.name == None
	def hasName(self):
		""" Test whether this element has a name.
		The name can be any non-null string, including the empty string.
		
		Returns:
		True if this element has a name, otherwise false """
		return self.name != None
	def isName(self, name):
		""" Test whether this element has a specific name.
		Providing null will perform the same test as with isAnon.
		
		Arguments:
		name, A name for comparison
		Returns:
		True if this element name matches the provided name, otherwise false """
		return self.getNameOr() == name
	
	def getName(self):
		""" Fetch this element name string.
		
		If this element content is anonymous, an exception is thrown.
		To avoid that possibility, use getNameOr to provide a default value.
		
		Throws:
		IDAOpException, if this element has no name.
		Returns:
		Element name. """
		if self.isAnon(): raise IDAOpException("Element has no name")
		return self.name
	def getNameOr(self, other = None):
		""" Fetch this element name string.
		
		If this element content is anonymous, the provided default value is returned.
		
		On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
		The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
		If possible, always call getNameOr() rather than getNameOr(null).
		
		Arguments:
		other, A default value to return if this element has no name.
		Returns:
		Element name if present, otherwise the default value. """
		return self.name if self.hasName() else other
	
	def clearName(self):
		""" Clear the name of this element, leaving it anonymous.
		
		Returns:
		This element. """
		return self.setName(None)
	def setName(self, name):
		""" Set the name of this element to be the provided string.
		
		If the provided string is null, this element will become anonymous as with calling clearName.
		
		Arguments:
		name, Element name string.
		Returns:
		This element. """
		self.name = name
		return self
	def copyName(self, src):
		""" Set the name of this element to match the name of the provided element.
		
		Arguments:
		src, An element from which to copy the name """
		self.name = src.name
	
	# Parameters
	# parameters
	def clearParams(self):
		""" Delete this element's parameters list.
		
		If this element has no parameters list, calling clearParams has no effect.
		
		Returns:
		This element. """
		self.parameters = None
		return self
	def setParams(self):
		""" Set this element's parameters to be an empty list.
		
		Any existing parameter elements will be deleted.
		
		Returns:
		This element. """
		self.parameters = []
		return self
	
	def getParams(self):
		""" Fetch this element's parameters list.
		
		If this element has no parameters list, an exception is thrown.
		To avoid that possibility, use getParamsOr to provide a default value.
		
		Throws:
		IDAOpException, if this element has no parameters list.
		Returns:
		List of parameters. """
		if not self.hasParams(): raise IDAOpException("No parameters list")
		return self.parameters
	def getParamsOr(self, other = None):
		""" Fetch this element's parameters list.
		
		If this element has no parameters list, the provided default value is returned.
		
		On platforms where lists/vectors are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
		The purpose is for this return type to be a guaranteed list, such that code will not be fully portable if null is provided.
		If possible, always call getParamsOr() rather than getParamsOr(null).
		
		Arguments:
		other, A default value to return if this element has no parameters list.
		Returns:
		List of parameters if present, otherwise the default value. """
		return self.parameters if self.hasParams() else other
	
	def hasParams(self):
		""" Test whether this element contains a parameters list.
		
		This reports whether a parameters list object exists at all, as opposed to whether at least one parameter is present.
		The primary purpose is to ensure the list exists before calling getParams and/or iterating over its elements.
		
		Use numParams to test for an empty list (0) or at least one parameter present (greater than 0).
		
		Returns:
		True if a parameters list exists for this element, otherwise false """
		return self.parameters != None
	def numParams(self):
		""" Fetch the number of parameters within this element.
		
		If no parameters list is present, the sentinel value -1 is returned.
		If 0 is returned, the list is present but empty.
		
		Returns:
		The parameters list size if a list is present, otherwise -1 """
		return len(self.parameters) if self.parameters != None else -1
	def hasParam(self, key):
		""" Test whether this element contains a parameter with the provided name.
		If it does, an equivalent call to getParam will return an element.
		
		If false is returned, this can be either due to having no parameter of that name or due to having no parameters list.
		Use hasParams to separate those two cases.
		
		Arguments:
		key, A parameter name to be checked
		Returns:
		True if a parameter exists with the provided name, otherwise false """
		return self.getParamOr(key) != None
	
	# Parameters/get
	def getParamAt(self, index):
		""" Fetch the parameter element at the provided index in this element's parameters list.
		
		If this element has no parameters list or has too few parameters in the list, an exception is thrown.
		To avoid that possibility, use getParamAtOr to provide a default value.
		
		Arguments:
		index, A parameter index to be recalled.
		Throws:
		IDAOpException, if this element has no such parameter.
		Returns:
		The indexed parameter element. """
		if self.numParams() <= index: raise IDAOpException("Invalid parameter index")
		return self.parameters[index]
	def getParamAtOr(self, index, other = None):
		""" Fetch the parameter element at the provided index in this element's parameters list.
		
		If this element has no parameters list or has too few parameters in the list, the provided default value is returned.
		
		Arguments:
		index, A parameter index to be recalled.
		other, A default value to return if this element has no such parameter.
		Returns:
		The indexed parameter element if present, otherwise the default value. """
		return self.parameters[index] if self.numParams() > index else other
	
	def getParam(self, key):
		""" Fetch the first parameter element with the provided name in this element's parameters list.
		
		If this element has no parameters list or has no parameter of that name, an exception is thrown.
		To avoid that possibility, use getParamOr to provide a default value.
		
		Arguments:
		key, A parameter name to be recalled.
		Throws:
		IDAOpException, if this element has no such parameter.
		Returns:
		The named parameter element. """
		p = self.getParamOr(key)
		if p == None: raise IDAOpException("Parameter not found")
		return p
	def getParamOr(self, key, other = None):
		""" Fetch the first parameter element with the provided name in this element's parameters list.
		
		If this element has no parameters list or has no parameter of that name, the provided default value is returned.
		
		Arguments:
		key, A parameter name to be recalled.
		other, A default value to return if this element has no such parameter.
		Returns:
		The named parameter element if present, otherwise the default value. """
		if self.hasParams():
			for p in self.parameters:
				if p.isName(key):
					return p
		return other
	
	def getParamStr(self, key):
		""" Cast to a string value the first parameter element with the provided name in this element's parameters list.
		
		If this element has no parameters list, has no parameter of that name or can not be interpreted as a string value, an exception is thrown.
		To avoid that possibility, use getParamStrOr to provide a default value.
		
		Arguments:
		key, A parameter name to be recalled.
		Throws:
		IDAOpException, if this element has no such parameter, or the parameter content cannot cast to a string value.
		Returns:
		String content interpretation of the named parameter element. """
		return self.getParam(key).toStr()
	def getParamStrOr(self, key, other = None):
		""" Cast to a string value the first parameter element with the provided name in this element's parameters list.
		
		If this element has no parameters list, has no parameter of that name or can not be interpreted as a string value, the provided default value is returned.
		
		On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
		The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
		If possible, always call getParamStrOr(key) rather than getParamStrOr(key, null).
		
		Arguments:
		key, A parameter name to be recalled.
		other, A default value to return if this element has no such parameter, or the parameter content cannot cast to a string value.
		Returns:
		String content interpretation of the named parameter element if possible, otherwise the default value. """
		param = self.getParamOr(key)
		return param.toStrOr(other) if param != None else other
	
	def getParamFirst(self):
		""" Fetch the first parameter element in this element's parameters list.
		
		If this element has no parameters list or has an empty parameters list, an exception is thrown.
		To avoid that possibility, use getParamFirstOr to provide a default value.
		
		Throws:
		IDAOpException, if this element has no such parameter.
		Returns:
		The first parameter element. """
		if self.numParams() < 1: raise IDAOpException("Parameters list is empty" if self.hasParams() else "Element has no parameters list")
		return self.parameters[0]
	def getParamFirstOr(self, other = None):
		""" Fetch the first parameter element in this element's parameters list.
		
		If this element has no parameters list or has an empty parameters list, the provided default value is returned.
		
		Arguments:
		other, A default value to return if this element has no such parameter.
		Returns:
		The first parameter element if present, otherwise the default value. """
		return self.parameters[0] if self.numParams() > 0 else other
	def getParamLast(self):
		""" Fetch the last parameter element in this element's parameters list.
		
		If this element has no parameters list or has an empty parameters list, an exception is thrown.
		To avoid that possibility, use getParamLastOr to provide a default value.
		
		Throws:
		IDAOpException, if this element has no such parameter.
		Returns:
		The last parameter element. """
		n = self.numParams()
		if n < 1: raise IDAOpException("Parameters list is empty" if self.hasParams() else "Element has no parameters list")
		return self.parameters[n-1]
	def getParamLastOr(self, other = None):
		""" Fetch the last parameter element in this element's parameters list.
		
		If this element has no parameters list or has an empty parameters list, the provided default value is returned.
		
		Arguments:
		other, A default value to return if this element has no such parameter.
		Returns:
		The last parameter element if present, otherwise the default value. """
		n = self.numParams()
		return self.parameters[n-1] if n > 0 else other
	
	# Parameters/add
	def newParam(self, name):
		""" Create and insert a new parameter at the end of the list, with the provided element name.
		If this element has no parameters list, a new list will be created.
		
		Arguments:
		name, Name string for the created parameter element.
		
		Returns:
		The created parameter element. """
		el = IDA(name)
		self.addParam(el)
		return el
	
	def addParam(self, param):
		""" Insert a parameter at the end of the list.
		If this element has no parameters list, a new list will be created.
		
		Arguments:
		param, The parameter element to be stored.
		
		Returns:
		This element. """
		if self.parameters == None:
			self.setParams()
		param.link(self.getParamLastOr(), None)
		self.parameters.append(param)
		return self
	def addParamAt(self, param, pos):
		""" Insert a parameter at the specified list index position.
		If this element has no parameters list, a new list will be created.
		
		Arguments:
		param, The parameter element to be stored.
		pos, The list index position at which the parameter should be inserted.
		
		Throws:
		IDAOpException, if the specified position is below zero or above the present list size.
		
		Returns:
		This element. """
		if self.parameters == None:
			self.setParams()
		IDA.list_add(self.getParams(), param, pos, "param")
		return self
	def setParam(self, param):
		""" Replace the first matching parameter name, or insert a new parameter at the end of the list.
		If this element has no parameters list, a new list will be created.
		
		Arguments:
		param, The parameter element to be stored.
		
		Returns:
		This element. """
		n = self.numParams()
		
		try:
			for i in range(n):
				p = self.getParamAt(i)
				if p.eqName(param):
					self.remParamAt(i)
					return self.addParamAt(param, i)
		except IDAOpException as ex:
			raise RuntimeError("Unexpected parameters state")
		
		return self.addParam(param) 
	
	def clearParam(self, key):
		""" Replace the first matching parameter name, or insert a new parameter.
		If this element has no parameters list, a new list will be created.
		
		This is the same as calling setParamNull.
		
		Arguments:
		key, The parameter element name.
		Returns:
		This element. """
		return self.setParamNull(key)
	def setParamNull(self, key):
		""" Replace the first matching parameter name, or insert a new parameter.
		If this element has no parameters list, a new list will be created.
		
		This is the same as calling clearParam.
		
		Arguments:
		key, The parameter element name.
		Returns:
		This element. """
		return self.setParam((IDA(key)).putNull())
	def setParamStr(self, key, content):
		""" Replace the first matching parameter name, or insert a new parameter.
		If this element has no parameters list, a new list will be created.
		
		Arguments:
		key, The parameter element name.
		content, String value to be stored.
		
		Returns:
		This element. """
		return self.setParam((IDA(key)).putStr(content))
	def setParamBool(self, key, content):
		""" Replace the first matching parameter name, or insert a new parameter.
		If this element has no parameters list, a new list will be created.
		
		Arguments:
		key, The parameter element name.
		content, Boolean value to be stored.
		
		Returns:
		This element. """
		return self.setParam((IDA(key)).putBool(content))
	def setParamDouble(self, key, content):
		""" Replace the first matching parameter name, or insert a new parameter.
		If this element has no parameters list, a new list will be created.
		
		Arguments:
		key, The parameter element name.
		content, Floating-point value to be stored.
		
		Returns:
		This element. """
		return self.setParam((IDA(key)).putDouble(content))
	def setParamFloat(self, key, content):
		""" Replace the first matching parameter name, or insert a new parameter.
		If this element has no parameters list, a new list will be created.
		
		Arguments:
		key, The parameter element name.
		content, Floating-point value to be stored.
		
		Returns:
		This element. """
		return self.setParam((IDA(key)).putFloat(content))
	def setParamLong(self, key, content):
		""" Replace the first matching parameter name, or insert a new parameter.
		If this element has no parameters list, a new list will be created.
		
		Arguments:
		key, The parameter element name.
		content, Integer value to be stored.
		
		Returns:
		This element. """
		return self.setParam((IDA(key)).putLong(content))
	def setParamInt(self, key, content):
		""" Replace the first matching parameter name, or insert a new parameter.
		If this element has no parameters list, a new list will be created.
		
		Arguments:
		key, The parameter element name.
		content, Integer value to be stored.
		
		Returns:
		This element. """
		return self.setParam((IDA(key)).putInt(content))
	def setParamRef(self, key, content):
		""" Replace the first matching parameter name, or insert a new parameter.
		If this element has no parameters list, a new list will be created.
		
		Arguments:
		key, The parameter element name.
		content, Referenced element to be stored.
		
		Returns:
		This element. """
		return self.setParam((IDA(key)).putRef(content))
	
	# Parameters/remove
	def remParamAt(self, pos):
		""" Remove the parameter at the specified list index position.
		If this element has no parameters list, an exception is thrown.
		
		Arguments:
		pos, The list index position of a parameter which should be removed.
		
		Throws:
		IDAOpException, if the specified position is below zero or beyond the present list size, or this element has no parameters list. """
		if not self.hasParams():
			raise IDAOpException("No parameters from which to remove")
		IDA.list_del(self.parameters, pos, "param")
	
	def remParam(self, param):
		""" Remove the provided parameter.
		If the parameter is absent or this element has no parameters list, the call has no effect.
		
		Arguments:
		param, The parameter element to be removed.
		
		Returns:
		True if a parameter was removed, otherwise false. """
		if self.hasParams() and IDA.list_rem(self.parameters, param):
			param.link(None, None)
			return True
		return False
	
	def unParam(self, key):
		""" Remove the first parameter with the provided element name.
		If no such parameter exists or this element has no parameters list, the call has no effect.
		
		Arguments:
		key, Name string of the parameter element to remove.
		
		Returns:
		True if a parameter was removed, otherwise false. """
		p = self.getParamOr(key)
		if p != None:
			self.remParam(p)
			return True
		return False
	
	# Parameters/other
	def copyParams(self, src):
		""" Set the parameters of this element to be clones of the parameters of the provided element.
		
		Arguments:
		src, An element from which to copy the parameters """
		self.clearParams()
		if src.hasParams():
			self.setParams()
			for p in src.parameters:
				self.addParam(p.copy_inner())
		pass
	
	# Content
	# cRaw
	# cStr
	# cNum, cDen, cUnit
	# cList
	# cBool
	# cRef
	# cRefObj
	# cRefTarget
	# cRefName
	
	def copyContent(self, src):
		""" Set the content of this element to match the content of the provided element.
		If the content in question is a list, its inner elements will be cloned.
		
		Arguments:
		src, An element from which to copy the content """
		self.cRaw = src.cRaw
		self.cStr = src.cStr
		self.cNum = src.cNum
		self.cDen = src.cDen
		self.cUnit = src.cUnit
		self.cBool = src.cBool
		self.cRef = None
		self.cRefObj = None
		self.cRefTarget = src.cRefTarget
		self.cRefName = src.cRefName
		self.clearList()
		if src.isList():
			self.putList()
			for x in src.asList():
				self.addItem(x.copy_inner())
		pass
	
	# Content/ref
	# Returns this element
	def clearRef(self):
		self.cRef = None
		self.cRefObj = None
		self.cRefTarget = None
		return self
	def isRef(self):
		""" Test whether the content type of this element is a reference to another element.
		
		Returns:
		True if this element content is a reference, otherwise false """
		return self.cRef != None
	def asRef(self):
		""" Fetch this element content as a reference (to another element).
		
		If this element content is not a reference, an exception is thrown.
		To avoid that possibility, use asRefOr to provide a default value.
		
		Throws:
		IDAOpException, if this element does not contain reference content.
		Returns:
		Reference content. """
		return self.cRef
	def asRefOr(self, other = None):
		""" Fetch this element content as a reference (to another element).
		
		If this element content is not a reference, the provided default value is returned.
		
		Arguments:
		other, A default value to return if this element does not contain reference content.
		Returns:
		Reference content if present, otherwise the default value. """
		return self.cRef if self.isRef() else other
	# Returns this element
	def putRef(self, ref):
		"""!Returns this element
		"""
		self.putNull()
		self.cRaw = "Ref"
		self.cRef = ref
		self.cRefObj = ref.repObj
		self.cRefTarget = ref.cRefName
		return self
	def setRefObj(self, refObj):
		""" Sets an object for which this element should reference the corresponding element as it is encoded.
		
		During a restore/decode process, the rebuilt object can be retrieved using getRefObj or withRefObj.
		The IDAObj codec interface implementation should ensure that any required type information is captured.
		
		Arguments:
		refObj, A target object which implements the IDAObj codec interface.
		Returns:
		This element. """
		self.putNull()
		self.cRaw = "Ref"
		self.cRefObj = refObj
		return self
	def withRefObj(self, callback):
		""" During a restore/decode process, the object corresponding with the referenced element will be sent to the callback when it becomes available.
		If that object is already available, the callback will be resolved before this call returns.
		
		If the object is needed immediately, the article structure must be designed such that the referenced object element is decoded in advance of this call.
		Use getRefObj instead if that design should be enforced.
		
		To ensure consistency, a top-level call to objRestore will throw an exception if any unresolved callbacks remain after decoding is complete.
		
		Calling code should use article information to establish the correct object type.
		
		Arguments:
		callback, A function which receives the decoded object associated with the referenced element.
		Throws:
		IDAOpException, if this element content is not a reference. """
		if self.cRefObj != None:
			callback(self.cRefObj)
		elif self.cRef == None:
			raise IDAOpException("Element is not a reference")
		else:
			self.cRef.on_rep_obj(callback)
	def getRefObj(self):
		""" Fetches the object corresponding with the referenced element, after it has been encoded or decoded.
		During a restore/decode process, an exception is thrown if the object is not yet available.
		
		To retrieve the object this way during a restore/decode process, the article structure must be designed such that the referenced object element is decoded in advance of this call.
		Use withRefObj instead if that design is not the case.
		
		Calling code should use article information to establish the correct object type.
		
		Throws:
		IDAOpException, if this element content is not a reference, or if no object is available immediately.
		Returns:
		The encoded or decoded object associated with the referenced element. """
		if self.cRefObj != None:
			return self.cRefObj
		if self.cRef == None:
			raise IDAOpException("Element is not a reference")
		if self.cRef.repObj != None:
			return self.cRef.repObj
		raise IDAOpException("Referenced object not available")
	def setRefTarget(self, refTarget):
		""" Set a transient target for this element.
		
		The targets match the identifiers assigned via setRefName, and are used only for capturing and restoring element pointer information.
		The string content itself is not information, insofar as it need not survive a loop test.
		
		Arguments:
		refTarget, A string to store as the transient target for this element. """
		self.putNull()
		self.cRaw = "Ref"
		self.cRefTarget = refTarget
	def getRefTarget(self):
		""" Fetch the transient target for this element.
		The returned value may be <i>stale</i>.
		
		The targets match the identifiers assigned via setRefName, and are used only for capturing and restoring element pointer information.
		The string content itself is not information, insofar as it need not survive a loop test.
		
		Returns:
		The transient target string for this element if it has a reference target, otherwise null. """
		return self.cRefTarget
	def hasRefTarget(self):
		""" Test whether this element has been assigned a transient target for encoding/decoding an outbound reference.
		The returned value may be <i>stale</i>.
		
		If an article has been decoded from markup, any reference target elements will have fresh identifiers from the markup document.
		Encoding, copying or testing equal a top-level article element will generate fresh identifiers.
		
		Call txRefLabel at top-level to generate fresh identifiers manually.
		
		Returns:
		True if this element has a transient target, otherwise false. """
		return self.cRefTarget != None
	def setRefName(self, refName):
		""" Set a transient identifier for this element.
		
		Identifiers are set automatically when encoding/decoding markup, testing equal, or calling txRefLabel directly.
		Encoding, copying or testing equal a top-level article element will generate fresh identifiers.
		
		These names are used only for capturing and restoring element pointer information.
		The string content itself is not information, insofar as it need not survive a loop test.
		
		Arguments:
		refName, A string to store as the transient identifier for this element. """
		self.cRefName = refName
	def getRefName(self):
		""" Fetch the transient identifier for this element.
		The returned value may be <i>stale</i>.
		
		Identifiers are set automatically when encoding/decoding markup, testing equal, or calling txRefLabel directly.
		Encoding, copying or testing equal a top-level article element will generate fresh identifiers.
		
		These names are used only for capturing and restoring element pointer information.
		The string content itself is not information, insofar as it need not survive a loop test.
		
		Returns:
		The transient identifier string for this element if it is a reference target, otherwise null. """
		return self.cRefName
	def hasRefName(self):
		""" Test whether this element has been assigned a transient identifier for encoding/decoding inbound references.
		The returned value may be <i>stale</i>.
		
		If an article has been decoded from markup, any reference target elements will have fresh identifiers from the markup document.
		Encoding, copying or testing equal a top-level article element will generate fresh identifiers.
		
		Call txRefLabel at top-level to generate fresh identifiers manually.
		
		Returns:
		True if this element has a transient identifier, otherwise false. """
		return self.cRefName != None
	
	# Returns the number of reference labels applied
	def txRefLabel(self, adj):
		# Ref walk
		rn = []
		rc = [0]
		self.walk(lambda x: IDA.ref_label_clear(x, rn), adj, adj)
		self.walk(lambda x: IDA.ref_label_count(x, rn, rc), adj, adj)
		
		# Null any foreign references at this stage (see IDA.verify)
		for r in rn: r.setRefName(None)
		
		return rc[0]
	@staticmethod
	def ref_label_clear(el, rn):
		el.setRefName(None)
		if el.isRef():
			r = el.asRefOr()
			if not IDA.port.list_contains(rn, r): rn.append(r)
	@staticmethod
	def ref_label_count(el, rn, rc):
		if IDA.port.list_contains(rn, el): IDA.port.list_rem(rn, el)
		
		if el.isRef():
			r = el.asRefOr()
			if not r.hasRefName():
				n = rc[0]
				rc[0] += 1
				n0 = n % 52
				id = "" + chr(0x61+n0 if n0 < 26 else 0x41+n0) + str(n//52)
				r.cRefName = id
				el.cRefTarget = id
	
	def rxRefResolve(self, adj):
		rs = {}
		self.walk(lambda x: IDA.ref_resolve_scan(x, rs), adj, adj)
		self.walk(lambda x: IDA.ref_resolve_put(x, rs), adj, adj)
	@staticmethod
	def ref_resolve_scan(el, rs):
		if el.hasRefName(): rs[el.getRefName()] = el
	@staticmethod
	def ref_resolve_put(el, rs):
		if el.hasRefTarget(): el.putRef(rs[el.getRefTarget()])
	
	# Content/bool
	def isBool(self):
		""" Test whether the content type of this element is a boolean value.
		
		Returns:
		True if this element has boolean content, otherwise false """
		return self.cBool != None
	def asBool(self):
		""" Fetch this element content as a boolean value.
		
		If this element content is not a boolean value, an exception is thrown.
		To avoid that possibility, use asBoolOr to provide a default value.
		One may also use isTrue for false by default, or not-isFalse for true by default.
		
		Use toBool/toBoolOr for casting other content types to be a boolean value.
		
		Throws:
		IDAOpException, if this element does not contain boolean content.
		Returns:
		Boolean content value. """
		if self.cBool == None: raise IDAOpException("Content is not a boolean value")
		return self.cBool == True
	def asBoolOr(self, other):
		""" Fetch this element content as a boolean value.
		
		If this element content is not a boolean value, the provided default value is returned.
		
		Arguments:
		other, A default value to return if this element does not contain boolean content.
		Returns:
		Boolean content value if present, otherwise the default value. """
		return self.cBool == True if self.cBool != None else other
	# Bool type and value is true, intended as a convenience for function references.
	def isTrue(self):
		""" Test whether the content of this element is boolean-true.
		
		If false is returned, this can be either due to being boolean-false or not being a boolean value.
		Use isBool and asBool for methods with those effects.
		
		This is intended as a convenience for function references.
		
		Returns:
		True if this element has boolean content with value true, otherwise false """
		return (self.cBool != None) and (self.cBool == True)
	# Bool type and value is false, intended as a convenience for function references.
	def isFalse(self):
		""" Test whether the content of this element is boolean-false.
		
		If false is returned, this can be either due to being boolean-true or not being a boolean value.
		Use isBool and (not-)asBool for methods with those effects.
		
		This is intended as a convenience for function references.
		
		Returns:
		True if this element has boolean content with value false, otherwise false """
		return (self.cBool != None) and (self.cBool != True)
	def putTrue(self):
		""" Place boolean content into this element with value true, replacing any existing content.
		
		Returns:
		This element """
		self.putNull()
		self.cRaw = "+"
		self.cBool = True
		return self
	def putFalse(self):
		""" Place boolean content into this element with value false, replacing any existing content.
		
		Returns:
		This element """
		self.putNull()
		self.cRaw = "-"
		self.cBool = False
		return self
	def putBool(self, b):
		""" Place boolean content into this element with the provided value, replacing any existing content.
		
		Arguments:
		b, Boolean content value.
		Returns:
		This element """
		return self.putTrue() if b else self.putFalse()
	def canBool(self):
		""" Test whether this element content can be interpreted as a boolean value (see toBool).
		This will always be the case for natural boolean content (see isBool).
		
		All numeric content can be cast to boolean, which is true if non-zero.
		String content can be "true" or "false".
		
		Returns:
		True if this element content can be cast to a boolean value, otherwise false """
		if self.isBool() or self.isNum(): return True
		s = self.toStrOr("")
		return s=="true" or s=="false"
	def toBool(self):
		""" Cast this element content to a boolean value.
		
		If this element content can not be interpreted as a boolean value, an exception is thrown.
		To avoid that possibility, use toBoolOr to provide a default value.
		
		Throws:
		IDAOpException, if this element content cannot cast to a boolean value.
		Returns:
		Boolean content interpretation. """
		if self.isBool(): return self.isTrue()
		if self.isNum(): return self.asDoubleOr(0.0) != 0.0
		s = self.toStrOr("")
		if s=="true": return True
		if s=="false": return False
		raise IDAOpException("Content cannot cast to a boolean value")
	def toBoolOr(self, other):
		try:
			return self.toBool()
		except:
			return other
	
	# Content/null
	def isNull(self):
		""" Test whether this element contains no content.
		
		This is the opposite of hasContent, intended as a convenience for function references.
		
		Returns:
		True if this element has no content, otherwise false. """
		return self.cRaw == None
	# Inverse of isNull, intended as a convenience for function references.
	def hasContent(self):
		""" Test whether this element contains some type of content.
		
		This is the opposite of isNull, intended as a convenience for function references.
		
		Returns:
		True if this element has content present, otherwise false. """
		return self.cRaw != None
	def putNull(self):
		""" Erase any existing content from this element.
		
		This is the same as calling clearContent.
		
		Returns:
		This element. """
		return self.clearContent()
	def clearContent(self):
		""" Erase any existing content from this element.
		
		This is the same as calling putNull.
		
		Returns:
		This element. """
		self.cRaw = None
		self.cStr = None
		self.cBool = None
		self.clearRef()
		self.clearList()
		self.clearNum()
		return self
	
	# Content/string
	def putStr(self, s):
		""" Place string content into this element with the provided value, replacing any existing content.
		
		Arguments:
		s, String content value.
		Returns:
		This element """
		self.putNull()
		self.cRaw = s
		self.cStr = s
		return self
	def isStr(self):
		""" Test whether the content type of this element is a string value.
		
		Returns:
		True if this element has string content, otherwise false """
		return self.cStr != None
	def toString(self):
		return self.cRaw if self.cRaw != None else "?"
	def asStr(self):
		""" Fetch this element content as a string value.
		
		If this element content is not a string value, an exception is thrown.
		To avoid that possibility, use asStrOr to provide a default value.
		
		Use toStr/toStrOr for casting other content types to be a string value.
		
		Throws:
		IDAOpException, if this element does not contain string content.
		Returns:
		String content value. """
		if self.cStr == None: raise IDAOpException("Content is not a string value")
		#return self.cStr.decode("UTF-8") if IDA.v2 and self.cStr != None else self.cStr
		return self.cStr
	def asStrOr(self, other=None):
		""" Fetch this element content as a string value.
		
		If this element content is not a string value, the provided default value is returned.
		
		On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
		The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
		If possible, always call asStrOr() rather than asStrOr(null).
		
		Arguments:
		other, A default value to return if this element does not contain string content.
		Returns:
		String content value if present, otherwise the default value. """
		return self.asStr() if self.cStr != None else other
	def canStr(self):
		""" Test whether this element content can be interpreted as a string value (see toStr).
		This will always be the case for natural string content (see isStr).
		
		Most content can be cast to string.
		Types which cannot be cast are lists, references and nulls.
		
		Returns:
		True if this element content can be cast to a string value, otherwise false """
		return self.hasContent() and (not self.isList()) and (not self.isRef())
	def toStr(self):
		""" Cast this element content to a string value.
		
		If this element content can not be interpreted as a string value, an exception is thrown.
		To avoid that possibility, use toStrOr to provide a default value.
		
		Throws:
		IDAOpException, if this element content cannot cast to a string value.
		Returns:
		String content interpretation. """
		if not self.canStr(): raise IDAOpException("Content cannot cast to a string value")
		return self.cRaw
	def toStrOr(self, other=None):
		""" Cast this element content to a string value.
		
		If this element content can not be interpreted as a string value, the provided default value is returned.
		
		On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
		The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
		If possible, always call toStrOr() rather than toStrOr(null).
		
		Arguments:
		other, A default value to return if this element content cannot cast to a string value.
		Returns:
		String content interpretation if able, otherwise the default value. """
		return self.cRaw if self.canStr() else other
	
	# Content/number/other
	def clearNum(self):
		self.cNum = ""
		self.cDen = "1"
		self.cUnit = ""
		pass
	
	# Content/number/set
	def refresh_num(self):
		self.cRaw = self.cNum + ("" if self.cDen == "1" else "/"+self.cDen) + self.cUnit
		self.cStr = None
	def setNumUnit(self, unit):
		""" Set the unit string appended to the number represented by this element content.
		
		Passing null to setNumUnit has no effect.
		To clear the unit, call setNumUnit with an empty string.
		
		If this element content is not numeric, an exception is thrown.
		
		Throws:
		IDAOpException, if this element does not contain numeric content.
		Returns:
		This element. """
		if unit != None:
			if not self.isNum(): raise IDAOpException("Content is not numeric")
			self.cUnit = unit
			self.refresh_num()
		return self
	
	def putNum(self, num, den="1", unit=None):
		if unit == None: unit = self.cUnit
		self.putNull()
		self.cNum = str(num)
		self.cDen = str(den)
		self.cUnit = unit
		self.refresh_num()
		return self
	
	def putDouble(self, num, unit=None):
		""" Place 64-bit floating-point content into this element with the provided value, replacing any existing content.
		
		If the provided unit string is non-null, also updates the unit string for this number (see setNumUnit).
		
		Arguments:
		num, 64-bit floating-point content value.
		unit, Number unit string.
		Returns:
		This element """
		return self.putNum(str(num)).setNumUnit(unit)
	def putFloat(self, num, unit=None):
		""" Place 32-bit floating-point content into this element with the provided value, replacing any existing content.
		
		If the provided unit string is non-null, also updates the unit string for this number (see setNumUnit).
		
		Arguments:
		num, 32-bit floating-point content value.
		unit, Number unit string.
		Returns:
		This element """
		return self.putNum(str(num)).setNumUnit(unit)
	def putLong(self, num, unit=None):
		""" Place 64-bit integer content into this element with the provided value, replacing any existing content.
		
		If the provided unit string is non-null, also updates the unit string for this number (see setNumUnit).
		
		Arguments:
		num, 64-bit integer content value.
		unit, Number unit string.
		Returns:
		This element """
		return self.putNum(str(num)).setNumUnit(unit)
	def putInt(self, num, unit=None):
		""" Place 32-bit integer content into this element with the provided value, replacing any existing content.
		
		If the provided unit string is non-null, also updates the unit string for this number (see setNumUnit).
		
		Arguments:
		num, 32-bit integer content value.
		unit, Number unit string.
		Returns:
		This element """
		return self.putNum(str(num)).setNumUnit(unit)
	
	# Content/number/get
	def isNum(self):
		""" Test whether the content type of this element is a numeric value.
		
		Returns:
		True if this element has numeric content, otherwise false """
		return len(self.cNum) > 0
	def getNumUnit(self):
		""" Fetch the unit string appended to the number represented by this element content.
		
		If this element content is not numeric, an exception is thrown.
		If numeric content had no unit specified, an empty string is returned.
		
		Throws:
		IDAOpException, if this element does not contain numeric content.
		Returns:
		Numeric unit string. """
		if not self.isNum(): raise IDAOpException("Content is not numeric")
		return self.cUnit
	def getNumN(self):
		""" Fetch the numerator string for the number represented by this element content.
		
		If this element content is not numeric, an exception is thrown.
		If the number was not a quotient, calling getNumN will return the same result as calling toStr.
		
		Throws:
		IDAOpException, if this element does not contain numeric content.
		Returns:
		Numerator as a string. """
		if not self.isNum(): raise IDAOpException("Content is not numeric")
		return self.cNum
	def getNumD(self):
		""" Fetch the denominator string for the number represented by this element content.
		
		If this element content is not numeric, an exception is thrown.
		If the number was not a quotient, calling getNumD will return "1".
		
		Throws:
		IDAOpException, if this element does not contain numeric content.
		Returns:
		Denominator as a string. """
		if not self.isNum(): raise IDAOpException("Content is not numeric")
		return self.cDen
	def asDoubleOr(self, other):
		""" Fetch this element content as a 64-bit floating-point value.
		
		If this element content is not numeric, the provided default value is returned.
		
		Arguments:
		other, A default value to return if this element does not contain numeric content.
		Returns:
		64-bit floating-point content value if present, otherwise the default value. """
		return self.c_double() if self.isNum() else other
	def asFloatOr(self, other):
		""" Fetch this element content as a 32-bit floating-point value.
		
		If this element content is not numeric, the provided default value is returned.
		
		Arguments:
		other, A default value to return if this element does not contain numeric content.
		Returns:
		32-bit floating-point content value if present, otherwise the default value. """
		return self.c_float() if self.isNum() else other
	def asLongOr(self, other):
		""" Fetch this element content as a 64-bit integer value.
		
		If this element content is not numeric, the provided default value is returned.
		
		Arguments:
		other, A default value to return if this element does not contain numeric content.
		Returns:
		64-bit integer content value if present, otherwise the default value. """
		return self.c_long() if self.isNum() else other
	def asIntOr(self, other):
		""" Fetch this element content as a 32-bit integer value.
		
		If this element content is not numeric, the provided default value is returned.
		
		Arguments:
		other, A default value to return if this element does not contain numeric content.
		Returns:
		32-bit integer content value if present, otherwise the default value. """
		return self.c_int() if self.isNum() else other
	def asDouble(self):
		""" Fetch this element content as a 64-bit floating-point value.
		
		If this element content is not numeric, an exception is thrown.
		To avoid that possibility, use asDoubleOr to provide a default value.
		
		Use toDouble/toDoubleOr for casting other content types to be a 64-bit floating-point value.
		
		Throws:
		IDAOpException, if this element does not contain numeric content.
		Returns:
		64-bit floating-point content value. """
		if not self.isNum(): raise IDAOpException("Content is not numeric")
		return self.c_double()
	def asFloat(self):
		""" Fetch this element content as a 32-bit floating-point value.
		
		If this element content is not numeric, an exception is thrown.
		To avoid that possibility, use asFloatOr to provide a default value.
		
		Use toFloat/toFloatOr for casting other content types to be a 32-bit floating-point value.
		
		Throws:
		IDAOpException, if this element does not contain numeric content.
		Returns:
		32-bit floating-point content value. """
		if not self.isNum(): raise IDAOpException("Content is not numeric")
		return self.c_float()
	def asLong(self):
		""" Fetch this element content as a 64-bit integer value.
		
		If this element content is not numeric, an exception is thrown.
		To avoid that possibility, use asLongOr to provide a default value.
		
		Use toLong/toLongOr for casting other content types to be a 64-bit integer value.
		
		Throws:
		IDAOpException, if this element does not contain numeric content.
		Returns:
		64-bit integer content value. """
		if not self.isNum(): raise IDAOpException("Content is not numeric")
		return self.c_long()
	def asInt(self):
		""" Fetch this element content as a 32-bit integer value.
		
		If this element content is not numeric, an exception is thrown.
		To avoid that possibility, use asIntOr to provide a default value.
		
		Use toInt/toIntOr for casting other content types to be a 32-bit integer value.
		
		Throws:
		IDAOpException, if this element does not contain numeric content.
		Returns:
		32-bit integer content value. """
		if not self.isNum(): raise IDAOpException("Content is not numeric")
		return self.c_int()
	
	def c_double(self):
		return self.n_float(self.cNum) if self.cDen == "1" else (self.n_float(self.cNum)/self.n_float(self.cDen))
	def c_float(self):
		return self.c_double()
	def c_long(self):
		return self.n_int(self.cNum) if self.cDen == "1" else int(self.n_float(self.cNum)/self.n_float(self.cDen))
	def c_int(self):
		return self.c_long()
	def n_int(self, s):
		if s.startswith("0x"):
			return self.n_hex(s)
		if s.startswith("0b"):
			return self.n_bin(s)
		try:
			return int(s)
		except ValueError as err:
			return int(float(s))
	def n_float(self, s):
		if s.startswith("0x"):
			return float(self.n_hex(s))
		if s.startswith("0b"):
			return float(self.n_bin(s))
		return float(s)
	def n_hex(self, s):
		return int(s[2:], 16)
	def n_bin(self, s):
		return int(s[2:], 2)
	
	# Content/list
	def clearList(self):
		self.cList = None
		pass
	def refresh_list(self):
		self.cRaw = "List("+str(self.numItems())+")"
		self.cStr = None
		pass
	
	def iterator(self):
		return IDA.port.list_i(self.cList)
	
	# Content/list/set
	# Returns this element
	def putList(self, l = False):
		self.putNull()
		if l != False:
			if l != None:
				self.cList = l
				self.refresh_list()
				last = None
				for item in l:
					item.link(last, None)
					last = item
		else:
			self.cList = []
			self.refresh_list()
		return self
		pass
	
	# Content/list/get
	def isList(self):
		""" Test whether the content type of this element is a list.
		
		This reports whether the content is a list or non-list, as opposed to whether at least one list item is present.
		The primary purpose is to ensure the list exists before calling asList and/or iterating over its elements.
		
		Use numItems to test for an empty list (0) or at least one item present (greater than 0).
		
		Returns:
		True if this element has list content, otherwise false """
		return self.cList != None
	def asList(self):
		""" Fetch this element content as a list (containing other elements).
		
		If this element content is not a list, an exception is thrown.
		To avoid that possibility, use asListOr to provide a default value.
		
		Throws:
		IDAOpException, if this element does not contain list content.
		Returns:
		List content. """
		if self.cList == None: raise IDAOpException("Content is not a list")
		return self.cList
	def asListOr(self, other = None):
		""" Fetch this element content as a list (containing other elements).
		
		If this element content is not a list, the provided default value is returned.
		
		On platforms where lists/vectors are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
		The purpose is for this return type to be a guaranteed list, such that code will not be fully portable if null is provided.
		If possible, always call asListOr() rather than asListOr(null).
		
		Arguments:
		other, A default value to return if this element does not contain list content.
		Returns:
		List content if present, otherwise the default value. """
		return self.cList if self.isList() else other
	def numItems(self):
		""" Fetch the number of content list items within this element.
		
		If no content list is present, the sentinel value -1 is returned.
		If 0 is returned, the list is present but empty.
		
		Returns:
		The content list size if a list is present, otherwise -1 """
		return len(self.cList) if self.isList() else -1
	
	def hasItem(self, key):
		""" Test whether this element contains a list item with the provided name.
		If it does, an equivalent call to getItem will return an element.
		
		If false is returned, this can be either due to having no list item of that name or due to being a non-list.
		Use isList to separate those two cases.
		
		Arguments:
		key, A list item name to be checked
		Returns:
		True if a content list item exists with the provided name, otherwise false """
		return self.getItemOr(key) != None
	
	def getItemAt(self, index):
		""" Fetch the item element at the provided index in this element's content list.
		
		If this element content is not a list or has too few items in the list, an exception is thrown.
		To avoid that possibility, use getItemAtOr to provide a default value.
		
		Arguments:
		index, A list item index to be recalled.
		Throws:
		IDAOpException, if this element has no such list item.
		Returns:
		The indexed content list item element. """
		if self.numItems() <= index: raise IDAOpException("Invalid list item index")
		return self.cList[index]
	def getItemAtOr(self, index, other = None):
		""" Fetch the item element at the provided index in this element's content list.
		
		If this element content is not a list or has too few items in the list, the provided default value is returned.
		
		Arguments:
		index, A list item index to be recalled.
		other, A default value to return if this element has no such list item.
		Returns:
		The indexed content list item element if present, otherwise the default value. """
		return self.cList[index] if self.numItems() > index else other
	
	def getItem(self, key):
		""" Fetch the first item element with the provided name in this element's content list.
		
		If this element content is not a list or has no item of that name, an exception is thrown.
		To avoid that possibility, use getItemOr to provide a default value.
		
		Arguments:
		index, A list item name to be recalled.
		Throws:
		IDAOpException, if this element has no such list item.
		Returns:
		The named content list item element. """
		item = self.getItemOr(key)
		if item == None: raise IDAOpException("Item not found")
		return item
	def getItemOr(self, key, other = None):
		""" Fetch the first item element with the provided name in this element's content list.
		
		If this element content is not a list or has no item of that name, the provided default value is returned.
		
		Arguments:
		key, A list item name to be recalled.
		other, A default value to return if this element has no such list item.
		Returns:
		The named content list item element if present, otherwise the default value. """
		if self.isList():
			for x in self.asList():
				if x.isName(key): return x
		return other
	
	def getItemStr(self, key):
		""" Cast to a string value the first item element with the provided name in this element's content list.
		
		If this element content is not a list, has no item of that name or can not be interpreted as a string value, an exception is thrown.
		To avoid that possibility, use getItemStrOr to provide a default value.
		
		Arguments:
		key, A list item name to be recalled.
		Throws:
		IDAOpException, if this element has no such list item, or the item content cannot cast to a string value.
		Returns:
		String content interpretation of the named list item element. """
		return self.getItem(key).toStr()
	def getItemStrOr(self, key, other=None):
		""" Cast to a string value the first item element with the provided name in this element's content list.
		
		If this element content is not a list, has no item of that name or can not be interpreted as a string value, the provided default value is returned.
		
		On platforms where strings are nullable/optional, the return value is guaranteed to be non-null if the default value is non-null.
		The purpose is for this return type to be a guaranteed string, such that code will not be fully portable if null is provided.
		If possible, always call getItemStrOr(key) rather than getItemStrOr(key, null).
		
		Arguments:
		key, A list item name to be recalled.
		other, A default value to return if this element has no such list item, or the item content cannot cast to a string value.
		Returns:
		String content interpretation of the named list item element if possible, otherwise the default value. """
		item = self.getItemOr(key)
		return item.toStrOr(other) if item != None else other
	
	def getItemFirst(self):
		""" Fetch the first item element in this element's content list.
		
		If this element content is not a list or is an empty list, an exception is thrown.
		To avoid that possibility, use getItemFirstOr to provide a default value.
		
		Throws:
		IDAOpException, if this element has no such list item.
		Returns:
		The first content list item element. """
		if self.numItems() < 1: raise IDAOpException("Content list is empty" if self.isList() else "Element content is not a list")
		return self.cList[0]
	def getItemFirstOr(self, other = None):
		""" Fetch the first item element in this element's content list.
		
		If this element content is not a list or is an empty list, the provided default value is returned.
		
		Arguments:
		other, A default value to return if this element has no such list item.
		Returns:
		The first content list item element if present, otherwise the default value. """
		return self.cList[0] if self.numItems() > 0 else other
	def getItemLast(self):
		""" Fetch the last item element in this element's content list.
		
		If this element content is not a list or is an empty list, an exception is thrown.
		To avoid that possibility, use getItemLastOr to provide a default value.
		
		Throws:
		IDAOpException, if this element has no such list item.
		Returns:
		The last content list item element. """
		n = self.numItems()
		if n < 1: raise IDAOpException("Content list is empty" if self.isList() else "Element content is not a list")
		return self.cList[n-1]
	def getItemLastOr(self, other = None):
		""" Fetch the last item element in this element's content list.
		
		If this element content is not a list or is an empty list, the provided default value is returned.
		
		Arguments:
		other, A default value to return if this element has no such list item.
		Returns:
		The last content list item element if present, otherwise the default value. """
		n = self.numItems()
		return self.cList[n-1] if n > 0 else other
	
	# Content/list/add
	
	def addItem(self, el, makeList=True):
		""" Insert a item element at the end of the content list.
		If this element does not have list content, a new list will be created if the makeList flag is set, otherwise an exception is thrown.
		
		Arguments:
		el, The item element to be stored.
		makeList, Whether to overwrite existing non-list content.
		
		Throws:
		IDAOpException, if this element content is not a list and the makeList flag is false.
		
		Returns:
		This element. """
		if (not self.isList()) and (not makeList):
			raise IDAOpException("Cannot append items onto a non-list")
		if not self.isList():
			self.putList()
		el.link(self.getItemLastOr(), None)
		self.cList.append(el)
		self.refresh_list()
		return self
	
	def addItemAt(self, el, pos):
		""" Insert a item element at the specified content list index position.
		If this element does not have list content, an exception is thrown.
		
		Arguments:
		el, The item element to be stored.
		pos, The list index position at which the item should be inserted.
		
		Throws:
		IDAOpException, if the specified position is below zero or above the present list size, or if this element content is not a list.
		
		Returns:
		This element. """
		if not self.isList():
			raise IDAOpException("Cannot merge items into a non-list")
		IDA.list_add(self.cList, el, pos, "item")
		self.refresh_list()
		return self
	
	def newItem(self, name, makeList=True):
		""" Create and insert a new item element at the end of the content list, with the provided element name.
		If this element does not have list content, a new list will be created if the makeList flag is set, otherwise an exception is thrown.
		
		Arguments:
		name, Name string for the created item element.
		makeList, Whether to overwrite existing non-list content.
		
		Throws:
		IDAOpException, if this element content is not a list and the makeList flag is false.
		
		Returns:
		The created item element. """
		el = IDA(name)
		self.addItem(el, makeList)
		return el
	
	# Content/list/remove
	
	def remItemAt(self, pos):
		""" Remove the item at the specified content list index position.
		If this element does not have list content, an exception is thrown.
		
		Arguments:
		pos, The list index position of an item which should be removed.
		
		Throws:
		IDAOpException, if the specified position is below zero or beyond the present list size, or this element does not have list content. """
		if not self.isList():
			raise IDAOpException("Cannot remove items from a non-list")
		IDA.list_del(self.cList, pos, "item")
		self.refresh_list()
	def remItem(self, el):
		""" Remove the provided content list item.
		If the item is absent or this element does not have list content, the call has no effect.
		
		Arguments:
		el, The item element to be removed.
		
		Returns:
		True if an item was removed, otherwise false. """
		if self.isList() and IDA.list_rem(self.cList, el):
			self.refresh_list()
			return True
		return False
	def unItem(self, name):
		""" Remove the first content list item with the provided element name.
		If no such item exists or this element does not have list content, the call has no effect.
		
		Arguments:
		name, Name string of the item element to remove.
		
		Returns:
		True if an item was removed, otherwise false. """
		item = self.getItemOr(name)
		if item != None:
			self.remItem(item)
			return True
		return False
	
	# Object/equals
	def eq_inner(self, obj):
		if obj == None or not isinstance(obj, IDA): return False
		if obj == self: return True
		if not self.eqName(obj): return False
		if not self.eqParams(obj): return False
		if not self.eqContent(obj): return False
		return True
	def eq(self, obj, adjacent=False):
		""" Perform a top-level comparison of this element and the provided element.
		
		To be equal, both elements must match when interpreted as full articles.
		The element names, parameters and content must all match recursively.
		
		Note that content matches consider whether the <i>information</i> is the same.
		See eqContent for details.
		
		Arguments:
		obj, An article element for comparison
		adjacents, Whether to include adjacent elements in the comparison
		Returns:
		True if the elements represent matching articles, otherwise false """
		if obj == self: return True
		if obj == None or not isinstance(obj, IDA): return False
		if self.txRefLabel(adjacent) != obj.txRefLabel(adjacent): return False
		if not self.eq_inner(obj): return False
		
		if adjacent:
			a = self
			b = obj
			el = b
			while a.hasPrev() and b.hasPrev():
				a = a.getPrevOr()
				b = b.getPrevOr()
				if not a.eq_inner(b):
					return False
			if a.hasPrev() or b.hasPrev():
				return False
			
			a = self
			b = el
			while a.hasNext() and b.hasNext():
				a = a.getNextOr()
				b = b.getNextOr()
				if not a.eq_inner(b):
					return False
			if a.hasNext() or b.hasNext():
				return False
		
		return True
	
	def eqName(self, obj):
		""" Compare the names of this element and the provided element.
		
		Arguments:
		obj, An element for name comparison
		
		Returns:
		True if the names match, otherwise false """
		return isinstance(obj, IDA) and self.isName(obj.getNameOr())
	def eqContent(self, obj):
		""" Compare the contents of this element and the provided element.
		
		The content type must match, and the contents must match in a type-dependent way
		such that each element contains the same <i>information</i>. Numbers must have the
		same precision, and scope references must target the same element relative to each
		scope. Foreign references are equal by default.
		
		Arguments:
		obj, An element for content comparison
		Returns:
		True if the contents match, otherwise false """
		if not isinstance(obj, IDA): return False
		if self.cRaw != obj.cRaw: return False
		if self.cStr != obj.cStr: return False
		if self.cNum != obj.cNum: return False
		# Assured by testing cRaw
		#if self.cDen != obj.cDen: return False
		# Assured by testing cRaw
		#if self.cUnit != obj.cUnit: return False
		if not IDA.list_equals(self.cList, obj.cList): return False
		if self.cBool != obj.cBool: return False
		if not self.eqContent_ref(self.cRef, obj.cRef): return False
		
		return True
	def eqContent_ref(self, ref_a, ref_b):
		if (ref_a != None) ^ (ref_b != None): return False
		if ref_a == ref_b: return True
		if ref_a.hasRefName() ^ ref_b.hasRefName(): return False
		if not ref_a.hasRefName(): return True
		return ref_a.getRefName() == ref_b.getRefName()
	def eqParams(self, obj):
		""" Compare recursively the parameters of this element and the provided element.
		
		Arguments:
		obj, An element for parameter comparison
		
		Returns:
		True if the parameters match, otherwise false """
		return isinstance(obj, IDA) and IDA.list_equals(self.parameters, obj.parameters)
	

	# Object/capture
	# repObj
	def objCapture(self, obj):
		""" Top-level article encode operation for a full object tree.
		This method calls objEncode, so the top-level object must implement idaEncode.
		
		
		After all encoding is complete, any object references declared via setRefObj will be established as element references in the article.
		
		Arguments:
		obj, A top-level object, which must implement idaEncode, to encode and record as this article element.
		Returns:
		This article element. """
		self.objEncode(obj)
		
		self.capture_objs()
		self.capture_refs()
		
		return self
	def objEncode(self, obj):
		""" Encode one member object via idaEncode, with automatic element name via idaName.
		This element will become "representative" of the object, which is used to encode references via setRefObj.
		
		
		This method should be called from within idaEncode itself, to record members which also implement the IDAObj codec interface.
		For the top-level operation, user code should call idaCapture instead to ensure that all object references are recorded.
		
		Arguments:
		obj, An object, which must implement idaEncode, to encode and record as this element.
		Returns:
		This element. """
		self.setName(obj.idaName())
		self.repObj = obj
		obj.idaEncode(self)
		
		return self
	def capture_objs(self):
		if self.repObj != None: self.repObj.idaCaptureObj = self
		if self.hasParams():
			for p in self.parameters: p.capture_objs()
		if self.isList():
			for i in self.cList: i.capture_objs()
	def capture_refs(self):
		if self.cRefObj != None: self.putRef(self.cRefObj.idaCaptureObj)
		if self.hasParams():
			for p in self.parameters: p.capture_refs()
		if self.isList():
			for i in self.cList: i.capture_refs()
	
	# Object/restore
	# repObjActions
	def on_rep_obj(self, callback):
		if self.repObj != None: callback(self.repObj)
		else:
			if self.repObjActions == None: self.repObjActions = []
			self.repObjActions.append(callback)
	def restore_check(self):
		if self.repObjActions != None: raise IDAOpException("Decode process failed to resolve an object reference")
	def objRestore(self, obj):
		""" Top-level article decode operation for a full object tree.
		This method calls objDecode, so the top-level object must implement idaDecode.
		
		
		After all decoding is complete, the integrity will be checked to ensure that all object reference requests were resolved.
		If any calls to withRefObj did not receive the deferred result, an exception is thrown.
		This occurs if the article contained an element reference for which no object was constructed via objDecode.
		
		Arguments:
		obj, A top-level object, which must implement idaDecode, to become the result of decoding this article element.
		Returns:
		The input object. """
		self.objDecode(obj)
		self.walk(IDA.restore_check, True, True)
		return obj
	def objDecode(self, obj):
		""" Decode one member object via idaDecode.
		This element will become "representative" of the object, which is used to decode references via getRefObj or withRefObj.
		
		
		This method should be called from within idaDecode itself, to create members which also implement the IDAObj codec interface.
		For the top-level operation, user code should call idaRestore instead to verify that all object reference requests were resolved.
		
		Arguments:
		obj, An object, which must implement idaDecode, to become the result of decoding this element.
		Returns:
		The input object. """
		self.repObj = obj
		if self.repObjActions != None:
			for c in self.repObjActions: c(obj)
			self.repObjActions = None
		obj.idaDecode(self)
		return obj
	
	# Deserialisation
	def inStream(self, input):
		""" Read from the provided input handle, parse it as markup and use the result to build this element.
		After the parser is done, this element will become the top-level entry element from the markup stream.
		
		Arguments:
		input, An open input handle for a stream containing markup data
		
		Returns:
		This element """
		return IDA_RX.inObj(self, input)
	def inBytes(self, input):
		""" Read the provided byte array, parse it as markup and use the result to build this element.
		After the parser is done, this element will become the top-level entry element from the markup stream.
		
		Arguments:
		input, A byte array containing markup data
		
		Returns:
		This element """
		return self.inStream(io.BytesIO(input))
	def inString(self, input):
		""" Read the provided string, parse it as markup and use the result to build this element.
		After the parser is done, this element will become the top-level entry element from the markup stream.
		
		Arguments:
		input, A string containing markup data
		
		Returns:
		This element """
		return self.inBytes(input.encode("UTF-8"))
	def inFile(self, path):
		""" Read from the file at the provided path, parse it as markup and use the result to build this element.
		After the parser is done, this element will become the top-level entry element from the markup stream.
		
		Arguments:
		input, A path to a file containing markup data
		
		Returns:
		This element """
		with open(path, "rb") as res:
			return self.inStream(res)
	
	# Serialisation
	def outStream(self, output, adjacents = False):
		""" Encode this element as top-level markup and write it to the provided output handle.
		If <i>adjacents</i> is true, all preceding and following elements will also be encoded.
		If a preceding element was encoded that way, the markup will contain an entry instruction.
		Any write errors will be sent back to the caller.
		
		Arguments:
		output, An open output handle to receive a stream of markup data
		adjacents, Whether to write markup for elements adjacent to this one """
		IDA_TX(output).idaWrite(self, adjacents)
	
	def outPrint(self, output, adjacents = False):
		""" Encode this element as top-level markup and write it to the provided output handle.
		If <i>adjacents</i> is true, all preceding and following elements will also be encoded.
		If a preceding element was encoded that way, the markup will contain an entry instruction.
		
		Arguments:
		output, An open output handle to receive a stream of markup data
		adjacents, Whether to write markup for elements adjacent to this one
		
		Returns:
		True on success, or false if a write error occurred """
		return IDA_TX(output).idaPrint(self, adjacents)
	
	def outBytes(self, adjacents = False):
		""" Encode this element as top-level markup and return it as a byte array.
		If <i>adjacents</i> is true, all preceding and following elements will also be encoded.
		If a preceding element was encoded that way, the markup will contain an entry instruction.
		
		Arguments:
		adjacents, Whether to write markup for elements adjacent to this one
		
		Returns:
		A byte array containing the markup data """
		buf = io.BytesIO()
		self.outPrint(buf, adjacents)
		output = buf.getvalue()
		buf.close()
		return output
	
	def outString(self, adjacents = False):
		""" Encode this element as top-level markup and return it as a byte array.
		If <i>adjacents</i> is true, all preceding and following elements will also be encoded.
		If a preceding element was encoded that way, the markup will contain an entry instruction.
		
		Arguments:
		adjacents, Whether to write markup for elements adjacent to this one.
		
		Returns:
		A string containing the markup data. """
		return self.outBytes(adjacents).decode("UTF-8")
	
	def outFile(self, path, adjacents = False):
		""" Encode this element as top-level markup and write it to the file at the provided path.
		If <i>adjacents</i> is true, all preceding and following elements will also be encoded.
		If a preceding element was encoded that way, the markup will contain an entry instruction.
		Any write errors will be sent back to the caller.
		
		Arguments:
		path, A destination file path for the markup data.
		adjacents, Whether to write markup for elements adjacent to this one. """
		with open(path, "w") as res:
			self.outStream(res, adjacents)
	
	# Construction
	def __init__(self, name = None):
		self.includePwd = IDA.gIncludePwd
		self.reset()
		self.setName(name)
	
	# Initialisation
	def reset(self):
		""" Initialisate this element anew.
		
		All existing information comprising this element will be discarded.
		The result will be an anonymous, null-content element with no parameters list, no adjacent elements and default input/output options.
		
		Returns:
		This element. """
		self.putNull()
		self.cRefName = None
		self.repObj = None
		self.repObjActions = None
		
		self.itemClasses = None
		self.name = None
		self.nextNode = None
		self.prevNode = None
		self.clearParams()
		
		self.opts = IDAOptions(IDAOptions.defaultOptions)
		self.scope = None
		self.itemTemplates = None
		
		self.inputListDepth = -1
		self.inputListLength = -1
		
		return self
	
	# Copies
	def copy(self, adjacent = False):
		""" Create a new element as a copy of all aspects of this element.
		Any parameters or list content will be cloned recursively.
		
		Arguments:
		adjacents, Whether to include adjacent elements in the cloning process
		Returns:
		The copy element """
		return self.copyTo(IDA(), adjacent)
	def copyTo(self, copy, adjacent = False):
		""" Modify the provided element to become a copy of all aspects of this element.
		Any parameters or list content will be cloned recursively.
		
		Arguments:
		copy, An element to become a copy of this element
		adjacents, Whether to include adjacent elements in the cloning process
		Returns:
		The copy element """
		has_ref = self.txRefLabel(adjacent) > 0
		IDA.copy_impl(self, copy, adjacent, adjacent)
		if has_ref: copy.rxRefResolve(adjacent)
		return copy
	
	def copy_inner(self):
		return IDA.copy_impl(self, IDA(), False, False)
	@staticmethod
	def copy_impl(orig, copy, scanPrev, scanNext):
		copy.copyName(orig)
		copy.copyParams(orig)
		copy.copyContent(orig)
		copy.copyOptions(orig)
		
		a = orig.getPrevOr()
		if scanPrev and (a != None):
			copy.linkPrev(IDA.copy_impl(a, IDA(), True, False))
		a = orig.getNextOr()
		if scanNext and (a != None):
			copy.linkNext(IDA.copy_impl(a, IDA(), False, True))
		
		return copy
	
	# Utility
	def walk(self, func, rev, fwd):
		""" Call the provided function once for each element within this article.
		
		This element is passed first, then its parameters (if any), then its list content items (if any).
		Parameters and list items are treated recursively until the full article has been traversed.
		
		Optionally, the walk may also traverse adjacent elements recursively, requested via the reverse and forward flags.
		
		Arguments:
		func, A callback function which will be supplied with article elements.
		rev, Whether to include elements adjacent to this one in the reverse direction.
		fwd, Whether to include elements adjacent to this one in the forward direction. """
		func(self)
		if self.hasParams():
			for p in self.parameters: p.walk(func, False, False)
		if self.isList():
			for i in self.cList: i.walk(func, False, False)
		if fwd and self.hasNext():
			self.getNextOr().walk(func, False, True)
		if rev and self.hasPrev():
			self.getPrevOr().walk(func, True, False)
	
	def validate(self, adjacents):
		""" Check the integrity of article element.
		If a problem is detected which would lead to errors or corrupted output, an exception is thrown with details in the exception text.
		
		The tests can be performed with adjacent elements enabled or disabled, matching the markup output modes.
		Note that disabling adjacents may avoid some problems (such as with linkage), but may also induce others (such as with references).
		
		The top-level element linkage is tested to ensure that there are no loops, and that adjacent elements are doubly-linked correctly.
		A bad double-link can disrupt the output flow, and a loop can cause infinite iteration.
		
		Any elements with reference content are tested for foreign references, which become lost/null during output.
		
		Arguments:
		adjacents, Whether to extend the test to include elements adjacent to this one.
		Throws:
		IDAOpException, if bad integrity is detected. """
		self.validate_loop(adjacents)
		self.validate_foreign(adjacents)
	def validate_loop(self, adjacents):
		if not adjacents: return
		fwd = lambda x: x.getNextOr()
		rev = lambda x: x.getPrevOr()
		self.validate_loop_f(fwd, rev)
		self.validate_loop_f(rev, fwd)
	def validate_loop_f(self, step, back):
		a = self
		b = self
		
		while True:
			last = b
			b = step(b)
			if b == None: return
			if back(b) != last: raise IDAOpException("Article contains a broken link")
			
			last = b
			b = step(b)
			if b == None: return
			if back(b) != last: raise IDAOpException("Article contains a broken link")
			
			a = step(a)
			if a == b: raise IDAOpException("Article contains an adjacency loop")
	def validate_foreign(self, adjacents):
		foreign = []
		self.walk(lambda x: IDA.validate_foreign_add(foreign, x), adjacents, adjacents)
		self.walk(lambda x: IDA.validate_foreign_rem(foreign, x), adjacents, adjacents)
		if len(foreign) > 0: raise IDAOpException("Article contains a foreign reference")
	@staticmethod
	def validate_foreign_add(foreign, x):
		if x.isRef():
			if not IDA.port.list_contains(foreign, x.asRefOr()):
				foreign.append(x.asRefOr())
	@staticmethod
	def validate_foreign_rem(foreign, x):
		if IDA.port.list_contains(foreign, x):
			IDA.port.list_rem(foreign, x)
	
	
	@staticmethod
	def list_add(l, el, pos, noun):
		size = len(l)
		
		if pos > 0:
			if pos > size:
				raise IDAOpException("Cannot add "+noun+" beyond position "+str(size)+" (request was "+str(pos)+")")
		elif pos < 0:
			if pos < -size:
				raise IDAOpException("Cannot add "+noun+" before position -"+str(size)+" (request was "+str(pos)+")")
			pos += size
		
		if pos == 0:
			if size > 0:
				el.link(None, l[0])
			l.insert(0, el)
		elif pos == size:
			el.link(l[size-1], None)
			l.append(el)
		else: # size must be at least 2, pos cannot be at either end
			el.link(l[pos-1], l[pos])
			l.insert(pos, el)
		
		pass
	
	@staticmethod
	def list_del(l, pos, noun):
		size = len(l)
		
		if pos > 0:
			if pos > size-1:
				raise IDAOpException("Cannot remove item beyond position "+str(size-1)+" (request was "+str(pos)+")")
		elif pos < 0:
			if pos < -size:
				raise IDAOpException("Cannot remove item before position -"+str(size)+" (request was "+str(pos)+")")
			pos += size
		
		return l.pop(pos).link(None, None)
	
	# Syntax
	@staticmethod
	def id0(c):
		return (c>='a' and c<='z') or (c>='A' and c<='Z') or c == '_' or c == '/'
	@staticmethod
	def id1(c):
		return IDA.id0(c) or (c>='0' and c<='9') or c == '-' or c == '.' or c == '!' or c == '@' or c == '^' or c == '&' or c == '?'
	@staticmethod
	def comment1(c):
		return c == '#'
	@staticmethod
	def comment2(c1, c2):
		return (c1 == '/' and (c2 == '/' or c2 == '*')) or (c1 == '<' and (c2 == '?' or c2 == '!'))
	
	# Compatibility/port through
	port = IDA_Port()
	
	# Compatibility
	@staticmethod
	def eq_obj(a, b):
		if isinstance(a, IDA) and isinstance(b, IDA):
			return a.eq(b)
		return (a == b) and (a.__class__ == b.__class__)
	
	@staticmethod
	def list_rem(l, el):
		if IDA.port.list_rem(l, el):
			el.link(None, None)
			return True
		else:
			return False
	
	@staticmethod
	def list_equals(la, lb):
		if (la == None) or (lb == None):
			return la == lb
		ia = IDA.port.list_i(la)
		ib = IDA.port.list_i(lb)
		while ia.hasNext() and ib.hasNext():
			if not ia.getNext().eq(ib.getNext()):
				return False
		return not (ia.hasNext() or ib.hasNext())
	
	pass

IDA.v2 = sys.version_info.major < 3

# Pass-throughs for alternative access
IDA.IDAOpException = IDAOpException
IDA.IDAFormatException = IDAFormatException
IDA.IDAObj = IDAObj

# Unit testing
IDA.IDA_Output = IDA_Output
IDA.IDA_TX = IDA_TX

# CLI
if __name__ == "__main__":
	exit(IDA_CLI.cli(len(sys.argv)-1, sys.argv[1:]))
